/*
 * tiny-login.c - This file is part of the tiny-utils package for Linux & ELKS,
 * Copyright (C) 1995, 1996 Nat Friedman <ndf@linux.mit.edu>.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "config.h"
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <termios.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <paths.h>
#ifdef SUPPORT_REMOTE
#include <netdb.h>
#endif
#include <utmp.h>
#include <pwd.h>
#include <grp.h>
#include "tiny-login.h"

void
report_error(const char *err_str1, const char *err_str2, const int fatal)
{
  int error_fd;

  /*
   * If we can't open the console, fall back to stderr.
   */
  if ((error_fd=open(_PATH_CONSOLE, O_RDWR))<0)
      error_fd=STDERR_FILENO;

  write(error_fd, PROGRAM_NAME, sizeof(PROGRAM_NAME));
  write(error_fd, ": ", 2);
  write(error_fd, err_str1, strlen(err_str1));
  write(error_fd, err_str2, strlen(err_str2));
  write(error_fd, "\n", 1);

  close(error_fd);

  if (fatal)
    exit(1);
}


void
parse_commandline(int argc, char **argv, struct settings *settings)
{
  int curr_arg;

  if (argc==1)
    return;

  if (argc==2)
    {
      strcpy(settings->username,argv[1]);
      settings->have_username=1;
      return;
    }

  if (argc>2)
    {
      curr_arg=1;
      while (curr_arg<argc)
	{
	  if (*argv[curr_arg]=='-')
	    switch(argv[curr_arg][1])
	      {
#ifdef SUPPORT_REMOTE
	      case 'h': /* "-h" == hostname */
		curr_arg++;
		if (curr_arg>=argc)
		  report_error("option -h requires an additional argument:",
			       " -h hostname", 1);
		strcpy(settings->hostname, argv[curr_arg]);
	        settings->remote_login=1;
		break;
#endif
	      case 'f': /* "-f" == pre-autenticated login (force) */
		curr_arg++;
		if (curr_arg>=argc)
		  report_error("option -f requires an additional argument:",
			       " -f username", 1);
		strcpy(settings->username, argv[curr_arg]);
		settings->have_username=1;
		settings->force=1;
		break;
	      }
	  curr_arg++;
	}
    }
}

int
get_username(struct settings *settings)
{
  int username_length;

  write(1, LOGIN_PROMPT, sizeof(LOGIN_PROMPT));

  if ((username_length=read(0, settings->username, MAX_USERNAME_LENGTH))<0)
    exit(1);

  /* If the user presses ^D at the login: prompt, we should
     report no error. */
  if (username_length==0)
    exit(0);

  /* remove the newline character */
  settings->username[username_length-1]='\0';

  return(1);  
}

int
get_password(struct settings *settings)
{
  struct termios old_termios;
  struct termios new_termios;
  int pw_length;
 
  ioctl(0, TCGETS, &old_termios);
  memcpy((void *) &new_termios, (void *) &old_termios, sizeof(struct termios));

  /* turn off echo */
  new_termios.c_lflag&=~(ECHO|ISIG);

  ioctl(0, TCSETS, &new_termios);

  write(1, PW_PROMPT, sizeof(PW_PROMPT));

  pw_length=read(0, settings->password, MAX_PW_LENGTH);
  if (pw_length<0)
      report_error("read returned an error", "", 1);

  /* remove the newline character */
  settings->password[pw_length-1]='\0';
  
  write(1, "\n", 1);

  ioctl(0, TCSETS, &old_termios);

  return(1);
}

int
verify_password(struct settings settings, char *password)
{
  char salt[3];

  /* First deal with things like locked passwords ... */
  if (strlen(password)<10)
    return 0;

  /* The salt is the first two charcters of the password entry
     in the passwd file. */
  *salt=*password;
  *(salt+1)=*(password+1);
  *(salt+2)='\0';

  return(!strcmp(password, crypt(settings.password, salt)));
}

#ifdef SUPPORT_UTMP
int
update_utmp(struct settings settings)
{
  struct utmp utmp_entry;
  int my_pid;
  int utmp_fd;
  int wtmp_fd;
  int new_entry=0;

  my_pid=getpid();
  
  utmp_fd=open(_PATH_UTMP, O_RDWR);

  /* step through the utmp file until we find out utmp entry. */
  while(read(utmp_fd, &utmp_entry, sizeof(struct utmp))==sizeof(struct utmp) &&
	utmp_entry.ut_pid!=my_pid);

  if (utmp_entry.ut_pid!=my_pid)
    {
      /* build a new utmp entry, since
	 we can't find ours in the file */
      utmp_entry.ut_pid=my_pid;
      new_entry=1;
    }
      

  /* modify the utmp entry */
  time(&utmp_entry.ut_time);
  memcpy(utmp_entry.ut_user, settings.username, sizeof(utmp_entry.ut_user));
  memcpy(utmp_entry.ut_line, ttyname(0)+5, sizeof(utmp_entry.ut_line));
  memcpy(utmp_entry.ut_id, ttyname(0)+strlen(ttyname(0))-2, sizeof(utmp_entry.ut_id));

  /* We were previously a LOGIN process.. but we're about to become a
     USER process (once the shell is executed). */
  utmp_entry.ut_type=USER_PROCESS;

#ifdef SUPPORT_REMOTE
  if (settings.remote_login)
    {
      memcpy(utmp_entry.ut_host, settings.hostname,
	      sizeof(utmp_entry.ut_host));
      memcpy(&utmp_entry.ut_addr, *settings.hostaddr.h_addr_list,
	      sizeof(utmp_entry.ut_addr));
    }
#endif

  if (!new_entry)
    lseek(utmp_fd, -sizeof(struct utmp), SEEK_CUR);
  write(utmp_fd, (char *) &utmp_entry, sizeof(struct utmp));
  close(utmp_fd);

  if ((wtmp_fd=open(_PATH_WTMP, O_APPEND|O_WRONLY))<0)
    report_error("cannot open wtmp file: ", _PATH_WTMP, 1);
  flock(wtmp_fd, LOCK_EX);
  write(wtmp_fd, (char *) &utmp_entry, sizeof(struct utmp));
  flock(wtmp_fd, LOCK_UN);
  close(wtmp_fd);

  return(1);
}
#endif

#ifdef SUPPORT_REMOTE
int
get_hostent(struct settings *settings)
{
  memcpy(&settings->hostaddr, (char *)gethostbyname(settings->hostname),
	 sizeof(struct hostent));
  return(1);
}
#endif

void
display_motd(void)
{
  char buff[BUFFER_SIZE];
  int bytes_read;
  int motd_fd;

  if ((motd_fd=open(MOTD_FILE, O_RDONLY))<0)
    report_error("error opening motd file: ", strerror(errno), 0);

  while ((bytes_read=read(motd_fd, &buff, BUFFER_SIZE))>0)
    write(1, buff, bytes_read);

  close(motd_fd);
}

void
login_failure(struct settings *settings, int *num_failures)
{

  (*num_failures)++;

  if (*num_failures==MAX_FAILURES)
#ifdef SUPPORT_REMOTE
    if (settings->remote_login)
	report_error("REPEATED LOGIN FAILURES FROM ", settings->hostname, 1);
    else
#endif
      report_error("REPEATED LOGIN FAILURES ON ", ttyname(0), 1);

  write(1, FAIL_MESSAGE, sizeof(FAIL_MESSAGE));
    
  settings->have_username=0;

}

void
check_nologin(void)
/* Check to see if there is a nologin file and if so, display
   it and exit. */
{
  int nologin_fd;
  int bytes_read;
  char buff[BUFFER_SIZE];

  if ((nologin_fd=open(_PATH_NOLOGIN, O_RDONLY))<0)
    return;

  while ((bytes_read=read(nologin_fd, &buff, BUFFER_SIZE))>0)
    write(1, buff, bytes_read);

  close(nologin_fd);

  exit(1);
}

void
log_in(struct settings settings, struct passwd *passwd_entry)
{
  char tmp_buff[64];
  char * tmp_ptr;

  /* set the tty permissions */
  fchown(STDIN_FILENO, passwd_entry->pw_uid, passwd_entry->pw_gid);
  fchmod(STDIN_FILENO, TTY_MODE);

  /* Set the user's group ID */
  setgid(passwd_entry->pw_gid);

  /* Initialize the user's secondary groups.  This basically
     just grabs the list of secondary groups from the group
     file and calls setgroups() on them. */
  initgroups(passwd_entry->pw_name, passwd_entry->pw_gid);

  /* Change to the user's UID. */
  setuid(passwd_entry->pw_uid);

  /* change to the home directory; prepare to exec the shell */
  if (chdir(passwd_entry->pw_dir)<0)
      report_error(passwd_entry->pw_dir, " does not exist!", 1);

  /* set the appropriate environment variables */
  strcpy(tmp_buff, "HOME=");
  strcpy(tmp_buff + 5, passwd_entry->pw_dir);
  putenv(tmp_buff);
  strcpy(tmp_buff, "SHELL=");
  strcpy(tmp_buff + 6, passwd_entry->pw_shell);
  putenv(tmp_buff);
  strcpy(tmp_buff, "LOGNAME=");
  strcpy(tmp_buff + 8, passwd_entry->pw_name);
  putenv(tmp_buff);
  
  if (passwd_entry->pw_uid)
    {
      strcpy(tmp_buff, "PATH=");
      putenv(strcat(tmp_buff, _PATH_DEFPATH));
    }  
  else
    {
      strcpy(tmp_buff, "PATH=");
      putenv(strcat(tmp_buff, _PATH_DEFPATH_ROOT));
    }

  display_motd();

  /* We must set the first character of argv[0] to - so the shell
     knows that it is a login shell */
  *tmp_buff='-';
  tmp_buff[1]='\0';
  tmp_ptr=strrchr(passwd_entry->pw_shell, '/')+1;
  if (tmp_ptr!=NULL)
    strcat(tmp_buff, tmp_ptr);
  else
    strcat(tmp_buff, passwd_entry->pw_shell);
  execl(passwd_entry->pw_shell, tmp_ptr, NULL);

  report_error("no shell: ", passwd_entry->pw_shell, 1);

  exit(1);
}

int
main(int argc, char **argv)
{
  struct settings settings;
  struct passwd *passwd_entry=NULL;
  int verified;
  int num_failures;

  /* Clear the settings structure */
  settings.force=settings.have_username=0;
#ifdef SUPPORT_REMOTE
  settings.remote_login=0;
#endif

  parse_commandline(argc, argv, &settings);

  check_nologin();

#ifdef SUPPORT_REMOTE
  if (settings.remote_login)
    get_hostent(&settings);
#endif

 /* Now begin the 'read login'->'read password'->'verify' loop */
  verified=settings.force;
  num_failures=0;
  while (!verified)
    {

      if (!settings.have_username)
	get_username(&settings);

      /* Snag the entry from the password file */
      passwd_entry=getpwnam(settings.username);

      /* Now get the password */
      get_password(&settings);

      if (passwd_entry!=NULL)
	  verified=verify_password(settings, passwd_entry->pw_passwd);

      /* clear the password entry - paranoia */
      memset(settings.password, 0, MAX_PW_LENGTH);

      if (!verified)
	  login_failure(&settings, &num_failures);
    }

#ifdef SUPPORT_UTMP
  /* Update the utmp and the wtmp entries for this login */
  update_utmp(settings);
#endif

  log_in(settings, passwd_entry);
  
  return(1);
}

