/* tiny-login.c <ndf@linux.mit.edu> */
/* Usage: login [ name ]
                -h name
		-f name
		*/

#include "config.h"
#include <unistd.h>
#include <sys/types.h>
#ifdef SUPPORT_REMOTE
#include <netdb.h>
#endif
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <termios.h>
#include <stdlib.h>
#include <pwd.h>
#include <utmp.h>
#include <string.h>
#include <paths.h>
#include <grp.h>
#include "tiny-login.h"

void
report_error(char *err_str1, char *err_str2, int fatal)
{
  int console_fd;

  if ((console_fd=open(_PATH_CONSOLE, O_RDWR))<0)
    {
      /* fall back to stderr */
      console_fd=2;
    }

  write(console_fd, PROGRAM_NAME, sizeof(PROGRAM_NAME));
  write(console_fd, ": ", 2);
  write(console_fd, err_str1, strlen(err_str1));
  write(console_fd, err_str2, strlen(err_str2));
  write(console_fd, "\n", 1);

  close(console_fd);

  if (fatal)
    exit(1);
}

void
parse_commandline(int argc, char **argv, struct settings *settings)
{
  int curr_arg;

  if (argc==1)
    return;

  if (argc==2)
    {
      strcpy(settings->username,argv[1]);
      settings->have_username=1;
      return;
    }

  if (argc>2)
    {
      curr_arg=1;
      while (curr_arg<argc)
	{
	  if (*argv[curr_arg]=='-')
	    switch(argv[curr_arg][1])
	      {
#ifdef REMOTE
	      case 'h': /* "-h" == hostname */
		curr_arg++;
		if (curr_arg>=argc)
		  report_error("option -h requires an additional argument:",
			       " -h hostname", 1);
		strcpy(settings->hostname, argv[curr_arg]);
	        settings->remote_login=1;
		break;
#endif
	      case 'f': /* "-f" == pre-autenticated login (force) */
		curr_arg++;
		if (curr_arg>=argc)
		  report_error("option -f requires an additional argument:",
			       " -f username", 1);
		strcpy(settings->username, argv[curr_arg]);
		settings->have_username=1;
		settings->force=1;
		break;
	      }
	  curr_arg++;
	}
    }
}

int
get_username(struct settings *settings)
{
  int username_length;

  write(1, LOGIN_PROMPT, sizeof(LOGIN_PROMPT));

  if ((username_length=read(0, settings->username, MAX_USERNAME_LENGTH))<0)
    exit(1);

  /* If the user presses ^D at the login: prompt, we should
     report no error. */
  if (username_length==0)
    exit(0);

  /* remove the newline character */
  settings->username[username_length-1]='\0';

  return(1);  
}

int
get_password(struct settings *settings)
{
  struct termios old_termios;
  struct termios new_termios;
  int pw_length;
 
  ioctl(0, TCGETS, &old_termios);
  memcpy((void *) &new_termios, (void *) &old_termios, sizeof(struct termios));

  /* turn off echo */
  new_termios.c_lflag&=~(ECHO|ISIG);

  ioctl(0, TCSETS, &new_termios);

  write(1, PW_PROMPT, sizeof(PW_PROMPT));

  pw_length=read(0, settings->password, MAX_PW_LENGTH);
  if (pw_length<0)
      report_error("read returned an error", "", 1);

  /* remove the newline character */
  settings->password[pw_length-1]='\0';
  
  write(1, "\n", 1);

  ioctl(0, TCSETS, &old_termios);

  return(1);
}

int
verify_password(struct settings settings, char *password)
{
  char salt[3];

  /* The salt is the first two charcters of the password entry
     in the passwd file. */

  *salt=*password;
  *(salt+1)=*(password+1);
  *(salt+2)='\0';

  return(!strcmp(password, crypt(settings.password, salt)));
}

#ifndef NO_UTMP
int
update_utmp(struct settings settings)
{
  struct utmp utmp_entry;
  int my_pid;
  int utmp_fd;
  int wtmp_fd;
  int ut_size;
  int new_entry=0;

  my_pid=getpid();
  ut_size=sizeof(struct utmp);
  
  utmp_fd=open(_PATH_UTMP, O_RDWR);

  /* step through the utmp file until we find out utmp entry. */
  while(read(utmp_fd, &utmp_entry, ut_size)==ut_size &&
	utmp_entry.ut_pid!=my_pid);

  if (utmp_entry.ut_pid!=my_pid)
    {
      /* build a new utmp entry, since
	 we can't find ours in the file */
      utmp_entry.ut_pid=my_pid;
      new_entry=1;
    }
      

  /* modify the utmp entry */
  time(&utmp_entry.ut_time);
  memcpy(utmp_entry.ut_user, settings.username, sizeof(utmp_entry.ut_user));
  memcpy(utmp_entry.ut_line, ttyname(0)+5, sizeof(utmp_entry.ut_line));
  memcpy(utmp_entry.ut_id, ttyname(0)+strlen(ttyname(0))-2, sizeof(utmp_entry.ut_id));

  /* We were previously a LOGIN process.. but we're about to become a
     USER process (once the shell is executed). */
  utmp_entry.ut_type=USER_PROCESS;

#ifdef REMOTE
  if (settings.remote_login)
    {
      memcpy(utmp_entry.ut_host, settings.hostname,
	      sizeof(utmp_entry.ut_host));
      memcpy(&utmp_entry.ut_addr, *settings.hostaddr.h_addr_list,
	      sizeof(utmp_entry.ut_addr));
    }
#endif

  if (!new_entry)
    lseek(utmp_fd, -ut_size, SEEK_CUR);
  write(utmp_fd, (char *) &utmp_entry, ut_size);
  close(utmp_fd);

  if ((wtmp_fd=open(_PATH_WTMP, O_APPEND|O_WRONLY))<0)
    report_error("cannot open wtmp file: ", _PATH_WTMP, 1);
  flock(wtmp_fd, LOCK_EX);
  write(wtmp_fd, (char *) &utmp_entry, ut_size);
  flock(wtmp_fd, LOCK_UN);
  close(wtmp_fd);

  return(1);
}
#endif

#ifdef REMOTE
int
get_hostent(struct settings *settings)
{
  memcpy(&settings->hostaddr, (char *)gethostbyname(settings->hostname),
	 sizeof(struct hostent));
  return(1);
}
#endif

void
display_motd(void)
{
  char buff[BUFFER_SIZE];
  int bytes_read;
  int motd_fd;

  if ((motd_fd=open(MOTD_FILE, O_RDONLY))<0)
    report_error("error opening motd file: ", strerror(errno), 0);

  while ((bytes_read=read(motd_fd, &buff, BUFFER_SIZE))>0)
    write(1, buff, bytes_read);

  close(motd_fd);
}

void
login_failure(struct settings *settings, int *num_failures)
{

  (*num_failures)++;

  if (*num_failures==MAX_FAILURES)
#ifdef REMOTE
    if (settings->remote_login)
	report_error("REPEATED LOGIN FAILURES FROM ", settings->hostname, 1);
    else
#endif
      report_error("REPEATED LOGIN FAILURES ON ", ttyname(0), 1);

  write(1, FAIL_MESSAGE, sizeof(FAIL_MESSAGE));
    
  settings->have_username=0;

}

void
check_nologin(void)
/* Check to see if there is a nologin file and if so, display
   it and exit. */
{
  int nologin_fd;
  int bytes_read;
  char buff[BUFFER_SIZE];

  if ((nologin_fd=open(_PATH_NOLOGIN, O_RDONLY))<0)
    return;

  while ((bytes_read=read(nologin_fd, &buff, BUFFER_SIZE))>0)
    write(1, buff, bytes_read);

  close(nologin_fd);

  exit(1);
}

void
log_in(struct settings settings, struct passwd *passwd_entry)
{
  char shell_name[512];

  /* set the tty permissions */
  fchown(0, passwd_entry->pw_uid, passwd_entry->pw_gid);
  fchmod(0, TTY_MODE);

  /* Set the user's group ID */
  setgid(passwd_entry->pw_gid);

  /* Initialize the user's secondary groups.  initgroups() needs
     to be run as root. */
  initgroups(passwd_entry->pw_name, passwd_entry->pw_gid);

  /* Change to the user's UID. */
  setuid(passwd_entry->pw_uid);

  /* change to the home directory, prepare to exec the shell */
  if (chdir(passwd_entry->pw_dir)<0)
      report_error(passwd_entry->pw_dir, " does not exist!", 1);

  /* set the appropriate environment variables */
  setenv("HOME", passwd_entry->pw_dir, 1);
  setenv("SHELL", passwd_entry->pw_shell, 1);
  setenv("PATH", _PATH_DEFPATH, 0);

  display_motd();

  /* We must set the first character of argv[0] to - so the shell
     knows that it is a login shell */
  *shell_name='-';
  shell_name[1]='\0';
  execl(passwd_entry->pw_shell, strcat(shell_name, passwd_entry->pw_shell),
	NULL);

  report_error("no shell: ", passwd_entry->pw_shell, 1);

  exit(1);
}

int
main(int argc, char **argv)
{
  struct settings settings;
  struct passwd *passwd_entry=NULL;
  int verified;
  int num_failures;

  /* Clear the settings structure */
  settings.force=DEFAULT_FORCE;
  settings.have_username=DEFAULT_HAVE_USERNAME;
#ifdef SUPPORT_REMOTE
  settings.remote_login=DEFAULT_REMOTE_LOGIN;
#endif

  parse_commandline(argc, argv, &settings);

  check_nologin();

#ifdef REMOTE
  if (settings.remote_login)
    get_hostent(&settings);
#endif

 /* Now begin the read login->read password->verify loop */
  verified=settings.force;
  num_failures=0;
  while (!verified)
    {

      if (!settings.have_username)
	get_username(&settings);

      /* Snag the entry from the password file */
      passwd_entry=getpwnam(settings.username);

      /* Now get the password */
      get_password(&settings);

      if (passwd_entry!=NULL)
	  verified=verify_password(settings, passwd_entry->pw_passwd);

      /* clear the password entry - paranoia */
      memset(settings.password, 0, MAX_PW_LENGTH);

      if (!verified)
	  login_failure(&settings, &num_failures);
    }

#ifndef NO_UTMP
  /* Write the utmp and the wtmp */
  update_utmp(settings);
#endif

  log_in(settings, passwd_entry);
  
  return(1);
}

