/* tiniest-login.c <ndf@linux.mit.edu> */
#include <unistd.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <termios.h>
#include <stdlib.h>
#include <pwd.h>
#include <utmp.h>
#include <string.h>
#include <paths.h>
#include <grp.h>
#include "tiniest-login.h"

void
init_settings(int argc, char **argv, struct settings *settings)
{

  settings->have_username=0;
  settings->verified=0;
  settings->num_failures=0;

  if (argc==1)
    return;

  if ((argc==2) && (*argv[1]!='-'))
    {
      strcpy(settings->username,argv[1]);
      settings->have_username=1;
    }
  else
    exit(1);
}

void
get_password(struct settings *settings)
{
  struct termios old_termios;
  struct termios new_termios;
  int bytes_read;

  ioctl (0, TCGETS, &old_termios);
  memcpy((void *)&new_termios, (void *) &old_termios, sizeof(struct termios));

  new_termios.c_lflag&=~(ECHO|ISIG);

  ioctl(0, TCSETS, &new_termios);

  write(1, PW_PROMPT, sizeof(PW_PROMPT));

  if ((bytes_read=read(0, settings->password, MAX_PW_LENGTH))<0)
    exit(1);

  /* remove the newline character */
  settings->password[bytes_read-1]='\0';
  
  write(1, "\n", 1);

  ioctl(0, TCSETS, &old_termios);
}

void
get_username(struct settings *settings)
{
  int bytes_read;

  write(1, LOGIN_PROMPT, sizeof(LOGIN_PROMPT));
  
  if ((bytes_read=read(0, settings->username, MAX_USERNAME_LENGTH))<0)
    exit(1);

  if (bytes_read==0) /* they hit ^D */
    exit(0);

  settings->username[bytes_read-1]='\0';
}

int
verify_password(struct settings settings, char *password)
{
  char salt[3];

  /* The salt is the first two charcters of the password entry
     in the passwd file. */

  *salt=*password;
  *(salt+1)=*(password+1);
  *(salt+2)='\0';

  return(!strcmp(password, crypt(settings.password, salt)));
}

void
log_in(struct settings *settings, struct passwd *passwd_entry)
{
  char shell_name[512];

  /* set the tty permissions */
  fchown(0, passwd_entry->pw_uid, passwd_entry->pw_gid);
  fchmod(0, TTY_MODE);

  /* Set the user's group ID */
  setgid(passwd_entry->pw_gid);

  /* Initialize the user's secondary groups.  initgroups() needs
     to be run as root. */
  initgroups(passwd_entry->pw_name, passwd_entry->pw_gid);

  /* Change to the user's UID. */
  setuid(passwd_entry->pw_uid);

  /* change to the home directory, prepare to exec the shell */
  if (chdir(passwd_entry->pw_dir)<0)
    exit(1);

  /* set the appropriate environment variables */
  setenv("HOME", passwd_entry->pw_dir, 1);
  setenv("SHELL", passwd_entry->pw_shell, 1);
  setenv("PATH", _PATH_DEFPATH, 0);

/*  display_motd(); */

  /* We must set the first character of argv[0] to - so the shell
     knows that it is a login shell */
  *shell_name='-';
  *(shell_name+sizeof(char))='\0';
  execl(passwd_entry->pw_shell, strcat(shell_name, passwd_entry->pw_shell),
	NULL);

  exit(1);
}

int
main(int argc, char **argv)
{
  struct settings settings;
  struct passwd *passwd_entry;

  init_settings(argc, argv, &settings);

  while (!settings.verified && (settings.num_failures<3))
    {
      if (!settings.have_username)
	get_username(&settings);

      get_password(&settings);
      passwd_entry=getpwnam(settings.username);
      if (passwd_entry!=NULL)
	settings.verified=verify_password(settings, passwd_entry->pw_passwd);

      if (!settings.verified)
	{
	  write(1, FAIL_MESSAGE, sizeof(FAIL_MESSAGE));
	  settings.num_failures++;
	  settings.have_username=0;
	}
      
    }

  if (settings.verified)
    log_in(&settings, passwd_entry);
}

