/* tiny-getty.c <ndf@linux.mit.edu> */
/* Usage: getty <tty> <baud rate> [-t terminal type] [-l login program] */
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <stdlib.h>
#include "tiny-getty.h"

void
report_error(char *err_str1, char *err_str2, int fatal)
{
  int console_fd;

  if ((console_fd=open("/dev/console", O_RDWR))<0)
    {
      /* fallback to stderr */
      console_fd=2;
    }

  write(console_fd, "getty: ", 7);
  write(console_fd, err_str1, strlen(err_str1));
  write(console_fd, err_str2, strlen(err_str2));
  write(console_fd, "\n", 1);

  close(console_fd);

  if (fatal)
    exit(1);
}

void 
parse_commandline(int argc, char **argv, struct settings *settings)
{
  int curr_arg;

#ifdef SUPPORT_SERIAL
  if (argc<3)
#else
  if (argc<2)
#endif
    report_error("Not enough arguments", "", 1);

  /* The first argument is the tty */
  settings->tty=argv[1];

#ifdef SUPPORT_SERIAL
  /* The second argument is the baud rate */
  if (!(settings->baud_rate=atoi(argv[2])))
    report_error("Invalid baud rate: ", argv[2], 1);

  curr_arg=3;
#else
  curr_arg=2;
#endif

  while (curr_arg<argc)
    {
      if (*argv[curr_arg]=='-')
	  switch(*(argv[curr_arg]+1))
	  {
	  case 'l': /* login program */
	    curr_arg++;
	    settings->login=argv[curr_arg];
	    break;
	  case 't': /* terminal type */
	    curr_arg++;
	    settings->set_term=1;
	    settings->term=argv[curr_arg];
	    break;
	  default:
	    report_error("unrecognized arg: ", argv[curr_arg], 1);
	    break;
	  }
      else
	report_error("unrecognized arg: ", argv[curr_arg], 1);
      curr_arg++;
    }
}

void
open_tty(char *filename)
{
  struct stat st;

  /* make sure it's a character device so "getty /dev/hda1" doesn't
     do to anyone else what it almost did to me */
  stat(filename, &st);
  if (!S_ISCHR(st))
    report_error("not a character device: ", filename, 1);

    
  close(0); /* stdin */
  close(1); /* stdout */
  close(2); /* stderr */

  if (open(filename, O_RDWR)) /* if it's not 0, it's not stdin */
    report_error("open fd not stdin", "", 1);

  /* set stdout and stderr equal to stdin */
  dup(0); /* stdout */
  dup(0); /* stderr */

  fchown(0, TTY_OWNER, TTY_GROUP);
  fchmod(0, TTY_MODE);
}

#ifdef SUPPORT_SERIAL
int
baud_constant(int baud_rate)
{
  int speed_num=0;

  while (speed_num<NUM_SPEEDS)
    {
      if (speed[speed_num].baud_rate==baud_rate)
	return(speed[speed_num].baud_const);
      speed_num++;
    }

  report_error("cannot locate baud rate", "", 1);

  exit(1);
}
#endif

void
display_issue(char *filename)
{
  int issue_fd;
  char buff[PAGE_SIZE];
  int bytes_read;

  if ((issue_fd=open(filename, O_RDONLY))<0)
    {
      report_error("error opening issue file", "", 0);
      return;
    }

  while ((bytes_read=read(issue_fd, &buff, PAGE_SIZE))>0)
    write(1, buff, bytes_read);
      
  close(issue_fd);
}

void
setup_termio(struct settings *settings)
{
  struct termio termio;

  /* flush the tty */
  ioctl(0, TCFLSH, 2);
  
  /* Get the current termio settings */
  if (ioctl(0, TCGETA, &termio)<0)
    report_error("ioctl: error", "", 1);

  termio.c_iflag=ICRNL|IXON;
  termio.c_lflag=ISIG|ICANON|IEXTEN|ECHO|ECHOE|ECHOK;
  termio.c_oflag=OPOST|ONLCR;
  termio.c_cflag=CS8|CREAD;
#ifdef SUPPORT_SERIAL
  termio.c_cflag|=baud_constant(settings->baud_rate);
#endif

  if (ioctl(0, TCSETA, &termio)<0)
    report_error("ioctl: error", "", 1);
}

void
get_login(char *login)
{
  int bytes_read;

  if (write(1, DEFAULT_PROMPT, sizeof(DEFAULT_PROMPT))<0)
    report_error("write error", "", 1);

  if ((bytes_read=read(0, login, MAX_LINE_LENGTH))<0)
    report_error("error on reading stdin", "", 1);

  /* remove the newline character */
  login[bytes_read-1]='\0';
}

int
main(int argc, char **argv)
{
  struct settings settings={ "" /* tty */,
			     DEFAULT_LOGIN,
			     DEFAULT_ISSUE,
			     DEFAULT_SET_TERM,
			     DEFAULT_TERM,
#ifdef SUPPORT_SERIAL
                             DEFAULT_BAUD
#endif
  };

  char login[MAX_LINE_LENGTH];

  /* Detach from the current tty (same as BSD demon()) */
  setsid();

  parse_commandline(argc, argv, &settings);

  if (settings.set_term)
    setenv("TERM", settings.term, 1);

  open_tty(settings.tty);

  /* Now that the tty has been opened and stdin/stdout/stderr
     point to the right places, we should vhangup() to clear the tty */
  vhangup();

  setup_termio(&settings);

  display_issue(settings.issue);

  get_login(login);

  execl(settings.login, settings.login, login, NULL);

  report_error("cannot exec ", settings.login, 1);

  return(1);
}








