/* tiny-getty.c <ndf@linux.mit.edu> */
/* Usage: getty tty [-l login program] */
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include "tiny-getty.h"

void
report_error(char *err_str1, char *err_str2)
{
  int console_fd;

  if ((console_fd=open("/dev/console", O_RDWR))<0)
    {
      /* fallback to stderr */
      console_fd=2;
    }

  write(console_fd, "getty: ", 7);
  write(console_fd, err_str1, strlen(err_str1));
  write(console_fd, err_str2, strlen(err_str2));
  write(console_fd, "\n", 1);

  close(console_fd);

  exit(1);
}

void 
parse_commandline(int argc, char **argv, struct settings *settings)
{
  int curr_arg=2;

  /* The first argument is the tty */
  settings->tty=argv[1];

  while (curr_arg<argc)
    {
      if (*argv[curr_arg]=='-')
	  switch(*(argv[curr_arg]+1))
	  {
	  case 'l': /* login program */
	    curr_arg++;
	    settings->login=argv[curr_arg];
	    break;
	  default:
	    report_error("unrecognized arg: ", argv[curr_arg]);
	    break;
	  }
      else
	report_error("unrecognized arg: ", argv[curr_arg]);
      curr_arg++;
    }
}

void
open_tty(char *filename)
{
  struct stat st;

  /* make sure it's a character device so "getty /dev/hda1" doesn't
     do to anyone else what it almost did to me */
  stat(filename, &st);
  if ((st.st_mode&S_IFMT)!=S_IFCHR)
    report_error("not a character device: ", filename);
    
  close(0); /* stdin */
  close(1); /* stdout */
  close(2); /* stderr */

  if (open(filename, O_RDWR)) /* if it's not 0, it's not stdin */
    report_error("open fd not stdin", "");

  /* set stdout and stderr equal to stdin */
  dup(0); /* stdout */
  dup(0); /* stderr */

  fchown(0, 0, 5);
  fchmod(0, 0622);
}

void
display_issue(char *filename)
{
  int issue_fd;
  char buff[PAGE_SIZE];
  int bytes_read;

  if ((issue_fd=open(filename, O_RDONLY))<0)
    report_error("error opening issue file", "");

  while ((bytes_read=read(issue_fd, &buff, PAGE_SIZE))>0)
      write(1, buff, bytes_read);

  close(issue_fd);
}

void
setup_termio(struct settings *settings)
{
  struct termio termio;

  /* flush the tty */
  ioctl(0, TCFLSH, 2);
  
  /* Get the current termio settings */
  if (ioctl(0, TCGETA, &termio)<0)
    report_error("ioctl: error", "");

  termio.c_iflag=ICRNL|IXON;
  termio.c_lflag=ISIG|ICANON|IEXTEN|ECHO|ECHOE|ECHOK;
  termio.c_oflag=OPOST|ONLCR;
  termio.c_cflag=CS8|CREAD;

  if (ioctl(0, TCSETA, &termio)<0)
    report_error("ioctl: error", "");
}

void
get_login(char *login)
{
  int bytes_read;

  if (write(1, "login: ", 7)<0)
    report_error("write error", "");

  if ((bytes_read=read(0, login, MAX_LINE_LENGTH))<0)
    report_error("error on reading stdin", "");

  /* remove the newline character */
  login[bytes_read-1]='\0';
}

int
main(int argc, char **argv)
{
  struct settings settings={ "" /* tty */,
			       DEFAULT_LOGIN,
			       DEFAULT_ISSUE};

  char login[MAX_LINE_LENGTH];


  /* Detach from the current tty (same as BSD demon()) */
  setsid();

  parse_commandline(argc, argv, &settings);

  open_tty(settings.tty);

  setup_termio(&settings);
   
  display_issue(settings.issue);

  get_login(login);

  execl(settings.login, settings.login, login, NULL);

  report_error(settings.login, ": cannot exec!");

  return(1);
}
