/*
 *	Memory management support for a swapping rather than paging kernel.
 *
 *	Memory allocator for ELKS. We keep a hole list so we can keep the
 *	malloc arena data in the kernel npot scattered into hard to read 
 *	user memory.
 */
 
#include <linuxmt/types.h>
#include <linuxmt/sched.h>
#include <linuxmt/mm.h>

/*
 *	Worst case is code+data segments for max tasks unique processes
 *	and one for the free space left.
 */
 
#define MAX_SEGMENTS	8+(MAX_TASKS*2)

struct malloc_hole
{
	unsigned short page_base;	/* Pages */
	unsigned short extent;		/* Pages */
	struct malloc_hole *next;	/* Next in list memory order */
	unsigned short refcount;
	unsigned short flags;		/* So we know if it is free */
#define HOLE_USED		1	
#define HOLE_FREE		2
#define HOLE_SPARE		3
};

static struct malloc_hole holes[MAX_SEGMENTS];

/*
 *	Find a spare hole.
 */
 
struct malloc_hole *alloc_hole()
{
	int ct=0;
	struct malloc_hole *m=&holes[0];
	while(ct<MAX_SEGMENTS)
	{
		if(m->flags==HOLE_SPARE)
			return m;
		m++;
		ct++;
	}
	panic("mm: too many holes");
}

/*
 *	Split a hole into two
 */
 
void split_hole(m, len)
struct malloc_hole *m;
int len;
{
	struct malloc_hole *n;
	int spare=m->extent-len;

	/*
	 *	Split into one allocated one free
	 */
	 
	m->extent=len;
	n=alloc_hole();
	n->page_base=m->page_base+len;
	n->extent=spare;
	n->next=m->next;
	m->next=n;
	n->flags=HOLE_FREE;
}

/*
 *	Merge adjacent free holes
 */
 
void sweep_holes()
{
	struct malloc_hole *m=&holes[0];
	while(m!=NULL && m->next!=NULL)
	{
		if(m->flags==HOLE_FREE && m->next->flags==HOLE_FREE && 
			m->page_base+m->extent==m->next->page_base)
		{
			m->extent+=m->next->extent;
			m->next->flags=HOLE_SPARE;
			m->next=m->next->next;
		}
		else
			m=m->next;
	}
}

/*
 *	Find the nearest fitting hole
 */
 
struct malloc_hole *best_fit_hole(size)
int size;
{
	struct malloc_hole *m=&holes[0];
	struct malloc_hole *best=NULL;
	while(m!=NULL)
	{
		if(m->extent>=size && m->flags==HOLE_FREE) 
			if(!best || best->extent > m->extent)
				best=m;
		m=m->next;
	}
	return best;
}

/*
 *	Find the hole starting at a given location
 */
 
struct malloc_hole *find_hole(base)
int base;
{
	struct malloc_hole *m=&holes[0];
	while(m!=NULL)
	{
		if(m->page_base==base)
			return m;
		m=m->next;
	}
	return NULL;
}		

/*
 *	Allocate a segment
 */
			
int mm_alloc(pages, priority)
int pages;
int priority;
{
	/*
	 *	Which hole fits best ?
	 */
	struct malloc_hole *m=best_fit_hole(pages);
	/*
	 *	No room , later check priority and swap
	 */
	if(m==NULL) {
		return -1;
	}
	/*
	 *	The hole is (probably) too big
	 */
	 
	if(m->extent > pages)
		split_hole(m, pages);
	m->flags=HOLE_USED;
	m->refcount = 1;
	return m->page_base;
}

/* 	Increase refcount */

int mm_realloc(base)
int base;
{
#if 1
	struct malloc_hole *m=find_hole(base);
	if (m==NULL)
		panic("arena corrupt\n");
	m->refcount++;
	return m->page_base;
#else
	return mm_dup(base);
#endif
}

/*
 *	Free a segment.
 */
 
int mm_free(base)
int base;
{
	struct malloc_hole *m=find_hole(base);
	if(m==NULL)
		panic("arena corrupt");
	if(m->flags!=HOLE_USED)
		panic("double free");
	m->refcount--;
	if (!m->refcount) {
		m->flags=HOLE_FREE;
		sweep_holes();
	}
	return 1;
}

/*
 *	Allocate a segment and copy it from another
 */
 
int mm_dup(base)
int base;
{
	struct malloc_hole *o, *m;
	int i;
	
	o=find_hole(base);
	if(o==NULL)
		panic("mm_dup given bad page base");
	m=best_fit_hole(o->extent);
	if(m==NULL)
		return -1;
	if(m->extent > o->extent)
		split_hole(m, o->extent);
	m->flags=HOLE_USED;
	m->refcount = 1;
	i = (o->extent << 4) - 2;
	fmemcpy(m->page_base, 0, o->page_base, 0, i);
	return m->page_base;
}

/*	This is just to keep malloc et. al happy - it dosen't really do anyting
 * 	- any memory is preallocated via chmem */

int sys_brk(len)
int len;
{
	if ((len < (current->t_endtext)) || 
	    (len > (current->t_endstack - USTACK_BYTES))) { 
		return -ENOMEM; 
	}

	current->t_endbrk = len;
	return 0;
}

/*
 *	Initialise the memory manager.
 */

void mm_init(start,end)
int start;
int end;
{
	int ct;
	/*
	 *	Mark pages free.
	 */
	for(ct=1;ct<MAX_SEGMENTS;ct++)
		holes[ct].flags=HOLE_SPARE;
	/*
	 *	Single hole containing all user memory.
	 */
	holes[0].flags=HOLE_FREE;
	holes[0].page_base = start;
	holes[0].extent = end-start;
	holes[0].refcount = 0;
	holes[1].flags=HOLE_FREE;
	holes[1].page_base = 0x900;
	holes[1].extent = (0xfff - 0x900);
	holes[1].refcount = 0;
	holes[0].next = &holes[1];
}

