/* doshd.c */
/* Copyright (C) 1994 Yggdrasil Computing, Incorporated
   4880 Stevens Creek Blvd. Suite 205
   San Jose, CA 95129-1034
   USA
   Tel (408) 261-6630
   Fax (408) 261-6631

   This file is part of the Linux Kernel

   Linux is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   Linux is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Linux; see the file COPYING.  If not, write to
   the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */


#include <linuxmt/types.h>
#include <linuxmt/kernel.h>
#include <linuxmt/sched.h>
#include <linuxmt/errno.h>
/*#include <linux/head.h> */
#include <linuxmt/genhd.h>
#include <linuxmt/hdreg.h>
#include <linuxmt/biosparm.h>
#include <linuxmt/major.h>
#include <linuxmt/bioshd.h>
#include <linuxmt/fs.h>
/*#include <linux/signal.h> */
#include <linuxmt/string.h>
#include <arch/segment.h>

#define MAJOR_NR BIOSHD_MAJOR
#define BIOSDISK
#include "blk.h"

static int bioshd_ioctl();
static int bioshd_open();
static void bioshd_release();

static struct file_operations bioshd_fops =
{
	NULL,			/* lseek - default */
	block_read,		/* read - general block-dev read */
	block_write,		/* write - general block-dev write */
	NULL,			/* readdir - bad */
	NULL,			/* select */
	bioshd_ioctl,		/* ioctl */
	bioshd_open,		/* open */
	bioshd_release,		/* release */
	NULL,			/* fsync */
	NULL,			/* check_media_change */
	NULL,			/* revalidate */
};

static struct wait_queue *busy_wait = NULL;
static struct wait_queue dma_wait;
static int dma_avail = 1;

static int bioshd_initialized = 0;
static int force_bioshd;
static int revalidate_hddisk();
static struct drive_infot
{
	int cylinders;
	int sectors;
	int heads;
	int fdtype; /* matches fd_types or -1 if hd */
}
drive_info[4];

struct drive_infot fd_types[] = {
	{40, 9, 2, 0},
	{80, 15, 2, 1},
	{80, 9, 2, 2},
	{80, 18, 2, 3},
	{80, 36, 2, 4},
};

static struct hd_struct hd[4 << 6];
static char busy[4] =
{0,};
static int access_count[4] =
{0,};
static unsigned char hd_drive_map[4] =
{
	0x80, 0x81,		/* hda, hdb */
	0x00, 0x01		/* fda, fdb */
};

static void bioshd_geninit();

static int bioshd_sizes[4 << 6] =
{0,};

static struct gendisk bioshd_gendisk =
{
	MAJOR_NR,		/* Major number */
	"hd",			/* Major name */
	6,			/* Bits to shift to get real from partition */
	1 << 6,			/* Number of partitions per real */
	4,			/* maximum number of real */
	bioshd_geninit,		/* init function */
	hd,			/* hd struct */
	bioshd_sizes,		/* sizes not blocksizes */
	0,			/* number */
	(void *) drive_info,	/* internal */
	NULL			/* next */
};

/* This function checks to see which hard drives are active and sets up the
 * drive_info[] table for them.  Ack this is darned confusing... */

int bioshd_gethdinfo()
{
	int drive, ndrives = 0;

	for (drive = 0; drive <= 1; drive++) {
		ndrives++;
		BD_AX = 0x0800;
		BD_DX = drive + 0x80; 
		BD_IRQ = 0x13;
		call_bios();
		if ((BD_AX != 0x100) && (!CARRY_SET)) {
			drive_info[drive].cylinders = ((BD_CX >> 8) & 255);
			drive_info[drive].cylinders += 
					(((BD_CX >> 6) & 3) * 256); 
			drive_info[drive].heads = ((BD_DX >> 8) & 63) + 1;
			drive_info[drive].sectors = (BD_CX & 63);
			drive_info[drive].fdtype = -1;
		}
		if ((BD_DX & 255) < 2) break;
	}
	return ndrives;
}

int bioshd_getfdinfo()
{
	int i, ndrives;

	/* We get the # of drives from the BPB, which is PC-friendly */
	ndrives = (peekb(0x40, 0x10) >> 6) + 1;
	printk("doshd: found %d floppy drives\n", ndrives);
	for (i = 0; i < ndrives; i++) {
		BD_AX = 0x800;
		BD_DX = i;
		BD_BX = 0;
		BD_IRQ = 0x13;
		printk("%x\n", BD_AX);
		call_bios();
		printk("%x\n", BD_AX);
		if ((!CARRY_SET) && (BD_AX != 0x100)) { 
			/* AT archecture, drive type in DX */
			printk("drive %d is %d\n", i, BD_BX);
			drive_info[i+2] = fd_types[BD_BX - 1];
		} else {	
			/* PC arch, assume 360K */
			printk("drive %d is 360k\n", i);
			drive_info[i+2] = fd_types[0];
		}
	}
	return ndrives;
}

static void
 bioshd_release(inode, filp)
    struct inode *inode;
    struct file *filp;
{
	int target;
	sync_dev(inode->i_rdev);

	target = DEVICE_NR(inode->i_rdev);
	access_count[target]--;
	return;
}

/* As far as I can tell this doesn't actually
   work, but we might as well try it. 
   -- Some XT controllers are happy with it.. [AC]
 */

static void reset_bioshd(minor)
    int minor;
{
	BD_IRQ = BIOSHD_INT;
	BD_AX = BIOSHD_RESET;
	BD_DX = hd_drive_map[DEVICE_NR(minor)];

	call_bios();
	if (CARRY_SET || (BD_AX & 0xff00) != 0)
		printk("hd: unable to reset.\n");
	return;
}

static int
 bioshd_open(inode, filp)
    struct inode *inode;
    struct file *filp;
{
	int target;
	int fdtype;
	target = DEVICE_NR(inode->i_rdev);

	fdtype = drive_info[target].fdtype;

	/* Bounds testing */

	if (bioshd_initialized == 0)
		return (-ENXIO);
	if (target >= 4)
		return (-ENXIO);
	if (hd[MINOR(inode->i_rdev)].start_sect == -1)
		return (-ENXIO);

	/* Wait until it's free */

	while (busy[target])
		sleep_on(&busy_wait);

	/* Register that we're using the device */

	access_count[target]++;

	/* Check the disk type */

	/* Currently, this can only differentiate between 1.4MB and 720kB
	   discs. It does this by trying to read sector 10 of track 0. If
	   it succeeds, it assumes that there are 18 sectors, if not 9. */

	if (target >= 2)
	{
		int bufseg;
		int count;
		int i;

		printk("hd: probing disc in /dev/fd%d\n", target % 2);

		/* The area between 32-64K is a 'scratch' area - we
		 * need a semaphore for it */

		while (!dma_avail) sleep_on(&dma_wait);
		dma_avail = 0;
		bufseg = 0x800;

		count = 0;
		for (i = 0; i <= fdtype; i++)
                {
		   count = 0;
 
		   while (count < 3)
		   {
/*			printk("type %d count %d\n", i, count);
*/			/* BIOS read sector */

			BD_IRQ = BIOSHD_INT;
			BD_AX = BIOSHD_READ | 1;	/* Read 1 sector */
			BD_BX = 0;	/* Seg offset = 0 */
			BD_ES = bufseg;		/* Segment to read to */
			BD_CX = (fd_types[i].cylinders<<8)|1;
			BD_CX += (0<<8)|fd_types[i].sectors;
			BD_DX = (0 << 8) | hd_drive_map[target];	/* Head 0, drive number */
			BD_FL = 0;

			sti();
			call_bios();

			if (CARRY_SET)
			{
				if (((BD_AX >> 8) == 0x04) && (count == 2)) 
				                         /* Sector not found */
					break;

				reset_bioshd(hd_drive_map[target]);
				count++;
			} else
			{
				drive_info[target].cylinders = fd_types[i].cylinders;
				drive_info[target].sectors = fd_types[i].sectors;
				break;
			}   
                }
		
		if (drive_info[target].cylinders == 0 || drive_info[target].sectors == 0)
			printk("hd: Autoprobe failed!\n");

		printk("hd: disc in /dev/fd%d probably has %d sectors and %d cylinders\n", target % 2,
		       drive_info[target].sectors, drive_info[target].cylinders);
	}

/*
 *	This is not a bugfix, hence no code, but coders should be aware
 *	that multi-sector reads from this point on depend on bootsect
 *	modifying the default Disk Parameter Block in BIOS.
 *	dpb[4] should be set to a high value such as 36 so that reads
 *	can go past what is hardwired in the BIOS.
 *	36 is the number of sectors in a 2.88 floppy track.
 *	If you are booting ELKS with anything other than bootsect you
 *	have to make equivalent arrangements.
 *	0:0x78 contains address of dpb (char dpb[12]), and dpb[4] is the End of
 *	Track parameter for the 765 Floppy Disk Controller.
 *	You may have to copy dpb to RAM as the original is in ROM.
 */
	}
	dma_avail = 1;
	wake_up(&dma_wait);
	return 0;
}

void init_bioshd()
{
	int i;
	int count = 0;
	int hdcount = 0;
	struct gendisk *ptr;
	int addr = 0x8c;

	printk("hd Driver Copyright (C) 1994 Yggdrasil Computing, Inc.\nExtended and modified for Linux8086 by Alan Cox.\n");

	bioshd_getfdinfo();
	bioshd_gethdinfo();
	for (i = 0; i <= 3; i++) {
		if (drive_info[i].heads) {count++; if (i <= 1) hdcount++;} 
	}

	if (!count)
		return;

	printk("hd: found %d drive%c\n", count, count == 1 ? ' ' : 's');

	for (i = 0; i < 4; i++)
	{
		if (drive_info[i].heads != 0)
		{
			printk("/dev/%cd%c: %d heads, %d cylinders, %d sectors = %ld %s\n",
			       (i < 2 ? 'h' : 'f'), (i % 2) + (i < 2 ? 'a' : '0'),
			       drive_info[i].heads,
			       drive_info[i].cylinders,
			       drive_info[i].sectors,
			       (long) drive_info[i].heads * drive_info[i].cylinders *
			       drive_info[i].sectors * 512L / (1024L * (i < 2 ? 1024L : 1L)),
			       (i < 2 ? "MB" : "kB"));
		}
	}
	bioshd_gendisk.nr_real = hdcount;
	i = register_blkdev(MAJOR_NR, DEVICE_NAME, &bioshd_fops);

	if (i == 0)
	{
		blk_dev[MAJOR_NR].request_fn = DEVICE_REQUEST;
		blksize_size[MAJOR_NR] = 1024;
		read_ahead[MAJOR_NR] = 2;
		if (gendisk_head == NULL)
		{
			bioshd_gendisk.next = gendisk_head;
			gendisk_head = &bioshd_gendisk;
		} else
		{
			for (ptr = gendisk_head; ptr->next != NULL; ptr = ptr->next)
				;
			ptr->next = &bioshd_gendisk;
			bioshd_gendisk.next = NULL;
		}
		bioshd_initialized = 1;
	} else
		printk("hd: unable to register\n");
}

static int bioshd_ioctl(inode, file, cmd, arg)
    struct inode *inode;
    struct file *file;
    unsigned int cmd;
    unsigned long arg;
{
	return -EINVAL;
}


static void do_bioshd_request()
{
	unsigned long count;
	unsigned long this_pass;
	short cylinder;
	short head;
	short sector;
	unsigned long start;
	int tmp;
	char *buff;
	int minor;
	int errs;
	int part;
	int drive;

	while (1)
	{
	      next_block:

		/* make sure we have a valid request */
		if (!CURRENT || CURRENT->dev < 0)
			return;

		/* now initialize it */
		INIT_REQUEST;

		/* make sure it's still valid */
		if (CURRENT == NULL || CURRENT->sector == -1)
			return;

		if (bioshd_initialized != 1)
		{
			end_request(0);
			continue;
		}
		minor = MINOR(CURRENT->dev);
		part = minor & ((1 << 6) - 1);
		drive = minor >> 6;

		/* make sure it's a disk we are dealing with. */
		if (drive > 3 || drive < 0 || drive_info[drive].heads == -0)
		{
			printk("hd: non-existant drive\n");
			end_request(0);
			continue;
		}
		count = CURRENT->nr_sectors;
		start = CURRENT->sector;
		buff = CURRENT->buffer;

		if (hd[minor].start_sect == -1 || start >= hd[minor].nr_sects)
		{
			printk("hd: bad partition start=%d sect=%d nr_sects=%d.\n",
			       start, (int) hd[minor].start_sect, (int) hd[minor].nr_sects);
			end_request(0);
			continue;
		}
		start += hd[minor].start_sect;
		errs = 0;

		while (count > 0)
		{
			sector = (start % drive_info[drive].sectors) + 1;
			tmp = start / drive_info[drive].sectors;
			head = tmp % drive_info[drive].heads;
			cylinder = tmp / drive_info[drive].heads;

			this_pass = count;
			if (count <= (drive_info[drive].sectors - sector + 1))
				this_pass = count;
			else
				this_pass = drive_info[drive].sectors - sector + 1;

			while (!dma_avail) sleep_on(&dma_wait);
			dma_avail = 0;

			BD_IRQ = BIOSHD_INT;
			if (CURRENT->cmd == WRITE) {
			   BD_AX = BIOSHD_WRITE | this_pass;
			   fmemcpy(0x800, 0, get_ds(), buff, (this_pass * 512));
			}
			else {
				BD_AX = BIOSHD_READ | this_pass;
			}
			
			BD_BX = 0;
			BD_ES = 0x800;
			BD_CX = (cylinder << 8) | ((cylinder >> 2) & 0xc0) | sector;
			BD_DX = (head << 8) | hd_drive_map[drive];
			BD_FL = 0;
#if 0
			printk("cylinder=%d head=%d sector=%d drive=%d CMD=%d\n",
			    cylinder, head, sector, drive, CURRENT->cmd);
			printk("blocks %d\n", this_pass);
#endif

			sti();
			call_bios();

			if (CARRY_SET)
			{
				reset_bioshd(MINOR(CURRENT->dev));
				dma_avail = 1;
				errs++;
				if (errs > MAX_ERRS)
				{
					printk("hd: error: AX=0x%x\n", BD_AX);
					end_request(0);
					wake_up(&dma_wait);
					goto next_block;
				}
				continue;	/* try again */
			}
			if (CURRENT->cmd==READ) {
			  fmemcpy(get_ds(), buff, 0x800, 0, (this_pass * 512)); 
			}
			/* In case it's already been freed */	
			if (!dma_avail) {
				dma_avail = 1;
				wake_up(&dma_wait);
			}
			count -= this_pass;
			start += this_pass;
			buff += this_pass * 512;
		}
		/* satisfied that request */
		end_request(1);
	}
}

#define DEVICE_BUSY busy[target]
#define USAGE access_count[target]
#define CAPACITY ((long)drive_info[target].heads*drive_info[target].sectors*drive_info[target].cylinders)
/* We assume that the the bios parameters do not change, so the disk capacity
   will not change */
#undef MAYBE_REINIT
#define GENDISK_STRUCT bioshd_gendisk

/*
 * This routine is called to flush all partitions and partition tables
 * for a changed cdrom drive, and then re-read the new partition table.
 * If we are revalidating a disk because of a media change, then we
 * enter with usage == 0.  If we are using an ioctl, we automatically have
 * usage == 1 (we need an open channel to use an ioctl :-), so this
 * is our limit.
 */
static int revalidate_hddisk(dev, maxusage)
    int dev;
    int maxusage;
{
	int target, major;
	struct gendisk *gdev;
	int max_p;
	int start;
	int i;

	target = DEVICE_NR(dev);
	gdev = &GENDISK_STRUCT;

	cli();
	if (DEVICE_BUSY || USAGE > maxusage)
	{
		sti();
		return -EBUSY;
	};
	DEVICE_BUSY = 1;
	sti();

	max_p = gdev->max_p;
	start = target << gdev->minor_shift;
	major = MAJOR_NR << 8;

	for (i = max_p - 1; i >= 0; i--)
	{
		sync_dev(major | start | i);
		invalidate_inodes(major | start | i);
		invalidate_buffers(major | start | i);
		gdev->part[start + i].start_sect = 0;
		gdev->part[start + i].nr_sects = 0;
	};

#ifdef MAYBE_REINIT
	MAYBE_REINIT;
#endif

	gdev->part[start].nr_sects = CAPACITY;
	resetup_one_dev(gdev, target);

	DEVICE_BUSY = 0;
	wake_up(&busy_wait);
	return 0;
}

static void bioshd_geninit()
{
	int i;

	for (i = 0; i < 4 << 6; i++)
	{
		if ((i & ((1 << 6) - 1)) == 0)
		{
			hd[i].start_sect = 0;
			hd[i].nr_sects = (long) drive_info[i >> 6].sectors *
			  drive_info[i >> 6].heads *
			  drive_info[i >> 6].cylinders;
		} else
		{
			hd[i].start_sect = -1;
			hd[i].nr_sects = 0;
		}
	}
	blksize_size[MAJOR_NR] = 1024;
}
