/**************************************
 * Direct video memory display driver *
 * Saku Airila 1996                   *
 **************************************/

#include<linuxmt/types.h>
#include<linuxmt/config.h>
#include<linuxmt/errno.h>
#include<linuxmt/fcntl.h>
#include<linuxmt/fs.h>
#include<linuxmt/major.h>
#include<linuxmt/sched.h>
#include <linuxmt/chqueue.h>
#include <linuxmt/ntty.h>

#ifdef CONFIG_CONSOLE_DIRECT

/****************************************************
 * This should come from the users configuration... *
 ****************************************************/
#define MAX_CONS 3

#define CONSOLE_NAME "console"

#define WAIT_MV1_PARM 1
#define WAIT_MV2_PARM 2
#define WAIT_P_PARM 3
#define WAIT_Q_PARM 4

#define MAX_ATTR 3

#define A_DEFAULT 0x07
#define A_BOLD 0x08
#define A_BLINK 0x80

typedef
struct ConsoleTag
{
   int cx, cy;
   int InCmd;
   int WaitParm, PTmp;
   unsigned Seg, Off;
   unsigned char Attr;
   char InUse;
} Console;

static int Width, Height, MaxRow, MaxCol;
static int CCBase;
static unsigned VideoSeg, PageSize;
static Console Con[ MAX_CONS ];
static Console * Visible;

/* from keyboard.c */
extern int Current_VCminor;

static unsigned AttrArry[] = {
   A_DEFAULT,
   A_BOLD,
   A_BLINK
   };

static void ScrollUp();
static void ScrollDown();
static void PositionCursor();
static void WriteChar();
static void CommandChar();
static void ClearRange();
static void WaitParameter();
static void Console_gotoxy();

extern int peekb();
extern int peekw();
extern void pokeb();
extern void pokew();
extern void outb();

extern int AddQueue(); /* From xt_key.c */
extern int GetQueue();

void con_charout( Ch )
char Ch;
{
   WriteChar( Visible, Ch );
   PositionCursor( Visible );
}

void WriteChar( C, Ch )
Console * C;
char Ch;
{
int CursorOfs;
   if( C->InCmd )
   {
      CommandChar( C, Ch );
      return;
   }
   if( Ch == 27 )
   {
      C->InCmd = 1;
      return;
   }
   if( Ch == '\b' )
   {
      if( C->cx > 0 )
      {
         C->cx--;
         WriteChar( C, ' ' );
         C->cx--;
         PositionCursor( C );
      }
      return;
   }
   if( Ch == '\n' )
   {
      C->cy++;
      C->cx = 0;
   }
   else if( Ch == '\r' )
      C->cx = 0;
   else
   {
      CursorOfs = ( C->cx << 1 ) + ( ( Width * C->cy ) << 1 );
      pokeb( C->Seg, C->Off + CursorOfs, Ch );
      pokeb( C->Seg, C->Off + CursorOfs + 1, C->Attr );
      if (C == Visible)
      {
         pokeb( VideoSeg, CursorOfs, Ch );
         pokeb( VideoSeg, CursorOfs + 1, C->Attr );
      }
      C->cx++;
   }
   if( C->cx > MaxCol )
#if 1 /* dodgy line wrap */
      C->cx = 0, C->cy++;
#else
      C->cx = MaxCol;
#endif
   if( C->cy >= Height )
   {
      ScrollUp( C, 1, Height );
      C->cy--;
   }
}

void ScrollUp( C, st, en )
Console * C;
int st, en;
{
unsigned rdofs, wrofs;
int cnt;
   if( st > en || st < 1 )
      return;
   rdofs = ( Width << 1 ) * st;
   wrofs = rdofs - ( Width << 1 );
   cnt = Width * ( en - st );
   far_memmove( C->Seg, C->Off + rdofs, C->Seg, C->Off + wrofs, cnt << 1 );
   if (C == Visible) 
      far_memmove( VideoSeg, rdofs, VideoSeg, wrofs, cnt << 1 );
   ClearRange( C, 0, MaxRow, Width, MaxRow );
}

void ScrollDown( C, st, en )
Console * C;
int st, en;
{
unsigned rdofs, wrofs;
int cnt;
   if( st > en || en > Height )
      return;
   rdofs = ( Width << 1 ) * st;
   wrofs = rdofs + ( Width << 1 );
   cnt = Width * ( en - st );
   far_memmove( C->Seg, C->Off + rdofs, C->Seg, C->Off + wrofs, cnt << 1 );
   if( C == Visible )
      far_memmove( VideoSeg, rdofs, VideoSeg, wrofs, 1 | ( cnt << 1 ) );
   ClearRange( C, 0, st, Width, st );
}

void PositionCursor( C )
Console * C;
{
int Pos;
   if( C != Visible )
      return;
   Pos = C->cx + Width * C->cy;
   outb( 14, CCBase );
   outb( ( unsigned char )( ( Pos >> 8 ) & 0xFF ), CCBase + 1 );
   outb( 15, CCBase );
   outb( ( unsigned char )( Pos & 0xFF ), CCBase + 1 );
}

void ClearRange( C, x, y, xx, yy )
Console * C;
int x, y, xx, yy;
{
unsigned st, en, ClrW, ofs;
   st = ( x << 1 ) + y * ( Width << 1 );
   en = ( xx << 1 ) + yy * ( Width << 1 );
   ClrW = A_DEFAULT << 8;
   for( ofs = st; ofs < en; ofs += 2 )
      pokew( C->Seg, C->Off + ofs, ClrW );
   if( C == Visible )
      for( ofs = st; ofs < en; ofs += 2 )
         pokew( VideoSeg, ofs, ClrW );
}

void CommandChar( C, Ch )
Console * C;
char Ch;
{
   if( C->WaitParm )
   {
      WaitParameter( C, Ch );
      return;
   }
   switch( Ch )
   {
      case 'H' : /* home */
         C->cx = C->cy = 0;
         break;
      case 'A' : /* up */
         if( C->cy )
            C->cy--;
         break;
      case 'B' : /* down */
         if( C->cy < MaxRow )
            C->cy++;
         break;
      case 'C' : /* right */
         if( C->cx < MaxCol )
            C->cx++;
         break;
      case 'D' : /* left */
         if( C->cx )
            C->cx--;
         break;
      case 'K' : /* clear to eol */
         ClearRange( C, C->cx, C->cy, Width, C->cy );
         break;
      case 'J' : /* clear to eoscreen */
         ClearRange( C, 0, 0, Width, Height );
         break;
      case 'I' : /* linefeed reverse */
         ScrollDown( C, 0, MaxRow );
         break;
      case 'Z' : /* identify */
/* Problem here */
         AddQueue( 27 );
         AddQueue( 'Z' );
         break;
      case 'P' : /* NOT VT52. insert/delete line */
         C->WaitParm = WAIT_P_PARM;
         return;
      case 'Q' : /* NOT VT52. My own additions. Attributes... */
         C->WaitParm = WAIT_Q_PARM;
         return;
      case 'Y' : /* cursor move */
         C->WaitParm = WAIT_MV1_PARM;
         return;
   }
   C->InCmd = 0;
}

void WaitParameter( C, Ch )
Console * C;
char Ch;
{
   switch( C->WaitParm )
   {
      case WAIT_MV1_PARM :
         C->PTmp = Ch - ' ';
         C->WaitParm = WAIT_MV2_PARM;
         return;
      case WAIT_MV2_PARM :
         Console_gotoxy( C, Ch - ' ', C->PTmp );
         break;
      case WAIT_P_PARM :
         switch( Ch )
         {
            case 'f' : /* insert line at crsr */
               ScrollDown( C, C->cy, MaxRow );
               break;
            case 'd' : /* delete line at crsr */
               ScrollUp( C, C->cy + 1, Height );
               break;
         }
         break;
      case WAIT_Q_PARM :
         Ch -= ' ';
         if( Ch > 0 && Ch < MAX_ATTR )
            C->Attr |= AttrArry[ Ch ];
         if( !Ch )
            C->Attr = A_DEFAULT;
         break;
      default :
         break;
   }
   C->WaitParm = C->InCmd = 0;
}

/* This also tells the keyboard driver which tty to direct it's output to...
 * CAUTION: It *WILL* break if the console driver doesn't get tty0-X. 
 */

void Console_set_vc( N )
{
   if( N < 0 || N >= MAX_CONS )
      return;
   if( Visible == &Con[ N ] )
      return;
   Visible = &Con[ N ];
   far_memmove(
      Visible->Seg, Visible->Off,
      VideoSeg, 0,
      ( Width * Height ) << 1 );
   PositionCursor( Visible );
   Current_VCminor = N;
}

void Console_gotoxy( C, x, y )
Console * C;
int x, y;
{
   if( x >= MaxCol )
      x = MaxCol;
   if( y >= MaxRow )
      y = MaxRow;
   if( x < 0 )
      x = 0;
   if( y < 0 )
      y = 0;
   C->cx = x;
   C->cy = y;
   PositionCursor( C );
}

int Console_write(tty)
struct tty * tty;
{
	int cnt = 0, chi;
	unsigned char ch;
	Console * C = &Con[tty->minor];
	while (tty->outq.len != 0) { 
		chq_getch(&tty->outq, &ch, 0);
		if (ch == '\n') WriteChar(C, '\r');
		WriteChar( C, ch);
		cnt++;
	};
	PositionCursor( C );
	return cnt;
}

void Console_release( inode, file )
struct inode *inode;
struct file *file;
{
int minor = MINOR( inode->i_rdev );
   if( minor < MAX_CONS )
      Con[ minor ].InUse = 0;
}

int Console_open( inode, file )
struct inode *inode;
struct file *file;
{
int minor = MINOR(inode->i_rdev);
   if( minor >= MAX_CONS )
      return -ENODEV;
   Con[ minor ].InUse = 1;
return 0;
}

struct tty_ops dircon_ops = {
	Console_open,
	Console_release,
	Console_write,
	NULL,
	NULL,
};

void init_console()
{
   unsigned int i, ConSeg;

   MaxCol = ( Width = peekb( 0x40, 0x4a ) ) - 1;
   /* Trust this. Cga does not support peeking at 0x40:0x84. */
   MaxRow = ( Height = 25 ) - 1;
   CCBase = peekw( 0x40, 0x63 );
   PageSize = peekw( 0x40, 0x4C );
   if( peekb( 0x40, 0x49 ) == 7 ) {
      VideoSeg = 0xB000;
      ConSeg = mm_alloc(0x400, 1); /* FIXME */
      for (i = 0; i < 0x4000; i+=2) pokew(ConSeg, i, 0); 
    }
    else
      ConSeg = VideoSeg = 0xb800;
 
   for( i = 0; i < MAX_CONS; i++ )
   {
      Con[ i ].cx = Con[ i ].cy = Con[ i ].InCmd = 0;
      Con[ i ].WaitParm = 0;
      Con[ i ].Attr = A_DEFAULT;
      Con[ i ].Seg = ConSeg + ( PageSize >> 4 ) * ( i + 1 );
      Con[ i ].Off = 0;
      Con[ i ].InUse = 0;
      ClearRange( &Con[ i ], 0, 0, Width, Height );
   }
   Con[ 0 ].cx = peekb( 0x40, 0x50 );
   Con[ 0 ].cy = peekb( 0x40, 0x51 );
   Con[ 0 ].InUse = 1;
   Visible = &Con[ 0 ];
   printk(
         "Console: Direct %dx%d emulating vt52 (%d virtual consoles)\n",
         Width,
         Height,
         MAX_CONS );
}

#endif

