/*
 * arch/i86/drivers/char/tty.c - A simple tty driver
 * (C) 1997 Chad Page et. al
 */

/* 
 * This new tty sub-system builds around the character queue to provide a
 * VFS interface to the character drivers (what a mouthful! :)  
 */

#include <linuxmt/types.h>
#include <linuxmt/config.h>
#include <linuxmt/sched.h>
#include <linuxmt/fs.h>
#include <linuxmt/fcntl.h>
#include <linuxmt/errno.h>
#include <linuxmt/termios.h>
#include <linuxmt/chqueue.h>
#include <linuxmt/ntty.h>
#include <linuxmt/major.h>

#define TAB_SPACES 8

#define MAX_TTYS 8
struct tty ttys[MAX_TTYS];
extern struct tty_ops dircon_ops;
extern struct tty_ops rs_ops;

/* Turns a dev_t variable into it's tty, or NULL if it's not valid */

struct tty * determine_tty(dev)
dev_t dev;
{
	int i, minor = MINOR(dev);

	for (i = 0; i < MAX_TTYS; i++) {
		if (ttys[i].minor == minor)
			return &ttys[i];
	}
	return 0; 
}

/* Just open it for now :) */
int tty_open(inode,file)
struct inode *inode;
struct file *file;
{
	struct tty * otty;

	if (otty = determine_tty(inode->i_rdev)) {
		return otty->ops->open(inode, file);
	}
	else
		return -ENODEV;

}

void tty_release(inode,file)
struct inode *inode;
struct file *file;
{
	struct tty *rtty;
	rtty = determine_tty(inode->i_rdev);
	if (rtty) 
		return rtty->ops->release(inode, file);
}

/* Write 1 byte to a terminal, with processing */

int tty_charout(tty, ch, defer)
struct tty *tty;
unsigned char ch;
int defer;
{
	int j;

	switch (ch) {
		case '\t':		
			for (j = 0; j < TAB_SPACES; j++) tty_charout(tty, ' ',1);
			break;
#if 1
		case '\n':
			tty_charout(tty, '\r',1);
#endif		
		default:
			while (chq_addch(&tty->outq, ch, 0) == -1) {
				tty->ops->write(tty);
			}
	};
	if (!defer) tty->ops->write(tty);
	return 1;
}

/*
 *	Write to a terminal
 *
 */
 
int tty_write(inode,file,data,len)
struct inode *inode;
struct file *file;
char *data;
int len;
{
   struct tty *tty=determine_tty(inode->i_rdev);
   int i = 0;

   while (i < len) {
	tty_charout(tty, peekb(current->t_regs.ds, data + i++), 1); 
   };
   tty->ops->write(tty);
   return i;
}

int tty_read(inode,file,data,len)
struct inode *inode;
struct file *file;
char *data;
int len;
{
	struct tty *tty=determine_tty(inode->i_rdev);
	int i = 0, j = 0, k;
	unsigned char ch, lch = 0;

	while (((i < len) && (j != 10))) { 
		lch = ch;
		j = chq_getch(&tty->inq, &ch, 1);
		if ((j != -1) && (j != '\b')) {
			pokeb(current->t_regs.ds, (data + i++), ch);		
			tty_charout(tty, ch, 0);
		}
		if (((j == '\b') && (i > 0))) {
		   i--;	
		   for (k = 0; k < (((lch == '\t') * (TAB_SPACES -1)) + 1); k++)
			tty_charout(tty, ch, 0);
		}	
	};	
	return i;
}

int tty_ioctl(inode,file, cmd, arg)
struct inode *inode;
struct file *file;
int cmd;
char *arg;
{
	return -EINVAL;
}

int tty_lseek(inode,file,offset,origin)
struct inode *inode;
struct file *file;
off_t offset;
int origin;
{
	return -ESPIPE;
}

int tty_readdir()
{
	return -ENOTDIR;
}

static struct file_operations tty_fops =
{
	tty_lseek,
	tty_read,
	tty_write,
	tty_readdir,
	NULL,		/* Select - needs doing */
	tty_ioctl,
	tty_open,
	tty_release,
	NULL,
	NULL,
	NULL
};

void tty_init()
{
	int i;

	for (i = 0; i < NUM_TTYS; i++) { 
		chq_init(&ttys[i].inq, ttys[i].inq_buf, INQ_SIZE);
		chq_init(&ttys[i].outq, ttys[i].outq_buf, OUTQ_SIZE);
	}

	for (i = 0; i < 3; i++) {
		ttys[i].ops = &dircon_ops;
		ttys[i].minor = i;
	}
#ifdef CONFIG_CHAR_DEV_RS
	for (i = 4; i < 8; i++) {
		ttys[i].ops = &rs_ops;
		ttys[i].minor = i;
	}
#endif

	register_chrdev(TTY_MAJOR,"tty",&tty_fops);
}

