/*
 * ELKS implmentation of memory devices
 * /dev/null, /dev/zero, /dev/mem, /dev/kmem, etc...
 *
 * Heavily inspired by linux/drivers/char/mem.c
 */

#include <linuxmt/config.h>

#ifdef CONFIG_CHAR_DEV_MEM
#include <linuxmt/major.h>
#include <linuxmt/fs.h>
#include <linuxmt/errno.h>
#include <linuxmt/mm.h>
#include <linuxmt/debug.h>

#define DEV_MEM_MINOR		1
#define DEV_KMEM_MINOR		2
#define DEV_NULL_MINOR		3
#define DEV_PORT_MINOR		4
#define DEV_ZERO_MINOR		5
#define DEV_FULL_MINOR		7
#define DEV_RANDOM_MINOR	8
#define DEV_URANDOM_MINOR	9

/*
 * generally useful code...
 */
int memory_lseek(inode, filp, offset, origin)
struct inode * inode;
struct file * filp;
off_t offset;
unsigned int origin;
{
	int tmp = -1;

	printd_mem("mem_lseek()\n");
	switch(origin) {
		case 0:
			tmp = offset;
			break;
		case 1:
			tmp = filp->f_pos + offset;
			break;
		default: 
			return -EINVAL;
	}
	if (tmp < 0)
		return -EINVAL;
	if (tmp != filp->f_pos) {
		filp->f_pos = tmp;
		filp->f_reada = 0;
		filp->f_version = ++event;
	}
	return filp->f_pos;	
}



/*
 * /dev/null code
 */
int null_lseek(inode, filp, offset, origin)
struct inode * inode;
struct file * filp;
off_t offset;
int origin;
{
	printd_mem("null_lseek()\n");
	return (filp->f_pos = 0);
}

int null_read(inode, filp, data, len)
struct inode * inode;
struct file * filp;
char * data;
int len;
{
	printd_mem("null_read()\n");
	return 0;
}

int null_write(inode, filp, data, len)
struct inode * inode;
struct file * filp;
char * data;
int len;
{
	printd_mem("null write: ignoring %d bytes!\n", len);
	return len;
}

static struct file_operations null_fops = {
	null_lseek,		/* lseek */
	null_read,		/* read */
	null_write,		/* write */
	NULL,			/* readdir */
	NULL,			/* select */
	NULL,			/* ioctl */
	NULL,			/* open */
	NULL,			/* release */
	NULL,			/* fsync */
	NULL,			/* check_media_change */
	NULL			/* revalidate */
};


/*
 * /dev/full code
 */
int full_read(inode, filp, data, len)
struct inode * inode;
struct file * filp;
char * data;
int len;
{
	printd_mem("full_read()\n");
	filp->f_pos += len;
	return len;
}

int full_write(inode, filp, data, len)
struct inode * inode;
struct file * filp;
char * data;
int len;
{
	printd_mem("full_write: objecting to %d bytes!\n", len);
	return -ENOSPC;
}

static struct file_operations full_fops = {
	memory_lseek,		/* lseek */
	full_read,		/* read */
	full_write,		/* write */
	NULL,			/* readdir */
	NULL,			/* select */
	NULL,			/* ioctl */
	NULL,			/* open */
	NULL,			/* release */
	NULL,			/* fsync */
	NULL,			/* check_media_change */
	NULL			/* revalidate */
};


/*
 * /dev/zero code
 */
int zero_read(inode, filp, data, len)
struct inode * inode;
struct file * filp;
char * data;
int len;
{
	int error;

	printd_mem("zero_read()\n");
	fmemset(data, current->mm.dseg, 0, len);
	filp->f_pos +=len;
	return len;
}

static struct file_operations zero_fops = {
	memory_lseek,		/* lseek */
	zero_read,		/* read */
	null_write,		/* write */
	NULL,			/* readdir */
	NULL,			/* select */
	NULL,			/* ioctl */
	NULL,			/* open */
	NULL,			/* release */
	NULL,			/* fsync */
	NULL,			/* check_media_change */
	NULL			/* revalidate */
};


/*
 * memory device open multiplexor
 */
int memory_open(inode, filp)
struct inode * inode;
struct file * filp;
{
	printd_mem("memory_open: minor = %d; it's /dev/", (int)MINOR(inode->i_rdev));
	switch(MINOR(inode->i_rdev)) {
		case DEV_NULL_MINOR:
			filp->f_op = & null_fops;
	printd_mem("null!\n");
			return 0;
		case DEV_FULL_MINOR:
			filp->f_op = & full_fops;
	printd_mem("full!\n");
			return 0;
		case DEV_ZERO_MINOR:
			filp->f_op = & zero_fops;
	printd_mem("zero!\n");
			return 0;
		default:
	printd_mem("???\n");
		return -ENXIO;
	}
}


static struct file_operations memory_fops = {
	NULL,			/* lseek */
	NULL,			/* read */
	NULL,			/* write */
	NULL,			/* readdir */
	NULL,			/* select */
	NULL,			/* ioctl */
	memory_open,		/* open */
	NULL,			/* release */
	NULL,			/* fsync */
	NULL,			/* check_media_change */
	NULL			/* revalidate */
};

void mem_dev_init()
{
	if(register_chrdev(MEM_MAJOR, "mem", &memory_fops))
		printd_mem("unable to get major %d for memory devices\n", MEM_MAJOR);
}

#endif /* CONFIG_CHAR_DEV_MEM */
