//   $Id: kvi_ulist.cpp,v 1.1 1998/09/23 12:25:38 pragma Exp $
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this library; see the file COPYING.LIB.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CLASS_NAME_ "KviUserManager"

#include "kvi_app.h"
#include "kvi_macros.h"
#include "kvi_debug.h"
#include "kvi_types.h"
#include "kvi_ulist.h"

#include <kmsgbox.h>


//============ KviUserList ============//

KviUserManager::KviUserManager()
{
	_debug_entertrace("KviUserManager");
	m_lpUserList=new QList<KviUserStruct>;
	m_lpUserList->setAutoDelete(true);
	m_lpCurNotifyList=new QList<QString>;
	m_lpCurNotifyList->setAutoDelete(true);
	m_lpCurIgnoreList=new QList<QString>;
	m_lpCurIgnoreList->setAutoDelete(true);
	readFromFile();
	fillIgnoreList();
	fillNotifyList();
	_debug_leavetrace("KviUserManager");
}

//============ ~KviUserManager ============//

KviUserManager::~KviUserManager()
{
	_debug_entertrace("~KviUserManager");
	saveToFile();
	while(!m_lpCurNotifyList->isEmpty())m_lpCurNotifyList->removeLast();
	while(!m_lpCurIgnoreList->isEmpty())m_lpCurIgnoreList->removeLast();
	while(!m_lpUserList->isEmpty())m_lpUserList->removeLast();
	delete m_lpCurNotifyList;
	delete m_lpCurIgnoreList;
	delete m_lpUserList;
	_debug_leavetrace("~KviUserManager");
}

//============ readFromFile ============//

void KviUserManager::readFromFile()
{
	//TODO:SessionManagement
	//     Move this class to use KConfig
	_debug_entertrace("readFromFile");
	QString szFileName(_macro_getKVircHomeDirectory("Config/kvi.user.conf"));
	QFile config(szFileName.data());
	if(!config.exists()){
		config.open(IO_WriteOnly);
		config.close();
		return;
	}
	if(!config.open(IO_ReadOnly)){ cantReadError(); return; }
	//Write the users num
	uint cnt;
	char dummyBuf[100];
	if(config.readBlock(dummyBuf,46)<0)return; //empty file
	if(config.readBlock((char *)&cnt,sizeof(cnt))<0)return; //no users
	if(cnt==0)return; //empty list
	for(uint i=0;i<cnt;i++){
		KviUserStruct *lpU=new KviUserStruct;
		clearUserStruct(lpU);
		//if we fail the QFile destructor will close the file
		if(!readString(&config,lpU->szNick)){ cantReadError(); delete lpU; return; };
		if(!readString(&config,lpU->szMask)){ cantReadError(); delete lpU; return; };
		if(!readString(&config,lpU->szNotes)){ cantReadError(); delete lpU; return; };
		if(!readString(&config,lpU->szActionOnJoin)){ cantReadError(); delete lpU; return; };
		if(!readString(&config,lpU->szActionOnNotify)){ cantReadError(); delete lpU; return; };
		if(!readString(&config,lpU->szActionOnCtcpCommand)){ cantReadError(); delete lpU; return; };
		QString szOptMask;
		if(!readString(&config,szOptMask)){ cantReadError(); delete lpU; return; };
		if(szOptMask.length()==5){
			lpU->bNotify=(szOptMask[0]=='1');
			lpU->bIgnore=(szOptMask[1]=='1');
			lpU->bEnableActionOnJoin=(szOptMask[2]=='1');
			lpU->bEnableActionOnNotify=(szOptMask[3]=='1');
			lpU->bEnableActionOnCtcpCommand=(szOptMask[4]=='1');
		}   //else keep the default mask , but we have the file messed...
		    //if we're lucky however it may be saved... , if not we will just fail at the end
		m_lpUserList->append(lpU);
	}
	config.close();
	_debug_leavetrace("readFromFile");
}

//============ readString ============//

bool KviUserManager::readString(QFile *lpFile,QString &szString)
{
	_debug_entertrace("readString");
	uint len;
	char buffer[4096];
	if(lpFile->readBlock((char *)&len,sizeof(len))<0)return false;
	if(len>4096){
		debug("User buffere too long");
		return false;
	}
	if(lpFile->readBlock(buffer,len)<0)return false;
	szString=QString(buffer,len+1);
	_debug_leavetrace("readString");
	return true;
}

//============ cantWriteError ============//
void KviUserManager::cantWriteError()
{
	_debug_entertrace("cantWriteError");
	KMsgBox::message(0,i18n("WARNING File I/O Error"),
			 i18n("Can't write to the kvi.user.conf file.\nThe current user list will be lost."));
	_debug_leavetrace("cantWriteError");
}

//============ cantReadError ============//
void KviUserManager::cantReadError()
{
	_debug_entertrace("cantReadError");
	KMsgBox::message(0,i18n("WARNING File I/O Error"),i18n("Can't read the kvi.user.conf file.\nThe current user list will be lost."));
	_debug_leavetrace("cantReadError");
}

//============ saveToFile ============//

void KviUserManager::saveToFile()
{
	_debug_entertrace("saveToFile");
	QString szFileName(_macro_getKVircHomeDirectory("Config/kvi.user.conf"));
	QFile config(szFileName.data());
	if(!config.open(IO_WriteOnly|IO_Truncate)){ cantWriteError(); return; }
	//Write the alias number
	uint cnt=m_lpUserList->count();
	if(config.writeBlock("KVIrc CONFIGURATION FILE - PLEASE DO NOT EDIT\n",46)<0){ cantWriteError(); return; };
	if(config.writeBlock((char *)&cnt,sizeof(cnt))<0){ cantWriteError(); return; };
	for(uint i=0;i<cnt;i++){
		KviUserStruct *lpU=m_lpUserList->at(i);
		if(lpU){
			uint len=lpU->szNick.length();
			if(config.writeBlock((char *)&len,sizeof(len))<0){ cantWriteError(); return; };
			if(config.writeBlock(lpU->szNick.data(),len)<0){ cantWriteError(); return; };
			len=lpU->szMask.length();
			if(config.writeBlock((char *)&len,sizeof(len))<0){ cantWriteError(); return; };
			if(config.writeBlock(lpU->szMask.data(),len)<0){ cantWriteError(); return; };
			len=lpU->szNotes.length();
			if(config.writeBlock((char *)&len,sizeof(len))<0){ cantWriteError(); return; };
			if(config.writeBlock(lpU->szNotes.data(),len)<0){ cantWriteError(); return; };
			len=lpU->szActionOnJoin.length();
			if(config.writeBlock((char *)&len,sizeof(len))<0){ cantWriteError(); return; };
			if(config.writeBlock(lpU->szActionOnJoin.data(),len)<0){ cantWriteError(); return; };
			len=lpU->szActionOnNotify.length();
			if(config.writeBlock((char *)&len,sizeof(len))<0){ cantWriteError(); return; };
			if(config.writeBlock(lpU->szActionOnNotify.data(),len)<0){ cantWriteError(); return; };
			len=lpU->szActionOnCtcpCommand.length();
			if(config.writeBlock((char *)&len,sizeof(len))<0){ cantWriteError(); return; };
			if(config.writeBlock(lpU->szActionOnCtcpCommand.data(),len)<0){ cantWriteError(); return; };
			QString szOptMask="00000";
			if(lpU->bNotify)szOptMask[0]='1';
			if(lpU->bIgnore)szOptMask[1]='1';
			if(lpU->bEnableActionOnJoin)szOptMask[2]='1';
			if(lpU->bEnableActionOnNotify)szOptMask[3]='1';
			if(lpU->bEnableActionOnCtcpCommand)szOptMask[4]='1';
			len=5;
			if(config.writeBlock((char *)&len,sizeof(len))<0){ cantWriteError(); return; };
			if(config.writeBlock(szOptMask.data(),5)<0){ cantWriteError(); return; };
		} else {
			_debug_fatal("Qt seems to be confused...");
		}
	}
	config.close();
	_debug_leavetrace("saveToFile");
}

//============ addUser ============//

void KviUserManager::addUser(KviUserStruct *lpS)
{
	_debug_entertrace("addUser");
	m_lpUserList->append(lpS);
	fillNotifyList();
	fillIgnoreList();
	_debug_leavetrace("addUser");
}

//============ getUser ============//

KviUserStruct * KviUserManager::getUser(const char *szNick)
{
	_debug_entertrace("getUser");
	KviUserStruct *lpU;
	for(lpU=m_lpUserList->first();lpU;lpU=m_lpUserList->next()){
		if(!strcasecmp(szNick,lpU->szNick.data()))return lpU;
	}
	_debug_leavetrace("getUser");
	return 0;
}

//============ delUser ============//

void KviUserManager::delUser(const char *szNick)
{
	_debug_entertrace("delUser");
	KviUserStruct *lpU=getUser(szNick);
	if(lpU){
		m_lpUserList->removeRef(lpU);
		fillNotifyList();
		fillIgnoreList();
	}
	_debug_leavetrace("delUser");
}

//============ fillNotifyList ============//

void KviUserManager::fillNotifyList()
{
	_debug_entertrace("fillNotifyList");
	while(!m_lpCurNotifyList->isEmpty())m_lpCurNotifyList->removeLast();
	KviUserStruct *lpU;
	for(lpU=m_lpUserList->first();lpU;lpU=m_lpUserList->next()){
		if(lpU->bNotify){
			QString *pS=new QString(lpU->szNick.data());
			m_lpCurNotifyList->append(pS);
		}
	}
	_debug_leavetrace("fillNotifyList");
}

//============ fillIgnoreList ============//

void KviUserManager::fillIgnoreList()
{
	_debug_entertrace("fillIgnoreList");
	while(!m_lpCurIgnoreList->isEmpty())m_lpCurIgnoreList->removeLast();
	KviUserStruct *lpU;
	for(lpU=m_lpUserList->first();lpU;lpU=m_lpUserList->next()){
		if(lpU->bIgnore){
			QString *pS=new QString(lpU->szNick.data());
			m_lpCurIgnoreList->append(pS);
		}
	}
	_debug_leavetrace("fillIgnoreList");
}

//============ clearUserStruct ============//

void KviUserManager::clearUserStruct(KviUserStruct *lpS)
{
	_debug_entertrace("clearUserStruct");
	lpS->szNick="";
	lpS->szMask="";
	lpS->szActionOnJoin="/say $2 hello!";
	lpS->szActionOnNotify="/whois $1";
	lpS->szActionOnCtcpCommand="/say $1 No actions set.";
	lpS->szNotes="";
	lpS->bNotify=false;
	lpS->bIgnore=false;
	lpS->bEnableActionOnCtcpCommand=false;
	lpS->bEnableActionOnJoin=false;
	lpS->bEnableActionOnNotify=false;
	_debug_leavetrace("clearUserStruct");
}

//============ copyUserStruct ============//

void KviUserManager::copyUserStruct(KviUserStruct *lpDest,KviUserStruct *lpSrc)
{
	_debug_entertrace("copyUserStruct");
	lpDest->szNick=lpSrc->szNick.copy();
	lpDest->szMask=lpSrc->szMask.copy();
	lpDest->szActionOnJoin=lpSrc->szActionOnJoin.copy();
	lpDest->szActionOnNotify=lpSrc->szActionOnNotify.copy();
	lpDest->szActionOnCtcpCommand=lpSrc->szActionOnCtcpCommand.copy();
	lpDest->szNotes=lpSrc->szNotes.copy();
	lpDest->bNotify=lpSrc->bNotify;
	lpDest->bIgnore=lpSrc->bIgnore;
	lpDest->bEnableActionOnCtcpCommand=lpSrc->bEnableActionOnCtcpCommand;
	lpDest->bEnableActionOnJoin=lpSrc->bEnableActionOnJoin;
	lpDest->bEnableActionOnNotify=lpSrc->bEnableActionOnNotify;
	_debug_leavetrace("copyUserStruct");
}


//
// $Log: kvi_ulist.cpp,v $
// Revision 1.1  1998/09/23 12:25:38  pragma
// User list system. Still not complete , but works.
//
//
