//   $Id: kvi_uglobal.cpp,v 1.1 1998/10/06 15:02:55 pragma Exp $
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//

//#define _KVI_DEBUG_CLASS_NAME_ "KviGlobalUserList"
#define _KVI_DEBUG_CHECK_RANGE_

#include "kvi_app.h"
#include "kvi_debug.h"
#include "kvi_uglobal.h"
#include "kvi_nick.h"
#include "kvi_child.h"
#include "kvi_defs.h"

#include <ctype.h>

//============ KviGlobalUserList ============//

KviGlobalUserList::KviGlobalUserList()
{
	_debug_entertrace("KviGlobalUserList");
	m_lpGlobalUserList=new QList<KviGlobalUserListEntry>;
	m_lpGlobalUserList->setAutoDelete(true);
	_debug_leavetrace("KviGlobalUserList");
}

//============ ~KviGlobalUserList ============//

KviGlobalUserList::~KviGlobalUserList()
{
	_debug_entertrace("~KviGlobalUserList");
	clear();
	delete m_lpGlobalUserList;
	_debug_leavetrace("~KviGlobalUserList");
}

//============ clear ============//

void KviGlobalUserList::clear()
{
	_debug_entertrace("clear");
	//
	// In fact if everything goes ok , this list should be empty when
	// clear() is called (in kvi_frame.cpp : handleSocketConnected).
	// If it is not , something gone wrong , so warn the developer :)
	//
	_range_valid(m_lpGlobalUserList->isEmpty());
	while(!m_lpGlobalUserList->isEmpty()){
		KviGlobalUserListEntry *lpE=m_lpGlobalUserList->first();
		delete lpE->lpNick;
		m_lpGlobalUserList->removeFirst();
	}
	_debug_leavetrace("clear");
}

//============ getNickEntry ============//

KviGlobalUserListEntry * KviGlobalUserList::getNickEntry(const char *szNick)
{
	_debug_entertrace("getNickEntry");
	_range_valid(szNick);
	KviGlobalUserListEntry *lpE=0;
	char upCase=toupper(*szNick);
	if(upCase<'N'){
		for(lpE=m_lpGlobalUserList->first();lpE;lpE=m_lpGlobalUserList->next()){
			_range_valid(lpE->lpNick);
			if(lpE->firstCharUpcase==upCase){
				if(!strcasecmp(szNick,lpE->lpNick->szNick.data()))return lpE;
			}
		}
	} else { //reverse search
		for(lpE=m_lpGlobalUserList->last();lpE;lpE=m_lpGlobalUserList->prev()){
			_range_valid(lpE->lpNick);
			if(lpE->firstCharUpcase==upCase){
				if(!strcasecmp(szNick,lpE->lpNick->szNick.data()))return lpE;
			}
		}
	}
	_debug_leavetrace("getNickEntry");
	return 0;
}

//============ getNick ============//

KviNewNick * KviGlobalUserList::getNick(const char *szNick)
{
	_debug_entertrace("getNick");
	KviGlobalUserListEntry *lpE=getNickEntry(szNick);
	_debug_leavetrace("getNick");
	if(lpE)return lpE->lpNick;
	else return 0;
}

//============ dump ============//

void KviGlobalUserList::dump(KviMdiChild *lpC)
{
	_debug_entertrace("dump");
	KviGlobalUserListEntry *lpE=0;
	lpC->doOutput(KVI_OUT_INTERNAL,"# Dumping internal user list.");
	int totRefs=0;
	for(lpE=m_lpGlobalUserList->first();lpE;lpE=m_lpGlobalUserList->next()){
		_range_valid(lpE->lpNick);
		lpC->doFmtOutput(KVI_OUT_INTERNAL,"# Entry :%s!%s@%s , nRefs :%d ,\n  char :%c , Entry ptr :%u , KviNick ptr :%u",
						lpE->lpNick->szNick.data(),
						lpE->lpNick->szUserName.data(),
						lpE->lpNick->szAddress.data(),
						lpE->nRefs,
						lpE->firstCharUpcase,
						(unsigned long)lpE,
						(unsigned long)lpE->lpNick);
		totRefs+=lpE->nRefs;
	}
	lpC->doFmtOutput(KVI_OUT_INTERNAL,"# Total entries : %d  , Total refs : %d",m_lpGlobalUserList->count(),totRefs);
	_debug_leavetrace("dump");
}

//============ updateNick ============//
//
//KviNewNick * KviGlobalUserList::updateNick(KviNewNick &kn)
//{
//	_debug_entertrace("updateNick");
//	KviGlobalUserListEntry *lpE=getNickEntry(kn.szNick.data());
//	_range_valid(lpE);
//	if(lpE){
//		_range_valid(lpE->lpNick);
//		lpE->lpNick->szAddress=kn.szAddress.data();
//		lpE->lpNick->szUserName=kn.szUserName.data();
//		return lpE->lpNick;
//	}
//	_debug_leavetrace("updateNick");
//	return 0;
//}

//============ updateUser ============//

KviNewNick * KviGlobalUserList::updateUser(KviNewNick &kn)
{
	_debug_entertrace("updateUser");
	KviGlobalUserListEntry *lpE=getNickEntry(kn.szNick.data());
//	_range_valid(lpE);
	if(lpE){
		_range_valid(lpE->lpNick);
		lpE->lpNick->szAddress=kn.szAddress.data();
		lpE->lpNick->szUserName=kn.szUserName.data();
		return lpE->lpNick;
	}
	_debug_leavetrace("updateUser");
	return 0;
}

//============ changeNickname ============//

KviNewNick * KviGlobalUserList::changeNickname(const char *szOldNick,const char *szNewNick)
{
	_debug_entertrace("changeNickname");
	KviGlobalUserListEntry *lpE=getNickEntry(szOldNick);
	if(!lpE)return 0;
	m_lpGlobalUserList->setAutoDelete(false);
	m_lpGlobalUserList->removeRef(lpE);
	m_lpGlobalUserList->setAutoDelete(true);
	_range_valid(lpE->lpNick);
	lpE->lpNick->szNick=szNewNick;
	lpE->firstCharUpcase=toupper(*szNewNick);
	inSort(lpE);
	_debug_leavetrace("changeNickname");
	return lpE->lpNick;
}

//============ addNick ============//

KviNewNick * KviGlobalUserList::addNick(KviNewNick &kn)
{
	_debug_entertrace("addNick");
	KviGlobalUserListEntry *lpE=getNickEntry(kn.szNick.data());
	if(lpE){
		lpE->nRefs++;
		// try also to update the nick if possible
		if(!strcmp(lpE->lpNick->szAddress.data(),KVI_STR_NULL)){
			//We have a null entry in the list
			if(strcmp(kn.szAddress.data(),KVI_STR_NULL)){
				//the new address is not null
				lpE->lpNick->szAddress=kn.szAddress.data();
				lpE->lpNick->szUserName=kn.szUserName.data();
			}
		}
		return lpE->lpNick;
	} else {
		lpE=new KviGlobalUserListEntry;
		lpE->lpNick=new KviNewNick(kn);
		lpE->nRefs=1;
		lpE->firstCharUpcase=toupper(*(kn.szNick.data()));
		inSort(lpE);
		return lpE->lpNick;
	}
	_debug_leavetrace("addNick");
}

//============ removeNick ============//

void KviGlobalUserList::removeNick(KviNewNick &kn)
{
	_debug_entertrace("removeNick");
	KviGlobalUserListEntry *lpE=getNickEntry(kn.szNick.data());
	_range_valid(lpE);
	if(lpE){
		lpE->nRefs--;
		_range_valid(lpE->nRefs >= 0);
		if(lpE->nRefs==0){
			delete lpE->lpNick;
			m_lpGlobalUserList->removeRef(lpE);
		}
	}
	_debug_leavetrace("removeNick");
}

//============ removeRef ============//

void KviGlobalUserList::removeRef(KviNewNick *lpNick)
{
	_debug_entertrace("removeRef");
	KviGlobalUserListEntry *lpE=getNickEntry(lpNick->szNick.data());
	_range_valid(lpE);
	_range_valid(lpNick);
	_range_valid(lpNick==lpE->lpNick);
	if(lpE){
		lpE->nRefs--;
		_range_valid(lpE->nRefs >= 0);
		if(lpE->nRefs==0){
			delete lpE->lpNick;
			m_lpGlobalUserList->removeRef(lpE);
		}
	}
	_debug_leavetrace("removeRef");
}


//============ inSort ============//

void KviGlobalUserList::inSort(KviGlobalUserListEntry *lpE)
{
	_debug_entertrace("inSort");
	KviGlobalUserListEntry *lpX=0;
	int index=0;
	for(lpX=m_lpGlobalUserList->first();lpX;lpX=m_lpGlobalUserList->next()){
		_range_valid(lpX->lpNick);
		if(lpX->firstCharUpcase==lpE->firstCharUpcase){
			if(strcasecmp(lpX->lpNick->szNick.data(),lpE->lpNick->szNick.data())>0){
				m_lpGlobalUserList->insert(index,lpE);
				return;
			}
		} else if(lpX->firstCharUpcase > lpE->firstCharUpcase){
			m_lpGlobalUserList->insert(index,lpE);
			return;
		}
		index++;
	}
	m_lpGlobalUserList->append(lpE);
	_debug_leavetrace("inSort");
}

//
// $Log: kvi_uglobal.cpp,v $
// Revision 1.1  1998/10/06 15:02:55  pragma
// Moved internal users list to one global list:
// faster updates and less server traffic
//
//
