//   $Id: kvi_translate.cpp,v 1.3 1998/09/20 20:24:11 fritz Exp $
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//

//
// Many parts of this source code were 'inspired' by the ircII4.4 translat.c source.
//

#define _KVI_DEBUG_CLASS_NAME_ "KviTranslator"

#include "kvi_translate.h"
#include "kvi_debug.h"
#include "kvi_app.h"

#include <kmsgbox.h>

//============ KviTranslator ============//

KviTranslator::KviTranslator()
{
	_debug_entertrace("KviTranslator");
	setDefaultTranslationMap();
	m_szMapFileName="";
	_debug_leavetrace("KviTranslator");
}

//============ ~KviTranslator ============//

KviTranslator::~KviTranslator()
{
	_debug_entertrace("~KviTranslator");
	_debug_leavetrace("~KviTranslator");
}

//============ setDefaultTranslationMap ============//

void KviTranslator::setDefaultTranslationMap()
{
	_debug_entertrace("setDefaultTranslationMap");
	for(int i=0;i<256;i++){
		serverToClientTab[i]=i;
		clientToServerTab[i]=i;
	}
	m_bDefault=true;
	m_szMapFileName="";
	_debug_leavetrace("setDefaultTranslationMap");
}

//============ setTranslationMap ============//

bool KviTranslator::setTranslationMap(const char *szMapFileName)
{
	_debug_entertrace("setTranslationMap");
	QFile mapFile(szMapFileName);
	QString szM=szMapFileName;
	if(szM.find("LATIN_1") != -1){
		setDefaultTranslationMap();
		return true;
	}
	if(!(mapFile.exists() && mapFile.open(IO_ReadOnly)) ){
		KMsgBox::message(0,i18n("Translation"),i18n("Could not open the translation file for reading."),KMsgBox::EXCLAMATION,i18n("OK"));
		setDefaultTranslationMap();
		return false;
	}

	// translat.c :
	// Any problems in the translation tables between hosts are
	// almost certain to be caused here.
	// many scanf implementations do not work as defined. In particular,
	// scanf should ignore white space including new lines (many stop
	// at the new line character, hence the fgets and sscanf workaround),
	// many fail to read 0xab as a hexadecimal number (failing on the
	// x) despite the 0x being defined as optionally existing on input,
	// and others zero out all the output variables if there is trailing
	// non white space in the format string which doesn't appear on the
	// input. Overall, the standard I/O libraries have a tendancy not
	// to be very standard.

	char buffer[80];
	int inputs[8];
	unsigned char temp_table[512];
	int c=0;

	while((mapFile.readLine(buffer,80)>0)&&(c<512)){
		if(
				sscanf(buffer, "0x%x,0x%x,0x%x,0x%x,0x%x,0x%x,0x%x,0x%x",
		    		inputs+0, inputs+1, inputs+2, inputs+3,
		    		inputs+4, inputs+5, inputs+6, inputs+7)   <8
		) {
			KMsgBox::message(0,i18n("Translation"),i18n("Syntax error in translation file."),KMsgBox::EXCLAMATION,i18n("OK"));
			setDefaultTranslationMap();
			mapFile.close();
			return false;
		}
		for (int j = 0; j<8; j++)temp_table[c++] = (unsigned char) inputs[j];
	}

	mapFile.close();

	if(c==512){
		for (c = 0; c < 256; c++)
		{
			serverToClientTab[c] = temp_table[c];
			clientToServerTab[c] = temp_table[c | 256];
		}
	} else {
		KMsgBox::message(0,i18n("Translation"),i18n("Translation file corrupted."),KMsgBox::EXCLAMATION,i18n("OK"));
		setDefaultTranslationMap();
		return false;
	}
	m_bDefault=false;
	m_szMapFileName=szMapFileName;
	return true;
	_debug_leavetrace("setTranslationMap");
}

//============ translateToClient ============//

void KviTranslator::translateToClient(QString &szString)
{
	_debug_entertrace("translateToClient");
	if(m_bDefault)return;
	char *pC=szString.data();
//	debug("Translating string %s",pC);
	while(*pC){
//		debug("Char %c (%d) becomes %c (%d)",*pC,(unsigned char)(*pC),serverToClientTab[(*pC)],serverToClientTab[(*pC)]);
		(*pC)=serverToClientTab[(unsigned char)(*pC)];
		pC++;
	}
	_debug_leavetrace("translateToClient");
}


//============ translateToServer ============//

void KviTranslator::translateToServer(QString &szString)
{
	_debug_entertrace("translateToServer");
	if(m_bDefault)return;
	char *pC=szString.data();
//	debug("Translating string %s",pC);
	while(*pC){
//		debug("Char %c (%d) becomes %c (%d)",*pC,(unsigned char)(*pC),clientToServerTab[(*pC)],clientToServerTab[(*pC)]);
		(*pC)=clientToServerTab[(unsigned char)(*pC)];
		pC++;
	}
	_debug_leavetrace("translateToServer");
}



