//
//   File : kvi_sparser.cpp (/usr/cvs/kvirc/kvirc/kvi_sparser.cpp)
//   Last modified : Fri Nov 27 1998 20:40:26 by root@localhost.localdomain
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998-1999 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation ; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING. If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//

//#define _KVI_DEBUG_CLASS_NAME_ "KviServerParser"

#include "kvi_defs.h"
#include "kvi_macros.h"
#include "kvi_sparser.h"
#include "kvi_debug.h"
#include "kvi_frame.h"
#include "kvi_ctcp.h"
#include "kvi_mdi.h"
#include "kvi_child.h"
#include "kvi_global.h"
#include "kvi_event.h"
#include "kvi_uparser.h"
#include "kvi_app.h"
#include "kvi_numeric.h"
#include "kvi_usr.h"
#include "kvi_socket.h"
#include "kvi_status.h"
#include "kvi_chan.h"
#include "kvi_opt.h"
#include "kvi_link.h"
#include "kvi_listbox.h"
#include "kvi_query.h"
#include "kvi_send.h"
#include "kvi_input.h"
#include "kvi_token.h"
#include "kvi_uglobal.h"
#include "kvi_chlist.h"

handleCmdTable KviServerParser::cmdTab[]={
	{ "PRIVMSG" , &KviServerParser::handlePrivmsg },
	{ "PING"    , &KviServerParser::handlePing    },
	{ "JOIN"    , &KviServerParser::handleJoin    },
	{ "PART"    , &KviServerParser::handlePart    },
	{ "MODE"    , &KviServerParser::handleMode    },
	{ "KICK"    , &KviServerParser::handleKick    },
	{ "TOPIC"   , &KviServerParser::handleTopic   },
	{ "NICK"    , &KviServerParser::handleNick    },
	{ "QUIT"    , &KviServerParser::handleQuit    },
	{ "NOTICE"  , &KviServerParser::handleNotice  },
	{ "ERROR"   , &KviServerParser::handleError   },
	{ "INVITE"  , &KviServerParser::handleInvite  },
	{ 0         , 0                               }
	//Manca KILL; ???
};
//============ KviServerParser ============//
KviServerParser::KviServerParser(KviFrame *lpFrame)
{
	_debug_entertrace("KviServerParser");
	m_lpFrm   = lpFrame;
	m_lpCons  = lpFrame->m_lpConsole;
	m_lpMdi   = lpFrame->m_lpMdi;
	m_lpUsr   = lpFrame->m_lpUsr;
	m_lpOpt   = lpFrame->m_lpOpt;
	m_iFileRequests=0;
	_debug_leavetrace("KviServerParser");
}
//============ ~KviServerParser ============//
KviServerParser::~KviServerParser()
{
	_debug_entertrace("~KviServerParser");
	delete m_lpCTCP;
	_debug_leavetrace("~KviServerParser");
}
//============ setStatus ============//
void KviServerParser::setStatus(const char *szFmt,const char *szData)
{
	_debug_entertrace("setStatus");
	QString szStat;
	szStat.sprintf(szFmt,szData);
	m_lpFrm->setStatusText(szStat.data());
	_debug_leavetrace("setStatus");
}
//============ activate ============//
void KviServerParser::activate(KviIrcSocket *lpSock)
{
	_debug_entertrace("activate");
	m_lpSock  = lpSock;
	m_lpEventManager=m_lpFrm->m_lpEventManager;
	m_lpUserParser=m_lpFrm->m_lpUserParser;
	m_lpCTCP=new KviCTCP(m_lpFrm,this);
	_debug_leavetrace("activate");
}
//============ parseMessage ============//
void KviServerParser::parseMessage(QString &szMessage)
{
	_debug_entertrace("parseMessage");
	//Split the string in three parts...
	register const char *pC=szMessage.data();
	int len=szMessage.length();
	QString szPrefix(len); //It should be enough
	QString szCommand(len); //I Know that this is too long but... we want to be sure....
	QString szParams(len);
	char *pPnt=szPrefix.data();
	const char *pStart=0;
	if(!(pC))return;
	while(((*pC)==' '))pC++; //skip the spaces... (is it really necessary?)
	if((*pC == '\0')) return;
	//Ok we point to a non space...
	if((*pC)==':'){
		pC++;
		//Point to the first letter (or null)
		pStart=pC;
		while((*pC) && (*pC != ' '))*pPnt++ = *pC++; //copy the first word (prefix);
		if(!(*pC))return; //Broken message...(only a senseless prefix)...
		//debug("prefix copied");
		//We point to a space...
		//:prefix command
		// |     |
		//pStart pC
		++(*pPnt)='\0';
		szPrefix.resize((int)((pC-pStart)+1));
		while(((*pC)==' '))pC++; //skip the spaces...
		if((*pC == '\0')) return; //Fucked message.
	}
	//Ok we point to the first letter of the command...
	pStart=pC;
	pPnt=szCommand.data();
	while((*pC) && (*pC != ' '))*pPnt++ = *pC++; //copy the second(?) word (command);
	//We point to a something after the command...
	//:prefix command???????????????.....
	//        |      |
	//       pStart  pC
	++(*pPnt)='\0';
	szCommand.resize((int)((pC-pStart)+1));
	while(((*pC)==' '))pC++; //skip the spaces...
	if((*pC)){
		//we have params too;
		pStart=pC;
		pPnt=szParams.data();
		while((*pC))*pPnt++ = *pC++; //copy the rest...
		++(*pPnt)='\0';
		szParams.resize((int) ((pC-pStart)+1));
	} else {
		//no params
		szParams="";
	}
	///////////////////////// Splitting DONE
	//Check if it is a numeric msg
	bool bOk=false;
	int nCommand=szCommand.toInt(&bOk);
	if(bOk){
		if((nCommand>399) && (nCommand<600))parseNumericError(szPrefix,nCommand,szParams);
		else parseNumericReply(szPrefix,nCommand,szParams);
	} else parseLiteral(szPrefix,szCommand,szParams);
	_debug_leavetrace("parseMessage");
}
//============ topWin ============//
KviMdiChild * KviServerParser::topWin()
{
	_debug_entertrace("topWin");
	KviMdiChild *lpC=m_lpMdi->topChildInZOrder();
	return ( lpC ? lpC : m_lpCons );
	_debug_leavetrace("topWin");
}
//============ findWin ============//
KviMdiChild * KviServerParser::findWin(const char *szName)
{
	_debug_entertrace("findWin");
	KviMdiChild *lpC=m_lpMdi->findWindow(szName);
	return ( lpC ? lpC : m_lpCons );
	_debug_leavetrace("findWin");
}
//============ getChanWnd ============//
KviChanWnd * KviServerParser::getChanWnd(const char *szName)
{
	_debug_entertrace("getChanWnd");
	KviMdiChild *lpC=m_lpMdi->findWindow(szName);
	if(lpC){
		if(lpC->type()==KVI_WND_TYPE_CHAN)return ((KviChanWnd *)lpC);
		else return 0;
	} else return 0;
	_debug_leavetrace("getChanWnd");
}
//============ getQueryWnd ============//
KviQueryWnd * KviServerParser::getQueryWnd(const char *szName)
{
	_debug_entertrace("getQueryWnd");
	KviMdiChild *lpC=m_lpMdi->findWindow(szName);
	if(lpC){
		if(lpC->type()==KVI_WND_TYPE_QUERY)return ((KviQueryWnd *)lpC);
		else return 0;
	} else return 0;
	_debug_leavetrace("getQueryWnd");
}
//============ isMe ============//
bool KviServerParser::isMe(QString &szNick)
{
	_debug_entertrace("isMe");
	const char *pD=szNick.data();
	if(pD){
		if(((*pD)=='@')||((*pD)=='+')){
			pD++;
			if(!(*pD))return false;
		}
		return (!strcasecmp(pD,m_lpFrm->m_lpGlb->szNick.data()));
	}
	return false;
	_debug_leavetrace("isMe");
}
//============ skipWord ============//
const char * KviServerParser::skipWord(QString &szString)
{
	_debug_entertrace("skipWord");
	char *cr=szString.data();
	while((*cr) && (*cr != ' '))cr++;
	if(*cr){
		while((*cr) && (*cr==' '))cr++;
	}
	return cr;
	_debug_leavetrace("skipWord");
}
//============ skipUpToTrailing ===========//
const char * KviServerParser::skipUpToTrailing(QString &szString)
{
	_debug_entertrace("skipUpToTrailing");
	char *cr=szString.data();
	while((*cr) && ((*cr)!= ':'))cr++;
	if(*cr){
		while((*cr) && ((*cr)==':'))cr++;
	}
	return cr;
	_debug_leavetrace("skipUpToTrailing");
}
//============ parseNumericError ============//
void KviServerParser::parseNumericError(QString &,int nCommand,QString &szParams)
{
	_debug_entertrace("parseNumericError");
	switch(nCommand){
		case ERR_NONICKNAMEGIVEN:
		case ERR_ERRONEUSNICKNAME:
		case ERR_NICKNAMEINUSE:
		case ERR_NICKCOLLISION:
		case ERR_UNAVAILRESOURCE: //same as ERR_BANNICKCHANGE
		case ERR_NICKTOOFAST:
			handleNickError(nCommand,szParams);
			break;
		default:
			if(m_lpEventManager->lpEvent[KVI_Event_OnUnhandledServerError]->bEnabled){
				QString szPara;
				szPara.setNum(nCommand);
				szPara+=" ";
				szPara+=szParams.data();
				if(m_lpUserParser->executeEvent(topWin(),
						m_lpEventManager->lpEvent[KVI_Event_OnUnhandledServerError],
						szPara)!=KVI_EventErr_Halt)
					topWin()->doFmtOutput(KVI_OUT_RPLERR,i18n("[SERVER ERROR %d] %s"),nCommand,szParams.data());
			} else topWin()->doFmtOutput(KVI_OUT_RPLERR,i18n("[SERVER REPLY %d] %s"),nCommand,szParams.data());
			break;
	}
//#define ERR_NOSUCHNICK	     401
//#define ERR_NOSUCHSERVER     402
//#define ERR_NOSUCHCHANNEL    403
//#define ERR_CANNOTSENDTOCHAN 404
//#define ERR_TOOMANYCHANNELS  405
//#define ERR_WASNOSUCHNICK    406
//#define ERR_TOOMANYTARGETS   407
//#define ERR_NOORIGIN	     409

//#define ERR_NORECIPIENT	     411
//#define ERR_NOTEXTTOSEND     412
//#define ERR_NOTOPLEVEL	     413
//#define ERR_WILDTOPLEVEL     414

//#define ERR_QUERYTOOLONG     416	/* Undernet extension */

//#define ERR_UNKNOWNCOMMAND   421
//#define ERR_NOMOTD	     422
//#define ERR_NOADMININFO	     423
/*      ERR_FILEERROR	     424           removed from RFC1459 */

//#define ERR_BANNICKCHANGE    437	/* Undernet extension */
//#define ERR_NICKTOOFAST	     438	/* Undernet extension */
//#define ERR_TARGETTOOFAST    439	/* Undernet extension */

//#define ERR_USERNOTINCHANNEL 441
//#define ERR_NOTONCHANNEL     442
//#define ERR_USERONCHANNEL    443
/*      ERR_NOLOGIN  	     444           removed from RFC1459 */
/*      ERR_SUMMONDISABLED   445           removed from RFC1459 */
/*      ERR_USERSDISABLED    446	   removed from RFC1459 */

//#define ERR_NOTREGISTERED    451

//#define ERR_NEEDMOREPARAMS   461
//#define ERR_ALREADYREGISTRED 462
//#define ERR_NOPERMFORHOST    463
//#define ERR_PASSWDMISMATCH   464
//#define ERR_YOUREBANNEDCREEP 465
//#define ERR_YOUWILLBEBANNED  466
//#define ERR_KEYSET	     467	/* Undernet extension */
//#define ERR_INVALIDUSERNAME  468	/* Undernet extension */

//#define ERR_CHANNELISFULL    471
//#define ERR_UNKNOWNMODE	     472
//#define ERR_INVITEONLYCHAN   473
//#define ERR_BANNEDFROMCHAN   474
//#define ERR_BADCHANNELKEY    475
//#define ERR_BADCHANMASK	     476	/* Undernet extension */
//#define ERR_MODELESS         477	/* Extension to RFC1459 */
//#define ERR_BANLISTFULL	     478	/* Undernet extension */

//#define ERR_NOPRIVILEGES     481
//#define ERR_CHANOPRIVSNEEDED 482
//#define ERR_CANTKILLSERVER   483
//#define ERR_ISCHANSERVICE    484	/* Undernet extension */

//#define ERR_NOOPERHOST	     491

//#define ERR_UMODEUNKNOWNFLAG 501
//#define ERR_USERSDONTMATCH   502

//#define ERR_SILELISTFULL     511	/* Undernet extension */

//#define ERR_NOSUCHGLINE	     512	/* Undernet extension */
//#define ERR_BADPING	     513	/* Undernet extension */
	_debug_leavetrace("parseNumericError");
}
//============ handleNickError ============//
void KviServerParser::handleNickError(int nCommand,QString &szParams)
{
	_debug_entertrace("handleNickError");
	if(m_lpFrm->m_bLoggedIn){	//Notify the error
		switch(nCommand){
			case ERR_NONICKNAMEGIVEN:
				topWin()->doFmtOutput(KVI_OUT_RPLERR,i18n("[NICKNAME ERROR %d] No nickname given."),nCommand);
				break;
			default:
				topWin()->doFmtOutput(KVI_OUT_RPLERR,i18n("[NICKNAME ERROR %d] %s"),nCommand,skipWord(szParams));
				break;
		}
	} else {					//Change the nick
		if((!(m_lpFrm->m_bUsingAltNick))&&(!(m_lpUsr->szAltNick.isEmpty()))){
			m_lpSock->sendFmtData("NICK %s",m_lpUsr->szAltNick.data());
			m_lpFrm->m_lpGlb->szNick=m_lpUsr->szAltNick.data();
			topWin()->doFmtOutput(KVI_OUT_INTERNAL,i18n("Login nickname refused by server, trying alternate nick %s"),
					m_lpUsr->szAltNick.data());
			m_lpFrm->m_bUsingAltNick=true;
		} else {
			QString szRandom=m_lpUsr->szNick.copy();
			if(szRandom[0]>'x' || szRandom[0]<'a')szRandom[0]='a';
			else szRandom[0]=(szRandom[0]+1);
			m_lpSock->sendFmtData("NICK %s",szRandom.data());
			m_lpFrm->m_lpGlb->szNick=szRandom.data();
			topWin()->doFmtOutput(KVI_OUT_INTERNAL,i18n("Login nickname refused by server, trying random alternate nick %s"),
					szRandom.data());
		}
	}
	_debug_leavetrace("handleNickError");
}
//============ parseNumericReply ============//
void KviServerParser::parseNumericReply(QString &szPrefix,int nCommand,QString &szParams)
{
	_debug_entertrace("parseNumericReply");
	switch(nCommand){
		case RPL_MOTDSTART:
		case RPL_MOTD:
		case RPL_MOTD2:
		case RPL_ENDOFMOTD:
			m_lpFrm->m_bLoggedIn=true;
			m_lpCons->doOutput(KVI_OUT_MOTD,skipUpToTrailing(szParams));
			break;
		case RPL_NAMREPLY:
		case RPL_ENDOFNAMES:
			handleNameReply(nCommand,szParams);
			break;
		case RPL_NOTOPIC:
		case RPL_TOPIC:
		case RPL_TOPICWHOTIME:
			handleTopicReply(nCommand,szParams);
			break;
		case RPL_CHANNELMODEIS:
			handleModeReply(szPrefix,szParams);
			break;
		case RPL_USERHOST:
			handleUserHost(szParams);
			break;
		case RPL_UNAWAY: // added by RAD Kade 1  
			topWin()->doOutput(KVI_OUT_BACK,skipUpToTrailing(szParams)); 
			setStatus(i18n("You are now %s"),i18n("back.")); 
			if(m_lpEventManager->lpEvent[KVI_Event_OnMeAwayChange]->bEnabled){
				QString szPara="";
				m_lpUserParser->executeEvent(m_lpCons,m_lpEventManager->lpEvent[KVI_Event_OnMeAwayChange],szPara);
			}
			break; 
		case RPL_NOWAWAY: // added by RAD Kade 1 
			topWin()->doOutput(KVI_OUT_AWAY,skipUpToTrailing(szParams)); 
			setStatus(i18n("You are now %s"),i18n("away.")); 
			if(m_lpEventManager->lpEvent[KVI_Event_OnMeAwayChange]->bEnabled){
				QString szPara=skipUpToTrailing(szParams);
				m_lpUserParser->executeEvent(m_lpCons,m_lpEventManager->lpEvent[KVI_Event_OnMeAwayChange],szPara);
			}
			break; 
		case RPL_WHOISUSER:
		case RPL_WHOISSERVER:
		case RPL_WHOISOPERATOR:
		case RPL_WHOISIDLE:
		case RPL_ENDOFWHOIS:
		case RPL_WHOISCHANNELS:
		case RPL_ENDOFWHOWAS:
		case RPL_WHOWASUSER:
		case RPL_AWAY:
			handleWhois(szPrefix,nCommand,szParams);
			break;
		case RPL_YOURHOST:
		case RPL_CREATED:
		case RPL_MYINFO:
		case RPL_MAP:			//I dunno...I have never seen the following ones...
		case RPL_MAPMORE:		//
		case RPL_MAPEND:		//
		case RPL_SNOMASK:		//
		case RPL_STATMEMTOT:	//
		case RPL_STATMEM:		//'Till here...
		case RPL_INFOSTART:
		case RPL_INFO:
		case RPL_ENDOFINFO:
		case RPL_YOUREOPER:		//This one is here just because it uses the same syntax...
		case RPL_LUSERCLIENT:
		case RPL_LUSERME:
		case RPL_LUSERLOCAL:
		case RPL_LUSERGLOBAL:
			m_lpFrm->m_bLoggedIn=true;
			m_lpCons->doOutput(KVI_OUT_INFO,skipUpToTrailing(szParams));
			break;
		case RPL_LUSEROP:
		case RPL_LUSERUNKNOWN:
		case RPL_LUSERCHANNELS:
			m_lpFrm->m_bLoggedIn=true;
			m_lpCons->doOutput(KVI_OUT_INFO,skipWord(szParams));
			break;
		case RPL_WELCOME:
			m_lpFrm->m_bLoggedIn=true;
			{
				//extract the last token from the Welcome message.
				//it should be our mask
				szParams=szParams.stripWhiteSpace();
				const char *theP=strrchr(szParams.data(),' ');
				m_lpFrm->m_lpGlb->szMaskFromServer=QString(theP);
				int idx=m_lpFrm->m_lpGlb->szMaskFromServer.find('!');
				if(idx==-1)m_lpCons->doOutput(KVI_OUT_ERROR,
							      i18n("The server missed to send us our mask."
								   "\nThe $_mymask and $_myaddress identifers may contain incorrect values."
								   "\nThis will be fixed when you join the first channel."));
 				else {
					KviNewNick nk(m_lpFrm->m_lpGlb->szMaskFromServer);
					if(nk.szNick != m_lpFrm->m_lpGlb->szNick){
						m_lpCons->doFmtOutput(KVI_OUT_INTERNAL,i18n("The server changed our nick to %s"),nk.szNick.data());
						m_lpFrm->m_lpGlb->szNick=nk.szNick.copy();
					}
					m_lpFrm->m_lpGlb->szHostName=nk.szAddress.copy();
				}
			}
			m_lpCons->doOutput(KVI_OUT_INFO,skipUpToTrailing(szParams));
			if(m_lpEventManager->lpEvent[KVI_Event_OnLoggedIn]->bEnabled){
				QString szPara=m_lpFrm->m_lpGlb->szNick+" "+m_lpFrm->m_lpGlb->szUserName+" "+m_lpFrm->m_lpGlb->szRealName;
				m_lpUserParser->executeEvent(m_lpCons,m_lpEventManager->lpEvent[KVI_Event_OnLoggedIn],szPara);
			}
			//set the user modes
			{
				QString szMode="";
				if(m_lpUsr->bool_wallops)szMode+='w';
				if(m_lpUsr->bool_servernotices)szMode+='s';
				if(m_lpUsr->bool_invisible)szMode+='i';
				if(!szMode.isEmpty())m_lpSock->sendFmtData("MODE %s %s",m_lpFrm->m_lpGlb->szNick.data(),szMode.data());
			}
			m_lpFrm->startNotifyList();
			break;
		case RPL_BANLIST:
		case RPL_ENDOFBANLIST:
			handleBanList(nCommand,szParams);
			break;
		case RPL_ISON:
			handleIsOn(szParams);
			break;
		case RPL_CREATIONTIME:
			handleCreationTime(szParams);
			break;
		case RPL_LINKS:
		case RPL_ENDOFLINKS:
			handleLinks(nCommand,szParams);
			break;
		case RPL_LISTSTART:
		case RPL_LIST:
		case RPL_LISTEND:
			handleChanList(nCommand,szParams);
			break;
		case RPL_CHANURL:
			handleChanUrl(szParams);
			break;
		default:
			if(m_lpEventManager->lpEvent[KVI_Event_OnUnhandledServerReply]->bEnabled){
				QString szPara;
				szPara.setNum(nCommand);
				szPara+=" ";
				szPara+=szParams.data();
				if(m_lpUserParser->executeEvent(topWin(),
						m_lpEventManager->lpEvent[KVI_Event_OnUnhandledServerReply],
						szPara)!=KVI_EventErr_Halt)
					topWin()->doFmtOutput(KVI_OUT_SERVER,i18n("[SERVER REPLY %d] %s"),nCommand,szParams.data());
			} else topWin()->doFmtOutput(KVI_OUT_SERVER,i18n("[SERVER REPLY %d] %s"),nCommand,szParams.data());
			break;
	}
//#define RPL_NONE	     300
//#define RPL_TEXT	     304
//#define RPL_USERIP	     307	/* Undernet extension */

//#define RPL_LISTUSAGE	     334	/* Undernet extension */
//#define RPL_INVITING	     341
///*      RPL_SUMMONING        342	   removed from RFC1459 */
//#define RPL_VERSION	     351
//
//#define RPL_WHOREPLY	     352	/* See also RPL_ENDOFWHO */
//#define RPL_WHOSPCRPL        354	/* Undernet extension,
//#define RPL_KILLDONE	     361
//#define RPL_CLOSING	     362
//#define RPL_CLOSEEND	     363

//#define RPL_ENDOFWHOWAS	     369
//#define RPL_REHASHING	     382
//#define RPL_MYPORTIS	     384
//#define RPL_NOTOPERANYMORE   385	/* Extension to RFC1459 */
//#define RPL_TIME	     391
//#define RPL_TRACELINK	     200
//#define RPL_TRACECONNECTING  201
//#define RPL_TRACEHANDSHAKE   202
//#define RPL_TRACEUNKNOWN     203
//#define RPL_TRACEOPERATOR    204
//#define RPL_TRACEUSER	     205
//#define RPL_TRACESERVER	     206
//#define RPL_TRACENEWTYPE     208
//#define RPL_TRACECLASS	     209
//#define RPL_STATSLINKINFO    211
//#define RPL_STATSCOMMANDS    212
//#define RPL_STATSCLINE	     213
//#define RPL_STATSNLINE	     214
//#define RPL_STATSILINE	     215
//#define RPL_STATSKLINE	     216
//#define RPL_STATSPLINE	     217	/* Undernet extenstion */
//#define RPL_STATSYLINE	     218
//#define RPL_ENDOFSTATS	     219	/* See also RPL_STATSDLINE */
//#define RPL_UMODEIS	     221
//#define RPL_SERVICEINFO	     231
//#define RPL_ENDOFSERVICES    232
//#define RPL_SERVICE	     233
//#define RPL_SERVLIST	     234
//#define RPL_SERVLISTEND	     235
//#define RPL_STATSLLINE	     241
//#define RPL_STATSUPTIME	     242
//#define RPL_STATSOLINE	     243
//#define RPL_STATSHLINE	     244
///*      RPL_STATSSLINE	     245	   Reserved */
//#define RPL_STATSTLINE	     246	/* Undernet extension */
//#define RPL_STATSGLINE	     247	/* Undernet extension */
//#define RPL_STATSULINE	     248	/* Undernet extension */
//#define RPL_STATSDEBUG	     249	/* Extension to RFC1459 */
//#define RPL_STATSCONN	     250	/* Undernet extension */
//#define RPL_ADMINME	     256
//#define RPL_ADMINLOC1	     257
//#define RPL_ADMINLOC2	     258
//#define RPL_ADMINEMAIL	     259
//#define RPL_TRACELOG	     261
//#define RPL_TRACEPING	     262	/* Extension to RFC1459 */
//#define RPL_SILELIST	     271	/* Undernet extension */
//#define RPL_ENDOFSILELIST    272	/* Undernet extension */
//#define RPL_STATSDLINE	     275	/* Undernet extension */
//#define RPL_GLIST	     280	/* Undernet extension */
//#define RPL_ENDOFGLIST	     281	/* Undernet extension */
	_debug_leavetrace("parseNumericReply");
}
//============ handleChanUrl ============//
void KviServerParser::handleChanUrl(QString &szParams)
{
	_debug_entertrace("handleChanUrl");
	KviIrcToken par(szParams);
	QString szChan=par.nextWord(); //remove our nick
	szChan=par.nextWord();
	par.stripUpToTrailing();
	getChanWnd(szChan.data())->doFmtOutput(KVI_OUT_TOPIC,i18n("Channel registered URL is : %s"),par.szString.data());
	_debug_leavetrace("handleChanUrl");
}
//============ handleInvite ============//
void KviServerParser::handleInvite(QString &szPrefix,QString &szParams)
{
	_debug_entertrace("handleInvite");
	KviIrcToken par(szParams);
	KviNewNick inviter(szPrefix);
	QString szChan=par.nextChannel();
	topWin()->doFmtOutput(KVI_OUT_INVITE,i18n("%s invites you to join channel %s"),inviter.szNick.data(),szChan.data());
	if(m_lpOpt->bAutoJoinOnInvite){
		m_lpCons->doFmtOutput(KVI_OUT_INVITE,i18n("Invited to %s by %s : autojoining channel."),szChan.data(),inviter.szNick.data());
		m_lpSock->sendFmtData("JOIN %s",szChan.data());
	}
	_debug_leavetrace("handleInvite");
}
//============ handleLinks ============//
void KviServerParser::handleLinks(int nCommand,QString &szParams)
{
	_debug_entertrace("handleLinks");
	KviIrcToken par(szParams);
	QString myNick=par.nextWord();
	bool doit=false;
	if(m_lpFrm->m_lpLinksWindow){
		if(!m_lpFrm->m_lpLinksWindow->m_bDone)doit=true;
	}
	if(doit){
		if(nCommand==RPL_LINKS){
			QString szHost=par.nextWord();
			QString szParenthost=par.nextWord();
			par.stripUpToTrailing();
			m_lpFrm->m_lpLinksWindow->insertLink(szHost.data(),szParenthost.data(),par.szString.data());
		} else m_lpFrm->m_lpLinksWindow->endOfLinks();
	} else {
		if(nCommand==RPL_LINKS){
			QString szHost=par.nextWord();
			QString szParenthost=par.nextWord();
			par.stripUpToTrailing();
			m_lpCons->doFmtOutput(KVI_OUT_SERVER,
				i18n("[LINKS REPLY] : host : %s linked to %s [ %s ]"),
				szHost.data(),szParenthost.data(),par.szString.data()); 	
		}
	}
	_debug_leavetrace("handleLinks");
}
//============ handleChanList ============//
void KviServerParser::handleChanList(int nCommand,QString &szParams)
{
	_debug_entertrace("handleChanList");
	KviIrcToken par(szParams);
	QString myNick=par.nextWord();
	if(m_lpFrm->m_lpChanListWindow){
		if(!m_lpFrm->m_lpChanListWindow->m_bDone){
			switch(nCommand){
				case RPL_LIST:
					m_lpFrm->m_lpChanListWindow->addChanEntry(par.szString);
					break;
				case RPL_LISTSTART: //Ignore
					m_lpFrm->m_lpChanListWindow->clear();
					break;
				case RPL_LISTEND:
					m_lpFrm->m_lpChanListWindow->endOfChannelList();
					break;
			}
			return;
		} else m_lpCons->doFmtOutput(KVI_OUT_SERVER,i18n("[LIST REPLY] : %s"),par.szString.data());
	} else {
		m_lpFrm->createChanListWindow(false);
		handleChanList(nCommand,szParams);  //and recall
	}
	_debug_leavetrace("handleChanList");
}
//============ handleIsOn ============//
void KviServerParser::handleIsOn(QString &szParams)
{
	_debug_entertrace("handleIsOn");
	KviIrcToken par(szParams);
	par.stripUpToTrailing();
	
	QStrIList lst(true);
	lst.setAutoDelete(true);
	while(!par.szString.isEmpty()){
		QString szNewNick=par.nextWord();
		lst.append(szNewNick.data());
	}
	KviIsonStruct *lpI=0;
	for(lpI=m_lpFrm->m_lpIsOnList->first();lpI;lpI=m_lpFrm->m_lpIsOnList->next()){
		int idx=lst.find(lpI->szNick.data());
		if(idx != -1){
			lst.remove(idx);
			if(!(lpI->isOn)){
				if(m_lpOpt->bNotifyListInActive)topWin()->doFmtOutput(KVI_OUT_WHO,i18n("%s is on irc"),lpI->szNick.data());
				else m_lpCons->doFmtOutput(KVI_OUT_WHO,i18n("%s is on irc"),lpI->szNick.data());
				lpI->isOn=true;
				KviNewNick nk(lpI->szNick.data());
				m_lpCons->m_lpNotifyList->nickJoin(nk);
				if(m_lpOpt->bEnableActionsOnNotify){
					if(lpI->lpUserStruct->bEnableActionOnNotify){
						KviEventStruct *lpE=new KviEventStruct;
						lpE->szName="Event_OnPersonalNotify";
						lpE->bEnabled=true;
						lpE->szBuffer=lpI->lpUserStruct->szActionOnNotify.copy();
						QString szParam=lpI->szNick.copy();
						m_lpFrm->m_lpUserParser->executeEvent(m_lpCons,lpE,szParam);
						delete lpE;
					}
				}
			}
		} else {
			if(lpI->isOn){
				if(m_lpOpt->bNotifyListInActive)topWin()->doFmtOutput(KVI_OUT_WHO,i18n("%s has left irc"),lpI->szNick.data());
				else m_lpCons->doFmtOutput(KVI_OUT_WHO,i18n("%s has left irc"),lpI->szNick.data());
				lpI->isOn=false;
				m_lpCons->m_lpNotifyList->nickPart(lpI->szNick.data());
			}
		}
	}
	while(!lst.isEmpty())lst.removeLast();
	m_lpFrm->m_bGotLastReply=true; //next time recheck
	uint msecs=m_lpFrm->m_lastNotifyCheck.msecsTo(QTime::currentTime());
	uint secs=msecs/1000;
	msecs=msecs%1000;
	if(secs > 1)m_lpCons->doFmtOutput(KVI_OUT_TIME,i18n("Lag from server : %u.%.3u seconds"),secs,msecs);
	QString szLag;
	szLag.setNum(secs);
	szLag+=" (sec.) ";
	QString szMsec;
	szMsec.setNum(msecs);
	szLag+=szMsec.data();
	szLag+=" (~msec.)";
	if(secs < 15)setStatus(i18n("Lag from server : %s"),szLag.data());
	else setStatus(i18n("Horrible lag : %s. Consider changing server..."),szLag.data());
	if(secs>((uint)m_lpOpt->iNotifyTimer))m_lpFrm->checkNotifyList(); //if not , wait for the timer to do it
	_debug_leavetrace("handleIsOn");
}
//============ handleCreationTime ============//
void KviServerParser::handleCreationTime(QString &szParams)
{
	_debug_entertrace("handleCreationTime");
	KviIrcToken par(szParams);
	QString channel=par.nextChannel();
	QString szTime=par.szString.stripWhiteSpace();
	KviChanWnd *lpWnd=getChanWnd(channel.data());
	if(lpWnd){
		bool bOK;
		uint crTime=szTime.toUInt(&bOK);
		if(bOK){
			QDateTime dt;
			dt.setTime_t(crTime);
			szTime=dt.toString();
			lpWnd->doFmtOutput(KVI_OUT_TIME,i18n("Channel was created on %s"),szTime.data());
		}
	}
	_debug_leavetrace("handleCreationTime");
}
//============ handleTopicReply ============//
void KviServerParser::handleTopicReply(int nCommand,QString &szParams)
{
	_debug_entertrace("handleTopicReply");
	KviIrcToken par(szParams);
	QString channel=par.nextChannel();
	par.stripUpToTrailing(); //contains the topic
	KviChanWnd *lpWnd=getChanWnd(channel.data());
	if(!lpWnd){
		topWin()->doFmtOutput(KVI_OUT_TOPIC,i18n("[TOPIC REPLY %d] : %s : %s"),nCommand,channel.data(),par.szString.data());
	} else {
		if(nCommand==RPL_NOTOPIC){
			lpWnd->setTopic(i18n("[ No topic is set ]"));
			return;
		}
		if(nCommand==RPL_TOPIC)lpWnd->setTopic(par.szString.data());
		else {
			QString topSetBy=par.nextWord();
			QString szTopTime=par.szString.stripWhiteSpace();
			bool bOK;
			uint topicTime=szTopTime.toUInt(&bOK);
			if(bOK){
				QDateTime dt;
				dt.setTime_t(topicTime);
				szTopTime=dt.toString();
				lpWnd->doFmtOutput(KVI_OUT_TOPIC,i18n("Topic was set by %s on %s"),topSetBy.data(),szTopTime.data());
			} else lpWnd->doFmtOutput(KVI_OUT_TOPIC,i18n("Topic was set by %s"),topSetBy.data());
		}
	}
	_debug_leavetrace("handleTopicReply");
}
//============ handleModeReply ============//
void KviServerParser::handleModeReply(QString &szPrefix,QString &szParams)
{
	_debug_entertrace("handleModeReply");
	KviIrcToken par(szParams);
	par.nextWord();
	handleMode(szPrefix,par.szString);
	_debug_leavetrace("handleModeReply");
}
//============ handleUserHost ============//
void KviServerParser::handleUserHost(QString &szParams)
{
	_debug_entertrace("handleUserHost");
	KviIrcToken par(szParams);
	par.stripUpToTrailing();
	QString szUH;
	QString szNK;
	QString szUN=KVI_STR_NULL;
	while(!par.szString.isEmpty()){
		szUH=par.nextWord();
		if(!szUH.isEmpty()){
			int idx=szUH.find('=');
			if(idx > 0){
				szNK=szUH.left(idx);
				szUH.remove(0,idx+1);
				if(szNK[idx-1]=='*')szNK.remove(idx-1,1);
				if(!szUH.isEmpty()){
					if((szUH[0]=='+')||(szUH[0]=='-'))szUH.remove(0,1);
					idx=szUH.find('@');
					if(idx != -1){
						szUN=szUH.left(idx);
						szUH.remove(0,idx+1);
					}
					KviNewNick nk;
					nk.szNick=szNK.data();
					nk.szAddress=szUH.data();
					nk.szUserName=szUN.data();
					m_lpFrm->m_lpGlobalUserList->updateUser(nk);
				}		
			}
		}
	}
	_debug_leavetrace("handleUserHost");
}
//============ handleBanList ============//
void KviServerParser::handleBanList(int nCommand,QString &szParams)
{
	_debug_entertrace("handleBanList");
	KviIrcToken par(szParams);
	QString channel=par.nextChannel();
	KviChanWnd *lpChan=getChanWnd(channel.data());
	if(!lpChan){
		topWin()->doFmtOutput(KVI_OUT_SERVER,i18n("[BANLIST REPLY %d] : %s : %s"),nCommand,channel.data(),szParams.data());
		return;
	}
	if(nCommand==RPL_BANLIST){
		// Fixed by RAD Kade 1
		if(!lpChan->m_bHasBanList)lpChan->changeBanList(par.nextWord(),par.nextWord(),true); 	
		else lpChan->doFmtOutput(KVI_OUT_SERVER,i18n("[BANLIST REPLY %d] : %s : %s"),nCommand,channel.data(),szParams.data());		
	} else lpChan->m_bHasBanList=true;
	_debug_leavetrace("handleBanList");
}
//============ handleNameReply ============//
void KviServerParser::handleNameReply(int nCommand,QString &szParams)
{
	_debug_entertrace("handleNameReply");
	KviIrcToken par(szParams);
	QString channel=par.nextChannel();

	KviChanWnd *lpChan=getChanWnd(channel.data());
	if(!lpChan){
		topWin()->doFmtOutput(KVI_OUT_SERVER,i18n("[NAMES REPLY %d] : %s : %s"),nCommand,channel.data(),szParams.data());
	//	m_lpCons->doFmtOutput(KVI_OUT_ERROR,i18n("[INTERNAL ERROR (NAME REPLY PARSING)] : Can't find channel %s"),channel.data());
		return;
	}
	if(nCommand==RPL_ENDOFNAMES){
		if(lpChan->m_bHasAllNames)lpChan->doFmtOutput(KVI_OUT_SERVER,i18n("[NAMES_REPLY %d] : %s : End of names list"),nCommand,channel.data());
		else {
			lpChan->m_bHasAllNames=true;
			if(m_lpEventManager->lpEvent[KVI_Event_OnMeJoin]->bEnabled){
				QString szPara=channel.data();
				m_lpUserParser->executeEvent(lpChan,m_lpEventManager->lpEvent[KVI_Event_OnMeJoin],szPara);
			}
			if(lpChan->m_lpListBox->m_iTotal==1){ 	//useful on efnet (see irc.nijenrode.nl)
										//that don't send our nick in names reply
										//so we need to op ourself
				if(!lpChan->m_bIsMeOp){
					lpChan->nickOp(m_lpFrm->m_lpGlb->szNick.data(),true);
					lpChan->m_bIsMeOp=true;
					lpChan->setLabelsPalette();
				}
			}
			if(m_lpOpt->bEnableUserListUpdate){
				lpChan->startUpdateUserList();
				lpChan->doOutput(KVI_OUT_INTERNAL,i18n("Starting compilation of the internal users list"));
			}
		}
	} else {
		par.stripUpToTrailing();
		if(lpChan->m_bHasAllNames)lpChan->doFmtOutput(KVI_OUT_SERVER,i18n("[NAMES REPLY %d] : %s : %s"),nCommand,channel.data(),szParams.data());
		else {
			QString szNick;
			lpChan->m_bAutoUpdateUsers=false;
			lpChan->m_lpListBox->m_bAutoUpdate=false;
			while(!par.szString.isEmpty()){
				szNick=par.nextWord();
				if(!szNick.isEmpty()){
					if(!isMe(szNick))lpChan->nickJoin(szNick);
				}
			}
			lpChan->m_bAutoUpdateUsers=true;
			lpChan->m_lpListBox->m_bAutoUpdate=true;
			lpChan->m_lpListBox->repaint();
			lpChan->updateUsersLabel();
			//Now we have a list of names here...
		}	
	}
	_debug_leavetrace("handleNameReply");
}
//============ parseLiteral ============//
void KviServerParser::parseLiteral(QString &szPrefix,QString &szCommand,QString &szParams)
{
	_debug_entertrace("parseLiteral");
	QString szCmd(szCommand.upper());
	if(szCommand.isEmpty())return;
	for (int i=0;cmdTab[i].szName!=0;i++){
		if (!strcmp(szCommand.data(),cmdTab[i].szName)){
	 	   (this->*(cmdTab[i].proc))(szPrefix,szParams);
			return;
		}
	}
	topWin()->doFmtOutput(KVI_OUT_SERVER,"[%s][%s] %s",szPrefix.data(),szCommand.data(),szParams.data()); 
	_debug_leavetrace("parseLiteral");
}
//============ handlePing ============//
void KviServerParser::handlePing(QString &szPrefix,QString &szParams)
{
	_debug_entertrace("handlePing");
	if(szPrefix.isEmpty())szPrefix=m_lpFrm->m_lpGlb->szServerName;
	m_lpCons->doFmtOutput(KVI_OUT_PING,i18n("Ping request from %s [PING %s] ... PONG!"),szPrefix.data(),szParams.data());
	m_lpSock->sendFmtData("PONG %s",szParams.data());
	setStatus(i18n("Received PING from %s"),szPrefix.data());
	_debug_leavetrace("handlePing");
}
//============ handleError ============//
void KviServerParser::handleError(QString &szPrefix,QString &szParams)
{
	_debug_entertrace("handleError");
	if(szPrefix.isEmpty())szPrefix=m_lpFrm->m_lpGlb->szServerName;
	m_lpCons->doFmtOutput(KVI_OUT_RPLERR,i18n("[ERROR from %s] : %s"),szPrefix.data(),szParams.data());
	setStatus(i18n("Received ERROR from %s"),szPrefix.data());
	_debug_leavetrace("handleError");
}
//============ handleJoin ============//
void KviServerParser::handleJoin(QString &szPrefix,QString &szParams)
{
	_debug_entertrace("handleJoin");
	KviNewNick joiner(szPrefix);
	QString channel(skipUpToTrailing(szParams));
	//
	//check for extended join syntax...
	//Nick!User@host JOIN :#channel\x07[o|v]
	//                     012345678    9 9
	const char *pExtended=channel.data();
	char cExtMode=0;
	while((*pExtended)&&(*pExtended != 0x07))pExtended++;
	if(*pExtended){
		pExtended++;
		//get the o or v flag (if any)
		cExtMode=(*pExtended); //may be 0 ???
		channel.resize(pExtended-channel.data());
	}
	QString szMask;
	joiner.mask(szMask,1);
	if(isMe(joiner.szNick)){
		joiner.mask(m_lpFrm->m_lpGlb->szMaskFromServer,0);
		m_lpFrm->m_lpGlb->szHostName=joiner.szAddress.data();
		KviChanWnd * lpChan=m_lpFrm->createChanWnd(channel.data());	
		lpChan->nickJoin(joiner);
		m_lpSock->sendFmtData("MODE %s",channel.data());
		lpChan->doFmtOutput(KVI_OUT_JOIN,i18n("You have joined %s [you're known as %s!%s]"),channel.data(),
					joiner.szNick.data(),szMask.data());
		if(cExtMode){
			//the user mode is passed here
			QString szWhoSetMode=m_lpFrm->m_lpGlb->szServerName.data();
			szWhoSetMode+=":new.channel.service";
			KviNewNick kN(szWhoSetMode);
			handleChanUserMode(lpChan,kN,cExtMode,true,joiner.szNick);
		}
		setStatus(i18n("Joined %s"),channel.data());
	} else {
		KviChanWnd *lpChan=getChanWnd(channel.data());
		if(lpChan){
			lpChan->nickJoin(joiner);
			if(m_lpEventManager->lpEvent[KVI_Event_OnJoin]->bEnabled){
				QString szPara=channel+" "+joiner.szNick+" "+joiner.szUserName+" "+joiner.szAddress;
				m_lpUserParser->executeEvent(lpChan,m_lpEventManager->lpEvent[KVI_Event_OnJoin],szPara);
			}
			if(m_lpOpt->bEnableActionsOnJoin){
				KviUserStruct *lpU=m_lpFrm->findUserStruct(joiner.szNick,szMask);
				if(lpU){
					if(lpU->bEnableActionOnJoin){
						KviEventStruct *lpE=new KviEventStruct;
						lpE->szName="Event_OnPersonalJoin";
						lpE->bEnabled=true;
						lpE->szBuffer=lpU->szActionOnJoin.copy();
						QString szParam=channel.data();
						szParam+=' ';
						szParam+=lpU->szNick;
						szParam+=' ';
						szParam+=lpU->szNick;
						szParam+='!';
						szParam+=szMask;
						m_lpFrm->m_lpUserParser->executeEvent(m_lpCons,lpE,szParam);
						delete lpE;
					}
				}
			}
			lpChan->doFmtOutput(KVI_OUT_JOIN,i18n("%s [%s] has joined %s"),joiner.szNick.data(),szMask.data(),channel.data());
			if(cExtMode){
				//the user mode is passed here
				QString szWhoSetMode=m_lpFrm->m_lpGlb->szServerName.data();
				szWhoSetMode+=":resync.service";
				KviNewNick kN(szWhoSetMode);
				handleChanUserMode(lpChan,kN,cExtMode,true,joiner.szNick);
			}
			if(m_lpOpt->bCloneScanOnJoin)lpChan->doSingleCloneScan(&joiner);	
		} else {
			m_lpCons->doFmtOutput(KVI_OUT_ERROR,i18n("[INTERNAL ERROR] : JOIN message from [%s] to an inexisting channel %s"),
							joiner.szNick.data(),channel.data());
		}
	}	
	_debug_leavetrace("handleJoin");
}
//============ handlePart ============//
void KviServerParser::handlePart(QString &szPrefix,QString &szParams)
{
	_debug_entertrace("handlePart");
	KviNewNick parter(szPrefix);
	KviIrcToken par(szParams);
	QString channel=par.nextChannel();
	par.stripUpToTrailing();
	if(par.szString.isEmpty())par.szString="Signoff";
	if(isMe(parter.szNick)){
		m_lpMdi->removeWindow(channel.data());
		m_lpCons->doFmtOutput(KVI_OUT_PART,i18n("You have left channel %s [%s]"),channel.data(),par.szString.data());
		if(m_lpEventManager->lpEvent[KVI_Event_OnMePart]->bEnabled){
			QString szPara=channel.copy();
			m_lpUserParser->executeEvent(m_lpCons,m_lpEventManager->lpEvent[KVI_Event_OnMePart],szPara);
		}	
		setStatus(i18n("Left %s"),channel.data());
		_debug_leavetrace("handlePart...chan window removed");
		return;
	}
	KviChanWnd *lpWnd=getChanWnd(channel.data());
	if(!lpWnd){
		m_lpCons->doFmtOutput(KVI_OUT_ERROR,i18n("[INTERNAL ERROR (PART PARSING)] : Can't find channel %s"),channel.data());
		return;
	} else {
		if(lpWnd->getNick(parter))lpWnd->nickPart(parter);
		QString szmask;
		parter.mask(szmask,1);
		if(m_lpEventManager->lpEvent[KVI_Event_OnPart]->bEnabled){
			QString szPara=channel+" "+parter.szNick+" "+parter.szUserName+" "+parter.szAddress;
			m_lpUserParser->executeEvent(lpWnd,m_lpEventManager->lpEvent[KVI_Event_OnPart],szPara);
		}
		lpWnd->doFmtOutput(KVI_OUT_PART,"%s [%s] has left %s [%s]",parter.szNick.data()
				,szmask.data(),channel.data(),par.szString.data());
	}
	_debug_leavetrace("handlePart");
}
//============ handleTopic ============//
void KviServerParser::handleTopic(QString &szPrefix,QString &szParams)
{
	_debug_entertrace("handleTopic");
	KviNewNick topiker(szPrefix); //nick that changes topic
	KviIrcToken par(szParams);
	QString channel=par.nextChannel();
	par.stripUpToTrailing();
	KviChanWnd *lpWnd=getChanWnd(channel.data());
	if(!lpWnd)m_lpCons->doFmtOutput(KVI_OUT_ERROR,i18n("[INTERNAL ERROR (TOPIC PARSING)] Can't find channel %s"),channel.data());
	else {	
		QString szmask;
		topiker.mask(szmask,1);
		if(par.szString.isEmpty()){
			lpWnd->doFmtOutput(KVI_OUT_TOPIC,i18n("%s [%s] has unset the topic"),topiker.szNick.data(),szmask.data()); 
			lpWnd->setTopic(i18n("[ No topic is set ]")); 
		} else { 
			lpWnd->doFmtOutput(KVI_OUT_TOPIC,i18n("%s [%s] changes topic to %s"),topiker.szNick.data(),szmask.data(),par.szString.data());
			lpWnd->setTopic(par.szString.data());
		}
		m_lpFrm->m_lpGlobalUserList->updateUser(topiker);
		if(m_lpEventManager->lpEvent[KVI_Event_OnTopicChange]->bEnabled){
			QString szPara=channel+" "+topiker.szNick+" "+par.szString;
			m_lpUserParser->executeEvent(lpWnd,m_lpEventManager->lpEvent[KVI_Event_OnTopicChange],szPara);
		}	
		setStatus(i18n("Topic change for %s"),channel.data());
	}
	_debug_leavetrace("handleTopic");
}
//============ handleKick ============//
void KviServerParser::handleKick(QString &szPrefix,QString &szParams)
{
	_debug_entertrace("handleKick");
	KviNewNick kicker(szPrefix);
	KviIrcToken par(szParams);
	QString channel=par.nextWord();
	QString kicked=par.nextWord();
	par.stripUpToTrailing();

	if(par.szString.isEmpty())par.szString=" *** No reason specified *** ";
	QString szmask;
	kicker.mask(szmask,1);

	if(isMe(kicked)){	
		m_lpMdi->removeWindow(channel.data());
		if(m_lpEventManager->lpEvent[KVI_Event_OnMeKick]->bEnabled){
			QString szPara=channel+" "+kicker.szNick+" "+par.szString;
			m_lpUserParser->executeEvent(m_lpCons,m_lpEventManager->lpEvent[KVI_Event_OnMeKick],szPara);
		}	
		topWin()->doFmtOutput(KVI_OUT_KICK,i18n("You have been kicked off %s by %s [%s] [%s]"),
				channel.data(),kicker.szNick.data(),szmask.data(),par.szString.data());
		setStatus(i18n("Kicked from %s"),channel.data());
		if(m_lpOpt->bAutoRejoinOnKick){
			m_lpCons->doFmtOutput(KVI_OUT_INTERNAL,i18n("Attempting to rejoin %s...."),channel.data());
			m_lpSock->sendFmtData("JOIN %s",channel.data());
		}
		_debug_leavetrace("handleKick");
		return;
	}

	KviChanWnd *lpWnd=getChanWnd(channel.data());
	if(!lpWnd)m_lpCons->doFmtOutput(KVI_OUT_ERROR,i18n("[INTERNAL ERROR (KICK PARSING)] : Can't find channel %s"),channel.data());
	else {
		if(lpWnd->getNick(kicked))lpWnd->nickPart(kicked);
		m_lpFrm->m_lpGlobalUserList->updateUser(kicker); //just to be fun...
		if(m_lpEventManager->lpEvent[KVI_Event_OnKick]->bEnabled){
			QString szPara=channel+" "+kicker.szNick+" "+kicked+" "+par.szString;
			m_lpUserParser->executeEvent(lpWnd,m_lpEventManager->lpEvent[KVI_Event_OnKick],szPara);
		}
		if(m_lpEventManager->lpEvent[KVI_Event_OnMassKick]->bEnabled){
			if(lpWnd->checkMassKick(kicker.szNick)){
				QString szPara=lpWnd->name();
				szPara+=' ';
				szPara+=kicker.szNick.data();
				m_lpUserParser->executeEvent(lpWnd,m_lpEventManager->lpEvent[KVI_Event_OnMassKick],szPara);
			}
		}
		lpWnd->doFmtOutput(KVI_OUT_KICK,i18n("%s was kicked from %s by %s [%s] [%s]"),kicked.data(),channel.data(),
					kicker.szNick.data(),szmask.data(),par.szString.data());
	}
	_debug_leavetrace("handleKick");
}
//============ handleNick ============//
void KviServerParser::handleNick(QString &szPrefix,QString &szParams)
{
	_debug_entertrace("handleNick");
	KviNewNick nicker(szPrefix); //nick that changes nick
	int idx=szParams.find(':');
	if(idx != -1)szParams.remove(0,idx+1);
	szParams=szParams.stripWhiteSpace();
	//check the notify list....if he is there remove him :)
	KviIsonStruct *lpI=0;
	for(lpI=m_lpFrm->m_lpIsOnList->first();lpI;lpI=m_lpFrm->m_lpIsOnList->next()){
		if(!strcasecmp(nicker.szNick.data(),lpI->szNick.data())){
			if(lpI->isOn){
				if(m_lpOpt->bNotifyListInActive)
					topWin()->doFmtOutput(KVI_OUT_WHO,
						i18n("Registered user %s changed nick to %s : will disappear from the notify list."),
						lpI->szNick.data(),szParams.data());
				else m_lpCons->doFmtOutput(KVI_OUT_WHO,
						i18n("Registered user %s changed nick to %s : will disappear from the notify list."),
						lpI->szNick.data());
				lpI->isOn=false;
				m_lpCons->m_lpNotifyList->nickPart(lpI->szNick.data());				
			}
			break;
		}
	}
	// if me changes nick , show it in the status bar and change global variables
	bool bIsMe=isMe(nicker.szNick);
	if(bIsMe){
		m_lpFrm->m_lpGlb->szNick=szParams.data();
		KviNewNick srv(m_lpFrm->m_lpGlb->szMaskFromServer);
		srv.szNick=szParams.data();
		srv.mask(m_lpFrm->m_lpGlb->szMaskFromServer);
		m_lpCons->doFmtOutput(KVI_OUT_NICK,i18n("You are now known as %s"),szParams.data());
		setStatus(i18n("Changed nick to %s"),szParams.data());
		m_lpFrm->setWindowCaption();
	}
	// fire the event too
	if(m_lpEventManager->lpEvent[KVI_Event_OnNick]->bEnabled){
		QString szPara=nicker.szNick.data();
		szPara+=" ";
		szPara+=szParams.data();
		m_lpUserParser->executeEvent(m_lpCons,m_lpEventManager->lpEvent[KVI_Event_OnNick],szPara);
	}
	// update the user address first
	m_lpFrm->m_lpGlobalUserList->updateUser(nicker);
	// change the nick globally
	KviNewNick *lpGlobalEntry=m_lpFrm->m_lpGlobalUserList->changeNickname(nicker.szNick.data(),szParams.data());
	if(!lpGlobalEntry){
		if(!bIsMe)m_lpCons->doFmtOutput(KVI_OUT_ERROR,i18n("[ Server drunk ] : Unknown user %s changed nick"),nicker.szNick.data());
		return;
	}
	//and now update the channel list boxes resorting it
	int nWinds=m_lpMdi->m_lpChildList->count();
	if(nWinds){
		QString szmask;
		nicker.mask(szmask,1);
		KviMdiChild *lpChild=0;
		for(lpChild=m_lpMdi->m_lpChildList->first();lpChild;lpChild=m_lpMdi->m_lpChildList->next()){
			if(lpChild->type()==KVI_WND_TYPE_CHAN){
				KviChanWnd *lpWnd=(KviChanWnd *)lpChild;
				if(lpWnd->m_lpListBox->resortNick(lpGlobalEntry)){
					lpWnd->doFmtOutput(KVI_OUT_NICK,
						i18n("%s [%s] is now known as %s"),nicker.szNick.data(),szmask.data(),szParams.data());
				}
			}
			QString szWindowName(lpChild->name());
			if((lpChild->type()==KVI_WND_TYPE_QUERY)&&(szWindowName.lower()==nicker.szNick.lower())){
				lpChild->setName(szParams.data());
				((KviQueryWnd *)lpChild)->m_lpInput->m_szParentName=szParams.copy();
				QString szmask;
				nicker.mask(szmask);
				lpChild->doFmtOutput(KVI_OUT_NICK,
					i18n("%s [%s] is now known as %s"),nicker.szNick.data(),szmask.data(),szParams.data());
			}
		}
	}
	_debug_leavetrace("handleNick");
}
//============ handlePrivmsg ============//
void KviServerParser::handlePrivmsg(QString &szPrefix,QString &szParams)
{
	_debug_entertrace("handlePrivmsg");
	if(szPrefix.isEmpty()){
		szPrefix=m_lpFrm->m_lpGlb->szServerName.data();
		szPrefix+=i18n(":auth.service");
	}
	KviNewNick talker(szPrefix); //from...
	KviIrcToken par(szParams);
	QString target=par.nextWord();
	par.stripUpToTrailing();
	if(par.szString.isEmpty())par.szString="( *** empty privmsg from server *** )";
	//check ignore list...
	if(m_lpOpt->bEnableIgnoreList){
		QString szmask;
		talker.mask(szmask,1);
		KviUserStruct *lpU=m_lpFrm->findUserStruct(talker.szNick,szmask);
		if(lpU){
			if(lpU->bIgnore){
				if(isMe(target))m_lpCons->doFmtOutput(KVI_OUT_INTERNAL,i18n("Ignoring privmsg from %s [%s]"),
							talker.szNick.data(),szmask.data());
				else m_lpCons->doFmtOutput(KVI_OUT_INTERNAL,i18n("Ignored user %s [%s] talks to %s"),
							talker.szNick.data(),szmask.data(),target.data());
			}
		}
	}
	//First check for CTCPs
	if((par.szString[0]==0x01)&&(par.szString[par.szString.length()-1]==0x01)){
		m_lpCTCP->handleCTCPRequest(talker,target,&par);
		return;
	}
	if(m_lpOpt->bListenToFileRequest){
		if(par.szString[0]=='!'){ //file request?
			QString szRequest=par.szString.data();
			szRequest.remove(0,1);
			if(!strncasecmp(szRequest.data(),m_lpFrm->m_lpGlb->szNick.data(),strlen(m_lpFrm->m_lpGlb->szNick.data()))){
				QString szFile=par.nextWord();
				szFile=par.nextWord();
				m_lpCons->doFmtOutput(KVI_OUT_INTERNAL,i18n("%s requests file %s."),talker.szNick.data(),szFile.data());
				if(!szFile.isEmpty()){ //try to send that file (if found somewhere)
					if(m_iFileRequests <= m_lpOpt->iMaxFileRequests){
						QString szFullFileName(_macro_getKVircHomeDirectory("Audio/"));
						szFullFileName+=szFile.data();
						QFile aFile(szFullFileName.data());
						if(!aFile.exists()){
							szFullFileName = QString(_macro_getKVircHomeDirectory("Scripts/Audio/"));
							szFullFileName+=szFile.data();
							aFile.setName(szFullFileName.data());
							if(!aFile.exists()){
								m_lpCons->doFmtOutput(KVI_OUT_INTERNAL,i18n("Ignoring file request from %s : file %s not found"),talker.szNick.data(),szFile.data());
//								m_lpSock->sendFmtData
								return;
							}
						}
						m_iFileRequests++;
						QString szSendName="[DCC-SEND]:-"+talker.szNick+" "+szFullFileName;
						if(m_lpMdi->findWindow(szSendName.data())||(szSendName==m_lpFrm->m_lpUserParser->m_szLastSendTarget)){
							m_lpCons->doFmtOutput(KVI_OUT_ERROR,i18n("[DCC SEND] : A previous DCC SEND with the same file name to %s already exists"),talker.szNick.data());
							m_iFileRequests--;
							return;
						} else {
							m_lpCons->doFmtOutput(KVI_OUT_INTERNAL,i18n("Sending file requested by %s : %s"),talker.szNick.data(),szFullFileName.data());
							m_lpFrm->m_lpUserParser->m_szLastSendTarget=szSendName.copy();
							KviSendWnd *lpW=m_lpFrm->createSendWnd(szSendName.data(),m_lpOpt->bMinimizeDCCSend);
							lpW->m_bIsFileRequest=true;
							lpW->requestSend(talker.szNick.data(),szFullFileName.data());
						}
					} else {
						m_lpCons->doFmtOutput(KVI_OUT_INTERNAL,i18n("Ignoring file request from %s : max count reached."),talker.szNick.data());
						return;
					}
				}
			}
		}
	}
	if(isMe(target)){
		KviQueryWnd *lpWnd=getQueryWnd(talker.szNick.data());
		QString szMask;
		talker.mask(szMask);
		if(!lpWnd){
			if(m_lpOpt->bBeepOnPrivmsg){
				_macro_beep(100);
				_macro_beep(80);
				_macro_beep(60);
				_macro_beep(40);
				_macro_beep(20);
			}
			lpWnd=m_lpFrm->createQueryWnd(talker.szNick.data());
			lpWnd->doFmtOutput(KVI_OUT_INTERNAL,i18n("Private conversation request by %s"),szMask.data());
			setStatus(i18n("Started private conversation with %s"),talker.szNick.data());
		}
		QString szNicko;
		m_lpFrm->formatNickOutputString(szNicko,talker.szNick);
		if(m_lpEventManager->lpEvent[KVI_Event_OnMePrivmsg]->bEnabled){
				QString szPara=talker.szNick.data();
				szPara+=' ';
				szPara+=par.szString.data();
				m_lpUserParser->executeEvent(lpWnd,m_lpEventManager->lpEvent[KVI_Event_OnMePrivmsg],szPara);
		}
		lpWnd->doFmtOutput(KVI_OUT_NONE,"%s%s",szNicko.data(),par.szString.data());
		_debug_leavetrace("handlePrivMsg");
		return;
	}
	KviChanWnd *lpWnd=getChanWnd(target.data());
	if(!lpWnd)m_lpCons->doFmtOutput(KVI_OUT_SERVER,i18n("[PRIVMSG to %s from %s] :%s"),target.data(),talker.szNick.data(),par.szString.data());
	else {
		QString szNicko;
		m_lpFrm->formatNickOutputString(szNicko,talker.szNick);
		if(m_lpEventManager->lpEvent[KVI_Event_OnChannelPrivmsg]->bEnabled){
				QString szPara=target.data();
				szPara+=' ';
				szPara+=talker.szNick.data();
				szPara+=' ';
				szPara+=par.szString.data();
				m_lpUserParser->executeEvent(lpWnd,m_lpEventManager->lpEvent[KVI_Event_OnChannelPrivmsg],szPara);
		}
		lpWnd->doFmtOutput(KVI_OUT_NONE,"%s%s",szNicko.data(),par.szString.data());
		m_lpFrm->m_lpGlobalUserList->updateUser(talker); //Just for fun...
		if(m_lpEventManager->lpEvent[KVI_Event_OnPublicFlood]->bEnabled){
			if(lpWnd->checkPublicFlood(talker.szNick)){
				QString szPara=lpWnd->name();
				szPara+=' ';
				szPara+=talker.szNick.data();
				m_lpUserParser->executeEvent(lpWnd,m_lpEventManager->lpEvent[KVI_Event_OnPublicFlood],szPara);
			}
		}
	}
	_debug_leavetrace("handlePrivmsg");
}
//============ handleNotice ============//
void KviServerParser::handleNotice(QString &szPrefix,QString &szParams)
{
	_debug_entertrace("handleNotice");
	if(szPrefix.isEmpty()){
		szPrefix=m_lpFrm->m_lpGlb->szServerName.data();
		szPrefix+=i18n(":auth.service");
	}
	KviNewNick talker(szPrefix); //from...
	KviIrcToken par(szParams);
	QString target=par.nextWord();
	par.stripUpToTrailing();
	if(par.szString.isEmpty())par.szString="( *** empty notice from server *** )";
	//First check for CTCP replies..
	//We grant a ctcp reply from ignored users too
	if((par.szString[0]==0x01)&&(par.szString[par.szString.length()-1]==0x01)){
		m_lpCTCP->handleCTCPReply(talker,target,&par);
		return;
	}
	//check ignore list...
	if(m_lpOpt->bEnableIgnoreList){
		QString szmask;
		talker.mask(szmask,1);
		KviUserStruct *lpU=m_lpFrm->findUserStruct(talker.szNick,szmask);
		if(lpU){
			if(lpU->bIgnore){
				if(isMe(target))m_lpCons->doFmtOutput(KVI_OUT_INTERNAL,i18n("Ignoring notice from %s [%s]"),
							talker.szNick.data(),szmask.data());
				else m_lpCons->doFmtOutput(KVI_OUT_INTERNAL,i18n("Ignored user %s [%s] sends notice to %s"),
							talker.szNick.data(),szmask.data(),target.data());
			}
		}
	}
	//check if it is a server notice
	bool bServerNotice=(talker.szNick.find('.') != -1);

	if(isMe(target)){
		KviMdiChild *lpWnd=((m_lpOpt->bPrivNoticeToActive) ? topWin() : m_lpCons);
		QString szMask;
		talker.mask(szMask);
		QString szNicko;
		m_lpFrm->formatNickOutputString(szNicko,talker.szNick);
		if(bServerNotice){
			if(m_lpEventManager->lpEvent[KVI_Event_OnServerNotice]->bEnabled){
					QString szPara=talker.szNick.data();
					szPara+=' ';
					szPara+=par.szString.data();
					if(m_lpUserParser->executeEvent(lpWnd,m_lpEventManager->lpEvent[KVI_Event_OnServerNotice]
						,szPara)==KVI_EventErr_Halt)return;
			}	
		} else {
			if(m_lpEventManager->lpEvent[KVI_Event_OnMeNotice]->bEnabled){
					QString szPara=talker.szNick.data();
					szPara+=' ';
					szPara+=par.szString.data();
					m_lpUserParser->executeEvent(lpWnd,m_lpEventManager->lpEvent[KVI_Event_OnMeNotice],szPara);
			}
		}
		lpWnd->doFmtOutput(KVI_OUT_NOTICE,"%s%s",szNicko.data(),par.szString.data());
		if(m_lpOpt->bBeepOnPrivmsg){
			_macro_beep(50);
			_macro_beep(100);
			_macro_beep(50);
		}
		_debug_leavetrace("handleNotice");
		return;
	}
	KviChanWnd *lpWnd=getChanWnd(target.data());
	if(!lpWnd)m_lpCons->doFmtOutput(KVI_OUT_SERVER,i18n("[NOTICE to %s from %s] :%s"),target.data(),talker.szNick.data(),par.szString.data());
	else {
		QString szNicko;
		m_lpFrm->formatNickOutputString(szNicko,talker.szNick);
		if(bServerNotice){
			if(m_lpEventManager->lpEvent[KVI_Event_OnServerNotice]->bEnabled){
						QString szPara=talker.szNick.data();
						szPara+=' ';
						szPara+=par.szString.data();
						if(m_lpUserParser->executeEvent(lpWnd,m_lpEventManager->lpEvent[KVI_Event_OnServerNotice]
							,szPara)==KVI_EventErr_Halt)return;
			}	
			if(m_lpEventManager->lpEvent[KVI_Event_OnChannelNotice]->bEnabled){
					QString szPara=target.data();
					szPara+=' ';
					szPara+=talker.szNick.data();
					szPara+=' ';
					szPara+=par.szString.data();
					m_lpUserParser->executeEvent(lpWnd,m_lpEventManager->lpEvent[KVI_Event_OnChannelNotice],szPara);
			}
		}
		lpWnd->doFmtOutput(KVI_OUT_NOTICE,"%s%s",szNicko.data(),par.szString.data());
		if(!bServerNotice)m_lpFrm->m_lpGlobalUserList->updateUser(talker); //Just for fun...
	}
	_debug_leavetrace("handleNotice");
}
//============ handleMode ============//
void KviServerParser::handleMode(QString &szPrefix,QString &szParams)
{
	_debug_entertrace("handleMode");
	debug("Mode change : %s",szParams.data());
	KviNewNick source(szPrefix);
	KviIrcToken par(szParams);
	if((par.szString[0]=='#')||(par.szString[0]=='&')){ //CHANNEL MODES
		QString channel	=par.nextWord();
		QString modes	=par.nextWord();
		par.szString.stripWhiteSpace();
		par.szString.simplifyWhiteSpace();
		int idx=par.szString.find(':');
		while(idx != -1){
			par.szString.remove(idx,1);
			idx=par.szString.find(':');
		}
		//We should newer get a ':' in the params now...
		KviChanWnd *lpW=getChanWnd(channel.data());
		if(!lpW){
			m_lpCons->doFmtOutput(KVI_OUT_ERROR,i18n("[INTERNAL ERROR (MODE PARSING)] : Can't find channel %s"),channel.data());
			return;
		}
		m_lpFrm->m_lpGlobalUserList->updateUser(source);
		if(modes.isEmpty()){
			m_lpCons->doOutput(KVI_OUT_ERROR,i18n("[INTERNAL ERROR] : Empty mode sequence."));
			return;
		}
		QString szMask;
		source.mask(szMask,1);		
		{ /////////BLOCK RECOGNIZE MODES
			//Now szModes contains something as -o+b Mimmo@*.molello.com
			char *lpC=modes.data();
			if((*lpC)==' ')lpC++;
			if(!(*lpC)){
				m_lpCons->doOutput(KVI_OUT_ERROR,i18n("[INTERNAL ERROR] : Empty mode sequence."));
				return;
			}
			//OK we have a + , - ora a mode letter here
			bool bModeAdd=true;
			QString parameter;
			while((*lpC)&&(*lpC != ' ')){
				switch(*lpC){
				case '+':
					bModeAdd=true;
					break;
				case '-':
					bModeAdd=false;
					break;
				case 'o':
				case 'b':
				case 'v':
					parameter=par.nextWord();
					handleChanUserMode(lpW,source,(*lpC),bModeAdd,parameter);
					break;
				case 'k':
				case 'l':
					if(bModeAdd){
						parameter=par.nextWord();
						if(parameter.isEmpty())parameter=i18n("[No parameter specified]");
						lpW->doFmtOutput(KVI_OUT_CHANMODE,i18n("%s (%s) sets mode +%c %s for %s"),
							source.szNick.data(),szMask.data(),(*lpC),parameter.data(),channel.data());
					} else {
						lpW->doFmtOutput(KVI_OUT_CHANMODE,i18n("%s (%s) sets mode -%c for %s"),
							source.szNick.data(),szMask.data(),(*lpC),channel.data());
					}
					lpW->setMode((*lpC),bModeAdd);
					break;
				default:
					lpW->doFmtOutput(KVI_OUT_CHANMODE,i18n("%s [%s] sets mode %c%c for %s"),source.szNick.data(),
							szMask.data(),(bModeAdd ? '+' : '-'),(*lpC),channel.data());
					lpW->setMode((*lpC),bModeAdd);
					break;
				}
				lpC++;
			}
			
		}
	} else {
		//Nick modes 
		QString szNicko;
		szNicko=par.nextWord();
		if(szNicko.isEmpty())szNicko=m_lpFrm->m_lpGlb->szNick;
		topWin()->doFmtOutput(KVI_OUT_SERVER,i18n("%s sets mode %s for %s")
					,source.szNick.data(),par.szString.data(),szNicko.data());
	}
	_debug_leavetrace("handleMode");
}
//============ handleChanUserMode ============//
void KviServerParser::handleChanUserMode(KviChanWnd *lpWnd,KviNewNick &source,char mode,bool bAdd,QString &szParam)
{
	_debug_entertrace("handleChanUserMode");
	QString szMask;
	source.mask(szMask,1);
	int type=KVI_OUT_CHANMODE;
	if(mode != 'b'){
		KviNewNick *lpN=lpWnd->getNick(szParam);
		if(!lpN){
			m_lpCons->doFmtOutput(KVI_OUT_ERROR,i18n("[INTERNAL ERROR] : Can't find %s on %s"),szParam.data(),lpWnd->name());
			return;
		}
	}
	QString szWhat;
	int even;
	bool meTarget=false;
	switch(mode){
		case 'o':
			type=(bAdd ? KVI_OUT_OP        : KVI_OUT_DEOP);
			if(isMe(szParam)){
				meTarget=true;
				even=(bAdd ? KVI_Event_OnMeOp  : KVI_Event_OnMeDeop);
			} else {
				even=(bAdd ? KVI_Event_OnOp    : KVI_Event_OnDeop);
			}
			szWhat=i18n("operator status"); // Fritz: check!!!
										    // Pragma: what??? :)	
			lpWnd->nickOp(szParam.data(),bAdd);
			if(bAdd){
				if(m_lpEventManager->lpEvent[KVI_Event_OnMassOp]->bEnabled){
					if(lpWnd->checkMassOp(source.szNick)){
						QString szPara=lpWnd->name();
						szPara+=' ';
						szPara+=source.szNick.data();
						m_lpUserParser->executeEvent(lpWnd,m_lpEventManager->lpEvent[KVI_Event_OnMassOp],szPara);
					}
				}
			} else {
				if(m_lpEventManager->lpEvent[KVI_Event_OnMassDeop]->bEnabled){
					if(lpWnd->checkMassDeop(source.szNick)){
						QString szPara=lpWnd->name();
						szPara+=' ';
						szPara+=source.szNick.data();
						m_lpUserParser->executeEvent(lpWnd,m_lpEventManager->lpEvent[KVI_Event_OnMassDeop],szPara);
					}
				}
			}
			break;
		case 'b':
			type=(bAdd ? KVI_OUT_BAN       : KVI_OUT_UNBAN);
			{
				KviNewNick *myNick=lpWnd->getNick(m_lpFrm->m_lpGlb->szNick.data());
				if(myNick->matches(szParam)){
					meTarget=true;
					even=(bAdd ? KVI_Event_OnMeBan : KVI_Event_OnMeUnban);
				} else {
					even=(bAdd ? KVI_Event_OnBan   : KVI_Event_OnUnban);
				}
				QString szNewMask;
				source.mask(szNewMask,0);
				//lpWnd->changeBanList(szParam.data(),szNewMask.data(),bAdd);
				// Fixed by RAD Kade 1
			    lpWnd->changeBanList(szNewMask.data(),szParam.data(),bAdd); 		
				//check mass ban
				if(bAdd){
					if(m_lpEventManager->lpEvent[KVI_Event_OnMassBan]->bEnabled){
						if(lpWnd->checkMassBan(source.szNick)){
							QString szPara=lpWnd->name();
							szPara+=' ';
							szPara+=source.szNick.data();
							m_lpUserParser->executeEvent(lpWnd,m_lpEventManager->lpEvent[KVI_Event_OnMassBan],szPara);
						}
					}
				}
			}
			break;
		case 'v':
			type=(bAdd ? KVI_OUT_VOICE     : KVI_OUT_DEVOICE);
			if(isMe(szParam)){
				meTarget=true;
				even=(bAdd ? KVI_Event_OnMeVoice : KVI_Event_OnMeDevoice);
			} else {
				even=(bAdd ? KVI_Event_OnVoice : KVI_Event_OnDevoice);
			}
			szWhat=i18n("voice status");
			lpWnd->nickVoice(szParam.data(),bAdd);
			break;
	}
	if(m_lpEventManager->lpEvent[even]->bEnabled){
		QString szPara=lpWnd->name();
		if(meTarget){
			szPara+=" "+source.szNick;
			if(mode=='b')szPara+=" "+szParam;
		} else {
			szPara+=" "+source.szNick+" "+szParam;
		}
		m_lpUserParser->executeEvent(lpWnd,m_lpEventManager->lpEvent[even],szPara);
	}
	lpWnd->doFmtOutput(type,i18n("%s [%s] sets mode %c%c %s"),source.szNick.data(),
				szMask.data(), (bAdd ? '+':'-') , mode , szParam.data());
	if(meTarget&&((mode=='o')||(mode=='v'))){
		QString szStat;
		szStat.sprintf(i18n("%s %s on %s"),(bAdd ? i18n("Gained") : i18n("Lost")),szWhat.data(),lpWnd->name());
		m_lpFrm->setStatusText(szStat.data());
		if(mode=='o'){
			lpWnd->m_bIsMeOp=bAdd;
			lpWnd->setLabelsPalette(); //show that we can change the topic etc...
		}
	}
	_debug_leavetrace("handleChanUserMode");
}
//============ handleQuit ============//
void KviServerParser::handleQuit(QString &szPrefix,QString &szParams)
{
	_debug_entertrace("handleQuit");
	KviNewNick quitter(szPrefix); //nick that quits
	KviIrcToken par(szParams);
	par.stripUpToTrailing();
	if(par.szString.isEmpty())par.szString="( *** I Quit *** )";
	else {	//check for netsplits
			//determine if signoff string matches "%.% %.%", and only one space
		char *p = strchr(par.szString.data(), ' ');
		if (p && (p == strrchr(par.szString.data(),' '))) {
			char *z1, *z2;
			*p = 0;
			z1 = strchr(p + 1, '.');
			z2 = strchr(par.szString.data(), '.');
			if (z1 && z2 && (*(z1 + 1) != 0) && (z1 - 1 != p) && (z2 + 1 != p) && (z2 != par.szString.data())) {
				// server split, or else it looked like it anyway
				*p=' ';
				setStatus("Netsplit detected : %s",par.szString.data());
				par.szString.prepend("NETSPLIT : ");
			} else  *p = ' ';
		}
	}
	if(m_lpEventManager->lpEvent[KVI_Event_OnQuit]->bEnabled){
		QString szPara=quitter.szNick+" "+par.szString;
		m_lpUserParser->executeEvent(m_lpCons,m_lpEventManager->lpEvent[KVI_Event_OnQuit],szPara);
	}
	int nWinds=m_lpMdi->m_lpChildList->count();
	if(nWinds){
		KviMdiChild *lpChild;
		for(lpChild=m_lpMdi->m_lpChildList->first();lpChild;lpChild=m_lpMdi->m_lpChildList->next()){
			if(lpChild->type()==KVI_WND_TYPE_CHAN){
				KviChanWnd *lpWnd=(KviChanWnd *)lpChild;
				if(lpWnd->getNick(quitter)){
					lpWnd->nickPart(quitter);
					QString szmask;
					quitter.mask(szmask,1);
					lpWnd->doFmtOutput(KVI_OUT_QUIT,
						i18n("%s [%s] has quit irc. [%s]"),quitter.szNick.data(),
						szmask.data(),par.szString.data());
				}
			}
			QString szWindowName(lpChild->name());
			if((lpChild->type()==KVI_WND_TYPE_QUERY)&&(szWindowName.lower()==quitter.szNick.lower())){
				QString szmask;
				quitter.mask(szmask,1);
				lpChild->doFmtOutput(KVI_OUT_QUIT,i18n("%s [%s] has quit irc. [%s]"),quitter.szNick.data(),
										szmask.data(),szParams.data());
			}
		}
	}
	_debug_leavetrace("handleQuit");
}
void KviServerParser::handleWhois(QString &szPrefix,int nCommand,QString &szParams){
	KviIrcToken par(szParams);
	QString szNick=par.nextWord(); //Cut our nickname
	szNick=par.nextWord();         //get the whois target nick
	KviNewNick nk;
	QString sz1,sz2;
	switch(nCommand){
	case RPL_WHOISUSER:
	case RPL_WHOWASUSER:
		sz1=par.nextWord(); //user
		sz2=par.nextWord(); //host
		topWin()->doFmtOutput(KVI_OUT_WHO,"%c%s%c %s           : %c%s!%s@%s",KVI_TEXT_BOLD,
				szNick.data(),KVI_TEXT_BOLD,((nCommand==RPL_WHOISUSER) ? i18n("is ") : i18n("was")),
							KVI_TEXT_BOLD,szNick.data(),sz1.data(),sz2.data());
		par.stripUpToTrailing();
		topWin()->doFmtOutput(KVI_OUT_WHO,i18n("%c%s%c real name     : %s"),KVI_TEXT_BOLD,
				szNick.data(),KVI_TEXT_BOLD,par.szString.data());
		//This one should be safe...(nick+user+host)
//		sz1.sprintf("%s!%s@%s",szNick.data(),sz1.data(),sz2.data()); <---that was funky! :) pragma
		if(nCommand==RPL_WHOISUSER){
			szNick+='!'; //gaining a variable in this way :)
			szNick+=sz1;
			szNick+='@';
			szNick+=sz2;
			nk.fromMask(szNick);
			m_lpFrm->m_lpGlobalUserList->updateUser(nk);
		}
		break;
	case RPL_WHOISSERVER:
		sz1=par.nextWord(); //server
		par.stripUpToTrailing(); //server info
		topWin()->doFmtOutput(KVI_OUT_WHO,i18n("%c%s%c using server  : %s [%s]"),KVI_TEXT_BOLD,
				szNick.data(),KVI_TEXT_BOLD,sz1.data(),par.szString.data());
		break;
	case RPL_WHOISOPERATOR:
		topWin()->doFmtOutput(KVI_OUT_WHO,i18n("%c%s%c status        : %cIRC Operator"),
				KVI_TEXT_BOLD,szNick.data(),KVI_TEXT_BOLD,KVI_TEXT_UNDERLINE);
		break;
	case RPL_WHOISIDLE:
		sz1=par.nextWord(); //seconds idle
		topWin()->doFmtOutput(KVI_OUT_WHO,i18n("%c%s%c idle time     : %s seconds"),KVI_TEXT_BOLD,
				szNick.data(),KVI_TEXT_BOLD,sz1.data());
		break;
	case RPL_WHOISCHANNELS:
		par.stripUpToTrailing(); //channels
		topWin()->doFmtOutput(KVI_OUT_WHO,i18n("%c%s%c channels      : %s"),KVI_TEXT_BOLD,
				szNick.data(),KVI_TEXT_BOLD,par.szString.data());
		break;
	case RPL_AWAY: //Added by RAD Kade 1
		par.stripUpToTrailing(); //Away message
		topWin()->doFmtOutput(KVI_OUT_WHO,i18n("%c%s%c is away       : %s"),KVI_TEXT_BOLD,
				szNick.data(),KVI_TEXT_BOLD,par.szString.data());
		break;
	case RPL_ENDOFWHOIS:
		topWin()->doFmtOutput(KVI_OUT_WHO,i18n("%c%s%c whois from    : %s"),KVI_TEXT_BOLD,
				szNick.data(),KVI_TEXT_BOLD,szPrefix.data());
		break;
	case RPL_ENDOFWHOWAS:
		topWin()->doFmtOutput(KVI_OUT_WHO,i18n("%c%s%c whowas from   : %s"),KVI_TEXT_BOLD,
				szNick.data(),KVI_TEXT_BOLD,szPrefix.data());	
		break;
	}
}

//
// $Log: kvi_sparser.cpp,v $
// Revision 1.12  1998/10/06 14:42:46  pragma
// Tons of changes
//
// Revision 1.11  1998/10/01 00:51:51  pragma
// Added handler for the 'Resource temporary unavailable' error reply
// Fixed the extended join syntax (user modes passed with the JOIN command)
//
// Revision 1.10  1998/09/30 03:10:05  pragma
// Bugfixes
//
// Revision 1.9  1998/09/25 15:58:35  pragma
// Moving ti use the KviProxy class.
//
// Revision 1.8  1998/09/23 23:15:13  pragma
// Ignore list. Still needs testing.
// Added some ASSERT-Like macros to kvi_chan.cpp and kvi_listbox.cpp
// to discover a strange bug :
// Sometimes the a single nick on a random channel gets wrong username...
// Just like a hollow pointer...but still can't guess how to reproduce it.
// It happened only a couple of times...
//
// Revision 1.7  1998/09/23 12:41:51  pragma
// Another big commit.
// Added user list (Still have to fix ignore things)
// User list dialogs
// Some code rearrangements
// Minor bugfixes
//
// Revision 1.6  1998/09/20 20:23:50  fritz
// reorganized includes.
// More work on srvdlg - still not finished.
//
// Revision 1.5  1998/09/18 03:57:09  pragma
// Not so much changes...only a couple of bug fixes.
//
// Revision 1.4  1998/09/17 19:30:07  fritz
// First translations.
// Fixed some typos and inconsistent messages.
//
// Revision 1.3  1998/09/16 17:16:58  fritz
// Starting i18n.
//
// Revision 1.2  1998/09/16 16:13:22  pragma
// Moving to use a dynamic kvirc home directory.
// Big commit :)
//
//
