//
//   File : kvi_send.cpp (/usr/cvs/kvirc/kvirc/kvi_send.cpp)
//   Last modified : Mon Dec 7 1998 14:58:04 by root@localhost.localdomain
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998-1999 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation ; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING. If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CLASS_NAME_ "KviSendWnd"

//
// WARNING : Extremely complex and ugly code :)
//           Don't use it as example of good C++ programming
//

#include "kvi_defs.h"
#include "kvi_macros.h"
#include "kvi_send.h"
#include "kvi_view.h"
#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_opt.h"
#include "kvi_sparser.h"
#include "kvi_uparser.h"
#include "kvi_debug.h"
#include "kvi_mdi.h"
#include "kvi_taskbar.h"
#include "kvi_int.h"
#include "kvi_socket.h"
#include "kvi_listen.h"
#include "kvi_support.h"
#include "kvi_global.h"
#include "kvi_usr.h"

#include <arpa/inet.h>
#include <stdarg.h>
#include <unistd.h>
#include <time.h>
#include <fcntl.h>
#include <errno.h>

//=============== KviSendWnd ==============//

KviSendWnd::KviSendWnd(KviMdiManager *parent,KviFrame *frame,const char *aname,int aid)
:KviMdiChild(parent,frame,aname,aid,KVI_WND_TYPE_SEND)
{
	_debug_entertrace("KviSendWnd");
	//GUI
	m_lpOutput=new KviView(this,frame,"KviViewClass");
	m_lpProgress=new KProgress(0,100,0,KProgress::Horizontal,this,"KviProgressClass");
	m_lpLabel=new QLabel(i18n("No connection active."),this);
	m_lpLabel->setFrameStyle(QFrame::Sunken | QFrame::Panel);

	m_szFileName        = "";
	m_lpFile            = 0;
	m_lpListeningSocket = 0;
	m_Sock              = 0;
	m_iPort             = 0;
	m_bConnected        = false;
	m_bSending          = false;
	m_lpReadNotifier    = 0;
	m_lpWriteNotifier   = 0;
	m_lpTimer           = 0;
	m_iNumAck           = 0;
	m_bIsFileRequest    = false;
	m_iPacketSize       = m_lpFrm->m_lpOpt->iDCCSendPacketSize;
	m_lpFastSendTimer   = 0;
	m_nFailedSends      = 0;
	m_bWaitForAck       = false;
	m_szRemoteIp        = "";
	setFocusOwner(m_lpProgress);
	applyOptions();
	connect(this,SIGNAL(closeButtonPressed()),this,SLOT(closeSlot()));
	_debug_leavetrace("KviSendWnd");
}

//============= ~KviSendWnd ============//

KviSendWnd::~KviSendWnd()
{
	_debug_entertrace("~KviSendWnd");
	if(m_bIsFileRequest){
		m_lpFrm->m_lpServerParser->m_iFileRequests--;
		if(m_lpFrm->m_lpServerParser->m_iFileRequests < 0)m_lpFrm->m_lpServerParser->m_iFileRequests=0;
	}
	m_lpFrm->m_lpUserParser->m_szLastSendTarget="";
	destroyFile();
	killSock();
	delete m_lpLabel;
	delete m_lpProgress;
	delete m_lpOutput;
	_debug_leavetrace("~KviSendWnd");
}

//============ closeSlot ============//

void KviSendWnd::closeSlot()
{
	_debug_entertrace("closeSlot");
	killSock();
	hide();
	_macro_kviApplication->processEvents();
	blockSignals(true);
	m_lpMdi->m_lpTaskBar->removeButton(id());
	m_lpMdi->m_lpChildList->setAutoDelete(false);
	m_lpMdi->m_lpChildList->removeRef(this);
	m_lpMdi->focusTopChild();
	m_lpMdi->fillWinListPopup();
	m_lpFrm->queryExternalDestroy(this);
	_debug_leavetrace("closeSlot");
}

//============== applyOptions =============//

void KviSendWnd::applyOptions()
{
	_debug_entertrace("applyOptions");
	m_lpOutput->setFont(m_lpInt->fnt_output);
	m_lpLabel->setFont(m_lpInt->fnt_chanlabels);
	QPalette labelPal=m_lpLabel->palette();
	QColorGroup aCG(labelPal.normal());
	QColorGroup nCG(aCG.foreground(),aCG.background(),aCG.light(),
			 	aCG.dark(), aCG.mid(),m_lpInt->clr_fr_chanlabels, aCG.base()); 
	labelPal.setNormal(nCG);
	m_lpOutput->m_bTimestamp=m_lpFrm->m_lpOpt->bTimestamp;
	m_lpOutput->m_bShowPixmaps=m_lpFrm->m_lpOpt->bShowPixmaps;
	m_lpLabel->setPalette(labelPal);
	m_lpLabel->setBackgroundColor(m_lpInt->clr_bk_chanlabels);
	m_lpLabel->update();
	_debug_leavetrace("applyOptions");
}

//============= resizeEvent ==============//

void KviSendWnd::resizeEvent(QResizeEvent *)
{
	_debug_entertrace("resizeEvent");
	updateRects();
	QRect rct=viewSizeHint();
	QFontMetrics fm=m_lpLabel->fontMetrics();
	int labl_hght=fm.lineSpacing()+6;
	m_lpOutput->setGeometry(rct.left(),
							rct.top()+labl_hght+KVI_PROGRESS_HEIGHT+(2*KVI_MDI_VIEW_SEP),
							rct.width(),
							rct.height()-(KVI_PROGRESS_HEIGHT+(2*KVI_MDI_VIEW_SEP)+labl_hght));
	m_lpProgress->setGeometry(rct.left(),rct.top(),rct.width(),KVI_PROGRESS_HEIGHT);
	m_lpLabel->setGeometry(rct.left(),rct.top()+KVI_PROGRESS_HEIGHT+KVI_MDI_VIEW_SEP,rct.width(),labl_hght);
	_debug_leavetrace("resizeEvent");
}

void KviSendWnd::doFmtOutput(int nType,const char *szFmt,...){
	char szText[600]; //It should be big enough... I hope...
	va_list list;
	va_start(list,szFmt);
	if(vsnprintf(szText,600,szFmt,list)==-1)debug("WARNING : Output string truncated"); // Fritz: vsnprintf
	va_end(list);
	m_lpMdi->highlightWindow(m_iId);
	m_lpOutput->appendText(nType,(const char *) &szText);
}
void KviSendWnd::doOutput(int nType,const char *szText){
	m_lpMdi->highlightWindow(m_iId);
	m_lpOutput->appendText(nType,szText);
}
////////////////////////////////////////

//============ initiateResumeGet ============//

void KviSendWnd::initiateResumeGet()
{
	_debug_entertrace("initiateResumeGet");
	QString szRealIp=QString(inet_ntoa(m_inAddress));
	doFmtOutput(KVI_OUT_INTERNAL,i18n("RESUME ACCEPTED : Connecting to %s on port %s"),szRealIp.data(),m_szPort.data());
	struct sockaddr_in sockAddrIn;
	m_lpFrm->m_lpSock->fillSockAddr(&sockAddrIn,m_inAddress,m_iPort);
	m_Sock = ::socket(PF_INET,SOCK_STREAM,0);
	if (m_Sock < 0){
		doOutput(KVI_OUT_ERROR,i18n("Unable to create STREAM socket for connection"));
		m_lpFrm->setStatusText(i18n("DCC GET failed"));
		return;
	}
	if(fcntl(m_Sock, F_SETFL, O_NONBLOCK) < 0){
		::close(m_Sock);
		doOutput(KVI_OUT_ERROR,i18n("Unable to create STREAM socket for connection (fcntl failed)"));
		m_lpFrm->setStatusText(i18n("DCC GET failed"));
		return;
	};
	m_lpTimer=new QTimer();
	QObject::connect(m_lpTimer,SIGNAL(timeout()),this,SLOT(connectTimedOut()));
	m_lpTimer->start(m_lpFrm->m_lpOpt->iDCCCHATTimeout * 1000);
	if(::connect( m_Sock, (struct sockaddr*)(&sockAddrIn), sizeof(sockAddrIn) )<0){
		if(errno != EINPROGRESS){
			killSock();
			int errore=m_lpFrm->m_lpSock->translateFromSystemError(errno);
			QString szError;
			m_lpFrm->m_lpSock->getErr(errore,szError);
			doFmtOutput(KVI_OUT_ERROR,i18n("Unable to connect to %s : %s"),szRealIp.data(),szError.data());
			return;
		}
	}
	m_lpReadNotifier = new QSocketNotifier(m_Sock, QSocketNotifier::Read,this);
	m_lpWriteNotifier= new QSocketNotifier(m_Sock, QSocketNotifier::Write,this);
	QObject::connect(m_lpReadNotifier, SIGNAL(activated(int)), this, SLOT(slotDataInBuffer(int)));
	QObject::connect(m_lpWriteNotifier, SIGNAL(activated(int)), this, SLOT(slotConnected(int)));
	_debug_leavetrace("initiateResumeGet");
}
//============ acceptConnectedSend ============//

bool KviSendWnd::acceptConnectedSend(int iSock,const char *szNick,const char *szIp,const char *szFileName,
			uint uSize,bool bResume)
{
	_debug_entertrace("acceptSend");
	//
	//  here we accept a DCC send request from another user
	//  If bResume is true...that means that we have another file with that name on the HD
	//
	QString szStat;
	//  set the status text
	szStat.sprintf(i18n("Accepting DCC SEND connection from %s@%s"),szNick,szIp);
	m_lpFrm->setStatusText(szStat.data());
	m_szRemoteNick=szNick;
	//setup the file
	m_iFileLength=uSize;
	m_szFileName=szFileName;
	if(m_lpFile||m_bConnected)_debug_fatal(i18n("Can not use this window two times"));
	//create the file
	m_lpFile=new QFile(szFileName);
	if(bResume){
		if(!m_lpFile->exists()){
			delete m_lpFile;
			m_lpFile=0;
			doFmtOutput(KVI_OUT_ERROR,i18n("Ops...file doesn't exist..cant start a RESUME session\n"\
											"Can not open file %s for appending"),szFileName);
			m_lpFrm->setStatusText(i18n("DCC RESUME GET failed"));
			killSock();
			return false;	
		}
		if(!m_lpFile->open(IO_WriteOnly | IO_Append)){
			delete m_lpFile;
			m_lpFile=0;
			doFmtOutput(KVI_OUT_ERROR,i18n("Can not open file %s for appending"),szFileName);
			m_lpFrm->setStatusText(i18n("DCC RESUME GET failed"));
			killSock();
			return false;
		}
	} else {
		if(!m_lpFile->open(IO_WriteOnly)){
			delete m_lpFile;
			m_lpFile=0;
			doFmtOutput(KVI_OUT_ERROR,i18n("Can not open file %s for writing"),szFileName);
			m_lpFrm->setStatusText(i18n("DCC GET failed"));
			killSock();
			return false;
		}
	}
	m_iPort=0;
	if(bResume){
		QFileInfo aFileInfo(*m_lpFile);
		m_iFileRead=aFileInfo.size();
		doFmtOutput(KVI_OUT_INTERNAL,i18n("Sent DCC RESUME [%s:%u of %u total] request to %s@%s"),
			aFileInfo.fileName().data(),m_iFileRead,m_iFileLength,szNick,szIp);
	} else m_iFileRead=0;
	m_szRemoteIp=szIp;
	m_Sock=iSock;
	m_lpReadNotifier = new QSocketNotifier(m_Sock, QSocketNotifier::Read,this);
	QObject::connect(m_lpReadNotifier, SIGNAL(activated(int)), this, SLOT(slotDataInBuffer(int)));
	QString szN=m_lpFrm->m_lpGlb->szNick.data();
	//send response : 121 myNick resumePos
	if(szN.isEmpty() || (!strcmp(szN.data(),KVI_STR_NULL))){
		szN=m_lpFrm->m_lpUsr->szNick.data();
		if(szN.isEmpty() || (!strcmp(szN.data(),KVI_STR_NULL))){
			doFmtOutput(KVI_OUT_INTERNAL,i18n("You have specified no nickname. Presenting myself as UNKNOWN"));
			szN=i18n("UNKNOWN");
		}
	}
	m_bConnected=true;
	QString szDataToSend;
	szDataToSend.sprintf("121 %s %i\r\n",szNick,m_iFileRead);
	::write(m_Sock,szDataToSend.data(),strlen(szDataToSend.data()));
	m_bSending=false;
	m_bWaitForAck=false;
	m_startTime=time(0);
	doOutput(KVI_OUT_INTERNAL,i18n("Connected"));
	_debug_leavetrace("acceptSend");
	return true;
}

//============ acceptSend ============//

bool KviSendWnd::acceptSend(const char *szNick,QString &szPort,QString &szIp,const char *szFileName,QString szSize,bool bResume)
{
	_debug_entertrace("acceptSend");
	//
	//  here we accept a DCC send request from another user
	//  If bResume is true...that means that we have another file with that name on the HD
	//
	QString szStat;
	//  set the status text
	szStat.sprintf(i18n("Accepting DCC SEND connection from %s"),szNick);
	m_lpFrm->setStatusText(szStat.data());
	m_szRemoteNick=QString(szNick);
	//setup the file
	bool bOK=false;
	m_iFileLength=szSize.toUInt(&bOK);
	if(!bOK)m_iFileLength=0;

	m_szFileName=szFileName;
	if(m_lpFile||m_bConnected)_debug_fatal(i18n("Can not use this window two times"));
	//create the file
	m_lpFile=new QFile(szFileName);
	if(bResume){
		if(!m_lpFile->exists()){
			delete m_lpFile;
			m_lpFile=0;
			doFmtOutput(KVI_OUT_ERROR,i18n("Ops...file doesn't exist..cant start a RESUME session\n"\
											"Can not open file %s for appending"),szFileName);
			m_lpFrm->setStatusText(i18n("DCC RESUME GET failed"));
			return false;	
		}
		if(!m_lpFile->open(IO_WriteOnly | IO_Append)){
			delete m_lpFile;
			m_lpFile=0;
			doFmtOutput(KVI_OUT_ERROR,i18n("Can not open file %s for appending"),szFileName);
			m_lpFrm->setStatusText(i18n("DCC RESUME GET failed"));
			return false;
		}
	} else {
		if(!m_lpFile->open(IO_WriteOnly)){
			delete m_lpFile;
			m_lpFile=0;
			doFmtOutput(KVI_OUT_ERROR,i18n("Can not open file %s for writing"),szFileName);
			m_lpFrm->setStatusText(i18n("DCC GET failed"));
			return false;
		}
	}
	m_iPort=(unsigned short int)szPort.toUInt(&bOK);
	if(!bOK){
		doOutput(KVI_OUT_ERROR,i18n("Can't retrieve the port to connect. DCC GET failed."));
		m_lpFrm->setStatusText(i18n("DCC GET failed"));
		return false;	
	}
	szIp=szIp.stripWhiteSpace();
	unsigned long iAddr=szIp.toULong(&bOK);
	if(!bOK){
		doFmtOutput(KVI_OUT_ERROR,i18n("Can't retrieve the address to connect. DCC GET failed (%s)."),szIp.data());
		m_lpFrm->setStatusText(i18n("DCC GET failed"));
		return false;	
	}
	//struct in_addr inAddress;
	m_inAddress.s_addr=ntohl(iAddr);

	if(bResume){
		QFileInfo aFileInfo(*m_lpFile);
		m_iFileRead=aFileInfo.size();
		m_lpFrm->m_lpSock->sendFmtData("PRIVMSG %s :%cDCC RESUME %s %s %u%c",szNick,0x01,
				aFileInfo.fileName().data(),szPort.data(),m_iFileRead,0x01);
		doFmtOutput(KVI_OUT_INTERNAL,i18n("Sent DCC RESUME [%s:%u of %u total] request to %s...waiting for reply"),
			aFileInfo.fileName().data(),m_iFileRead,m_iFileLength,szNick);
		m_szPort=szPort.data();
		return true;
	}

	m_szRemoteIp=QString(inet_ntoa(m_inAddress));
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Connecting to %s on port %s"),m_szRemoteIp.data(),szPort.data());
	struct sockaddr_in sockAddrIn;
	m_lpFrm->m_lpSock->fillSockAddr(&sockAddrIn,m_inAddress,m_iPort);
	m_Sock = ::socket(PF_INET,SOCK_STREAM,0);
	if (m_Sock < 0){
		doOutput(KVI_OUT_ERROR,i18n("Unable to create STREAM socket for connection"));
		m_lpFrm->setStatusText(i18n("DCC GET failed"));
		return false;
	}
	if(fcntl(m_Sock, F_SETFL, O_NONBLOCK) < 0){
		::close(m_Sock);
		doOutput(KVI_OUT_ERROR,i18n("Unable to create STREAM socket for connection (fcntl failed)"));
		m_lpFrm->setStatusText(i18n("DCC GET failed"));
		return false;
	};
	m_lpTimer=new QTimer();
	QObject::connect(m_lpTimer,SIGNAL(timeout()),this,SLOT(connectTimedOut()));
	m_lpTimer->start(m_lpFrm->m_lpOpt->iDCCCHATTimeout * 1000);
	if(::connect( m_Sock, (struct sockaddr*)(&sockAddrIn), sizeof(sockAddrIn) )<0){
		if(errno != EINPROGRESS){
			killSock();
			int errore=m_lpFrm->m_lpSock->translateFromSystemError(errno);
			QString szError;
			m_lpFrm->m_lpSock->getErr(errore,szError);
			doFmtOutput(KVI_OUT_ERROR,i18n("Unable to connect to %s : %s"),m_szRemoteIp.data(),szError.data());
			return false;
		}
	}
	m_iFileRead=0;
	m_lpReadNotifier = new QSocketNotifier(m_Sock, QSocketNotifier::Read,this);
	m_lpWriteNotifier= new QSocketNotifier(m_Sock, QSocketNotifier::Write,this);
	QObject::connect(m_lpReadNotifier, SIGNAL(activated(int)), this, SLOT(slotDataInBuffer(int)));
	QObject::connect(m_lpWriteNotifier, SIGNAL(activated(int)), this, SLOT(slotConnected(int)));
	_debug_leavetrace("acceptSend");
	return true;
}
//============ slotConnected ============//

void KviSendWnd::slotConnected(int)
{
	_debug_entertrace("slotConnected");
	if(m_lpWriteNotifier){
		delete m_lpWriteNotifier;
		m_lpWriteNotifier=0;
	}
	destroyTimer();
	m_bConnected=true;
	m_bSending=false;
	m_startTime=time(0);
	doOutput(KVI_OUT_INTERNAL,i18n("Connected."));
	_debug_leavetrace("slotConnected");
}

//============ requestFailed ============//

void KviSendWnd::requestFailed(const char *text)
{
	_debug_entertrace("requestFailed");
	doOutput(KVI_OUT_ERROR,text);
	doOutput(KVI_OUT_ERROR,i18n("DCC SEND Request failed"));
	m_lpFrm->setStatusText(i18n("DCC SEND Request failed"));
	_debug_leavetrace("requestFailed");
}

//============ sendToDaemon ============//

bool KviSendWnd::sendToDaemon(QString &szPort,QString &szIp,const char *szFileName)
{
	_debug_entertrace("sendToDaemon");
	QString szStat;
	m_lpFrm->setStatusText((szStat.sprintf(i18n("Connecting to %s"),szIp.data())).data());
	m_szRemoteNick=i18n("Unknown");
	m_szFileName=szFileName;
	//Check for the file to send...
	if(m_lpFile||m_bConnected)_debug_fatal("WARNING Double used!");
	m_lpFile=new QFile(szFileName);
	if(!m_lpFile->open(IO_ReadOnly)){
		//No file....
		delete m_lpFile;
		m_lpFile=0;
		requestFailed((szStat.sprintf(i18n("Can not open file %s for reading"),szFileName)).data());
		return false;
	}
	//Good the file exists...
	QFileInfo aFileInfo(*m_lpFile);
	m_iFileLength=aFileInfo.size();
	//Check for the port
	bool bOK=false;
	unsigned short int iPort=(unsigned short int)szPort.toUInt(&bOK);
	if(!bOK){
		requestFailed(i18n("Can't retrieve the port to connect"));
		return false;	
	}
	//Check the IP
	szIp=szIp.stripWhiteSpace();
	struct in_addr inAddress;
	if(!inet_aton(szIp.data(),&inAddress)){
		requestFailed(i18n("Can't retrieve the address to connect"));
		return false;	
	}
	//Ready....
	m_szRemoteIp=szIp.data();
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Connecting to %s on port %s"),szIp.data(),szPort.data());
	//Go!
	struct sockaddr_in sockAddrIn;
	m_lpFrm->m_lpSock->fillSockAddr(&sockAddrIn,inAddress,iPort);
	m_Sock = ::socket(PF_INET,SOCK_STREAM,0);
	if (m_Sock < 0){
		requestFailed(i18n("Unable to create STREAM socket for connection"));
		return false;
	}
	if(fcntl(m_Sock, F_SETFL, O_NONBLOCK) < 0){
		::close(m_Sock);
		requestFailed(i18n("Unable to create STREAM socket for connection (fcntl failed)"));
		return false;
	};
	m_lpTimer=new QTimer();
	QObject::connect(m_lpTimer,SIGNAL(timeout()),this,SLOT(connectTimedOut()));
	m_lpTimer->start(m_lpFrm->m_lpOpt->iDCCCHATTimeout * 1000);
	if(::connect( m_Sock, (struct sockaddr*)(&sockAddrIn), sizeof(sockAddrIn) )<0){
		if(errno != EINPROGRESS){
			killSock();
			QString szError;
			m_lpFrm->m_lpSock->getErr(m_lpFrm->m_lpSock->translateFromSystemError(errno),szError);
			requestFailed((szStat.sprintf(i18n("Unable to connect to %s : %s"),m_szRemoteIp.data(),szError.data())).data());
			return false;
		}
	}
	m_lpReadNotifier = new QSocketNotifier(m_Sock, QSocketNotifier::Read,this);
	m_lpWriteNotifier= new QSocketNotifier(m_Sock, QSocketNotifier::Write,this);
	QObject::connect(m_lpReadNotifier, SIGNAL(activated(int)), this, SLOT(slotDataInBuffer(int)));
	QObject::connect(m_lpWriteNotifier, SIGNAL(activated(int)), this, SLOT(slotConnectedToDaemon(int)));
	return true;
	_debug_leavetrace("sendToDaemon");
}

//============ slotConnectedToDaemon ============//

void KviSendWnd::slotConnectedToDaemon(int)
{
	_debug_entertrace("slotConnectedToDaemon");
	if(m_lpWriteNotifier){
		delete m_lpWriteNotifier;
		m_lpWriteNotifier=0;
	}
	destroyTimer();
	m_bConnected=true;

	int nSockErr;
	ksize_t iSize=sizeof(int);
	if(getsockopt(m_Sock,SOL_SOCKET,SO_ERROR,(void *)&nSockErr,&iSize)==-1)nSockErr=EBADF;
	if(nSockErr!=0){
		QString szErr;
		m_lpFrm->m_lpSock->getErr(m_lpFrm->m_lpSock->translateFromSystemError(nSockErr),szErr);
		doFmtOutput(KVI_OUT_ERROR,i18n("Connect failed : %s"),szErr.data());
		killSock();
		return;
	}
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Connected to daemon (%s)"),m_szRemoteIp.data());
	doOutput(KVI_OUT_INTERNAL,i18n("Logging in..."));
	// Accepted connection send : 120 MyNickname Filesize Filename
	// and wait for acknowledgement
	QString szN=m_lpFrm->m_lpGlb->szNick.data();
	if(szN.isEmpty() ||(!strcmp(szN.data(),KVI_STR_NULL))){
		szN=m_lpFrm->m_lpUsr->szNick.data();
		if(szN.isEmpty()||(!strcmp(szN.data(),KVI_STR_NULL))){
			doFmtOutput(KVI_OUT_INTERNAL,i18n("You have specified no nickname; Presenting myself as UNKNOWN"));
			szN=i18n("UNKNOWN");
		}
	}
	QString szDataToSend;
	QFileInfo fInfo(m_szFileName.data());
	szDataToSend.sprintf("120 %s %d %s\r\n",szN.data(),m_iFileLength,fInfo.fileName().data());
	::write(m_Sock,szDataToSend.data(),strlen(szDataToSend.data()));
	// Now wait for response : 121 ServerNickname ResumePosition
	m_bWaitForAck=true;
	_debug_leavetrace("slotConnectedToDaemon");
}


//============ handleDaemonResponse ============//

void KviSendWnd::handleDaemonResponse()
{
	_debug_entertrace("handleDaemonResponse");
	m_bWaitForAck=false;
	char szIncomingData[1025];
   	bzero(szIncomingData, 1025); //just a space for the terminator
	int readLength = read(m_Sock,szIncomingData, 1024);
	if (readLength <= 0){
		if(readLength==0){
			QString szStat;
			szStat.sprintf(i18n("DCC SEND to %s@%s terminated"),m_szRemoteNick.data(),m_szRemoteIp.data());
			m_lpFrm->setStatusText(szStat.data());
			doFmtOutput(KVI_OUT_INTERNAL,i18n("DCC SEND to %s@%s terminated (Remote end closed connection)"),
				m_szRemoteNick.data(),m_szRemoteIp.data());
			killSock();
		} else {
			if((errno != EINTR)&&(errno != EAGAIN)){
				QString szStat;
				szStat.sprintf(i18n("DCC SEND with %s@%s terminated"),m_szRemoteNick.data(),m_szRemoteIp.data());
				m_lpFrm->setStatusText(szStat.data());
				if(errno==111)doFmtOutput(KVI_OUT_INTERNAL,i18n("DCC SEND to %s%s failed : Connection refused (Remote end closed listening socket)"),
						m_szRemoteNick.data(),m_szRemoteIp.data());
				else {
					QString szErr;
					m_lpFrm->m_lpSock->getErr(m_lpFrm->m_lpSock->translateFromSystemError(errno),szErr);
					doFmtOutput(KVI_OUT_INTERNAL,i18n("DCC SEND to %s@%s terminated by a transmission error (%s)"),
						m_szRemoteNick.data(),m_szRemoteIp.data(),szErr.data());
				}
				killSock();
			}
		}
  	} else {
		QString szIncomingString(szIncomingData,readLength+1);
		bool bGotResponse=false;
		if(!szIncomingString.isEmpty()){
			if(!strncmp(szIncomingString.data(),"121 ",4)){
				szIncomingString.remove(0,4);
				if(!szIncomingString.isEmpty()){
					int idx=szIncomingString.find(' ');
					if(idx != -1){
						m_szRemoteNick=szIncomingString.left(idx);
						szIncomingString.remove(0,idx+1);
						QString szResumePos;
						idx=szIncomingString.find("\r");
						if(idx != -1){
							szResumePos=szIncomingString.left(idx);
							idx=szResumePos.find("\n");
							if(idx != -1)szResumePos=szResumePos.left(idx);
						} else {
							idx=szIncomingString.find("\n");
							if(idx != -1)szResumePos=szIncomingString.left(idx);
							else szResumePos=szIncomingString.data();
						}
						bool bOk=false;
						uint resumePos=szIncomingString.toUInt(&bOk);
						if(bOk){
							if(resumePos > 0){
								if(dccResumeForCurrentSend(resumePos))bGotResponse=true;
							} else {
								bGotResponse=true;
								m_iFileSent=0;
							}
						}
					}
				}
			}
		}
		if(bGotResponse){
			doFmtOutput(KVI_OUT_INTERNAL,i18n("Negotiation complete : remote nick is %s"),m_szRemoteNick.data());
			m_bConnected=true;
			m_bSending=true;
			m_startTime=time(0);
			if(m_lpFrm->m_lpOpt->bUseFastSend){
				m_iPacketSize=m_lpFrm->m_lpOpt->iDCCSendPacketSize;
				m_lpFastSendTimer=new QTimer();
				connect(m_lpFastSendTimer,SIGNAL(timeout()),this,SLOT(sendFileBuffer()));
				if((m_lpFrm->m_lpOpt->iFSPacketsPerSec < 1)||(m_lpFrm->m_lpOpt->iFSPacketsPerSec>20))
							m_lpFrm->m_lpOpt->iFSPacketsPerSec=2;
				//WARNING : 20 pck/sec means 50ms timer interval : needs 500MhZ processor to work without blocking!
				doFmtOutput(KVI_OUT_INTERNAL,i18n("Enabled fast send with %d packets/sec. of %d bytes each one....good luck!"),
						m_lpFrm->m_lpOpt->iFSPacketsPerSec,m_iPacketSize);
				m_lpFastSendTimer->start(1000 / m_lpFrm->m_lpOpt->iFSPacketsPerSec);
			} else sendFileBuffer();
		} else {
			QString szStat;
			szStat.sprintf(i18n("DCC SEND with %s@%s terminated"),m_szRemoteNick.data(),m_szRemoteIp.data());
			m_lpFrm->setStatusText(szStat.data());
			doFmtOutput(KVI_OUT_INTERNAL,i18n("DCC SEND to %s@%s terminated : incomplete response"),
					m_szRemoteNick.data(),m_szRemoteIp.data());
			killSock();
		}
	}
	_debug_leavetrace("handleDaemonResponse");
}

//============ requestSend ============//

bool KviSendWnd::requestSend(const char *szNick,const char *szFileName)
{
	_debug_entertrace("requestSend");
	QString szStat;	
	m_lpFrm->setStatusText((szStat.sprintf(i18n("Requesting DCC SEND connection to %s"),szNick)).data());
	m_szFileName=szFileName;
	if(m_lpFile||m_bConnected)_debug_fatal("Can not use this window two times");
	m_lpFile=new QFile(szFileName);
	if(!m_lpFile->open(IO_ReadOnly)){
		delete m_lpFile;
		m_lpFile=0;
		doFmtOutput(KVI_OUT_ERROR,i18n("Can not open file %s for reading"),szFileName);
		m_lpFrm->setStatusText(i18n("DCC SEND Request failed"));
		return false;
	}
	QFileInfo aFileInfo(*m_lpFile);
	m_iFileLength=aFileInfo.size();
	m_szRemoteNick=szNick;
	if(!m_lpFrm->m_lpSock->m_bConnected){
		doOutput(KVI_OUT_ERROR,i18n("DCC SEND Requires an active IRC connection"));
		m_lpFrm->setStatusText(i18n("DCC SEND Request failed"));
		return false;
	}
	unsigned long myAddr=m_lpFrm->m_lpSock->getSockAddress();
	if(!myAddr){
		doOutput(KVI_OUT_ERROR,i18n("Unable to resolve localhost, can't request DCC SEND"));
		m_lpFrm->setStatusText(i18n("DCC SEND Request failed"));
		return false;
	}
	m_lpListeningSocket=new KviListeningSocket();
	if(!m_lpListeningSocket->listenForConnection(0,&m_Sock,m_lpFrm)){
		doOutput(KVI_OUT_ERROR,i18n("Unable to setup a listening socket, can't request a DCC SEND"));
		m_lpFrm->setStatusText(i18n("DCC SEND Request failed"));
		killSock();
		return false;
	}
	QObject::connect(m_lpListeningSocket,SIGNAL(gotConnection()),this,SLOT(connectedSlot()));
	QObject::connect(m_lpListeningSocket,SIGNAL(connectTimedOut()),this,SLOT(connectTimedOut()));
	QString szAddr;
	QString szPort;
	szAddr.setNum(myAddr);
	szPort.setNum(m_lpListeningSocket->m_iPort);
	m_iPort=m_lpListeningSocket->m_iPort;
	m_lpFrm->m_lpSock->sendFmtData("PRIVMSG %s :%cDCC SEND %s %s %s %u%c",szNick,0x01,
				aFileInfo.fileName().data(),szAddr.data(),szPort.data(),m_iFileLength,0x01);
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Sent DCC SEND [%s:%u] request to %s...waiting for reply"),aFileInfo.fileName().data(),m_iFileLength,szNick);
	m_iFileSent=0;
	m_bSending=true;
	_debug_leavetrace("requestSend");
	return true;
}

//============ dccResumeForCurrentSend ============//

bool KviSendWnd::dccResumeForCurrentSend(uint iPosition)
{
	_debug_entertrace("dccResumeForCurrentSend");
	// iPosition is the file size already sent (say 100 bytes)
	m_iFileSent=iPosition; //so sent 100 bytes
	doFmtOutput(KVI_OUT_INTERNAL,i18n("%s requested a DCC RESUME for %s (%u bytes)"),m_szRemoteNick.data(),m_szFileName.data(),m_iFileLength);
	if(m_iFileLength<=iPosition){
		doFmtOutput(KVI_OUT_INTERNAL,i18n("WARNING : File position requested is bigger that file size.Resuming NORMAL dcc send!"));
		m_iFileSent=0;
		return false;
	}
	m_lpFile->at(iPosition); //and the 101st byte is at position 100 in the file (0 based index)
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Acknowledged DCC RESUME : Transfer will start from position %u"),iPosition);
	_debug_leavetrace("dccResumeForCurrentSend");
	return true;
}

//============ destoryTimer ============//

void KviSendWnd::destroyTimer()
{
	_debug_entertrace("destoryTimer");
	if(m_lpTimer){
		if(m_lpTimer->isActive())m_lpTimer->stop();
		delete m_lpTimer;
		m_lpTimer=0;
	}
	_debug_leavetrace("destoryTimer");
}


//============ destroyReadNotifier ============//

void KviSendWnd::destroyReadNotifier()
{
	_debug_entertrace("destroyReadNotifier");
	if(m_lpReadNotifier){
		if(m_lpReadNotifier->isEnabled())m_lpReadNotifier->setEnabled(false);
		delete m_lpReadNotifier;
		m_lpReadNotifier=0;
	}
	_debug_leavetrace("destroyReadNotifier");
}


//============ destroyWriteNotifier ============//

void KviSendWnd::destroyWriteNotifier()
{
	_debug_entertrace("destroyWriteNotifier");
	if(m_lpWriteNotifier){
		if(m_lpWriteNotifier->isEnabled())m_lpWriteNotifier->setEnabled(false);
		delete m_lpWriteNotifier;
		m_lpWriteNotifier=0;
	}
	_debug_leavetrace("destroyWriteNotifier");
}


//============ killNotifiers ============//

void KviSendWnd::killNotifiers()
{
	_debug_entertrace("killNotifiers");
	destroyWriteNotifier();
	destroyReadNotifier();
	_debug_leavetrace("killNotifiers");
}

//============ killSock ============//

void KviSendWnd::killSock()
{
	_debug_entertrace("killSock");
	destroyListeningSocket();
	killNotifiers();
	destroyTimer();
	destroyFile();
	if(m_Sock != -1)::close(m_Sock);
	m_bConnected=false;
	_debug_leavetrace("killSock");
}
//============ connectedSlot ============//
void KviSendWnd::connectedSlot()
{
	_debug_entertrace("connectedSlot");
	m_bSending=true;
	QString szName=QString(inet_ntoa(m_lpListeningSocket->connectedAddr.sin_addr));
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Connected to %s on port %u"),szName.data(),m_iPort);
	m_lpReadNotifier=new QSocketNotifier(m_Sock,QSocketNotifier::Read,this);
	QObject::connect(m_lpReadNotifier, SIGNAL(activated(int)), this, SLOT(slotDataInBuffer(int)));
	m_bConnected=true;
	m_startTime=time(0);
	if(m_lpFrm->m_lpOpt->bUseFastSend){
		m_iPacketSize=m_lpFrm->m_lpOpt->iDCCSendPacketSize;
		m_lpFastSendTimer=new QTimer();
		connect(m_lpFastSendTimer,SIGNAL(timeout()),this,SLOT(sendFileBuffer()));
		if((m_lpFrm->m_lpOpt->iFSPacketsPerSec < 1)||(m_lpFrm->m_lpOpt->iFSPacketsPerSec>20))
					m_lpFrm->m_lpOpt->iFSPacketsPerSec=2;
		//WARNING : 20 pck/sec means 50ms timer interval : needs 500MhZ processor to work without blocking!
		doFmtOutput(KVI_OUT_INTERNAL,i18n("Enabled fast send with %d packets/sec. of %d bytes each one....good luck!"),
				m_lpFrm->m_lpOpt->iFSPacketsPerSec,m_iPacketSize);
		m_lpFastSendTimer->start(1000 / m_lpFrm->m_lpOpt->iFSPacketsPerSec);
	} else sendFileBuffer();
	_debug_leavetrace("connectedSlot");
}
//============ connectTimedOut ============//
void KviSendWnd::connectTimedOut()
{
	_debug_entertrace("connectTimedOut");
	terminateByError("SEND",i18n("by a socket timeout"));
	_debug_leavetrace("connectTimedOut");
}

//============ terminateByError ============//

void KviSendWnd::terminateByError(const char *szType,const char *szErr)
{
	_debug_entertrace("terminateByError");
	QString szStat;
	szStat.sprintf(i18n("DCC %s session with %s terminated"),szType,m_szRemoteNick.data());
	m_lpFrm->setStatusText(szStat.data());
	doFmtOutput(KVI_OUT_ERROR,i18n("DCC %s session with %s terminated %s"),szType,m_szRemoteNick.data(),szErr);
	if(m_lpListeningSocket)m_lpListeningSocket->stopListening();
	destroyReadNotifier();
	destroyWriteNotifier();
	destroyTimer();
	destroyFile();
	if(m_Sock != -1)::close(m_Sock);
	m_bConnected=false;
	_debug_leavetrace("terminateByError");
}


//============ terminateBySuccess ============//

void KviSendWnd::terminateBySuccess(const char *szType)
{
	_debug_entertrace("terminateBySuccess");
	QString szStat;
	szStat.sprintf(i18n("DCC %s with %s terminated"),szType,m_szRemoteNick.data());
	m_lpFrm->setStatusText(szStat.data());
	doFmtOutput(KVI_OUT_INTERNAL,i18n("DCC %s with %s succesfully terminated"),szType,m_szRemoteNick.data(),errno);
	killSock();
	if(m_lpFrm->m_lpOpt->bAutoCloseDCCChatOnSuccess)emit closeButtonPressed();
	_debug_leavetrace("terminateBySuccess");
}


//============ destroyFile ============//

void KviSendWnd::destroyFile()
{
	_debug_entertrace("destroyFile");
	if(m_lpFile){
		m_lpFile->close();
		delete m_lpFile;
		m_lpFile=0;
	}
	if(m_lpFastSendTimer){
		if(m_lpFastSendTimer->isActive())m_lpFastSendTimer->stop();
		delete m_lpFastSendTimer;
		m_lpFastSendTimer=0;
	}
	_debug_leavetrace("destroyFile");
}


//============ destroyListeningSocket ============//

void KviSendWnd::destroyListeningSocket()
{
	_debug_entertrace("destroyListeningSocket");
	if(m_lpListeningSocket){
		delete m_lpListeningSocket;
		m_lpListeningSocket=0;
	}
	_debug_leavetrace("destroyListeningSocket");
}


//============ setProgress ============//

void KviSendWnd::setProgress(int progress)
{
	_debug_entertrace("setProgress");
	m_lpProgress->setValue(progress);
	m_lpMdi->m_lpTaskBar->setButtonProgress(id(),progress);
	_debug_leavetrace("setProgress");
}


//============ slotDataInBuffer ============//

void KviSendWnd::slotDataInBuffer(int)
{
	_debug_entertrace("slotDataInBuffer");
	if(m_lpListeningSocket)destroyListeningSocket();
	if(m_bWaitForAck)handleDaemonResponse();
	if(m_bSending){
		uint ack;
		int readLength = read(m_Sock,&ack,sizeof(int));
		m_iNumAck=ntohl(ack);
		if(readLength<=0){
			if(readLength==0)terminateByError("SEND",i18n("(remote end closed connection)"));
			else {
				if((errno != EINTR)&&(errno != EAGAIN))terminateByError("SEND",i18n("by a transmission error"));
			}
		} else {
			if(m_iNumAck>=m_iFileLength){
				setLabelInfo(m_iNumAck,m_iFileSent);
				if(m_iFileLength)setProgress((m_iNumAck * 100)/m_iFileLength);
				destroyFile();
				terminateBySuccess("SEND");
				return;
			} else {
				if(!m_lpFastSendTimer){
					if(m_iNumAck==m_iFileSent){
						setLabelInfo(m_iNumAck,m_iFileSent);
						if(m_iFileLength)setProgress((m_iNumAck * 100)/m_iFileLength);
						sendFileBuffer();
					} else {
						setLabelInfo(m_iNumAck,m_iFileSent,i18n("stalled : partial ack"));
						if(m_iFileLength)setProgress((m_iNumAck * 100)/m_iFileLength);
						// else the remote side has smaller read buffer , wait for next ack
					}
				} else {
					setLabelInfo(m_iNumAck,m_iFileSent);
					if(m_iFileLength)setProgress((m_iNumAck * 100)/m_iFileLength);
				}
			}
		}
	} else {
		char szIncomingData[2048];
   		bzero(szIncomingData, 2048);
		int readLength = read(m_Sock,szIncomingData, 2048);
		if (readLength <= 0){
			if(readLength==0){
				if(m_iFileLength == m_iFileRead)terminateBySuccess("GET");
				else terminateByError("GET",i18n("(remote end closed connection)"));
			} else if((errno != EINTR)&&(errno != EAGAIN)){
				QString szErr=i18n("by a transmission error");
				switch(errno){
					case EIO:
						szErr+=i18n(" (Low level I/O error)");
						break;
					case EBADF:
						szErr+=i18n(" (Bad file descriptor)");
						break;
					case EPIPE:
						szErr+=i18n(" (Broken pipe)");
						break;
					case EREMOTEIO:
						szErr+=i18n(" (Remote io error)");
						break;
					case ENOBUFS:
						szErr+=i18n(" (No buffer space available)");
						break;
					case 0:
						szErr+=i18n(" (Error 0)");
						break;
				}
				terminateByError("GET",szErr.data());
			}
  		} else {
			m_iFileRead+=readLength;
			setLabelInfo(m_iFileRead,0);
			if(m_iFileLength)setProgress((m_iFileRead * 100)/m_iFileLength);
			uint toAckBytes=htonl(m_iFileRead);
			if(m_lpFile->writeBlock(szIncomingData,readLength)<0)terminateByError("GET",i18n("by a file I/O Error"));
			else {
				::write(m_Sock,&toAckBytes,sizeof(int));
				if(m_iFileLength==m_iFileRead)doFmtOutput(KVI_OUT_INTERNAL,i18n("File transfer complete"));
			}
		}
	}
	_debug_leavetrace("slotDataInBuffer");
}

//============ setLabelInfo ============//

void KviSendWnd::setLabelInfo(uint iBytesLeft,uint iBytesRight,const char *msg)
{
	_debug_entertrace("setLabelInfo");
	uint dur=(time(0)-m_startTime);
	if(dur==0)dur=1;
	uint bps=iBytesLeft/dur;
	QString szTxt;
	szTxt.sprintf(i18n("[File: %s ("),m_szFileName.data());
	QString szStats;
	if(m_iFileLength){
		szStats.sprintf(i18n("%u bytes)]"),m_iFileLength);
		szTxt+=szStats.data();
	} else szTxt+=i18n("unknown size)]");
	if(m_bSending){
		szStats.sprintf(i18n("[Ack: %u bytes][Sent: %u bytes][Time: %u sec.]"),iBytesLeft,iBytesRight,dur);
	} else {
		szStats.sprintf(i18n("[Received: %u bytes][Time: %u sec]"),iBytesLeft,dur); 
	}
	szTxt+=szStats.data();
	if(msg){
		szTxt+="[";
		szTxt+=msg;
		szTxt+="]";
	} else {
		szStats.sprintf(i18n("[%u bytes/sec.]"),bps);
		szTxt+=szStats.data();
	}
	m_lpLabel->setText(szTxt.data());
	_debug_leavetrace("setLabelInfo");
}


//============ sendFileBuffer ============//

void KviSendWnd::sendFileBuffer()
{
	_debug_entertrace("sendFileBuffer");
	if(m_lpFastSendTimer){
		FD_ZERO(&m_SocketSet);
		FD_SET(m_Sock, &m_SocketSet);
		// Wait 1 msec
		m_timeval.tv_sec = 0;
		m_timeval.tv_usec = 1;
		if(::select(m_Sock+1,0,&m_SocketSet,0,&m_timeval))m_nFailedSends=0;
		else {
			if(m_nFailedSends < 5)m_nFailedSends++;
			else { //Stalled : queue full...watinig
				if(m_nFailedSends==5){
					m_nFailedSends++;
					setLabelInfo(m_iNumAck,m_iFileSent,i18n("stalled : queue full"));
				}
			}
			return;
		}
	}
	int sizeToRead=m_iFileLength-m_iFileSent;
	if(sizeToRead<=0){
		if(m_lpFastSendTimer){
			if(m_lpFastSendTimer->isActive())m_lpFastSendTimer->stop();
		}
		return;
	}
	if(sizeToRead>m_iPacketSize)sizeToRead=m_iPacketSize;
	char szBlock[4096];
	if(m_lpFile->readBlock(szBlock,sizeToRead)<0){
		//Just to prevent further timeouts when the timer is really fast
		if(m_lpFastSendTimer){
			if(m_lpFastSendTimer->isActive())m_lpFastSendTimer->stop();
		}
		terminateByError("SEND",i18n("by a file I/O Error"));
		return;
	}
	int wroteBytes=::write(m_Sock,szBlock,sizeToRead);
	if(wroteBytes<sizeToRead){
		if(wroteBytes<0){
			if(errno==EAGAIN){
				if(!m_lpFastSendTimer){
					terminateByError("SEND",i18n("by a socket I/O Error (Outgoing buffer overflow)"));
					return;
				}
				doFmtOutput(KVI_OUT_INTERNAL,i18n("Warning : no room for data in the outgoing buffer : skipping. (Not dangerous,yet...)"));
			} else {
				//Just to prevent further timeouts when the timer is really fast
				if(m_lpFastSendTimer){
					if(m_lpFastSendTimer->isActive())m_lpFastSendTimer->stop();
				}
				QString szErr=i18n("by a socket I/O Error");
				if(errno==EINTR)szErr+=i18n(" (Socket write interrupted by a signal)");
				if(errno==EPIPE)szErr+=i18n(" (Broken pipe)");
				terminateByError("SEND",szErr.data());
				return;
			}
		} else {
			doFmtOutput(KVI_OUT_INTERNAL,i18n("Warning : socket write operation reduced to %d bytes of %d. (Not dangerous,yet...)"),wroteBytes,sizeToRead);
			sizeToRead=wroteBytes;
		}
	}
	m_iFileSent+=sizeToRead;
	setLabelInfo(m_iNumAck,m_iFileSent);
	_debug_leavetrace("sendFileBuffer");
}

#include "m_kvi_send.moc"

//
// $Log: kvi_send.cpp,v $
// Revision 1.10  1998/10/06 14:42:44  pragma
// Tons of changes
//
// Revision 1.9  1998/09/23 12:41:49  pragma
// Another big commit.
// Added user list (Still have to fix ignore things)
// User list dialogs
// Some code rearrangements
// Minor bugfixes
//
// Revision 1.8  1998/09/20 20:23:38  fritz
// reorganized includes.
// More work on srvdlg - still not finished.
//
// Revision 1.7  1998/09/16 17:16:51  fritz
// Starting i18n.
//
// Revision 1.6  1998/09/16 12:58:07  pragma
// Fixed some message strings.
//
// Revision 1.5  1998/09/16 03:29:40  pragma
// The fsend will make me go mad...
// Fixed the ::write(sock) return value handling to avoid stopping on EAGAIN
// if we're using a fsend.
// In 'normal' DCC send we have no chance to try again so we terminate
// with an error (something like 'buffer overflow'...but it should not happen)
// Added also better error handling...(Some new specific error msgs.)
// Curious : closing a fsend DCC before it's finished causes ::read(sock) to
// return EPIPE on the remote end. Why?
//
// Revision 1.4  1998/09/15 13:21:43  pragma
// Another small hack in the DCC fsend system.
//
//
