//   $Id: kvi_script.cpp,v 1.4 1998/09/20 20:23:35 fritz Exp $
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CLASS_NAME_ "KviScriptManager"

#include "kvi_macros.h"
#include "kvi_script.h"
#include "kvi_debug.h"
#include "kvi_frame.h"
#include "kvi_app.h"
#include "kvi_mirc.h"
#include "kvi_opt.h"
#include "kvi_event.h"
#include "kvi_alias.h"
#include "kvi_popup.h"
#include "kvi_int.h"

#include <kmsgbox.h>

#define _macro_readAndCheck(__stream,__var) __stream >> __var; if(__stream.device()->status() != IO_Ok)return false
#define _macro_log(__text) m_lpLog->append(__text); m_lpLog->append("\n")

#define KVI_SCRIPT_TAG_COLOR_ENTRY  ((Q_UINT32)101)
#define KVI_SCRIPT_TAG_COLOR_MAP    ((Q_UINT32)102)
#define KVI_SCRIPT_TAG_OPTIONS      ((Q_UINT32)103)
#define KVI_SCRIPT_TAG_ALIASES      ((Q_UINT32)104)
#define KVI_SCRIPT_TAG_SINGLE_ALIAS ((Q_UINT32)1040)
#define KVI_SCRIPT_TAG_EVENTS       ((Q_UINT32)105)
#define KVI_SCRIPT_TAG_SINGLE_EVENT ((Q_UINT32)1050)
#define KVI_SCRIPT_TAG_POPUPS       ((Q_UINT32)106)
#define KVI_SCRIPT_TAG_SINGLE_POPUP ((Q_UINT32)1060)
#define KVI_SCRIPT_TAG_INTERFACE    ((Q_UINT32)107)
#define KVI_SCRIPT_TAG_DIRECTORY    ((Q_UINT32)108)
#define KVI_SCRIPT_TAG_FILE         ((Q_UINT32)109)

//============ KviScriptManager ============//

KviScriptManager::KviScriptManager(KviFrame *lpFrm)
{
	_debug_entertrace("KviScriptManager");
	m_lpFrm=lpFrm;
	m_lpClr=m_lpFrm->m_lpClr;
	m_lpOpt=m_lpFrm->m_lpOpt;
	m_lpInt=m_lpFrm->m_lpInt;
	m_lpAli=m_lpFrm->m_lpAliasManager;
	m_lpPop=m_lpFrm->m_lpPopupManager;
	m_lpEve=m_lpFrm->m_lpEventManager;
	_debug_leavetrace("KviScriptManager");
}

//============ ~KviScriptManager ============//

KviScriptManager::~KviScriptManager()
{
	_debug_entertrace("~KviScriptManager");
	_debug_leavetrace("~KviScriptManager");
}

//============ scriptExit ============//

void KviScriptManager::scriptExit(const char *szMsg)
{
	_debug_entertrace("scriptExit");
	QString szM;
	szM.sprintf(i18n("The script you're trying to load seems to be corrupted.\n"
			 "This may be due to a download error.\n"
			 "Error message : %s'\n"
			 "KVirc will now exit without saving the changes."), szMsg);
	KMsgBox::message(m_lpFrm,i18n("Script manager error"),szM.data());
	exit(-1);
	_debug_leavetrace("scriptExit");
}

//============ saveToFile ============//

bool KviScriptManager::saveToFile(const char *fileName,KviScriptDescription *pD)
{
	_debug_entertrace("saveToFile");
	QFile theFile(fileName);
	if(!theFile.open(IO_WriteOnly|IO_Truncate)){
		KMsgBox::message(m_lpFrm,i18n("Script manager error"),i18n("Unable to open the specified file for writing."));
		return false;
	}
	QDataStream stream(&theFile);
	//write the signature
	Q_INT8 dummy='K';
	stream << dummy;
	dummy=(Q_INT8)'V';
	stream << dummy;
	dummy=(Q_INT8)'I';
	stream << dummy;
	dummy=(Q_INT8)'S';
	stream << dummy;
	dummy=(Q_INT8)'c';
	stream << dummy;
	dummy=(Q_INT8)'r';
	stream << dummy;
	dummy=(Q_INT8)'p';
	stream << dummy;
	dummy=(Q_INT8)'t';
	stream << dummy;
	//kvirc version
	stream << (Q_UINT16)KVI_UINT_VERSION;
	stream << (Q_UINT16)KVI_UINT_REVISION;
	//author and description
	stream << pD->szAuthor;
	stream << pD->szDescription;
	//entries
	stream << pD->hasColorEntries;
	stream << pD->hasColorMap;
	stream << pD->hasOptions;
	stream << pD->hasEvents;
	stream << pD->hasAliases;
	stream << pD->hasPopups;
	stream << pD->hasInterface;
	stream << pD->hasAudioFiles;
	stream << pD->hasImages;
	stream << pD->hasMiscFiles;
	stream << pD->hasAsciiFiles;
	//write color entries
	if(pD->hasColorEntries)writeColorEntries(stream);
	if(pD->hasColorMap)writeColorMap(stream);
	if(pD->hasOptions)writeOptions(stream);
	if(pD->hasEvents)writeEvents(stream);
	if(pD->hasAliases)writeAliases(stream);
	if(pD->hasPopups)writePopups(stream);
	if(pD->hasInterface)writeInterface(stream);
	if(pD->hasAudioFiles)writeDirContents("Scripts/Audio",stream);
	if(pD->hasImages)writeDirContents("Scripts/Images",stream);
	if(pD->hasMiscFiles)writeDirContents("Scripts/Misc",stream);
	if(pD->hasAsciiFiles)writeDirContents("Scripts/Ascii",stream);
	theFile.close();
	return true;
	_debug_leavetrace("saveToFile");
}


//============ readHeaderFromFile ============//

bool KviScriptManager::readHeaderFromFile(QDataStream &stream,KviScriptDescription *pD)
{
	_debug_entertrace("readHeaderFromFile");
	Q_INT8 dummy;
	_macro_readAndCheck(stream,dummy);
	if(dummy != (Q_UINT8)'K')return false;
	_macro_readAndCheck(stream,dummy);
	if(dummy != (Q_UINT8)'V')return false;
	_macro_readAndCheck(stream,dummy);
	if(dummy != (Q_UINT8)'I')return false;
	_macro_readAndCheck(stream,dummy);
	if(dummy != (Q_UINT8)'S')return false;
	_macro_readAndCheck(stream,dummy);
	if(dummy != (Q_UINT8)'c')return false;
	_macro_readAndCheck(stream,dummy);
	if(dummy != (Q_UINT8)'r')return false;
	_macro_readAndCheck(stream,dummy);
	if(dummy != (Q_UINT8)'p')return false;
	_macro_readAndCheck(stream,dummy);
	if(dummy != (Q_UINT8)'t')return false;
	//kvirc version
	_macro_readAndCheck(stream,pD->version);
	_macro_readAndCheck(stream,pD->revision);
	//author and description
	_macro_readAndCheck(stream,pD->szAuthor);
	_macro_readAndCheck(stream,pD->szDescription);
	//entries
	_macro_readAndCheck(stream,pD->hasColorEntries);
	_macro_readAndCheck(stream,pD->hasColorMap);
	_macro_readAndCheck(stream,pD->hasOptions);
	_macro_readAndCheck(stream,pD->hasEvents);
	_macro_readAndCheck(stream,pD->hasAliases);
	_macro_readAndCheck(stream,pD->hasPopups);
	_macro_readAndCheck(stream,pD->hasInterface);
	_macro_readAndCheck(stream,pD->hasAudioFiles);
	_macro_readAndCheck(stream,pD->hasImages);
	_macro_readAndCheck(stream,pD->hasMiscFiles);
	_macro_readAndCheck(stream,pD->hasAsciiFiles);
	_debug_leavetrace("readHeaderFromFile");
	return true;
}

//============ preloadHeaderAndCheckFile ============//

bool KviScriptManager::preloadHeaderAndCheckFile(const char *fileName,KviScriptDescription *pD)
{
	_debug_entertrace("preloadHeaderAndCheckFile");
	QFile theFile(fileName);
	if(!theFile.open(IO_ReadOnly)){
		KMsgBox::message(m_lpFrm,i18n("Script manager error"),i18n("Unable to open the specified file for reading."));
		return false;
	}
	QDataStream stream(&theFile);
	if(!readHeaderFromFile(stream,pD))return false;
	bool bCheck=true;
	m_lpLog=&(pD->szLog);
	_macro_log(i18n("CHECK-LOG FOR SCRIPT FILE:"));
	_macro_log(fileName);
	QFileInfo aInfo(theFile);
	QString szLog;
	szLog.sprintf(i18n(" - file size : %d bytes"),aInfo.size());
	_macro_log(szLog.data());
	szLog.sprintf(i18n(" - last modified : %s"),aInfo.lastModified().toString().data());
	_macro_log(szLog.data());
	_macro_log("---");
	_macro_log(i18n("Checking tagged blocks:"));
	if(pD->hasColorEntries)bCheck=checkColorEntries(stream);
	if(bCheck && pD->hasColorMap)bCheck=checkColorMap(stream);
	if(bCheck && pD->hasOptions)bCheck=checkOptions(stream);
	if(bCheck && pD->hasEvents)bCheck=checkEvents(stream);
	if(bCheck && pD->hasAliases)bCheck=checkAliases(stream);
	if(bCheck && pD->hasPopups)bCheck=checkPopups(stream);
	if(bCheck && pD->hasInterface)bCheck=checkInterface(stream);
	if(bCheck && pD->hasAudioFiles)bCheck=checkDirContents("Audio",stream);
	if(bCheck && pD->hasImages)bCheck=checkDirContents("Images",stream);
	if(bCheck && pD->hasMiscFiles)bCheck=checkDirContents("Misc",stream);
	if(bCheck && pD->hasAsciiFiles)bCheck=checkDirContents("Ascii",stream);
	_macro_log("---");
	_macro_log(i18n("Check succesful : script installable"));
	theFile.close();
	return bCheck;
	_debug_leavetrace("preloadHeaderAndCheckFile");
}

//============ installFromFile ============//

bool KviScriptManager::installFromFile(const char *fileName)
{
	_debug_entertrace("installFromFile");
	QFile theFile(fileName);
	if(!theFile.open(IO_ReadOnly)){
		KMsgBox::message(m_lpFrm,i18n("Script manager error"),i18n("Unable to open the specified file for reading."));
		return false;
	}
	QDataStream stream(&theFile);
	KviScriptDescription d;
	if(!readHeaderFromFile(stream,&d))return false;
	if(d.hasColorEntries)readColorEntries(stream);
	if(d.hasColorMap)readColorMap(stream);
	if(d.hasOptions)readOptions(stream);
	if(d.hasEvents)readEvents(stream);
	if(d.hasAliases)readAliases(stream);
	if(d.hasPopups)readPopups(stream);
	if(d.hasInterface)readInterface(stream);
	if(d.hasAudioFiles)readDirContents("Audio",stream);
	if(d.hasImages)readDirContents("Images",stream);
	if(d.hasMiscFiles)readDirContents("Misc",stream);
	if(d.hasAsciiFiles)readDirContents("Ascii",stream);
	theFile.close();
	return true;
	_debug_leavetrace("installFromFile");
}

//============ writeColorEntries ============//

void KviScriptManager::writeColorEntries(QDataStream &stream)
{
	_debug_entertrace("writeColorEntries");
	stream << KVI_SCRIPT_TAG_COLOR_ENTRY;
	stream << (Q_UINT32)KVI_NUM_PIXMAPS;
	Q_INT8 dummy;
	for(int i=0;i<KVI_NUM_PIXMAPS;i++){
		dummy=m_lpClr->chOutForeColor[i];
		stream << dummy;
		dummy = m_lpClr->chOutBackColor[i];
		stream << dummy;
	}
	dummy = m_lpClr->chNormalTextColor;
	stream << dummy;
	dummy = m_lpClr->chNormalBackColor;
	stream << dummy;
	_debug_leavetrace("writeColorEntries");
}

//============ readColorEntries ============//

void KviScriptManager::readColorEntries(QDataStream &stream)
{
	_debug_entertrace("readColorEntries");
	Q_UINT32 tag;
	stream >> tag;
	if(tag != KVI_SCRIPT_TAG_COLOR_ENTRY){
		scriptExit("Color entries with wrong tag");
	}
	Q_UINT32 numPixmaps;
	stream >> numPixmaps;
	Q_INT8 dummy;
	for(Q_UINT32 i=0;i<numPixmaps;i++){
		stream >> dummy;
		m_lpClr->chOutForeColor[i] = (char)dummy;
		stream >> dummy;
		m_lpClr->chOutBackColor[i] = (char)dummy;	
	}
	stream >> dummy;
	m_lpClr->chNormalTextColor = (char)dummy;
	stream >> dummy;
	m_lpClr->chNormalBackColor = (char)dummy;
	_debug_leavetrace("readColorEntries");
}

//============ checkColorEntries ============//

bool KviScriptManager::checkColorEntries(QDataStream &stream)
{
	_debug_entertrace("checkColorEntries");
	_macro_log(i18n("Block tagged : 'color entries'"));
	Q_UINT32 tag;
	_macro_readAndCheck(stream,tag);
	if(tag != KVI_SCRIPT_TAG_COLOR_ENTRY)return false;
	Q_UINT32 numEntries;
	_macro_readAndCheck(stream,numEntries);
	if(numEntries > KVI_NUM_PIXMAPS)return false;
	Q_INT8 dummy1;
	for(Q_UINT32 i=0;i<numEntries;i++){
		_macro_readAndCheck(stream,dummy1);
		_macro_readAndCheck(stream,dummy1);
	}
	_macro_readAndCheck(stream,dummy1);
	_macro_readAndCheck(stream,dummy1);
	QString szLog;
	szLog.sprintf(i18n(" # %d color setting entries"),(numEntries+1)*2);
	_macro_log(szLog.data());
	_debug_leavetrace("checkColorEntries");
	return true;
}
//============ writeColorMap ============//

void KviScriptManager::writeColorMap(QDataStream &stream)
{
	_debug_entertrace("writeColorMap");
	stream << KVI_SCRIPT_TAG_COLOR_MAP;
	for(int i=0;i<16;i++){
		stream << (*m_lpFrm->m_lpColor[i]);
	}
	_debug_leavetrace("writeColorMap");
}

//============ readColorMap ============//

void KviScriptManager::readColorMap(QDataStream &stream)
{
	_debug_entertrace("readColorMap");
	Q_UINT32 tag;
	stream >> tag;
	if(tag != KVI_SCRIPT_TAG_COLOR_MAP){
		scriptExit(i18n("Color map with wrong tag"));
	}
	for(int i=0;i<16;i++){
		stream >> (*m_lpFrm->m_lpColor[i]);
	}
	_debug_leavetrace("readColorMap");
}

//============ checkColorMap ============//

bool KviScriptManager::checkColorMap(QDataStream &stream)
{
	_debug_entertrace("checkColorMap");
	_macro_log(i18n("Block tagged : 'color map'"));
	Q_UINT32 tag;
	_macro_readAndCheck(stream,tag);
	if(tag != KVI_SCRIPT_TAG_COLOR_MAP)return false;
	QColor clr;
	for(int i=0;i<16;i++){
		_macro_readAndCheck(stream,clr);
	}
	_macro_log(i18n(" # 16 rgb color map entries"));
	_debug_leavetrace("checkColorMap");
	return true;
}

//============ writeOptions ============//

void KviScriptManager::writeOptions(QDataStream &stream)
{
	_debug_entertrace("writeOptions");
	stream << KVI_SCRIPT_TAG_OPTIONS;
	stream << m_lpOpt->szCaption;
	stream << m_lpOpt->szQuitMsg;
	stream << m_lpOpt->szPartMsg;
	stream << m_lpOpt->szKickMsg;
	stream << m_lpOpt->szAwayMsg;
	stream << m_lpOpt->szFinger;
	stream << m_lpOpt->szUserInfo;
	stream << m_lpOpt->szScriptVer;
	stream << m_lpOpt->massDeopDeadline;
	stream << m_lpOpt->massDeopTime;
	stream << m_lpOpt->massOpDeadline;
	stream << m_lpOpt->massOpTime;
	stream << m_lpOpt->massKickDeadline;
	stream << m_lpOpt->massKickTime;
	stream << m_lpOpt->massBanDeadline;
	stream << m_lpOpt->massBanTime;
	stream << m_lpOpt->publicFloodDeadline;
	stream << m_lpOpt->publicFloodTime;
	_debug_leavetrace("writeOptions");
}

//============ readOptions ============//

void KviScriptManager::readOptions(QDataStream &stream)
{
	_debug_entertrace("readOptions");
	Q_UINT32 tag;
	stream >> tag;
	if(tag != KVI_SCRIPT_TAG_OPTIONS){
		scriptExit(i18n("Options block with wrong delimiter tag"));
	}
	stream >> m_lpOpt->szCaption;
	stream >> m_lpOpt->szQuitMsg;
	stream >> m_lpOpt->szPartMsg;
	stream >> m_lpOpt->szKickMsg;
	stream >> m_lpOpt->szAwayMsg;
	stream >> m_lpOpt->szFinger;
	stream >> m_lpOpt->szUserInfo;
	stream >> m_lpOpt->szScriptVer;
	stream >> m_lpOpt->massDeopDeadline;
	stream >> m_lpOpt->massDeopTime;
	stream >> m_lpOpt->massOpDeadline;
	stream >> m_lpOpt->massOpTime;
	stream >> m_lpOpt->massKickDeadline;
	stream >> m_lpOpt->massKickTime;
	stream >> m_lpOpt->massBanDeadline;
	stream >> m_lpOpt->massBanTime;
	stream >> m_lpOpt->publicFloodDeadline;
	stream >> m_lpOpt->publicFloodTime;
	_debug_leavetrace("readOptions");
}

//============ checkOptions ============//

bool KviScriptManager::checkOptions(QDataStream &stream)
{
	_debug_entertrace("checkOptions");
	_macro_log(i18n("Block tagged : 'options'"));
	Q_UINT32 tag;
	_macro_readAndCheck(stream,tag);
	if(tag != KVI_SCRIPT_TAG_OPTIONS)return false;
	QString dummy;
	for(int i=0;i<8;i++){
		_macro_readAndCheck(stream,dummy);
	}
	_macro_log(i18n(" # 8 string entries [default messages]"));
	Q_UINT32 dumuint;
	for(int k=0;k<10;k++){
		_macro_readAndCheck(stream,dumuint);
	}
	_macro_log(i18n(" # 10 unsigned int entries [protection times and deadlines]"));
	_debug_leavetrace("checkOptions");
	return true;
}
//============ writeEvents ============//

void KviScriptManager::writeEvents(QDataStream &stream)
{
	_debug_entertrace("writeEvents");
	stream << KVI_SCRIPT_TAG_EVENTS;
	stream << ((Q_UINT32)KVI_EVENTS_NUMBER);
	for(int i=0;i<KVI_EVENTS_NUMBER;i++){
		stream << KVI_SCRIPT_TAG_SINGLE_EVENT;
		stream << m_lpEve->lpEvent[i]->szName;
		stream << m_lpEve->lpEvent[i]->szBuffer;
		Q_UINT32 dummy=m_lpEve->lpEvent[i]->bEnabled;
		stream << dummy;
	}
	_debug_leavetrace("writeEvents");
}

//============ readEvents ============//

void KviScriptManager::readEvents(QDataStream &stream)
{
	_debug_entertrace("readEvents");
	Q_UINT32 tag;
	stream >> tag;
	if(tag != KVI_SCRIPT_TAG_EVENTS){
		scriptExit("Events block with wrong header tag");
	}
	Q_UINT32 numEvents;
	stream >> numEvents;
	for(Q_UINT32 i=0;i<numEvents;i++){
		stream >> tag;
		if(tag != KVI_SCRIPT_TAG_SINGLE_EVENT){
			scriptExit(i18n("Event structure with wrong tag"));
		}
		QString dummyName;
		QString dummyBuffer;
		Q_UINT32 dummyEnabled;
		stream >> dummyName;
		stream >> dummyBuffer;
		stream >> dummyEnabled;
		bool bFound=false;
		for(int k=0;(k<KVI_EVENTS_NUMBER) && (!bFound);k++){
			if(!strcasecmp(dummyName.data(),m_lpEve->lpEvent[k]->szName.data())){
				bFound=true;
				m_lpEve->lpEvent[k]->szName=dummyName.copy();
				m_lpEve->lpEvent[k]->szBuffer=dummyBuffer.copy();
				m_lpEve->lpEvent[k]->bEnabled=(bool)dummyEnabled;
			}
		}
	}
	_debug_leavetrace("readEvents");
}

//============ checkEvents ============//

bool KviScriptManager::checkEvents(QDataStream &stream)
{
	_debug_entertrace("checkEvents");
	_macro_log(i18n("Block tagged : 'event handlers'"));
	Q_UINT32 tag;
	_macro_readAndCheck(stream,tag);
	if(tag != KVI_SCRIPT_TAG_EVENTS)return false;
	Q_UINT32 numEvents;
	_macro_readAndCheck(stream,numEvents);
	for(Q_UINT32 i=0;i<numEvents;i++){
		_macro_readAndCheck(stream,tag);
		if(tag != KVI_SCRIPT_TAG_SINGLE_EVENT)return false;
		QString dummy;
		Q_UINT32 dummyEnabled;
		QString szD;
		_macro_readAndCheck(stream,dummy);
		szD=dummy.copy();
		szD.prepend(" - event :");
		_macro_readAndCheck(stream,dummy);
		QString szLen;
		szLen.sprintf(i18n(" (buffer :%d bytes) "),dummy.length());
		szD.append(szLen.data());
		_macro_readAndCheck(stream,dummyEnabled);
		if(dummyEnabled)szD.append(i18n("enabled"));
		else szD.append(i18n("disabled"));
		_macro_log(szD.data());
	}
	QString szLog;
	szLog.sprintf(i18n(" # total %d event handler entries"),numEvents);
	_macro_log(szLog.data());
	_debug_leavetrace("checkEvents");
	return true;
}

//============ writeAliases ============//

void KviScriptManager::writeAliases(QDataStream &stream)
{
	_debug_entertrace("writeAliases");
	stream << KVI_SCRIPT_TAG_ALIASES;
	Q_UINT32 dummy=m_lpAli->m_lpAliasList->count();
	stream << dummy;
	for(uint i=0;i<m_lpAli->m_lpAliasList->count();i++){
		stream << KVI_SCRIPT_TAG_SINGLE_ALIAS;
		stream << m_lpAli->m_lpAliasList->at(i)->szName;
		stream << m_lpAli->m_lpAliasList->at(i)->szBuffer;
	}
	_debug_leavetrace("writeAliases");
}

//============ readAliases ============//

void KviScriptManager::readAliases(QDataStream &stream)
{
	_debug_entertrace("readAliases");
	Q_UINT32 tag;
	stream >> tag;
	if(tag != KVI_SCRIPT_TAG_ALIASES){
		scriptExit(i18n("Aliases block with wrong header tag"));
	}
	Q_UINT32 numAliases;
	stream >> numAliases;
	for(Q_UINT32 i=0;i<numAliases;i++){
		stream >> tag;
		if(tag != KVI_SCRIPT_TAG_SINGLE_ALIAS){
			scriptExit(i18n("Alias structure with wrong tag"));
		}
		QString dummyName;
		QString dummyBuffer;
		stream >> dummyName;
		stream >> dummyBuffer;
		if(m_lpAli->getAlias(dummyName.data()))m_lpAli->removeAlias(dummyName.data());
		m_lpAli->addAlias(dummyName.data(),dummyBuffer);
	}
	_debug_leavetrace("readAliases");
}

//============ checkAliases ============//

bool KviScriptManager::checkAliases(QDataStream &stream)
{
	_debug_entertrace("checkAliases");
	_macro_log(i18n("Block tagged : 'aliases'"));
	Q_UINT32 tag;
	_macro_readAndCheck(stream,tag);
	if(tag != KVI_SCRIPT_TAG_ALIASES)return false;
	Q_UINT32 numAliases;
	_macro_readAndCheck(stream,numAliases);
	for(Q_UINT32 i=0;i<numAliases;i++){
		_macro_readAndCheck(stream,tag);
		if(tag != KVI_SCRIPT_TAG_SINGLE_ALIAS)return false;
		QString dummy;
		_macro_readAndCheck(stream,dummy);
		QString szD=dummy.copy();
		szD.prepend(" - alias :");
		_macro_readAndCheck(stream,dummy);
		QString szLen;
		szLen.sprintf(i18n(" (buffer :%d bytes)"),dummy.length());
		szD.append(szLen.data());
		_macro_log(szD.data());
	}
	QString szLog;
	szLog.sprintf(i18n(" # total : %d alias entries"),numAliases);
	_macro_log(szLog.data());
	_debug_leavetrace("checkAliases");
	return true;
}

//============ writePopups ============//

void KviScriptManager::writePopups(QDataStream &stream)
{
	_debug_entertrace("writePopups");
	stream << KVI_SCRIPT_TAG_POPUPS;
	stream << m_lpPop->szNickListPopup;
	writePopup(stream,m_lpPop->m_lpUserNickListPopup);
	stream << m_lpPop->szChanPopup;
	writePopup(stream,m_lpPop->m_lpUserChanPopup);
	stream << m_lpPop->szActionPopup;
	writePopup(stream,m_lpPop->m_lpUserActionPopup);
	stream << m_lpPop->szQueryPopup;
	writePopup(stream,m_lpPop->m_lpUserQueryPopup);
	stream << m_lpPop->szChatPopup;
	writePopup(stream,m_lpPop->m_lpUserChatPopup);
	_debug_leavetrace("writePopups");
}

//============ readPopups ============//

void KviScriptManager::readPopups(QDataStream &stream)
{
	_debug_entertrace("readPopups");
	Q_UINT32 tag;
	stream >> tag;
	if(tag != KVI_SCRIPT_TAG_POPUPS){
		scriptExit(i18n("Popups block with wrong header tag"));
	}
	stream >> m_lpPop->szNickListPopup;
	readPopup(stream,m_lpPop->m_lpUserNickListPopup);
	stream >> m_lpPop->szChanPopup;
	readPopup(stream,m_lpPop->m_lpUserChanPopup);
	stream >> m_lpPop->szActionPopup;
	readPopup(stream,m_lpPop->m_lpUserActionPopup);
	stream >> m_lpPop->szQueryPopup;
	readPopup(stream,m_lpPop->m_lpUserQueryPopup);
	stream >> m_lpPop->szChatPopup;
	readPopup(stream,m_lpPop->m_lpUserChatPopup);
	_debug_leavetrace("readPopups");
}

//============ checkPopups ============//

bool KviScriptManager::checkPopups(QDataStream &stream)
{
	_debug_entertrace("checkPopups");
	_macro_log(i18n("Block tagged : 'popups'"));
	Q_UINT32 tag;
	_macro_readAndCheck(stream,tag);
	if(tag != KVI_SCRIPT_TAG_POPUPS)return false;
	QString dummy;
	QString szLog;
	_macro_readAndCheck(stream,dummy);
	szLog.sprintf(i18n(" @ User popup menu [Nick list] %s"),dummy.data());
	_macro_log(szLog.data());
	if(!checkPopup(stream,m_lpPop->m_lpUserNickListPopup))return false;
	_macro_readAndCheck(stream,dummy);
	szLog.sprintf(i18n(" @ User popup menu [Channel] %s"),dummy.data());
	_macro_log(szLog.data());
	if(!checkPopup(stream,m_lpPop->m_lpUserChanPopup))return false;
	_macro_readAndCheck(stream,dummy);
	szLog.sprintf(i18n(" @ User popup menu [Menu bar] %s"),dummy.data());
	_macro_log(szLog.data());
	if(!checkPopup(stream,m_lpPop->m_lpUserActionPopup))return false;
	_macro_readAndCheck(stream,dummy);
	szLog.sprintf(i18n(" @ User popup menu [Query] %s"),dummy.data());
	_macro_log(szLog.data());
	if(!checkPopup(stream,m_lpPop->m_lpUserQueryPopup))return false;
	_macro_readAndCheck(stream,dummy);
	szLog.sprintf(i18n(" @ User popup menu [DCC Chat] %s"),dummy.data());
	_macro_log(szLog.data());
	if(!checkPopup(stream,m_lpPop->m_lpUserChatPopup))return false;
	_macro_log(i18n(" # total : 5 user popup menu entries"));
	_debug_leavetrace("checkPopups");
	return true;
}
//============ writePopup ============//

void KviScriptManager::writePopup(QDataStream &stream,KviPopupMenu *m)
{
	_debug_entertrace("writePopup");
	stream << KVI_SCRIPT_TAG_SINGLE_POPUP;
	Q_UINT32 cnt=m->m_lpData->count();
	stream << cnt;
	KviPopupItem *lpI=0;
	for(lpI=m->m_lpData->first();lpI;lpI=m->m_lpData->next()){
		Q_INT32 dummy=lpI->id;
		stream << dummy;
		Q_INT32 type=2;
		if(lpI->isSep)type=0;
		else {
			if(lpI->lpMenu)type=1;
		}
		stream << type;
		if(type){
			stream << lpI->szText;
			if(type==1){
				writePopup(stream,lpI->lpMenu);
			} else {
				stream << lpI->szAction;
			}
		}	
	}
	_debug_leavetrace("writePopup");
}

//============ readPopup ============//

void KviScriptManager::readPopup(QDataStream &stream,KviPopupMenu *m)
{
	_debug_entertrace("readPopup");
	Q_UINT32 tag;
	stream >> tag;
	if(tag != KVI_SCRIPT_TAG_SINGLE_POPUP){
		scriptExit(i18n("Popup structure with wrong tag"));
	}
	m->clearPopup();
	Q_UINT32 cnt;
	stream >> cnt;
	for(Q_UINT32 i=0;i<cnt;i++){
		Q_INT32 id;
		stream >> id;
		Q_INT32 type;
		stream >> type;
		KviPopupItem *lpI=new KviPopupItem;
		switch(type){
			case 0:
				lpI->isSep=true;
				lpI->lpMenu=0;
				lpI->id=id;
				lpI->szAction="";
				m->addItem(lpI);		
				break;
			case 1:
				lpI->id=id;
				lpI->isSep=false;
				lpI->lpMenu=new KviPopupMenu();
				stream >> lpI->szText;
				lpI->szAction="";
				readPopup(stream,lpI->lpMenu);
				m->addItem(lpI);
				break;
			case 2:
				lpI->id=id;
				lpI->isSep=false;
				lpI->lpMenu=0;
				stream >> lpI->szText;
				stream >> lpI->szAction;
				m->addItem(lpI);
				break;
			default:
				KMsgBox::message(m_lpFrm,i18n("Script manager error"),
						 i18n("Corrupted popup menu entry.\n"
						      "The file is unusable.\nExiting without saving changes.\n"
						      "Please restart kvirc."));
				exit(-1);
				break;
		}
	}
	_debug_leavetrace("readPopup");
}

//============ checkPopup ============//

bool KviScriptManager::checkPopup(QDataStream &stream,KviPopupMenu *)
{
	_debug_entertrace("readPopup");
	_macro_log(i18n(" [ Internal block : popup menu ]"));
	Q_UINT32 tag;
	_macro_readAndCheck(stream,tag);
	if(tag != KVI_SCRIPT_TAG_SINGLE_POPUP)return false;
	Q_UINT32 cnt;
	_macro_readAndCheck(stream,cnt);
	QString szLog;
	szLog.sprintf(i18n(" --- %d menu entries"),cnt);
	_macro_log(szLog.data());
	for(Q_UINT32 i=0;i<cnt;i++){
		Q_INT32 id;
		_macro_readAndCheck(stream,id);
		Q_INT32 type;
		_macro_readAndCheck(stream,type);
		KviPopupItem *lpI=new KviPopupItem;
		QString szLog;
		switch(type){
			case 0:
				_macro_log(i18n(" ----- Separator")); 
				break;
			case 1:
				lpI->lpMenu=new KviPopupMenu();
				_macro_readAndCheck(stream,lpI->szText);
				szLog.sprintf(i18n(" ----- Popup %s"),lpI->szText.data());
				_macro_log(szLog.data());
				if(!checkPopup(stream,lpI->lpMenu))return false;
				break;
			case 2:
				_macro_readAndCheck(stream,lpI->szText);
				_macro_readAndCheck(stream,lpI->szAction);
				szLog.sprintf(i18n(" ----- Item %s (buffer %d bytes)"),lpI->szText.data(),lpI->szAction.length());
				_macro_log(szLog.data());
				break;
			default:
				return false;
				break;
		}
		delete lpI;
	}
	_macro_log(i18n(" [ End of popup block ] "));
	_debug_leavetrace("readPopup");
	return true;
}
//============ writeInterface ============//

void KviScriptManager::writeInterface(QDataStream &stream)
{
	_debug_entertrace("writeInterface");
	stream << KVI_SCRIPT_TAG_INTERFACE;
//	Q_UINT32 dummy;
	stream << m_lpInt->clr_bk_mdi;
	stream << m_lpInt->str_bk_mdi;
	stream << m_lpInt->clr_bk_act_caption;
	stream << m_lpInt->clr_bk_ina_caption;
	stream << m_lpInt->clr_fr_act_caption;
	stream << m_lpInt->clr_fr_ina_caption;
	stream << m_lpInt->fnt_caption;
	stream << m_lpInt->str_bk_caption;
	stream << m_lpInt->clr_bk_input;
	stream << m_lpInt->clr_fr_input;
	stream << m_lpInt->clr_bk_sel_input;
	stream << m_lpInt->clr_fr_sel_input;
	stream << m_lpInt->clr_cursor_input;
	stream << m_lpInt->fnt_input;
	stream << m_lpInt->str_bk_input;
	stream << m_lpInt->clr_bk_output;
	stream << m_lpInt->clr_bk_sel_output;
	stream << m_lpInt->clr_fr_sel_output;
	stream << m_lpInt->fnt_output;
	stream << m_lpInt->str_bk_output;
	stream << m_lpInt->clr_bk_listbox;
	stream << m_lpInt->clr_fr_normal_listbox;
	stream << m_lpInt->clr_fr_op_listbox;
	stream << m_lpInt->clr_fr_voice_listbox;
	stream << m_lpInt->clr_bk_sel_listbox;
	stream << m_lpInt->clr_fr_sel_listbox;
	stream << m_lpInt->str_bk_listbox;
	stream << m_lpInt->fnt_listbox;
	stream << m_lpInt->clr_bk_chanlabels;
	stream << m_lpInt->clr_fr_chanlabels;
	stream << m_lpInt->clr_fr_act_chanlabels;
	stream << m_lpInt->str_bk_chanlabels;
	stream << m_lpInt->fnt_chanlabels;

	_debug_leavetrace("writeInterface");
}

//============ readInterface ============//

void KviScriptManager::readInterface(QDataStream &stream)
{
	_debug_entertrace("readInterface");
	Q_UINT32 tag;
	stream >> tag;
	if(tag != KVI_SCRIPT_TAG_INTERFACE){
		scriptExit(i18n("Interface block with wrong header tag"));
	}
//	Q_UINT32 dummy;
	stream >> m_lpInt->clr_bk_mdi;
	stream >> m_lpInt->str_bk_mdi;
	stream >> m_lpInt->clr_bk_act_caption;
	stream >> m_lpInt->clr_bk_ina_caption;
	stream >> m_lpInt->clr_fr_act_caption;
	stream >> m_lpInt->clr_fr_ina_caption;
	stream >> m_lpInt->fnt_caption;
	stream >> m_lpInt->str_bk_caption;
	stream >> m_lpInt->clr_bk_input;
	stream >> m_lpInt->clr_fr_input;
	stream >> m_lpInt->clr_bk_sel_input;
	stream >> m_lpInt->clr_fr_sel_input;
	stream >> m_lpInt->clr_cursor_input;
	stream >> m_lpInt->fnt_input;
	stream >> m_lpInt->str_bk_input;
	stream >> m_lpInt->clr_bk_output;
	stream >> m_lpInt->clr_bk_sel_output;
	stream >> m_lpInt->clr_fr_sel_output;
	stream >> m_lpInt->fnt_output;
	stream >> m_lpInt->str_bk_output;
	stream >> m_lpInt->clr_bk_listbox;
	stream >> m_lpInt->clr_fr_normal_listbox;
	stream >> m_lpInt->clr_fr_op_listbox;
	stream >> m_lpInt->clr_fr_voice_listbox;
	stream >> m_lpInt->clr_bk_sel_listbox;
	stream >> m_lpInt->clr_fr_sel_listbox;
	stream >> m_lpInt->str_bk_listbox;
	stream >> m_lpInt->fnt_listbox;
	stream >> m_lpInt->clr_bk_chanlabels;
	stream >> m_lpInt->clr_fr_chanlabels;
	stream >> m_lpInt->clr_fr_act_chanlabels;
	stream >> m_lpInt->str_bk_chanlabels;
	stream >> m_lpInt->fnt_chanlabels;
	_debug_leavetrace("readInterface");
}

//============ checkInterface ============//

bool KviScriptManager::checkInterface(QDataStream &stream)
{
	_debug_entertrace("checkInterface");
	_macro_log(i18n("Block tagged : 'interface settings'"));
	Q_UINT32 tag;
	_macro_readAndCheck(stream,tag);
	if(tag != KVI_SCRIPT_TAG_INTERFACE)return false;
//	Q_UINT32 dUInt;
	QColor dColor;
	QString dString;
	QFont dFont;

	#define __chk_clr stream >> dColor;  if(stream.device()->status() != IO_Ok)return false
	#define __chk_str stream >> dString; if(stream.device()->status() != IO_Ok)return false
	#define __chk_fnt stream >> dFont;   if(stream.device()->status() != IO_Ok)return false
//	#define __chk_num stream >> dUInt;   if(stream.device()->status() != IO_Ok)return false

	#define __log_pix dString.prepend(i18n(" - Reference to image: "));m_lpLog->append(dString.data());m_lpLog->append("\n")
	#define __log_fnt m_lpLog->append(i18n(" - Reference to external font: "));m_lpLog->append(dFont.key().data());m_lpLog->append("\n")

	__chk_clr;
	__chk_str;
	__log_pix;
	__chk_clr;
	__chk_clr;
	__chk_clr;
	__chk_clr;
	__chk_fnt;
	__log_fnt;
	__chk_str;
	__log_pix;
	__chk_clr;
	__chk_clr;
	__chk_clr;
	__chk_clr;
	__chk_clr;
	__chk_fnt;
	__log_fnt;
	__chk_str;
	__log_pix;
//	__chk_num;
	__chk_clr;
	__chk_clr;
	__chk_clr;
	__chk_fnt;
	__log_fnt;
	__chk_str;
	__log_pix;
//	__chk_num;
//	__chk_num;
//	__chk_num;
	__chk_clr;
	__chk_clr;
	__chk_clr;
	__chk_clr;
	__chk_clr;
	__chk_clr;
	__chk_str;
	__log_pix;
	__chk_fnt;
	__chk_clr;
	__chk_clr;
	__chk_clr;
	__chk_str;
	__log_pix;
	__chk_fnt;
	__log_fnt;

	#undef __chk_clr
	#undef __chk_str
	#undef __chk_num
	#undef __chk_fnt
	#undef __log_pix
	#undef __log_fnt

	_debug_leavetrace("readInterface");
	return true;
}
//============ writeDirContents ============//

void KviScriptManager::writeDirContents(const char *szDir,QDataStream &stream)
{
	_debug_entertrace("writeDirContents");
	stream << KVI_SCRIPT_TAG_DIRECTORY;
	QString szHome=_macro_getKVircHomeDirectory(szDir);
	QDir aDir(szHome);
	aDir.setFilter(QDir::Files | QDir::NoSymLinks | QDir::Readable);
	uint numFiles=0;
	const QFileInfoList *l=aDir.entryInfoList();
	if(aDir.exists()){
		numFiles=l->count();
	}
	stream << (Q_UINT32)numFiles;
	if(numFiles==0)return;
	QFileInfoListIterator it(*l);
	for( ; it.current() ; ++it){
		stream << KVI_SCRIPT_TAG_FILE;
		QFileInfo *lpInfo=it.current();
		stream << lpInfo->fileName();
		QString szFileName=_macro_getKVircHomeDirectory(szDir);
		szFileName+='/';
		szFileName+=lpInfo->fileName();
		QFile theFile(szFileName.data());
		if((!theFile.open(IO_ReadOnly)) || (!lpInfo->size())){
			stream << (Q_UINT32)0;
			theFile.close();
		} else { 
			char * pBuf=new char[lpInfo->size()];
			if(theFile.readBlock(pBuf,lpInfo->size())<(int)lpInfo->size()){
				stream << (Q_UINT32)0;
			} else {
				stream << (Q_UINT32)lpInfo->size();
				stream.writeRawBytes(pBuf,lpInfo->size());	
			}
			delete[] pBuf;
			theFile.close();
		}
	}
	_debug_leavetrace("writeDirContents");
}

//============ readDirContents ============//

void KviScriptManager::readDirContents(const char *szDir,QDataStream &stream)
{
	_debug_entertrace("readDirContents");
	Q_UINT32 tag;
	stream >> tag;
	if(tag != KVI_SCRIPT_TAG_DIRECTORY){
		scriptExit(i18n("Directory block with wrong header tag"));
	}
	Q_UINT32 numFiles;
	stream >> numFiles;
	for(Q_UINT32 i=0;i<numFiles;i++){
		stream >> tag;
		if(tag != KVI_SCRIPT_TAG_FILE){
			scriptExit(i18n("File block with wrong header tag"));
		}
		QString szFileName;
		stream >> szFileName;
		Q_UINT32 uFileSize;
		stream >> uFileSize;
		QString szHome=_macro_getKVircHomeDirectory(szDir);
		szHome+='/';
		szHome+=szFileName;
		//check if exists
		QFile aFile(szHome.data());
		if(aFile.exists()){
			QFileInfo aInfo(aFile);
			if(aInfo.size()!=uFileSize){
				QString szNewFileName=szHome.copy();
				while(aFile.exists()){ //rename it
					szNewFileName+=".rnm";
					aFile.setName(szNewFileName.data());
				}
				rename(szHome.data(),szNewFileName.data());
			}
		}
		aFile.setName(szHome.data());
		if(!aFile.open(IO_WriteOnly|IO_Truncate)){
			aFile.close();
			KMsgBox::message(m_lpFrm,i18n("Script manager error"),
					 i18n("Unable to write unpacked file.\n"
					      "This may be due to a missing write permission to $HOME/kvirc\n"
					      "or insufficient disk space."));
			return;
 		}
		if(uFileSize){
			char * pChar = new char[uFileSize];
			stream.readRawBytes(pChar,uFileSize);
			aFile.writeBlock(pChar,uFileSize);
			delete[] pChar;
		}
		aFile.close();
	}
	_debug_leavetrace("readDirContents");
}

//============ checkDirContents ============//

bool KviScriptManager::checkDirContents(const char *szDir,QDataStream &stream)
{
	_debug_entertrace("checkDirContents");
	_macro_log(i18n("Block tagged : 'directory entry'"));
	QString szLog=i18n(" - Directory $HOME/kvirc/");
	szLog+=szDir;
	_macro_log(szLog.data());

	Q_UINT32 tag;
	_macro_readAndCheck(stream,tag);
	if(tag != KVI_SCRIPT_TAG_DIRECTORY)return false;
	Q_UINT32 numFiles;
	_macro_readAndCheck(stream,numFiles);
	for(Q_UINT32 i=0;i<numFiles;i++){
		_macro_readAndCheck(stream,tag);
		if(tag != KVI_SCRIPT_TAG_FILE)return false;
		QString szFileName;
		_macro_readAndCheck(stream,szFileName);
		Q_UINT32 uFileSize;
		_macro_readAndCheck(stream,uFileSize);
		szLog.sprintf(i18n(" - File entry : %s (%d bytes long)"),szFileName.data(),uFileSize);
		_macro_log(szLog.data());
		QString szHome=_macro_getKVircHomeDirectory(szDir);
		szHome+='/';
		szHome+=szFileName;
		//check if exists
		QFile aFile(szHome.data());
		if(aFile.exists()){
			szLog.sprintf(i18n(" !  File confilcts with existing %s"),szHome.data());
			_macro_log(szLog.data());
			QFileInfo aInfo(aFile);
			_macro_log(i18n(" !  Comparing file sizes"));
			if(aInfo.size()!=uFileSize){
				_macro_log(i18n(" !  Different file sizes : Old file will be renamed"));
			} else _macro_log(i18n(" !  Equal file sizes : Old file will be overwritten"));
		}
		if(uFileSize){
			char * pChar = new char[uFileSize];
			stream.readRawBytes(pChar,uFileSize);
			if(stream.device()->status() != IO_Ok)return false;
			delete[] pChar;
		}
	}
	szLog.sprintf(i18n(" # total : %d files"),numFiles);
	_macro_log(szLog.data());
	_debug_leavetrace("checkDirContents");
	return true;
}

//
// $Log: kvi_script.cpp,v $
// Revision 1.4  1998/09/20 20:23:35  fritz
// reorganized includes.
// More work on srvdlg - still not finished.
//
// Revision 1.3  1998/09/16 17:16:48  fritz
// Starting i18n.
//
// Revision 1.2  1998/09/16 16:13:19  pragma
// Moving to use a dynamic kvirc home directory.
// Big commit :)
//
//
