//   $Id: kvi_proxy.cpp,v 1.1 1998/09/25 15:51:45 pragma Exp $
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CLASS_NAME_ "KviProxy"
#define _KVI_DEBUG_CHECK_RANGE_

#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_status.h"
#include "kvi_debug.h"
#include "kvi_socket.h"
#include "kvi_proxy.h"
#include "kvi_support.h"
//#include <qstring.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

//The way to use a proxy:
// construct a KviProxy instance from KviFrame
// call setup selecting the method , a host and a port
// the IP may be passed or resolved by the class
// call beginNegotiate and wait until the socket calls handleProxyReply in KviFrame class.

//============ KviProxy ============//

KviProxy::KviProxy(KviIrcSocket *lpSocket,KviFrame *lpFrm)
{
	_debug_entertrace("KviProxy");
	clear();
	m_lpSock=lpSocket;
	m_lpFrm=lpFrm;
	m_lpConsole=m_lpFrm->m_lpConsole;
	_debug_leavetrace("KviProxy");
}

//============ ~KviProxy ============//

KviProxy::~KviProxy()
{
	_debug_entertrace("~KviProxy");
	_debug_leavetrace("~KviProxy");
}

//============ clear ============//

void KviProxy::clear()
{
	//turns off the proxy manager
	_debug_entertrace("clear");
	m_iProto=0; //no protocol set
	m_host.szName="";
	m_host.szIp="";
	m_host.szAlias1="";
	m_host.szAlias2="";
	m_host.iPort=0;
	m_szUserName="";
	m_szPassword="";
	m_bIsActive=false; //not ready
	_debug_leavetrace("clear");
}

//============ isActive ============//

bool KviProxy::isActive()
{
	_debug_entertrace("isActive");
	return m_bIsActive;
	_debug_leavetrace("isActive");
}

//============ getState ============//

int KviProxy::state()
{
	_debug_entertrace("getState");
	return m_iProto;
	_debug_leavetrace("getState");
}

//============ setup ============//

bool KviProxy::setup(bool bUseV5,QString &szHost,QString &szIp,unsigned short int iPort)
{
	//selects the method , and sets up the socket
	_debug_entertrace("setup");
	m_lpConsole->doOutput(KVI_OUT_INTERNAL,i18n("[ SOCKS ] : Preparing proxy data"));
	if(!setHost(szHost,szIp,iPort)){
		m_lpConsole->doOutput(KVI_OUT_INTERNAL,i18n("[ SOCKS ] : Invalid proxy address or port."));
		clear();
		return false;
	}
	QString szStat=i18n("Looking up host ")+szHost;
	m_lpFrm->setStatusText(szStat.data());
	m_lpConsole->doFmtOutput(KVI_OUT_INTERNAL,i18n("[ SOCKS ] : Looking up host %s"),szHost.data());
	if(!resolveHost()){
		m_lpConsole->doFmtOutput(KVI_OUT_INTERNAL,i18n("[ SOCKS ] : Unable to resolve proxy name."));
		clear();
		return false;		
	} else m_lpConsole->doFmtOutput(KVI_OUT_INTERNAL,i18n("[ SOCKS ] : Proxy name resolved : %s"),m_host.szIp.data());
	m_iProto=(bUseV5 ? 5 : 4);
	m_bIsActive=true;
	m_lpConsole->doFmtOutput(KVI_OUT_INTERNAL,i18n("[ SOCKS ] : Selecting protocol SOCKSV%d"),m_iProto);
	_debug_leavetrace("setup");
	return true;
}

//============ setHost ============//

bool KviProxy::setHost(QString &szHost,QString &szIp,unsigned short int iPort)
{
	//sets the proxy host to contact
	_debug_entertrace("setHost");
	if(szHost.isEmpty())return false;
	clear();
	m_host.szName   = szHost.stripWhiteSpace();
	m_host.szIp     = szIp.copy();
	if(m_host.szIp==KVI_EMPTY_IP)m_host.szIp="";
	m_host.szAlias1 = i18n("unknown");
	m_host.szAlias2 = i18n("unknown");
	m_host.iPort = iPort;
	_range_valid(m_host.iPort); // !=0
	_debug_leavetrace("setHost");
	return true;
}

//============ resolveHost ============//

bool KviProxy::resolveHost()
{
	//resolves the host
	_debug_entertrace("resolveHost");
	if(!(inet_aton(m_host.szIp.data(),&(m_host.iAddress)))){
		if(m_lpSock->getHostByName(m_host.szName.data(),&m_host) != KVI_SOCKERR_SUCCESS)return false;
	}
	_debug_leavetrace("resolveHost");
	return true;
}


//============ beginNegotiate ============//

void KviProxy::beginNegotiate(KviHostDescription *kh,QString &szUsername,QString &szPass)
{
	_debug_entertrace("beginNegotiate");
	_range_valid(kh->iPort); //!=0
	if(m_iProto==4){
		//first fill the userid field.
		//The socks proto V4 does not require user+pass
		//and event the user id may be ignored
		//however I saw some clients using it
		//so if someone provides it...
		QString userAndPass=szUsername.stripWhiteSpace();
		if((!szPass.isEmpty()) && (szPass != KVI_EMPTY_PASSWORD)){
			if(!userAndPass.isEmpty())userAndPass+=" ";
			userAndPass+=szPass;
		}
		int iLen=userAndPass.length()+9;
		char *bufToSend=new char[iLen];
		bzero(bufToSend,iLen);
		bufToSend[0]=(unsigned char)4;
		bufToSend[1]=(unsigned char)1;
		Q_UINT16 port=htons(kh->iPort);
		memmove(bufToSend+2,(char *)&port,2);
		Q_UINT32 host=(Q_UINT32)kh->iAddress.s_addr;
		memmove(bufToSend+4,(char *)&host,4);
		memmove(bufToSend+8,userAndPass.data(),userAndPass.length());
		m_iProto=KVI_PROXY_FINAL; //just wait for final ack
		m_lpSock->writeRawData(bufToSend,iLen);
		delete[] bufToSend;
		m_lpConsole->doOutput(KVI_OUT_INTERNAL,i18n("[ SOCKS ] : Sent target request...Waiting for reply."));
	} else {
		// proto==5
		// The client connects to the server, and sends a version
		// identifier/method selection message:
		//    +----+----------+----------+
		//    |VER | NMETHODS | METHODS  |
		//    +----+----------+----------+
		//    | 1  |    1     | 1 to 255 |
		//    +----+----------+----------+
		m_szUserName=szUsername.stripWhiteSpace(); //remember username and password
		m_szPassword=szPass.stripWhiteSpace();
		m_iPort=htons(kh->iPort);                  //remember the target host and port
		m_iHost=(Q_UINT32)kh->iAddress.s_addr;
		char *bufToSend=new char[3];
		bufToSend[0]=(unsigned char)5; //use version 5
		bufToSend[1]=(unsigned char)1; //select one method
		if(m_szUserName.isEmpty() || m_szPassword.isEmpty() || (m_szPassword==KVI_EMPTY_PASSWORD)){
			m_szUserName=""; //clear completly...do not use it...
			m_szPassword="";
			bufToSend[2]=(unsigned char)0; //select method 0 : no auth needed
			m_lpConsole->doOutput(KVI_OUT_INTERNAL,i18n("[ SOCKS ] : Selecting authentication method 0 : No auth"));
		} else {
			bufToSend[2]=(unsigned char)2; //username/pass auth
			m_lpConsole->doOutput(KVI_OUT_INTERNAL,i18n("[ SOCKS ] : Selecting authentication method 2 : User/pass"));
		}
		m_iProto=KVI_PROXY_SELECT;
		m_lpSock->writeRawData(bufToSend,3);
		delete[] bufToSend;
	}
	_debug_leavetrace("beginNegotiate");
}

//============ methodSelected ============//

void KviProxy::methodSelected(unsigned char chReply)
{
	_debug_entertrace("methodSelected");
	if(chReply>3){
		//replied something strange (!= (0==no_auth) and != (2==user_pass))
		//1 is a complete nonsense for US! ...so we do the nonsense.
		m_lpConsole->doOutput(KVI_OUT_INTERNAL,i18n("[ SOCKS ] : Warning , this proxy does not support the V5 protocol."));
		return;
	}
	if(chReply==0){
		m_lpConsole->doOutput(KVI_OUT_INTERNAL,i18n("[ SOCKS ] : Method OK : no auth needed."));
		authComplete();
	} else { //user and pass
		m_lpConsole->doOutput(KVI_OUT_INTERNAL,i18n("[ SOCKS ] : Method OK : sending user and pass."));
		m_iProto=KVI_PROXY_AUTH;
		// Send user and pass packet.
		//   +----+------+----------+------+----------+
		//   |VER | ULEN |  UNAME   | PLEN |  PASSWD  |
		//   +----+------+----------+------+----------+
		//   | 1  |  1   | 1 to 255 |  1   | 1 to 255 |
		//   +----+------+----------+------+----------+	
		//
		uint lPass=m_szPassword.length();
		if(lPass>255)lPass=255;
		uint lUser=m_szUserName.length();
		if(lUser>255)lUser=255;
		int iLen=lPass+lUser+3;
		char *bufToSend=new char[iLen];
		bzero(bufToSend,iLen);
		//54user4pass
		//0123456789
		bufToSend[0]=(unsigned char)1;                        //version x'01'
		bufToSend[1]=(unsigned char)lUser;                    //length of the username
		memmove(bufToSend+2,m_szUserName.data(),lUser);       //username
		bufToSend[2+lUser]=(unsigned char)lPass;                             //length of the password
		memmove(bufToSend+3+lUser,m_szPassword.data(),lPass);
		m_lpSock->writeRawData(bufToSend,iLen);
		delete[] bufToSend;
	}
	_debug_leavetrace("methodSelected");
}

//============ authComplete ============//

void KviProxy::authComplete()
{
	_debug_entertrace("authComplete");
	m_lpConsole->doOutput(KVI_OUT_INTERNAL,i18n("[ SOCKS ] : Auth complete : sending target data."));
	m_iProto=KVI_PROXY_FINAL;
	// now select the target to connect:
	//        +----+-----+-------+------+----------+----------+
	//        |VER | CMD |  RSV  | ATYP | DST.ADDR | DST.PORT |
	//        +----+-----+-------+------+----------+----------+
	//        | 1  |  1  | X'00' |  1   | Variable |    2     |
	//        +----+-----+-------+------+----------+----------+
	//
	//     Where:
	//
	//          o  VER    protocol version: X'05'
	//          o  CMD
	//             o  CONNECT X'01'
	//          o  RSV    RESERVED
	//          o  ATYP   address type of following address
	//             o  IP V4 address: X'01'
	//          o  DST.ADDR       desired destination address
	//          o  DST.PORT desired destination port in network octet
	//             order
	char *bufToSend=new char[10];
	bzero(bufToSend,10);
	bufToSend[0]=(unsigned char)5;           //Proto 5
	bufToSend[1]=(unsigned char)1;           //CONNECT
	bufToSend[2]=(unsigned char)0;           //RSV
	bufToSend[3]=(unsigned char)1;           //IPV4
	memmove(bufToSend+4,(char *)&m_iHost,4); //ipaddr
	memmove(bufToSend+8,(char *)&m_iPort,2); //port
	m_iProto=KVI_PROXY_FINAL;
	m_lpSock->writeRawData(bufToSend,10);
	delete[] bufToSend;
	_debug_leavetrace("authComplete");
}

//============ getProxyReply ============//

bool KviProxy::getProxyReply(unsigned char replycode,QString &szBuffer)
{
	_debug_entertrace("getProxyReply");
	switch(replycode){
		case 0:
			szBuffer=i18n("[00] :SOCKSV5: Success");
			return true;
			break;
		case 1:
			szBuffer=i18n("[01] :SOCKSV5: General SOCKS failure");
			return false;
			break;
		case 2:
			szBuffer=i18n("[02] :SOCKSV5: Connection not allowed by ruleset");
			return false;
			break;
		case 3:
			szBuffer=i18n("[03] :SOCKSV5: Network unreachable");
			return false;
			break;
		case 4:
			szBuffer=i18n("[04] :SOCKSV5: Host unreachable");
			return false;
			break;
		case 5:
			szBuffer=i18n("[05] :SOCKSV5: Connection refused");
			return false;
			break;
		case 6:
			szBuffer=i18n("[06] :SOCKSV5: TTL expired");
			return false;
			break;
		case 7:
			szBuffer=i18n("[07] :SOCKSV5: Command not supported");
			return false;
			break;
		case 8:
			szBuffer=i18n("[08] :SOCKSV5: Address type not supported");
			return false;
			break;
		case 90:
			szBuffer=i18n("[90] :SOCKSV4: Request granted");
			return true;
			break;
		case 91:
			szBuffer=i18n("[91] :SOCKSV4: Request rejected");
			return false;
			break;
		case 92:
			szBuffer=i18n("[92] :SOCKSV4: Request rejected : Unable to connect to your ident server");
			return false;
			break;
		case 93:
			szBuffer=i18n("[94] :SOCKSV4: Request rejected : Ident mismatch");
			return false;
			break;
		default:
			szBuffer.sprintf("[%d] :SOCKSV?: Unknown reply code : assuming success",replycode);
			return true;
			break;
	}
	_debug_leavetrace("getProxyReply");
}

//
// $Log: kvi_proxy.cpp,v $
// Revision 1.1  1998/09/25 15:51:45  pragma
// New KviProxy class.
// Moving proxy protocol to make KviFrame slimmer.
// Added V5 support and fixed V4.
// Fritz : If you have a fire , can you check the V5 user/pass authentication
// proto?
//
