//   $Id: kvi_proc.cpp,v 1.4 1998/09/20 20:23:30 fritz Exp $
//
//   This file is part of the KVIrc+ irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//
#define _KVI_DEBUG_CLASS_NAME_ "KviProcessManager"

#include "kvi_proc.h"
#include "kvi_debug.h"
#include "kvi_frame.h"
#include "kvi_mdi.h"
#include "kvi_status.h"
#include "kvi_app.h"
#include "kvi_event.h"
#include "kvi_uparser.h"

//============ KviProcessManager ============//
KviProcessManager::KviProcessManager(KviFrame *lpFrame):QObject(0,0)
{
	_debug_entertrace("KviProcessManager");
	m_lpFrm      = lpFrame;
	m_lpCons     = lpFrame->m_lpConsole;
	m_lpProcList = new QList<KviProcParams>;
	m_lpProcList->setAutoDelete(true);
	_debug_leavetrace("KviProcessManager");
}
//============ ~KviProcessManager ============//
KviProcessManager::~KviProcessManager()
{
	_debug_entertrace("~KviProcessManager");
	killAll();
	delete m_lpProcList;
	_debug_leavetrace("~KviProcessManager");
}
//============ killAll ============//
void KviProcessManager::killAll()
{
	_debug_entertrace("killAll");
	while(!m_lpProcList->isEmpty()){
		KviProcParams *lpP=m_lpProcList->first();
		lpP->lpProc->kill();
		delete lpP->lpProc;
		m_lpProcList->removeFirst();
	}
	_debug_leavetrace("killAll");
}
//============ topWin ============//
KviMdiChild * KviProcessManager::topWin()
{
	_debug_entertrace("topWin");
	KviMdiChild *lpC=m_lpFrm->m_lpMdi->topChildInZOrder();
	if(lpC)return lpC;
	else return m_lpCons;
	_debug_leavetrace("topWin");
}
//============ runProcess ============//
void KviProcessManager::runProcess(const char *procCmdLine)
{
	_debug_entertrace("runProcess");
	KProcess *lpP=new KProcess();
	QString szProcLine(procCmdLine);
	szProcLine.stripWhiteSpace();
	int idx=szProcLine.find(' ');
	while(idx != -1){
		QString szProcArg=szProcLine.left(idx);
		szProcLine.remove(0,idx+1);
		(*lpP)<<szProcArg.data();
		szProcLine=szProcLine.stripWhiteSpace();
		idx=szProcLine.find(' ');
	}
	(*lpP) << szProcLine.data();
	if(!lpP->start(KProcess::DontCare,KProcess::NoCommunication)){
		topWin()->doFmtOutput(KVI_OUT_ERROR,i18n("Could not execute process %s"),procCmdLine);
		delete lpP;
	} else {
		int aPid=lpP->getPid();
		topWin()->doFmtOutput(KVI_OUT_STDIN,i18n("Running external process '%s' with pid %d"),procCmdLine,aPid);
		delete lpP;
	}
	_debug_leavetrace("runProcess");
}
//============ execProcess ============//
void KviProcessManager::execProcess(const char *procCmdLine)
{
	_debug_entertrace("execProcess");
	KProcess *lpP=new KProcess();
	QString szProcLine(procCmdLine);
	szProcLine.stripWhiteSpace();
	int idx=szProcLine.find(' ');
	while(idx != -1){
		QString szProcArg=szProcLine.left(idx);
		szProcLine.remove(0,idx+1);
		(*lpP)<<szProcArg.data();
		szProcLine=szProcLine.stripWhiteSpace();
		idx=szProcLine.find(' ');
	}
	(*lpP) << szProcLine.data();
	if(!lpP->start(KProcess::NotifyOnExit,KProcess::All)){
		topWin()->doFmtOutput(KVI_OUT_ERROR,i18n("Could not execute process %s"),procCmdLine);
		delete lpP;
	} else {
		bool bHalt=false;
		int aPid=lpP->getPid();
		if(m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnProcessStarted]->bEnabled){
			QString szPara;
			szPara.setNum(aPid);
			szPara+=" ";
			szPara+=procCmdLine;
			if(m_lpFrm->m_lpUserParser->executeEvent(topWin(),
					m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnProcessStarted],
					szPara)==KVI_EventErr_Halt)bHalt=true;;
		}
		if(!bHalt)topWin()->doFmtOutput(KVI_OUT_STDIN,i18n("Starting child process '%s' with pid %d"),procCmdLine,aPid);
		KviProcParams *lpPara=new KviProcParams;
		lpPara->lpProc=lpP;
		lpPara->iPid=aPid;
		lpPara->bCanWrite=true;
		lpPara->szStdIn="";
		lpPara->szCmdLine=procCmdLine;
		m_lpProcList->append(lpPara);
		QObject::connect(lpP,SIGNAL(receivedStdout(KProcess *,char *,int)),this,SLOT(slotGotStdout(KProcess *,char *,int)));
		QObject::connect(lpP,SIGNAL(receivedStderr(KProcess *,char *,int)),this,SLOT(slotGotStderr(KProcess *,char *,int)));
		QObject::connect(lpP,SIGNAL(processExited(KProcess *)),this,SLOT(slotExited(KProcess *)));
		QObject::connect(lpP,SIGNAL(wroteStdin(KProcess *)),this,SLOT(slotWrote(KProcess *)));
	}
	_debug_leavetrace("execProcess");
}
//============ slotWrote ============//
void KviProcessManager::slotWrote(KProcess *proc)
{
	_debug_entertrace("slotWrote");
	KviProcParams *lpP=getProcByPtr(proc);
	if(lpP)lpP->bCanWrite=true;
	else _debug_warning("INTERNAL ERROR : Can't find the process in the list.");
	_debug_leavetrace("slotWrote");
}
//============ writeProcess ============//
void KviProcessManager::writeProcess(int aPid,const char *buffer)
{
	_debug_entertrace("writeProcess");
	KviProcParams *lpP=getProcByPid(aPid);
	if(lpP){
		if(!lpP->bCanWrite){
			topWin()->doFmtOutput(KVI_OUT_ERROR,i18n("Cannot write to process %d : another write operation in progress."),aPid);
			return;
		}
		lpP->szStdIn=QString(buffer);
		topWin()->doFmtOutput(KVI_OUT_STDIN,"[ stdin  %d ] : %s",lpP->lpProc->getPid(),lpP->szStdIn.data());
		lpP->lpProc->writeStdin(lpP->szStdIn.data(),lpP->szStdIn.length());
		lpP->bCanWrite=false;
	} else topWin()->doFmtOutput(KVI_OUT_ERROR,i18n("Pid %d : No such child process."),aPid);
	_debug_leavetrace("writeProcess");
}
//============ killProcess ============//
void KviProcessManager::killProcess(int aPid)
{
	_debug_entertrace("killProcess");
	KviProcParams *lpP=getProcByPid(aPid);
	if(lpP){
		lpP->lpProc->kill();
		delete lpP->lpProc;
		m_lpProcList->removeRef(lpP);
		bool bHalt=false;
		if(m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnProcessExited]->bEnabled){
				QString szPara;
				szPara.setNum(aPid);
				if(m_lpFrm->m_lpUserParser->executeEvent(topWin(),
						m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnProcessExited],
						szPara)==KVI_EventErr_Halt)bHalt=true;;
		}
		if(!bHalt)topWin()->doFmtOutput(KVI_OUT_STDIN,i18n("Process %d killed."),aPid);
	} else topWin()->doFmtOutput(KVI_OUT_ERROR,i18n("Pid %d : No such child process."),aPid);
	_debug_leavetrace("killProcess");
}
//============ slotExited ============//
void KviProcessManager::slotExited(KProcess *proc)
{
	_debug_entertrace("slotExited");
	KviProcParams *lpP=getProcByPtr(proc);
	if(lpP){
		bool bHalt=false;
		if(m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnProcessExited]->bEnabled){
				QString szPara;
				szPara.setNum(lpP->iPid);
				if(m_lpFrm->m_lpUserParser->executeEvent(topWin(),
						m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnProcessExited],
						szPara)==KVI_EventErr_Halt)bHalt=true;;
		}
		if(!bHalt)topWin()->doFmtOutput(KVI_OUT_STDIN,i18n("Process %d exited."),proc->getPid());
		delete lpP->lpProc;
		m_lpProcList->removeRef(lpP);
	} else _debug_warning("INTERNAL ERROR : Can't find the process.");
	_debug_leavetrace("slotExited");
}
//============ getProcByPid ============//
KviProcParams * KviProcessManager::getProcByPid(int aPid)
{
	_debug_entertrace("getProcByPid");
	KviProcParams *lpP=0;
	for(lpP=m_lpProcList->first();lpP;lpP=m_lpProcList->next()){
		if(lpP->iPid==aPid){
			return lpP;
		}
	}
	return 0;
	_debug_leavetrace("getProcByPid");
}
//============ getProcByPtr ============//
KviProcParams * KviProcessManager::getProcByPtr(KProcess * proc)
{
	_debug_entertrace("getProcByPtr");
	KviProcParams *lpP=0;
	for(lpP=m_lpProcList->first();lpP;lpP=m_lpProcList->next()){
		if(lpP->lpProc==proc){
			return lpP;
		}
	}
	return 0;
	_debug_leavetrace("getProcByPtr");
}
void KviProcessManager::slotGotStdout(KProcess *proc,char *buffer,int buflen)
{
	_debug_entertrace("slotGotStdout");
	KviProcParams *lpP=0;
	for(lpP=m_lpProcList->first();lpP;lpP=m_lpProcList->next()){
		if(lpP->lpProc==proc){
			QString buf(buffer,buflen+1);
			int idx=buf.find('\n');
			while(idx != -1){
				QString szB=buf.left(idx);
				buf.remove(0,idx+1);
				bool bHalt=false;
				if(m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnProcessStdout]->bEnabled){
					QString szPara;
					szPara.setNum(proc->getPid());
					szPara+=" ";
					szPara+=szB;
					if(m_lpFrm->m_lpUserParser->executeEvent(topWin(),
							m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnProcessStdout],
							szPara)==KVI_EventErr_Halt)bHalt=true;;
				}		
				if(!bHalt)topWin()->doFmtOutput(KVI_OUT_STDOUT,"[ stdout %d ] : %s",proc->getPid(),szB.data());
				idx=buf.find('\n');
			}
			if(!buf.isEmpty()){
				bool bHalt=false;
				if(m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnProcessStdout]->bEnabled){
					QString szPara;
					szPara.setNum(proc->getPid());
					szPara+=" ";
					szPara+=buf;
					if(m_lpFrm->m_lpUserParser->executeEvent(topWin(),
							m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnProcessStdout],
							szPara)==KVI_EventErr_Halt)bHalt=true;;
				}
				if(!bHalt)topWin()->doFmtOutput(KVI_OUT_STDOUT,"[ stdout %d ] : %s",proc->getPid(),buf.data());
			}
			return;
		}
	}
	_debug_warning("Got stdout from an unknown process.");
	_debug_leavetrace("slotGotStdout");
}
void KviProcessManager::slotGotStderr(KProcess *proc,char *buffer,int buflen)
{
	_debug_entertrace("slotGotStderr");
	KviProcParams *lpP=0;
	for(lpP=m_lpProcList->first();lpP;lpP=m_lpProcList->next()){
		if(lpP->lpProc==proc){
			QString buf(buffer,buflen+1);
			int idx=buf.find('\n');
			while(idx != -1){
				QString szB=buf.left(idx);
				buf.remove(0,idx+1);
				bool bHalt=false;
				if(m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnProcessStderr]->bEnabled){
					QString szPara;
					szPara.setNum(proc->getPid());
					szPara+=" ";
					szPara+=szB;
					if(m_lpFrm->m_lpUserParser->executeEvent(topWin(),
							m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnProcessStderr],
							szPara)==KVI_EventErr_Halt)bHalt=true;;
				}	
				if(!bHalt)topWin()->doFmtOutput(KVI_OUT_STDOUT,"[ stdout %d ] : %s",proc->getPid(),szB.data());
				idx=buf.find('\n');
			}
			if(!buf.isEmpty()){
				bool bHalt=false;
				if(m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnProcessStderr]->bEnabled){
					QString szPara;
					szPara.setNum(proc->getPid());
					szPara+=" ";
					szPara+=buf;
					if(m_lpFrm->m_lpUserParser->executeEvent(topWin(),
							m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnProcessStderr],
							szPara)==KVI_EventErr_Halt)bHalt=true;;
				}		
				if(!bHalt)topWin()->doFmtOutput(KVI_OUT_STDOUT,"[ stdout %d ] : %s",proc->getPid(),buf.data());
			}
			return;
		}
	}
	_debug_warning("Got stderr from an unknown process.");
	_debug_leavetrace("slotGotStderr");
}

#include "m_kvi_proc.moc"
