//   $Id: kvi_file.cpp,v 1.3 1998/09/20 20:22:29 fritz Exp $
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//

#include "kvi_defs.h"
#include "kvi_file.h"
#include "kvi_debug.h"
#include "kvi_mdi.h"
#include "kvi_support.h"

#include <kapp.h>

#include <qtimer.h>

#include <stdlib.h>

//============ KviFileManager ============//

KviFileManager::KviFileManager()
{
	_debug_entertrace("KviFileManager");
	m_lpFileList=new QList<KviFileStruct>;
	m_lpFileList->setAutoDelete(false);
	m_lpClearTimer=new QTimer();
	connect(m_lpClearTimer,SIGNAL(timeout()),this,SLOT(clearFiles()));
	_debug_leavetrace("KviFileManager");
}

//============ ~KviFileManager ============//

KviFileManager::~KviFileManager()
{
	_debug_entertrace("~KviFileManager");
	if(m_lpClearTimer->isActive())m_lpClearTimer->stop();
	delete m_lpClearTimer;
	while(!m_lpFileList->isEmpty()){
		KviFileStruct *lpF=m_lpFileList->last();
		lpF->lpFile->close();
		delete lpF->lpFile;
		m_lpFileList->removeLast();
		delete lpF;
	}
	delete m_lpFileList;
	_debug_leavetrace("~KviFileManager");
}

//============ getFileStruct ============//

KviFileStruct * KviFileManager::getFileStruct(const char *identifier)
{
	_debug_entertrace("getFileStruct");
	KviFileStruct *lpS=0;
	for(lpS=m_lpFileList->first();lpS;lpS=m_lpFileList->next()){
		if(!strcasecmp(lpS->szIdentifier.data(),identifier)){
			lpS->timeLastOp=QTime::currentTime();
			return lpS;
		}
	}
	return lpS;
	_debug_leavetrace("getFileStruct");
}

//============ fileOpen ============//

bool KviFileManager::fileOpen(const char *identifier,const char *filename,int openflags)
{
	_debug_entertrace("fileOpen");
	KviFileStruct *lpS=getFileStruct(identifier);
	if(lpS)fileClose(identifier);
	int myflags=0;
	if(openflags & KVI_FILE_OPEN_WRITE){
		myflags=IO_WriteOnly;
		if(openflags & KVI_FILE_OPEN_TRUNCATE)myflags |= IO_Truncate;
        else myflags |= IO_Append;
		if(openflags & KVI_FILE_OPEN_READ)return false;
		if(openflags & KVI_FILE_OPEN_CIRCULAR)return false;
	} else {
		myflags=IO_ReadOnly;
		openflags |= KVI_FILE_OPEN_READ;
	}
	lpS=new KviFileStruct;
	lpS->szIdentifier=identifier;
	lpS->openflags=openflags;
	lpS->lpFile=new QFile(filename);
	lpS->szFileName=filename;
	lpS->timeLastOp=QTime::currentTime();
	if(!lpS->lpFile->open(myflags)){
		delete lpS->lpFile;
		delete lpS;
		return false;
	}
	if(m_lpFileList->isEmpty())m_lpClearTimer->start(300000);
	m_lpFileList->append(lpS);
	return true;
	_debug_leavetrace("fileOpen");
}

//============ fileClose ============//

bool KviFileManager::fileClose(const char *identifier)
{
	_debug_entertrace("fileClose");
	KviFileStruct *lpS=getFileStruct(identifier);
	if(lpS){
		lpS->lpFile->close();
		delete lpS->lpFile;
		m_lpFileList->removeRef(lpS);
		if(m_lpFileList->isEmpty())m_lpClearTimer->stop();
		delete lpS;
		return true;
	}
	return false;
	_debug_leavetrace("fileClose");
}


//============ fileReadNextLine ============//

bool KviFileManager::fileReadNextLine(const char *identifier,QString &szData)
{
	_debug_entertrace("fileReadNextLine");
	KviFileStruct *lpS=getFileStruct(identifier);
	if(lpS){
		if(lpS->openflags & KVI_FILE_OPEN_READ){
			if(lpS->lpFile->atEnd()){
				szData=KVI_STR_NULL;
				if(lpS->openflags & KVI_FILE_OPEN_CIRCULAR){
					lpS->lpFile->at(0);
					if(lpS->lpFile->atEnd())return true;
				} else return true;
			}
			char *buffer=new char[513];
			bzero(buffer,513);
			if(lpS->lpFile->readLine(buffer,512)==-1){
				delete[] buffer;
				return false;
			}
			szData=buffer;
			szData=szData.stripWhiteSpace();
			delete[] buffer;
		} else return false;
	} else return false;
	_debug_leavetrace("fileReadNextLine");
	return true;
}
//============ fileReadRandLine ============//

bool KviFileManager::fileReadRandLine(const char *identifier,QString &szData)
{
	_debug_entertrace("fileReadRandLine");
	KviFileStruct *lpS=getFileStruct(identifier);
	if(lpS){
			uint r=(uint)(rand() % 100);
			//debug("rand = %d",r);
			char *buffer=new char[513];
			for(uint i=0;i<r;i++){
				if(lpS->lpFile->atEnd())lpS->lpFile->at(0);
				if(lpS->lpFile->readLine(buffer,512)==-1){
					delete[] buffer;
					return false;
				}
			}
			bzero(buffer,513);
			if(lpS->lpFile->atEnd())lpS->lpFile->at(0);
			if(lpS->lpFile->readLine(buffer,512)==-1){
				delete[] buffer;
				return false;
			}
			szData=buffer;
			szData=szData.stripWhiteSpace();
			delete[] buffer;
	} else return false;
	_debug_leavetrace("fileReadRandLine");
	return true;
}

//============ fileWriteLine ============//

bool KviFileManager::fileWriteLine(const char *identifier,QString &szData)
{
	_debug_entertrace("fileWriteLine");
	KviFileStruct *lpS=getFileStruct(identifier);
	if(lpS){
		if(lpS->openflags & KVI_FILE_OPEN_WRITE){
			szData+='\n';
			if(lpS->lpFile->writeBlock(szData.data(),strlen(szData.data()))==-1){
				return false;
			}
		} else return false;
	} else return false;
	_debug_leavetrace("fileWriteLine");
	return true;
}

//============ showFiles ============//

void KviFileManager::showFiles(KviMdiChild *lpC)
{
	_debug_entertrace("showFiles");
	lpC->doOutput(KVI_OUT_INTERNAL,i18n("[FILE] : Open files list"));
	if(m_lpFileList->isEmpty()){
		lpC->doOutput(KVI_OUT_INTERNAL,i18n("[FILE] : No files open"));
		return;
	}
	KviFileStruct *lpS=0;
	for(lpS=m_lpFileList->first();lpS;lpS=m_lpFileList->next()){
		QString szFlags="";
		if(lpS->openflags & KVI_FILE_OPEN_READ)szFlags+=i18n("read ");
		if(lpS->openflags & KVI_FILE_OPEN_WRITE)szFlags+=i18n("write ");
		if(lpS->openflags & KVI_FILE_OPEN_TRUNCATE)szFlags+=i18n("truncate ");
		if(lpS->openflags & KVI_FILE_OPEN_CIRCULAR)szFlags+=i18n("circular");
		int secs=lpS->timeLastOp.secsTo(QTime::currentTime());
		lpC->doFmtOutput(KVI_OUT_INTERNAL,i18n("[FILE] : file:%s ident:%s flags:%s idle:%d sec."),
			lpS->szFileName.data(),lpS->szIdentifier.data(),szFlags.data(),secs);
	}
	_debug_leavetrace("showFiles");
}

//============ clearFiles ============//

void KviFileManager::clearFiles()
{
	_debug_entertrace("clearFiles");
	//kill all the files that are idle for more than 5 minutes
	KviFileStruct *lpS=0;
	QList<KviFileStruct> *l=new QList<KviFileStruct>;
	l->setAutoDelete(false);
	for(lpS=m_lpFileList->first();lpS;lpS=m_lpFileList->next()){
		int secondsIdle=lpS->timeLastOp.secsTo(QTime::currentTime());
		if(secondsIdle>300)l->append(lpS);
	}
	while(!l->isEmpty()){
		lpS=l->last();
		lpS->lpFile->close();
		delete lpS->lpFile;
		m_lpFileList->removeRef(lpS);
		if(m_lpFileList->isEmpty())m_lpClearTimer->stop();
		l->removeLast();
		delete lpS;
	}
	delete l;
	_debug_leavetrace("clearFiles");
}

#include "m_kvi_file.moc"
