//
//   File : kvi_dccd.cpp (/usr/cvs/kvirc/kvirc/kvi_dccd.cpp)
//   Last modified : Sat Dec 5 1998 02:10:32 by root@localhost.localdomain
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998-1999 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation ; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING. If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//
#define _KVI_DEBUG_CLASS_NAME_ "KviDCCDaemon"

#include "kvi_dccd.h"
#include "kvi_defs.h"
#include "kvi_debug.h"
#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_opt.h"
#include "kvi_support.h"
#include "kvi_status.h"
#include "kvi_chat.h"
#include "kvi_macros.h"
#include "kvi_send.h"
#include "kvi_voice.h"
#include "kvi_sparser.h"
#include "kvi_ctcp.h"

#include <kmsgbox.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>     
#include <unistd.h>


//============ KviDCCMajorDomo ============//

KviDCCMajorDomo::KviDCCMajorDomo(KviFrame *lpFrm,KviDCCDaemon *dccd,int sockFd,const char *szIp)
:QObject(0,0)
{
	_debug_entertrace("KviDCCMajorDomo");
	m_lpDCCD=dccd;
	m_szRemoteIp=szIp;
	m_iSock=sockFd;
	m_lpFrm=lpFrm;
	m_lpCTCP=lpFrm->m_lpServerParser->m_lpCTCP;
	QString szNotice=i18n("Incoming DCC connection request from ");
	szNotice+=szIp;
	doOutput(szNotice.data());
	m_lpSocketNotifier=new QSocketNotifier(m_iSock, QSocketNotifier::Read, this);
	QObject::connect(m_lpSocketNotifier, SIGNAL(activated(int)), this, SLOT(dataInBuffer(int)));
	m_lpTimer=new QTimer();
	QObject::connect(m_lpTimer,SIGNAL(timeout()),this,SLOT(timedOut()));
	m_lpTimer->start(m_lpFrm->m_lpOpt->iDCCDLoginTimeout * 1000); //mIrc uses 15 secs.
	_debug_leavetrace("KviDCCMajorDomo");
}
//============ ~KviDCCMajorDomo ============//

KviDCCMajorDomo::~KviDCCMajorDomo()
{
	_debug_entertrace("~KviDCCMajorDomo");
	delete m_lpTimer;
	delete m_lpSocketNotifier;
	_debug_leavetrace("~KviDCCMajorDomo");
}

//============ dataInBuffer ============//

void KviDCCMajorDomo::dataInBuffer(int)
{
	_debug_entertrace("dataInBuffer");
	m_lpSocketNotifier->setEnabled(false);
	m_lpTimer->stop();
	char szIncomingData[1025];
   	bzero(szIncomingData, 1025); //just a space for the terminator
	int readLength = read(m_iSock,szIncomingData,1024);
	if (readLength > 4){
		QString szS(szIncomingData,readLength+1);
		//get the code of request
		QString szRequest=szS.left(3);
		szS.remove(0,4);
		int idx=szS.find('\r');
		QString szData;
		if(idx != -1)szData=szS.left(idx);
		else {
			idx=szS.find('\n');
			if(idx != -1)szData=szS.left(idx);
			else szData=szS.data();
		}
		QString szOut=i18n("Client request : ");
		szOut+=szRequest.data();
		szOut+=" ";
		szOut+=szData.data();
		doOutput(szOut.data());
		if(szRequest.isEmpty() || szData.isEmpty()){ //Impossible message : die
			doError(i18n("Huh ? Senseless request from client , closing connection"));
			::close(m_iSock);
		} else {
			//OK got the request code , parse it
			bool bOk=false;
			int  iCode=szRequest.toInt(&bOk);
			if(!bOk){ //Not a number at all , die
				doError(i18n("Huh ? Senseless request from client , closing connection"));
				::close(m_iSock);
			} else {
				switch(iCode){
					case 100:
						//Chat Protocol 
						//Client connects to Server and sends: 
						//100 clientnickname 
						//When Server receives this, it sends: 
						//101 servernickname 
						//Connection is established, users can now chat. 
						handleDCCChat(szData.data());
						break;
//					case 110:
//						//Fserve Protocol 
						//Client connects to Server and sends: 
						//110 clientnickname 
						//When Server receives this, it sends: 
						//111 servernickname 
						//Connection is established, user can now access fserve. 
//						break;
					case 120:
						//Send Protocol 
						//Client connects to Server and sends: 
						//120 clientnickname filesize filename 
						//When Server receives this, it sends: 
						//121 servernickname resumeposition 
						//Where resumeposition is between 0 and filesize, and is required. 
						//Connection is established, and Server dcc gets the file. 
						{
							int idsp=szData.find(" ");
							if(idsp != -1){
							} else {
								doError(i18n("Correct syntax , but missing parameters , closing connection"));
								serviceNotAvailable();
								::close(m_iSock);
							}
							QString szNk=szData.left(idsp);
							szData.remove(0,idsp+1);
							idsp=szData.find(" ");
							if(idsp != -1){
							} else {
								doError(i18n("Correct syntax , but missing file parameters, closing connection"));
								serviceNotAvailable();
								::close(m_iSock);
							}
							QString szFS=szData.left(idsp);
							szData.remove(0,idsp+1);
							szData.stripWhiteSpace();
							if(szData.isEmpty()){
								doError(i18n("Correct syntax , but senseless parameters , closing connection"));
								serviceNotAvailable();
								::close(m_iSock);
							}
							bool bOk=false;
							uint iFSize=szFS.toUInt(&bOk);
							if(bOk)handleDCCSend(szNk.data(),szData,iFSize);
							else {
								doError(i18n("Correct syntax , but missing parameters , closing connection"));
								serviceNotAvailable();
								::close(m_iSock);
							}
						}
						break;
//					case 130:
						//Get Protocol 
						//Client connects to Server and sends: 
						//130 clientnickname filename 
						//When Server receives this, it sends: 
						//131 servernickname filesize 
						//When Client receives this, it sends: 
						//132 clientnickname resumeposition 
						//Where resumeposition is between 0 and filesize, and is required. 
						//Connection is established, and Server dcc sends the file. 
//						break;
					default:
						//Just die
						doError(i18n("Correct syntax , but unrecognized service code , closing connection"));
						serviceNotAvailable();
						::close(m_iSock);
						break;
				}
			}
		}
	} else {
		doError(i18n("Incomplete request from client , closing connection"));
		::close(m_iSock);
	}
	m_lpDCCD->killMajorDomo(this);
	_debug_leavetrace("dataInBuffer");
}

//============ timedOut ============//

void KviDCCMajorDomo::timedOut()
{
	_debug_entertrace("timedOut");
	doError(i18n("Timed out while waiting for login , closing connection"));
	::close(m_iSock);
	m_lpDCCD->killMajorDomo(this);
	_debug_leavetrace("timedOut");
}


//============ serviceNotAvailable ============//

void KviDCCMajorDomo::serviceNotAvailable()
{
	_debug_entertrace("serviceNotAvailable");
	::write(m_iSock,"150 DccDaemon : Service not available\r\n",39);
	//should we sleep here ?
	_macro_kviApplication->processEvents();
	_debug_leavetrace("serviceNotAvailable");
}

//============ requestRejected ============//

void KviDCCMajorDomo::requestRejected()
{
	_debug_entertrace("requestRejected");
	::write(m_iSock,"151 DccDaemon : Request rejected\r\n",34);
	//should we sleep here ?
	_macro_kviApplication->processEvents();
	_debug_leavetrace("requestRejected");
}

//============ handleDccSend ============//

void KviDCCMajorDomo::handleDCCSend(const char *fromNick,QString &szFileName,uint iFileSize)
{
	_debug_entertrace("handleDccSend");
	if(!m_lpFrm->m_lpOpt->bIgnoreDCCSend){
		if(!m_lpFrm->m_lpOpt->bDCCSendServiceAvailable){
			//reply the service not available message
			QString szRef=i18n("Refusing DCC Send connection from ");
			szRef+=fromNick;
			szRef+="@";
			szRef+=m_szRemoteIp.data();
			szRef+=" [Service not available]";
			doOutput(szRef.data());
			//reply the service not available error message
			serviceNotAvailable();
			::close(m_iSock);
			return;
		}
		QString szReq;
		szReq.sprintf(i18n("%s@%s\nrequests a DCC SEND\nfor file %s %u bytes long\nDo you accept?"),
			      fromNick,m_szRemoteIp.data(), szFileName.data(), iFileSize);
		if(!m_lpFrm->m_lpOpt->bAutoAcceptDCCSend){
			if(KMsgBox::yesNo(m_lpFrm, i18n("DCC SEND Request"), szReq.data())==1){
				//Make the $home/Incoming directory (if is not there)
				QString szSaveFileName;
				QString szFileSize;
				szFileSize.setNum(iFileSize);
				bool bResume=m_lpCTCP->getDCCSaveFileName(szSaveFileName, szFileName, szFileSize);
				if(!szFileName.isEmpty()){
					QString szChatName="[DCC-GET]:-";
					szChatName+=fromNick;
					szChatName+="@";
					szChatName+=m_szRemoteIp+" "+szFileName;		
					KviSendWnd *lpWnd=m_lpFrm->createSendWnd(szChatName.data(),m_lpFrm->m_lpOpt->bMinimizeDCCSend);
					lpWnd->acceptConnectedSend(m_iSock,fromNick,m_szRemoteIp.data(),szSaveFileName.data(),iFileSize,bResume);
				} else {
					QString szRef=i18n("Refusing DCC Send connection from ");
					szRef+=fromNick;
					szRef+="@";
					szRef+=m_szRemoteIp.data();
					doOutput(szRef.data());
					requestRejected();
					::close(m_iSock);
				}
			} else {
				QString szRef=i18n("Refusing DCC Send connection from ");
				szRef+=fromNick;
				szRef+="@";
				szRef+=m_szRemoteIp.data();
				doOutput(szRef.data());
				requestRejected();
				::close(m_iSock);
			}
		} else {
			QString szSaveFileName(m_lpCTCP->getDCCSaveDirectory(szFileName));
			QDir aDir(szSaveFileName.data());
			if(!aDir.exists())aDir.mkdir(szSaveFileName.data(),true);
			if(szFileName[0] != '/')szSaveFileName+='/';
			szSaveFileName+=szFileName;
			bool bExists=true;
			while(bExists){ //Do not resume
				QFile aFile(szSaveFileName.data());
				bExists=aFile.exists();
				if(bExists)szSaveFileName+=".rnm";
			}
			QString szChatName="[DCC-GET]:-";
			szChatName+=fromNick;
			szChatName+="@";
			szChatName+=m_szRemoteIp+" "+szFileName;		
			KviSendWnd *lpWnd=m_lpFrm->createSendWnd(szChatName.data(),m_lpFrm->m_lpOpt->bMinimizeDCCSend);
			lpWnd->acceptConnectedSend(m_iSock,fromNick,m_szRemoteIp.data(),szSaveFileName.data(),iFileSize,false);
		}
	} else {
		QString szRef=i18n("Ignoring DCC Send connection from ");
		szRef+=fromNick;
		szRef+="@";
		szRef+=m_szRemoteIp.data();
		doOutput(szRef.data());
		requestRejected();
		::close(m_iSock);
	}
	_debug_leavetrace("handleDccSend");
}

//============ handleDCCChat ============//

void KviDCCMajorDomo::handleDCCChat(const char *fromNick)
{
	_debug_entertrace("handleDCCChat");
	if(!m_lpFrm->m_lpOpt->bIgnoreDCCChat){
		if(!m_lpFrm->m_lpOpt->bDCCChatServiceAvailable){
			//reply the service not available message
			QString szRef=i18n("Refusing DCC Chat connection from ");
			szRef+=fromNick;
			szRef+="@";
			szRef+=m_szRemoteIp.data();
			szRef+=" [Service not available]";
			doOutput(szRef.data());
			//reply the service not available error message
			serviceNotAvailable();
			::close(m_iSock);
			return;
		}
		QString szReq=fromNick;
		szReq+="@";
		szReq+=m_szRemoteIp.data();
		szReq+=i18n(" requests a DCC CHAT.\nDo you accept?");
		QString szChatName="[DCC-CHAT]:-";
		szChatName+=fromNick;
		szChatName+="@";
		szChatName+=m_szRemoteIp.data();
		if(!m_lpFrm->m_lpOpt->bAutoAcceptDCCChat){
			if(KMsgBox::yesNo(m_lpFrm, i18n("DCC Daemon"),szReq.data())==1){ // Fritz: check retcode
				KviChatWnd *lpWnd=m_lpFrm->createChatWnd(szChatName.data());
				lpWnd->acceptConnectedChat(m_iSock,m_szRemoteIp.data(),fromNick);
			} else {
				QString szRef=i18n("Refising DCC Chat connection from ");
				szRef+=fromNick;
				szRef+="@";
				szRef+=m_szRemoteIp.data();
				doOutput(szRef.data());
				requestRejected();
				::close(m_iSock);
			}
		} else {
			KviChatWnd *lpWnd=m_lpFrm->createChatWnd(szChatName.data());
			lpWnd->acceptConnectedChat(m_iSock,m_szRemoteIp.data(),fromNick);
		}
	} else {
		QString szRef=i18n("Ignoring DCC Chat connection from ");
		szRef+=fromNick;
		szRef+="@";
		szRef+=m_szRemoteIp.data();
		doOutput(szRef.data());
		requestRejected();
		::close(m_iSock);
	}
	_debug_leavetrace("handleDCCChat");
}

//============ doOutput ============//

void KviDCCMajorDomo::doOutput(const char *text)
{
	_debug_entertrace("doOutput");
	m_lpFrm->m_lpConsole->doFmtOutput(KVI_OUT_DCCD,"[DCCD:MajorDomo] : %s",text);
	_debug_leavetrace("doOutput");
}

//============ doError ============//

void KviDCCMajorDomo::doError(const char *text)
{
	_debug_entertrace("doError");
	m_lpFrm->m_lpConsole->doFmtOutput(KVI_OUT_ERROR,"[DCCD:MajorDomo] : Connection failed [%s] : %s",m_szRemoteIp.data(),text);
	_debug_leavetrace("doError");
}

//
// DCC Daemon
//

//============ KviDCCDaemon ============//

KviDCCDaemon::KviDCCDaemon(KviFrame *lpFrm,KviOptions *lpOpt)
:QObject(0,0)
{
	_debug_entertrace("KviDCCDaemon");
	m_lpOpt=lpOpt;
	m_lpFrm=lpFrm;
	m_iSock=-1;
	m_lpSocketNotifier=0;
	m_iPort=0;
	m_lpMajorDomoList=new QList<KviDCCMajorDomo>;
	m_lpMajorDomoList->setAutoDelete(true);
	_debug_leavetrace("KviDCCDaemon");
}
//============ ~KviDCCDaemon ============//

KviDCCDaemon::~KviDCCDaemon()
{
	_debug_entertrace("~KviDCCDaemon");
	stop();
	while(!m_lpMajorDomoList->isEmpty())m_lpMajorDomoList->removeLast();
	delete m_lpMajorDomoList;
	_debug_leavetrace("~KviDCCDaemon");
}

//============ doOutput ============//

void KviDCCDaemon::doOutput(const char *text)
{
	_debug_entertrace("doOutput");
	m_lpFrm->m_lpConsole->doFmtOutput(KVI_OUT_DCCD,"[DCCD] : %s",text);
	_debug_leavetrace("doOutput");
}

//============ doError ============//

void KviDCCDaemon::doError(const char *text)
{
	_debug_entertrace("doError");
	m_lpFrm->m_lpConsole->doFmtOutput(KVI_OUT_ERROR,"[DCCD] : %s",text);
	_debug_leavetrace("doError");
}

//============ stop ============//

void KviDCCDaemon::stop()
{
	_debug_entertrace("stop");
	if(m_iSock != -1){
		::close(m_iSock);
		m_iSock=-1;
	}
	if(m_lpSocketNotifier){
		if(m_lpSocketNotifier->isEnabled())m_lpSocketNotifier->setEnabled(false);
		delete m_lpSocketNotifier;
		m_lpSocketNotifier=0;
	}
	m_iPort=0;
	_debug_leavetrace("stop");
}

//============ terminateIfRunning ============//

void KviDCCDaemon::terminateIfRunning()
{
	_debug_entertrace("terminateIfRunning");
	if(m_lpSocketNotifier){
		stop();
		doOutput(i18n("Service stopped"));
	}
	_debug_leavetrace("terminateIfRunning");
}

//============ isActive ============//

bool KviDCCDaemon::isActive()
{
	_debug_entertrace("isActive");
	_debug_leavetrace("isActive");
	return (m_lpSocketNotifier != 0);
}

//============ start ============//

bool KviDCCDaemon::start(unsigned short int iPort)
{
	_debug_entertrace("start");
	stop();
	QString szErr;
	if(iPort<=0){
		doError((szErr.sprintf(i18n("Can not listen on port %d , dying"),iPort)).data());
		return false; //Need a defined port to listen on
	}
	m_iPort=iPort;
 	struct sockaddr_in sockAddress;
	bzero((char *)&sockAddress,sizeof(sockAddress));
	sockAddress.sin_family = AF_INET;
	sockAddress.sin_port   = htons(iPort);
	sockAddress.sin_addr.s_addr = INADDR_ANY;
	if((m_iSock = socket(PF_INET,SOCK_STREAM,0)) < 0){
		stop();
		doError(i18n("Unable to create a STREAM socket , dying"));
		return false;
	}
	if((bind(m_iSock,(struct sockaddr *) &sockAddress,sizeof(sockAddress))<0)){
		stop();
		doError(i18n("Can not listen for connections : bind() call failed , dying"));
		return false;
	}
	if((listen(m_iSock,100)<0)){
		stop();
		doError(i18n("Can not listen for connections : listen() call failed , dying"));
		return false;
	}
	m_lpSocketNotifier=new QSocketNotifier(m_iSock, QSocketNotifier::Read, this);
	QObject::connect(m_lpSocketNotifier, SIGNAL(activated(int)), this, SLOT(connectionRequest(int)));
	doOutput((szErr.sprintf(i18n("Service started : accepting connections on port %d"),iPort)).data());
	_debug_leavetrace("start");
	return true;
}

//============ connectionRequest ============//

void KviDCCDaemon::connectionRequest(int)
{
	_debug_entertrace("connectionRequest");
	struct sockaddr_in connectedAddr;
	ksize_t size = sizeof(connectedAddr);
	int theFd = accept(m_iSock,(struct sockaddr*)&connectedAddr,&size);
	if(theFd!=-1){
		//Got an estabilished connection
		//retrieve the ip address
		QString szRemoteIp=inet_ntoa(connectedAddr.sin_addr);
		KviDCCMajorDomo *james=new KviDCCMajorDomo(m_lpFrm,this,theFd,szRemoteIp.data());
		m_lpMajorDomoList->append(james);
	}
	_debug_leavetrace("connectionRequest");
}

//============ killMajorDomo ============//

void KviDCCDaemon::killMajorDomo(KviDCCMajorDomo *james)
{
	_debug_entertrace("killMajorDomo");
	m_lpMajorDomoList->removeRef(james);
	_debug_leavetrace("killMajorDomo");
}


#include "m_kvi_dccd.moc"
