//   $Id:...$
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//
#define _KVI_DEBUG_CLASS_NAME_ "KviChanListWnd"

#include "kvi_app.h"
#include "kvi_defs.h"
#include "kvi_macros.h"
#include "kvi_chlist.h"
#include "kvi_debug.h"
#include "kvi_view.h"
#include "kvi_frame.h"
#include "kvi_socket.h"
#include "kvi_mdi.h"
#include "kvi_taskbar.h"
#include "kvi_opt.h"
#include "kvi_int.h"
#include "kvi_support.h"

#include <stdarg.h>
#include <kmsgbox.h>
#include <qdatetime.h>

KviChanListWnd::KviChanListWnd(KviMdiManager *parent,KviFrame *frame,int aid,bool getFromFile)
				:KviMdiChild(parent,frame,KVI_CHANLIST_WND_NAME,aid,KVI_WND_TYPE_CHANLIST)
{
	m_lpPanner=new KNewPanner(this,"KviPanner",KNewPanner::Horizontal,KNewPanner::Percent,70);
	m_lpView = new KTabListBox(m_lpPanner,0,3);
	m_lpOutput=new KviView(m_lpPanner,frame,"KviViewClass");
	m_lpPanner->activate(m_lpView,m_lpOutput);
	m_lpView->setColumn(0, i18n("Channel"),160,KTabListBox::TextColumn,KTabListBox::SimpleOrder,KTabListBox::Ascending);
	m_lpView->setColumn(1, i18n("Users"),35,KTabListBox::TextColumn,KTabListBox::SimpleOrder,KTabListBox::Ascending);
	m_lpView->setColumn(2, i18n("Topic"),160,KTabListBox::TextColumn,KTabListBox::SimpleOrder,KTabListBox::Ascending);
	m_lpView->setSeparator('~');
	setFocusOwner(m_lpView);
	connect(this,SIGNAL(closeButtonPressed()),this,SLOT(closeSlot()));
	applyOptions();
	m_bDone=false;
	m_lpChanList=new QList<KviChanListEntry>;
	m_lpChanList->setAutoDelete(true);
	m_iNumChans=0;
	m_iMaxUsers=0;
	m_iAvgUsers=0;
	m_iPrvChans=0;
	m_iTotUsers=0;
	m_szMaxUsersChan=i18n("none");
	if(getFromFile)readFromFile();
	connect(m_lpView,SIGNAL(selected(int,int)),this,SLOT(itemDblClk(int,int)));
}

KviChanListWnd::~KviChanListWnd()
{
	writeToFile();
	while(!m_lpChanList->isEmpty())m_lpChanList->removeLast();
	delete m_lpChanList;
	delete m_lpView;
	delete m_lpOutput;
	delete m_lpPanner;
}
//============ endOfChannelList ============//
void KviChanListWnd::endOfChannelList()
{
	_debug_entertrace("endOfChannelList");
	KviChanListEntry *e=0;
	m_lpView->setAutoUpdate(false);
	for(e=m_lpChanList->first();e;e=m_lpChanList->next()){
		appendViewItem(e);
	}
	m_lpView->setAutoUpdate(true);
	m_lpView->repaint();
	m_bDone=true;
	if(m_iNumChans>0)m_iAvgUsers=m_iTotUsers / m_iNumChans;
	doOutput(KVI_OUT_INTERNAL,i18n("End of channels list"));
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Total            : %d users on %d channels"),m_iTotUsers,m_iNumChans);
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Max users        : %d on channel %s"),m_iMaxUsers,m_szMaxUsersChan.data());
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Average users    : %d"),m_iAvgUsers);
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Private channels : %d"),m_iPrvChans);
	_debug_leavetrace("endOfChannelList");
}
//============ itemDblClk ============//
void KviChanListWnd::itemDblClk(int index,int)
{
	_debug_entertrace("itemDblClk");
	QString szTxt=m_lpView->text(index);
	int idx=szTxt.find('~');
	QString szCh;
	if(idx != -1)szCh=szTxt.left(idx);
	else szCh=szTxt.data();
	if(m_lpFrm->m_lpSock->m_bConnected){
		m_lpFrm->m_lpSock->sendFmtData("JOIN %s",szCh.data());
		doFmtOutput(KVI_OUT_INTERNAL,i18n("Attempting to join %s"),szCh.data());
	} else doFmtOutput(KVI_OUT_ERROR,i18n("Can not join %s : no connection active"),szCh.data());
	_debug_leavetrace("itemDblClk");
}
//============ clear ============//
void KviChanListWnd::clear()
{
	_debug_entertrace("clear");
	while(!m_lpChanList->isEmpty())m_lpChanList->removeLast();
	m_lpView->clear();
	m_bDone=false;
	_debug_leavetrace("clear");
}
//============ addChanEntry ============//
void KviChanListWnd::addChanEntry(QString &szEntry)
{
	_debug_entertrace("addChanEntry");
	KviChanListEntry *e=new KviChanListEntry;
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Processing : %s"),szEntry.data());
	szEntry=szEntry.stripWhiteSpace();
	int idx=szEntry.find(' ');
	if(idx != -1){
		e->szChannel=szEntry.left(idx);
		szEntry.remove(0,idx+1);
		idx=szEntry.find(' ');
		if(idx != -1){
			e->szUsers=szEntry.left(idx);
			szEntry.remove(0,idx+1);
			int idx=szEntry.find(':');
			if(idx != -1)szEntry.remove(0,idx+1);
			e->szTopic=szEntry.data();
		} else e->szUsers="?";
	} else e->szUsers="?";
	if(e->szTopic.isEmpty())e->szTopic="[unknown]";
	m_lpChanList->append(e);
	_debug_leavetrace("addChanEntry");
}
//============ appendViewItem ============//
void KviChanListWnd::appendViewItem(KviChanListEntry *e)
{
	_debug_entertrace("appendViewItem");
	QString szIt=e->szChannel.data();
	szIt+='~';
	szIt+=e->szUsers.data();
	szIt+='~';
	szIt+=e->szTopic.data();
	if(e->szTopic.isEmpty() || (e->szTopic=="[unknown]"))m_iPrvChans++;
	bool bOk=false;
	uint nUsers=e->szUsers.toUInt(&bOk);
	if(!bOk){
		nUsers=1;
		doFmtOutput(KVI_OUT_ERROR,i18n("WARNING : Parse error while calculating users on %s"),e->szChannel.data());
	}
	if(m_iMaxUsers<nUsers){
		m_iMaxUsers=nUsers;
		m_szMaxUsersChan=e->szChannel.data();
	}
	m_iTotUsers+=nUsers;
	m_lpView->appendItem(szIt.data());
	m_iNumChans++;
	_debug_leavetrace("appendViewItem");
}
//============ readFromFile ============//
void KviChanListWnd::readFromFile()
{
	_debug_entertrace("readFromFile");
	m_bDone=true;
	QString szFileName(_macro_getKVircHomeDirectory("Config/kvi.chanlist.conf"));
	QFile config(szFileName.data());
	if(!config.exists()){
		doFmtOutput(KVI_OUT_ERROR,"The file kvi.chanlist.conf seems to not exist.\nYou will need to get a new channels list from a server.");
		return;
	}
	if(!config.open(IO_ReadOnly)){ cantReadError(); return; }
	uint cnt;
	char dummyBuf[1024];
	if(config.readBlock(dummyBuf,46)<0){ cantReadError(); return; }
	if(config.readBlock((char *)&cnt,sizeof(cnt))<0){ cantReadError(); return; }
	if(cnt==0){ cantReadError(); return; }
	int len=0;
	if(config.readBlock((char *)&len,sizeof(len))<0){ cantReadError(); return; }
	if(len>1024){ cantReadError(); return; }
	if(config.readBlock(dummyBuf,len)<0){ cantReadError(); return; }
	QString szDtm(dummyBuf,len+1);
	for(uint i=0;i<cnt;i++){
		KviChanListEntry *e=new KviChanListEntry;
		if(!readChanEntry(&config,e)){
			delete e;
			cantReadError();
			return;
		} else m_lpChanList->append(e);
	}
	doFmtOutput(KVI_OUT_INTERNAL,"List from %s",szDtm.data());
	endOfChannelList();
	config.close();
	_debug_leavetrace("readFromFile");
}
//============ cantWriteError ============//
void KviChanListWnd::cantWriteError()
{
	_debug_entertrace("cantWriteError");
	KMsgBox::message(this,i18n("File I/O error"),i18n("Can't write to kvi.chanlist.conf file.\nUnable to save this channel list."),KMsgBox::EXCLAMATION);
	_debug_leavetrace("cantWriteError");
}
//============ cantReadError ============//
void KviChanListWnd::cantReadError()
{
	_debug_entertrace("cantReadError");
	KMsgBox::message(this,i18n("File I/O error"),i18n("Can't read the kvi.chanlist.conf file.\nUnable to get the latest channel list."),KMsgBox::EXCLAMATION);
	_debug_leavetrace("cantReadError");
}
//============ writeToFile ============//
void KviChanListWnd::writeToFile()
{
	_debug_entertrace("writeToFile");
	//writes the channel list to a file if we have a full list
	if(!m_bDone)return;
	if(m_lpChanList->isEmpty())return;
	QString szFileName(_macro_getKVircHomeDirectory("Config/kvi.chanlist.conf"));

	QFile config(szFileName.data());
	if(!config.open(IO_WriteOnly|IO_Truncate)){ cantWriteError(); return; }
	//Write the alias number
	uint cnt=m_lpChanList->count();
	if(config.writeBlock("KVIrc CONFIGURATION FILE - PLEASE DO NOT EDIT\n",46)<0){ cantWriteError(); return; }
	if(config.writeBlock((char *)&cnt,sizeof(cnt))<0){ cantWriteError(); return; }
	QDateTime dtm(QDateTime::currentDateTime());
	QString szDtm=dtm.toString();
	int len=szDtm.length();
	if(config.writeBlock((char *)&len,sizeof(len))<0){ cantWriteError(); return; }
	if(config.writeBlock(szDtm.data(),len)<0){ cantWriteError(); return; }
	KviChanListEntry *e=0;
	for(e=m_lpChanList->first();e;e=m_lpChanList->next()){
		if(!writeChanEntry(&config,e)){ cantWriteError(); return; }
	}
	config.close();
	_debug_leavetrace("writeToFile");
}
//============ writeChanEntry ============//
bool KviChanListWnd::writeChanEntry(QFile *f,KviChanListEntry *e)
{
	_debug_entertrace("writeChanEntry");
	int len=e->szChannel.length();
	if(f->writeBlock((char *)&len,sizeof(len))<0)return false;
	if(f->writeBlock(e->szChannel.data(),len)<0)return false;
	len=e->szUsers.length();
	if(f->writeBlock((char *)&len,sizeof(len))<0)return false;
	if(f->writeBlock(e->szUsers.data(),len)<0)return false;
	len=e->szTopic.length();
	if(f->writeBlock((char *)&len,sizeof(len))<0)return false;
	if(f->writeBlock(e->szTopic.data(),len)<0)return false;
	_debug_leavetrace("writeChanEntry");
	return true;
}
//============ readChanEntry ============//
bool KviChanListWnd::readChanEntry(QFile *f,KviChanListEntry *e)
{
	_debug_entertrace("readChanEntry");
	char buffer[1024];
	int len;
	if(f->readBlock((char *)&len,sizeof(len))<0)return false;
	if(len>1024)return false;
	if(f->readBlock(buffer,len)<0)return false;
	e->szChannel=QString(buffer,len+1);
	if(f->readBlock((char *)&len,sizeof(len))<0)return false;
	if(len>1024)return false;
	if(f->readBlock(buffer,len)<0)return false;
	e->szUsers=QString(buffer,len+1);
	if(f->readBlock((char *)&len,sizeof(len))<0)return false;
	if(len>1024)return false;
	if(f->readBlock(buffer,len)<0)return false;
	e->szTopic=QString(buffer,len+1);
	_debug_leavetrace("readChanEntry");
	return true;
}
//============ standard mdi child functions ============//
void KviChanListWnd::closeSlot()
{
	_debug_entertrace("closeSlot");
	hide();
	_macro_kviApplication->processEvents();
	blockSignals(true);
	m_lpMdi->m_lpTaskBar->removeButton(id());
	m_lpMdi->m_lpChildList->setAutoDelete(false);
	m_lpMdi->m_lpChildList->removeRef(this);
	m_lpMdi->focusTopChild();
	m_lpMdi->fillWinListPopup();
	m_lpFrm->queryExternalDestroy(this);
	_debug_leavetrace("closeSlot");
}
void KviChanListWnd::applyOptions(){
	m_lpOutput->m_bShowPixmaps=m_lpFrm->m_lpOpt->bShowPixmaps;
	m_lpOutput->m_bTimestamp=m_lpFrm->m_lpOpt->bTimestamp;
	m_lpOutput->setFont(m_lpInt->fnt_output);
	const QSize sz=size();
	QResizeEvent e(sz,sz);
	resizeEvent(&e);
}
void KviChanListWnd::resizeEvent(QResizeEvent *){
	_debug_entertrace("resizeEvent");
	updateRects();
	QRect rct=viewSizeHint();
	m_lpPanner->setGeometry(rct);
	_debug_leavetrace("resizeEvent");
}
void KviChanListWnd::doFmtOutput(int nType,const char *szFmt,...){
	char szText[600]; //It should be big enough... I hope...
	va_list list;
	va_start(list,szFmt);
	vsprintf(szText,szFmt,list); // Fritz: vsnprintf
	va_end(list);
	m_lpMdi->highlightWindow(m_iId);
	m_lpOutput->appendText(nType,(const char *) &szText);
}
void KviChanListWnd::doOutput(int nType,const char *szText){
	m_lpMdi->highlightWindow(m_iId);
	m_lpOutput->appendText(nType,szText);
}

#include "m_kvi_chlist.moc"

//
// $Log:...$
//
