/*
 * Copyright (c) 2000 Silicon Graphics, Inc.  All Rights Reserved.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it would be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * Further, this software is distributed without any warranty that it is
 * free of the rightful claim of any third person regarding infringement
 * or the like.  Any license provided herein, whether implied or
 * otherwise, applies only to this software file.  Patent licenses, if
 * any, provided herein do not apply to combinations of this program with
 * other software, or any other product whatsoever.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write the Free Software Foundation, Inc., 59
 * Temple Place - Suite 330, Boston MA 02111-1307, USA.
 * 
 * Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 * Mountain View, CA  94043, or:
 * 
 * http://www.sgi.com 
 * 
 * For further information regarding this notice, see: 
 * 
 * http://oss.sgi.com/projects/GenInfo/SGIGPLNoticeExplan/
 */
#include <libxfs.h>
#include <jdm.h>

#include <sys/resource.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>

#include "types.h"
#include "mlog.h"

#define STACKFUDGE	( 8 * pgsz )
#define STACKMARGIN	( 8 * pgsz )

struct stkchk {
	pid_t sc_pid;
	char *sc_initsp;
	char *sc_minsp;
	int sc_sz;
};

typedef struct stkchk stkchk_t;

/* REFERENCED */
static size_t stkchk_pidcntmax = 0;

static stkchk_t *stkchk_list;

static size_t stkchk_pidcnt = 0;

static int get_stacksz( void );

extern size_t pgsz;

void
stkchk_init( size_t maxpidcnt )
{
	ASSERT( maxpidcnt > 0 );
	ASSERT( stkchk_pidcntmax == 0 );
	stkchk_list = ( stkchk_t * )calloc( maxpidcnt, sizeof( stkchk_t ));
	ASSERT( stkchk_list );
	stkchk_pidcntmax = maxpidcnt;
	ASSERT( stkchk_pidcnt == 0 );
}

void
stkchk_register( void )
{
	char dummy;
	pid_t pid;
	stkchk_t *stkchkp;

	ASSERT( stkchk_pidcntmax );
	pid = getpid( );
	ASSERT( stkchk_pidcnt < stkchk_pidcntmax );
	stkchkp = &stkchk_list[ stkchk_pidcnt++ ];
	stkchkp->sc_pid = getpid( );
	stkchkp->sc_initsp = &dummy;
	stkchkp->sc_sz = get_stacksz( );
	ASSERT( stkchkp->sc_initsp >= (char *)stkchkp->sc_sz );
	stkchkp->sc_minsp = stkchkp->sc_initsp - stkchkp->sc_sz;
	mlog( MLOG_DEBUG | MLOG_PROC,
	      "stack pid %u: "
	      "sz 0x%lx min 0x%lx max 0x%lx\n",
	      pid,
	      stkchkp->sc_sz,
	      stkchkp->sc_minsp,
	      stkchkp->sc_initsp );
}

int
stkchk( void )
{
	char dummy = 1; /* keep lint happy */
	pid_t pid;
	size_t pidix;
	stkchk_t *stkchkp = NULL;

	ASSERT( stkchk_pidcntmax );
	pid = getpid( );

	for ( pidix = 0 ; pidix < stkchk_pidcnt ; pidix++ ) {
		stkchkp = &stkchk_list[ pidix ];
		if ( stkchkp->sc_pid == pid ) {
			break;
		}
	}
	ASSERT( pidix < stkchk_pidcnt );

	if ( &dummy > stkchkp->sc_initsp ) {
		mlog( MLOG_NORMAL | MLOG_ERROR | MLOG_PROC,
		      "stack underrun pid %d "
		      "sp 0x%lx min 0x%lx max 0x%lx\n",
		      pid,
		      &dummy,
		      stkchkp->sc_minsp,
		      stkchkp->sc_initsp );
		      return 3;
	} else if ( &dummy < stkchkp->sc_minsp + STACKFUDGE ) {
		mlog( MLOG_NORMAL | MLOG_ERROR | MLOG_PROC,
		      "stack overrun pid %d "
		      "sp 0x%lx min 0x%lx max 0x%lx\n",
		      pid,
		      &dummy,
		      stkchkp->sc_minsp,
		      stkchkp->sc_initsp );
		      return 2;
	} else if ( &dummy < stkchkp->sc_minsp + STACKFUDGE + STACKMARGIN ) {
		mlog( MLOG_VERBOSE | MLOG_WARNING | MLOG_PROC,
		      "stack near overrun pid %d "
		      "sp 0x%lx min 0x%lx max 0x%lx\n",
		      pid,
		      &dummy,
		      stkchkp->sc_minsp,
		      stkchkp->sc_initsp );
		      return 1;
	}

	mlog( MLOG_DEBUG | MLOG_PROC,
	      "stack ok pid %d "
	      "sp 0x%lx min 0x%lx max 0x%lx\n",
	      pid,
	      &dummy,
	      stkchkp->sc_minsp,
	      stkchkp->sc_initsp );
	return 0;
}

static int
get_stacksz( void )
{
	struct rlimit rlimit;
	/* REFERENCED */
	intgen_t rval;

	rval = getrlimit( RLIMIT_STACK, &rlimit );
	ASSERT( ! rval );
	return rlimit.rlim_cur;
}
