/*
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 *  Routines for control of CS4231(A)/InterWave chip
 *
 *  Note: Code isn't tested with CS4231 version of chip, only with CS4231A.
 *
 *  Bugs:
 *     - sometimes record brokes playback with WSS portion of 
 *       Yamaha OPL3-SA3 chip
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "cs4231.h"
#include "ulaw.h"

#if 0
#define SND_DEBUG_MCE
#endif

/*
 *  Some variables
 */

static struct snd_stru_cs4231_freq snd_cs4231_freq[14] =
{
	{5, 5510, 0x00 | CS4231_XTAL2},
	{6, 6620, 0x0E | CS4231_XTAL2},
	{8, 8000, 0x00 | CS4231_XTAL1},
	{9, 9600, 0x0E | CS4231_XTAL1},
	{11, 11025, 0x02 | CS4231_XTAL2},
	{16, 16000, 0x02 | CS4231_XTAL1},
	{18, 18900, 0x04 | CS4231_XTAL2},
	{22, 22050, 0x06 | CS4231_XTAL2},
	{27, 27042, 0x04 | CS4231_XTAL1},
	{32, 32000, 0x06 | CS4231_XTAL1},
	{33, 33075, 0x0C | CS4231_XTAL2},
	{37, 37800, 0x08 | CS4231_XTAL2},
	{44, 44100, 0x0A | CS4231_XTAL2},
	{48, 48000, 0x0C | CS4231_XTAL1}
};

static struct snd_stru_cs4231_image snd_cs4231_original_image =
{
	0x00,			/* 00/00 - lic */
	0x00,			/* 01/01 - ric */
	0x80,			/* 02/02 - la1ic */
	0x80,			/* 03/03 - ra1ic */
	0x80,			/* 04/04 - la2ic */
	0x80,			/* 05/05 - ra2ic */
	0x80,			/* 06/06 - loc */
	0x80,			/* 07/07 - roc */
	0x20,			/* 08/08 - pdfr */
	CS4231_AUTOCALIB,	/* 09/09 - ic */
	0x00,			/* 0a/10 - pc */
	0x00,			/* 0b/11 - ti */
	CS4231_MODE2,		/* 0c/12 - mi */
	0x00,			/* 0d/13 - lbc */
	0x00,			/* 0e/14 - pbru */
	0x00,			/* 0f/15 - pbrl */
	0x80,			/* 10/16 - afei */
	0x01,			/* 11/17 - afeii */
	0x00,			/* 12/18 - llic */
	0x00,			/* 13/19 - rlic */
	0x00,			/* 14/20 - tlb */
	0x00,			/* 15/21 - thb */
	0x00,			/* 16/22 - la3mic/reserved */
	0x00,			/* 17/23 - ra3mic/reserved */
	0x00,			/* 18/24 - afs */
	0x00,			/* 19/25 - lamoc/version */
	0xc0,			/* 1a/26 - mioc */
	0x00,			/* 1b/27 - ramoc/reserved */
	0x20,			/* 1c/28 - cdfr */
	0x00,			/* 1d/29 - res4 */
	0x00,			/* 1e/30 - cbru */
	0x00,			/* 1f/31 - cbrl */
};

/*
 *  Basic I/O functions
 */

void snd_cs4231_outm(cs4231_t * codec, unsigned char reg,
		     unsigned char mask, unsigned char value)
{
	int timeout;
	unsigned char tmp;

	for (timeout = 250;
	     timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT);
	     timeout--)
		snd_delay(1);
#ifdef SNDCFG_DEBUG
	if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
		snd_printk("snd_cs4231_outm: auto calibration time out - reg = 0x%x, value = 0x%x\n", reg, value);
#endif
	outb(codec->mce_bit | reg, CS4231P(codec, REGSEL));
	mb();
	tmp = (inb(CS4231P(codec, REG)) & mask) | value;
	outb(tmp, CS4231P(codec, REG));
	mb();
}

void snd_cs4231_out(cs4231_t * codec, unsigned char reg, unsigned char value)
{
	int timeout;

	for (timeout = 250;
	     timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT);
	     timeout--)
		snd_delay(1);
#ifdef SNDCFG_DEBUG
	if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
		snd_printk("snd_cs4231_out: auto calibration time out - reg = 0x%x, value = 0x%x\n", reg, value);
#endif
	outb(codec->mce_bit | reg, CS4231P(codec, REGSEL));
	outb(value, CS4231P(codec, REG));
	mb();
#if 0
	printk("codec out - reg 0x%x = 0x%x\n", codec->mce_bit | reg, value);
#endif
}

unsigned char snd_cs4231_in(cs4231_t * codec, unsigned char reg)
{
	int timeout;

	for (timeout = 250;
	     timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT);
	     timeout--)
		snd_delay(1);
#ifdef SNDCFG_DEBUG
	if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
		snd_printk("snd_cs4231_in: auto calibration time out - reg = 0x%x\n", reg);
#endif
	outb(codec->mce_bit | reg, CS4231P(codec, REGSEL));
	mb();
	return inb(CS4231P(codec, REG));
}

void snd_cs4236_ext_out(cs4231_t * codec, unsigned char reg, unsigned char val)
{
	outb(codec->mce_bit | 0x17, CS4231P(codec, REGSEL));
	outb(reg | (codec->image.ra3ic & 0x01), CS4231P(codec, REG));
	outb(val, CS4231P(codec, REG));
#if 0
	printk("ext out : reg = 0x%x, val = 0x%x\n", reg, val);
#endif
}

unsigned char snd_cs4236_ext_in(cs4231_t * codec, unsigned char reg)
{
	outb(codec->mce_bit | 0x17, CS4231P(codec, REGSEL));
	outb(reg | (codec->image.ra3ic & 0x01), CS4231P(codec, REG));
#if 1
	return inb(CS4231P(codec, REG));
#else
	{
		unsigned char res;
		res = inb(CS4231P(codec, REG));
		printk("ext in : reg = 0x%x, val = 0x%x\n", reg, res);
		return res;
	}
#endif
}

#ifdef SNDCFG_DEBUG

void snd_cs4231_debug(cs4231_t * codec)
{
	printk("CS4231 REGS:      INDEX = 0x%02x  ", inb(CS4231P(codec, REGSEL)));
	printk("                 STATUS = 0x%02x\n", inb(CS4231P(codec, STATUS)));
	printk("  0x00: left input      = 0x%02x  ", snd_cs4231_in(codec, 0x00));
	printk("  0x10: alt 1 (CFIG 2)  = 0x%02x\n", snd_cs4231_in(codec, 0x10));
	printk("  0x01: right input     = 0x%02x  ", snd_cs4231_in(codec, 0x01));
	printk("  0x11: alt 2 (CFIG 3)  = 0x%02x\n", snd_cs4231_in(codec, 0x11));
	printk("  0x02: GF1 left input  = 0x%02x  ", snd_cs4231_in(codec, 0x02));
	printk("  0x12: left line in    = 0x%02x\n", snd_cs4231_in(codec, 0x12));
	printk("  0x03: GF1 right input = 0x%02x  ", snd_cs4231_in(codec, 0x03));
	printk("  0x13: right line in   = 0x%02x\n", snd_cs4231_in(codec, 0x13));
	printk("  0x04: CD left input   = 0x%02x  ", snd_cs4231_in(codec, 0x04));
	printk("  0x14: timer low       = 0x%02x\n", snd_cs4231_in(codec, 0x14));
	printk("  0x05: CD right input  = 0x%02x  ", snd_cs4231_in(codec, 0x05));
	printk("  0x15: timer high      = 0x%02x\n", snd_cs4231_in(codec, 0x15));
	printk("  0x06: left output     = 0x%02x  ", snd_cs4231_in(codec, 0x06));
	printk("  0x16: left MIC (PnP)  = 0x%02x\n", snd_cs4231_in(codec, 0x16));
	printk("  0x07: right output    = 0x%02x  ", snd_cs4231_in(codec, 0x07));
	printk("  0x17: right MIC (PnP) = 0x%02x\n", snd_cs4231_in(codec, 0x17));
	printk("  0x08: playback format = 0x%02x  ", snd_cs4231_in(codec, 0x08));
	printk("  0x18: IRQ status      = 0x%02x\n", snd_cs4231_in(codec, 0x18));
	printk("  0x09: iface (CFIG 1)  = 0x%02x  ", snd_cs4231_in(codec, 0x09));
	printk("  0x19: left line out   = 0x%02x\n", snd_cs4231_in(codec, 0x19));
	printk("  0x0a: pin control     = 0x%02x  ", snd_cs4231_in(codec, 0x0a));
	printk("  0x1a: mono control    = 0x%02x\n", snd_cs4231_in(codec, 0x1a));
	printk("  0x0b: init & status   = 0x%02x  ", snd_cs4231_in(codec, 0x0b));
	printk("  0x1b: right line out  = 0x%02x\n", snd_cs4231_in(codec, 0x1b));
	printk("  0x0c: revision & mode = 0x%02x  ", snd_cs4231_in(codec, 0x0c));
	printk("  0x1c: record format   = 0x%02x\n", snd_cs4231_in(codec, 0x1c));
	printk("  0x0d: loopback        = 0x%02x  ", snd_cs4231_in(codec, 0x0d));
	printk("  0x1d: var freq (PnP)  = 0x%02x\n", snd_cs4231_in(codec, 0x1d));
	printk("  0x0e: ply upr count   = 0x%02x  ", snd_cs4231_in(codec, 0x0e));
	printk("  0x1e: ply lwr count   = 0x%02x\n", snd_cs4231_in(codec, 0x1e));
	printk("  0x0f: rec upr count   = 0x%02x  ", snd_cs4231_in(codec, 0x0f));
	printk("  0x1f: rec lwr count   = 0x%02x\n", snd_cs4231_in(codec, 0x1f));
}

#endif

/*
 *  CS4231 detection / MCE routines
 */

static void snd_cs4231_busy_wait(cs4231_t * codec)
{
	int timeout;

	/* huh.. looks like this sequence is proper for CS4231A chip (GUS MAX) */
	for (timeout = 5; timeout > 0; timeout--)
		inb(CS4231P(codec, REGSEL));
	/* end of cleanup sequence */
	for (timeout = 250;
	     timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT);
	     timeout--)
		snd_delay(1);
}

void snd_cs4231_mce_up(cs4231_t * codec)
{
	unsigned long flags;
	int timeout;

	snd_spin_lock(codec, reg, &flags);
	for (timeout = 250; timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT); timeout--)
		snd_delay(1);
#ifdef SNDCFG_DEBUG
	if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
		snd_printk("cs4231_mce_up - auto calibration time out (0)\n");
#endif
	codec->mce_bit |= CS4231_MCE;
	timeout = inb(CS4231P(codec, REGSEL));
	if (timeout == 0x80)
		snd_printk("snd_cs4231_up [0x%x]: serious init problem - codec still busy\n", codec->port);
	if (!(timeout & CS4231_MCE))
		outb(codec->mce_bit | (timeout & 0x1f), CS4231P(codec, REGSEL));
	snd_spin_unlock(codec, reg, &flags);
}

void snd_cs4231_mce_down(cs4231_t * codec)
{
	unsigned long flags;
	int timeout;
	signed long time;

	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_busy_wait(codec);
#if 0
	printk("(1) timeout = %i\n", timeout);
#endif
#ifdef SNDCFG_DEBUG
	if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
		snd_printk("cs4231_mce_down [0x%x] - auto calibration time out (0)\n", CS4231P(codec, REGSEL));
#endif
	codec->mce_bit &= ~CS4231_MCE;
	timeout = inb(CS4231P(codec, REGSEL));
	outb(codec->mce_bit | (timeout & 0x1f), CS4231P(codec, REGSEL));
	if (timeout == 0x80)
		snd_printk("snd_cs4231_down [0x%x]: serious init problem - codec still busy\n", codec->port);
	if ((timeout & CS4231_MCE) == 0 ||
	    !(codec->hardware & (CS4231_HW_CS4231_MASK | CS4231_HW_CS4232_MASK))) {
		snd_spin_unlock(codec, reg, &flags);
		return;
	}
	snd_cs4231_busy_wait(codec);

	/* calibration process */

	for (timeout = 500; timeout > 0 && (snd_cs4231_in(codec, CS4231_TEST_INIT) & CS4231_CALIB_IN_PROGRESS) == 0; timeout--);
	if ((snd_cs4231_in(codec, CS4231_TEST_INIT) & CS4231_CALIB_IN_PROGRESS) == 0) {
		snd_printd("cs4231_mce_down - auto calibration time out (1)\n");
		snd_spin_unlock(codec, reg, &flags);
		return;
	}
#if 0
	printk("(2) timeout = %i, jiffies = %li\n", timeout, jiffies);
#endif
	time = HZ / 4;
	while (snd_cs4231_in(codec, CS4231_TEST_INIT) & CS4231_CALIB_IN_PROGRESS) {
		snd_spin_unlock(codec, reg, &flags);
		if (time <= 0) {
			snd_printk("cs4231_mce_down - auto calibration time out (2)\n");
			return;
		}
		snd_schedule(codec, mce, time);
		time = snd_timevalue(codec, mce);
		snd_spin_lock(codec, reg, &flags);
	}
#if 0
	printk("(3) jiffies = %li\n", jiffies);
#endif
	time = HZ / 10;
	while (inb(CS4231P(codec, REGSEL)) & CS4231_INIT) {
		snd_spin_unlock(codec, reg, &flags);
		if (time <= 0) {
			snd_printk("cs4231_mce_down - auto calibration time out (3)\n");
			return;
		}
		snd_schedule(codec, mce, time);
		time = snd_timevalue(codec, mce);
		snd_spin_lock(codec, reg, &flags);
	}
	snd_spin_unlock(codec, reg, &flags);
#if 0
	printk("(4) jiffies = %li\n", jiffies);
	snd_printk("cs4231_mce_down - exit = 0x%x\n", inb(CS4231P(codec, REGSEL)));
#endif
}

static unsigned int snd_cs4231_get_count(unsigned char format, unsigned int size)
{
	switch (format & 0xe0) {
	case CS4231_LINEAR_16:
	case CS4231_LINEAR_16_BIG:
		size >>= 1;
		break;
	case CS4231_ADPCM_16:
		return size >> 2;
	}
	if (format & CS4231_STEREO)
		size >>= 1;
	return size;
}

static void snd_cs4231_trigger(cs4231_t * codec, unsigned char what, int enable)
{
	unsigned long flags;

#if 0
	printk("codec trigger!!! - what = %i, enable = %i, status = 0x%x\n", what, enable, inb(CS4231P(card, STATUS)));
#endif
	snd_spin_lock(codec, reg, &flags);
	if (enable) {
		if (codec->image.ic & what) {
			snd_spin_unlock(codec, reg, &flags);
			return;
		}
		snd_cs4231_out(codec, CS4231_IFACE_CTRL, codec->image.ic |= what);
	} else {
		if (!(codec->image.ic & what)) {
			snd_spin_unlock(codec, reg, &flags);
			return;
		}
		snd_cs4231_out(codec, CS4231_IFACE_CTRL, codec->image.ic &= ~what);
	}
	snd_spin_unlock(codec, reg, &flags);
#if 0
	snd_cs4231_debug(codec);
#endif
}

/*
 *  CODEC I/O
 */

static unsigned char snd_cs4231_get_freq(unsigned int freq)
{				/* freq in Hz */
	int i;

	freq /= 1000;
#if 0
	snd_printk("pcm_rate: %d\n", freq);
#endif
	if (freq > 48)
		freq = 48;
	for (i = 0; i < 14; i++)
		if (freq <= snd_cs4231_freq[i].hertz)
			return snd_cs4231_freq[i].bits;
	return snd_cs4231_freq[13].bits;
}

static unsigned int snd_cs4231_get_real_freq(unsigned int freq)
{				/* freq in Hz */
	int i;

	freq /= 1000;
#if 0
	snd_printk("pcm_rate: %d\n", freq);
#endif
	if (freq > 48)
		freq = 48;
	for (i = 0; i < 14; i++)
		if (freq <= snd_cs4231_freq[i].hertz)
			return snd_cs4231_freq[i].rate;
	return snd_cs4231_freq[13].rate;
}

static unsigned int snd_cs4231_xrate(snd_pcm1_t * pcm1, cs4231_t * codec,
                                     unsigned int rate)
{
	return snd_cs4231_get_real_freq(rate);
}

static int snd_cs4231_playback_ioctl(snd_pcm1_t * pcm1, unsigned int cmd,
                                     unsigned long *arg)
{
	cs4231_t *codec = (cs4231_t *) pcm1->private_data;

	switch (cmd) {
	case SND_PCM1_IOCTL_RATE:
		pcm1->playback.real_rate = codec->set_playback_rate(pcm1, codec, pcm1->playback.rate);
		return 0;
	}
	return -ENXIO;
}

static int snd_cs4231_record_ioctl(snd_pcm1_t * pcm1, unsigned int cmd,
                                   unsigned long *arg)
{
	cs4231_t *codec = (cs4231_t *) pcm1->private_data;

	switch (cmd) {
	case SND_PCM1_IOCTL_RATE:
		pcm1->record.real_rate = codec->set_record_rate(pcm1, codec, pcm1->record.rate);
		return 0;
	}
	return -ENXIO;
}

static unsigned char snd_cs4231_get_format(cs4231_t * codec,
				           unsigned int mode,
                                           int voices)
{
	unsigned char format;

	format = 0;
	if (mode & SND_PCM1_MODE_16) {
		if (codec->hardware == CS4231_HW_CS4235 ||
		    codec->hardware == CS4231_HW_CS4239) {
			format |= CS4231_LINEAR_16;
		} else {
			if (mode & SND_PCM1_MODE_ADPCM)
				format |= CS4231_ADPCM_16;
			else if (mode & SND_PCM1_MODE_BIG)
				format |= CS4231_LINEAR_16_BIG;
			else
				format |= CS4231_LINEAR_16;
		}
	} else {
		if (codec->hardware == CS4231_HW_CS4235 ||
		    codec->hardware == CS4231_HW_CS4239) {
			/* nothing */
		} else {
			if (mode & SND_PCM1_MODE_ALAW)
				format |= CS4231_ALAW_8;
			else if (mode & SND_PCM1_MODE_ULAW)
				format |= CS4231_ULAW_8;
#if 0				/* CS4231_LINEAR_8 == 0 */
			else
				format |= CS4231_LINEAR_8;
#endif
		}
	}
	if (voices > 1)
		format |= CS4231_STEREO;
#if 0
	snd_printk("snd_cs4231_get_format: 0x%x (mode=0x%x)\n", format, mode);
#endif
	return format;
}

static void snd_cs4231_calibrate_mute(cs4231_t * codec, unsigned int mute)
{
	static unsigned int sources[] =
	{
		SND_MIXER_PRI_AUXA,
		SND_MIXER_PRI_SYNTHESIZER,	/* GUS MAX */
		SND_MIXER_PRI_AUXB,
		SND_MIXER_PRI_CD,		/* GUS MAX */
		SND_MIXER_PRI_LOOPBACK,
		SND_MIXER_PRI_LINE,
		SND_MIXER_PRI_MIC,		/* GUS MAX */
		SND_MIXER_PRI_MONO,
		SND_MIXER_PRI_PARENT
	};
	unsigned int *src = sources;

	if (!(codec->hardware & CS4231_HW_CS4231_MASK))
		return;
	/* CS4231 chips produces clicks when calibration is in progress */
	mute = mute ? SND_MIX_MUTE : 0;
	while (*src != SND_MIXER_PRI_PARENT) {
		snd_mixer_set_kernel_mute(codec->mixer, *src, mute, 0);
		src++;
	}
#if 0
	printk("calibrate mute = %i\n", mute);
	snd_cs4231_debug(codec);
#endif
}

static void snd_cs4231_playback_format(snd_pcm1_t * pcm1,
				       cs4231_t * codec,
				       unsigned char pdfr)
{
	unsigned long flags;
	int full_calib = 1;

	snd_mutex_down(codec, mce);
	snd_cs4231_calibrate_mute(codec, 1);
	if (codec->hardware == CS4231_HW_CS4231A ||
	    (codec->hardware & CS4231_HW_CS4232_MASK)) {
		snd_spin_lock(codec, reg, &flags);
		if ((codec->image.pdfr & 0x0f) == (pdfr & 0x0f)) {	/* rate is same? */
			snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image.afei | 0x10);
			snd_cs4231_out(codec, CS4231_PLAYBK_FORMAT, codec->image.pdfr = pdfr);
			snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image.afei);
			full_calib = 0;
		}
		snd_spin_unlock(codec, reg, &flags);
	}
	if (full_calib) {
		snd_cs4231_mce_up(codec);
		snd_spin_lock(codec, reg, &flags);
		if (codec->hardware != CS4231_HW_INTERWAVE && !codec->single_dma) {
			snd_cs4231_out(codec, CS4231_PLAYBK_FORMAT,
				       codec->image.pdfr =
					(codec->image.ic & CS4231_RECORD_ENABLE) ?
					(pdfr & 0xf0) | (codec->image.cdfr & 0x0f) :
				        pdfr);
		} else {
			snd_cs4231_out(codec, CS4231_PLAYBK_FORMAT, codec->image.pdfr = pdfr);
		}
		snd_spin_unlock(codec, reg, &flags);
		snd_cs4231_mce_down(codec);
	}
	snd_cs4231_calibrate_mute(codec, 0);
	snd_mutex_up(codec, mce);
}

static void snd_cs4231_record_format(snd_pcm1_t * pcm1,
                                     cs4231_t * codec,
                                     unsigned char cdfr)
{
	unsigned long flags;
	int full_calib = 1;

	snd_mutex_down(codec, mce);
	snd_cs4231_calibrate_mute(codec, 1);
	if (codec->hardware == CS4231_HW_CS4231A ||
	    (codec->hardware & CS4231_HW_CS4232_MASK)) {
		snd_spin_lock(codec, reg, &flags);
		if ((codec->image.pdfr & 0x0f) == (cdfr & 0x0f) ||	/* rate is same? */
		    (codec->image.ic & CS4231_PLAYBACK_ENABLE)) {
			snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image.afei | 0x20);
			snd_cs4231_out(codec, CS4231_REC_FORMAT, (codec->image.cdfr = cdfr) & 0xf0);
			snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image.afei);
			full_calib = 0;
		}
		snd_spin_unlock(codec, reg, &flags);
	}
	if (full_calib) {
		snd_cs4231_mce_up(codec);
		snd_spin_lock(codec, reg, &flags);
		if (codec->hardware != CS4231_HW_INTERWAVE) {
			if (!(codec->image.ic & CS4231_PLAYBACK_ENABLE)) {
				snd_cs4231_out(codec, CS4231_PLAYBK_FORMAT,
					       codec->image.pdfr =
					       ((codec->single_dma ? cdfr : codec->image.pdfr) & 0xf0) |
					       (cdfr & 0x0f));
				snd_spin_unlock(codec, reg, &flags);
				snd_cs4231_mce_down(codec);
				snd_cs4231_mce_up(codec);
				snd_spin_lock(codec, reg, &flags);
			}
			snd_cs4231_out(codec, CS4231_REC_FORMAT, (codec->image.cdfr = cdfr) & 0xf0);
		} else {
			snd_cs4231_out(codec, CS4231_REC_FORMAT, codec->image.cdfr = cdfr);
		}
		snd_spin_unlock(codec, reg, &flags);
		snd_cs4231_mce_down(codec);
	}
	snd_cs4231_calibrate_mute(codec, 0);
	snd_mutex_up(codec, mce);
}

/*
 *  Timer interface
 */

static unsigned int snd_cs4231_timer_resolution(snd_timer_t * timer)
{
	cs4231_t *codec;

	codec = (cs4231_t *) timer->private_data;
	if (codec->hardware & CS4231_HW_CS4236_MASK)
		return 14467;
	else
		return codec->image.pdfr & 1 ? 9969 : 9920;
}

static void snd_cs4231_timer_start(snd_timer_t * timer)
{
	unsigned long flags;
	unsigned int ticks;
	cs4231_t *codec;

	codec = (cs4231_t *) timer->private_data;
	snd_spin_lock(codec, reg, &flags);
	ticks = timer->cticks;
	if (ticks > 65535)
		ticks = 65535;
	timer->cticks -= ticks;
	if ((codec->image.afei & CS4231_TIMER_ENABLE) == 0 ||
	    (unsigned char)(ticks >> 8) != codec->image.thb ||
	    (unsigned char)ticks != codec->image.tlb) {
		codec->image.afei |= CS4231_TIMER_ENABLE;
		snd_cs4231_out(codec, CS4231_TIMER_HIGH, codec->image.thb = (unsigned char) (ticks >> 8));
		snd_cs4231_out(codec, CS4231_TIMER_LOW, codec->image.tlb = (unsigned char) ticks);
		snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image.afei);
		timer->flags |= SND_TIMER_FLG_RUNNING;
	}
	snd_spin_unlock(codec, reg, &flags);
}

static void snd_cs4231_timer_stop(snd_timer_t * timer)
{
	unsigned long flags;
	cs4231_t *codec;

	codec = (cs4231_t *) timer->private_data;
	snd_spin_lock(codec, reg, &flags);
	timer->flags &= ~SND_TIMER_FLG_RUNNING;
	codec->image.afei &= ~CS4231_TIMER_ENABLE;
	snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image.afei);
	snd_spin_unlock(codec, reg, &flags);
}

static void snd_cs4231_init(cs4231_t * codec)
{
	unsigned long flags;

	snd_cs4231_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_cs4231_init: (1)\n");
#endif
	snd_cs4231_mce_up(codec);
	snd_spin_lock(codec, reg, &flags);
	codec->image.ic &= ~(CS4231_PLAYBACK_ENABLE | CS4231_PLAYBACK_PIO |
			     CS4231_RECORD_ENABLE | CS4231_RECORD_PIO |
			     CS4231_CALIB_MODE);
	codec->image.ic |= CS4231_AUTOCALIB;
	snd_cs4231_out(codec, CS4231_IFACE_CTRL, codec->image.ic);
	snd_spin_unlock(codec, reg, &flags);
	snd_cs4231_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_cs4231_init: (2)\n");
#endif

	snd_cs4231_mce_up(codec);
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image.afei);
	snd_spin_unlock(codec, reg, &flags);
	snd_cs4231_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_cs4231_init: (3) - afei = 0x%x\n", codec->image.afei);
#endif

	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_ALT_FEATURE_2, codec->image.afeii);
	snd_spin_unlock(codec, reg, &flags);

	snd_cs4231_mce_up(codec);
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_PLAYBK_FORMAT, codec->image.pdfr);
	snd_spin_unlock(codec, reg, &flags);
	snd_cs4231_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_cs4231_init: (4)\n");
#endif

	snd_cs4231_mce_up(codec);
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_REC_FORMAT,
		       codec->hardware == CS4231_HW_INTERWAVE ?
		       codec->image.cdfr :
		       (codec->image.cdfr & 0xf0));
	snd_spin_unlock(codec, reg, &flags);
	snd_cs4231_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_cs4231_init: (5)\n");
#endif
}

static void snd_cs4231_open(cs4231_t * codec, unsigned int mode)
{
	unsigned long flags;

	snd_mutex_down(codec, open);
	if (codec->mode & CS4231_MODE_OPEN) {
		codec->mode |= mode;
		snd_mutex_up(codec, open);
		return;
	}
	/* ok. now enable and ack CODEC IRQ */
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, CS4231_PLAYBACK_IRQ |
		       CS4231_RECORD_IRQ |
		       CS4231_TIMER_IRQ);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, 0);
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	codec->image.pc |= CS4231_IRQ_ENABLE;
	snd_cs4231_out(codec, CS4231_PIN_CTRL, codec->image.pc);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, CS4231_PLAYBACK_IRQ |
		       CS4231_RECORD_IRQ |
		       CS4231_TIMER_IRQ);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, 0);
	snd_spin_unlock(codec, reg, &flags);

	codec->mode = mode;
	snd_mutex_up(codec, open);
}

static void snd_cs4231_close(cs4231_t * codec, unsigned int mode)
{
	unsigned long flags;

	snd_mutex_down(codec, open);
	codec->mode &= ~mode;
	if (codec->mode & CS4231_MODE_OPEN) {
		snd_mutex_up(codec, open);
		return;
	}
	snd_cs4231_calibrate_mute(codec, 1);

	/* disable IRQ */
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, 0);
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	codec->image.pc &= ~CS4231_IRQ_ENABLE;
	snd_cs4231_out(codec, CS4231_PIN_CTRL, codec->image.pc);
	snd_spin_unlock(codec, reg, &flags);

	/* now disable record & playback */

	snd_cs4231_mce_up(codec);
	snd_spin_lock(codec, reg, &flags);
	codec->image.ic &= ~(CS4231_PLAYBACK_ENABLE | CS4231_PLAYBACK_PIO |
			     CS4231_RECORD_ENABLE | CS4231_RECORD_PIO);
	snd_cs4231_out(codec, CS4231_IFACE_CTRL, codec->image.ic);
	snd_spin_unlock(codec, reg, &flags);
	snd_cs4231_mce_down(codec);

	/* clear IRQ again */
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, 0);
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	snd_spin_unlock(codec, reg, &flags);

	snd_cs4231_calibrate_mute(codec, 0);

	codec->mode = 0;
	snd_mutex_up(codec, open);
}

/*
 *  timer open/close
 */

static int snd_cs4231_timer_open(snd_timer_t * timer)
{
	cs4231_t *codec;

	codec = (cs4231_t *) timer->private_data;
	snd_cs4231_open(codec, CS4231_MODE_TIMER);
	return 0;
}

static void snd_cs4231_timer_close(snd_timer_t * timer)
{
	cs4231_t *codec;

	codec = (cs4231_t *) timer->private_data;
	snd_cs4231_close(codec, CS4231_MODE_TIMER);
}

static struct snd_stru_timer_hardware snd_cs4231_timer =
{
	9945,			/* resolution in us * 1000 */
	1,			/* low timer mark */
	65535,			/* high timer mark */
	snd_cs4231_timer_open,	/* open */
	snd_cs4231_timer_close,	/* close */
	snd_cs4231_timer_resolution, /* resolution */
	snd_cs4231_timer_start,	/* start */
	snd_cs4231_timer_stop,	/* stop */
	NULL,			/* continue */
};

/*
 *  ok.. exported functions..
 */

static void snd_cs4231_playback_trigger(snd_pcm1_t * pcm1, int up)
{
	cs4231_t *codec;

	codec = (cs4231_t *) pcm1->private_data;

	if (codec->mixer && !up)
		snd_mixer_set_kernel_mute(codec->mixer, SND_MIXER_PRI_PCM, SND_MIX_MUTE, 0);

	snd_cs4231_trigger(codec, CS4231_PLAYBACK_ENABLE, up);

	if (codec->mixer && up)
		snd_mixer_set_kernel_mute(codec->mixer, SND_MIXER_PRI_PCM, 0, 0);

#if 0
	if (up)
		snd_cs4231_debug(codec);
#endif
}

static void snd_cs4231_record_trigger(snd_pcm1_t * pcm1, int up)
{
	cs4231_t *codec;

	codec = (cs4231_t *) pcm1->private_data;
	snd_cs4231_trigger(codec, CS4231_RECORD_ENABLE, up);
}

static void snd_cs4231_playback_prepare(snd_pcm1_t * pcm1,
					unsigned char *buffer,
					unsigned int size,
					unsigned int offset,
					unsigned int count)
{
	cs4231_t *codec;
	unsigned long flags;
	unsigned int mode;
	unsigned char new_pdfr;

	codec = (cs4231_t *) pcm1->private_data;
	/* Note: offset is always 0 if AUTO DMA */
#if 0
	snd_printk("cs4231_start_playback - dma = %i, buffer = 0x%lx, size = 0x%x, offset = 0x%x, count = 0x%x\n", codec->dma1, (long) buffer, size, offset, count);
#endif
	mode = pcm1->playback.mode;
	if ((mode & SND_PCM1_MODE_ULAW) &&
	    codec->hardware == CS4231_HW_INTERWAVE &&
	    codec->dma1 > 3) {
		mode = SND_PCM1_MODE_U;
	}
	new_pdfr = snd_cs4231_get_format(codec, mode, pcm1->playback.voices) |
	    snd_cs4231_get_freq(pcm1->playback.real_rate);
	codec->image.ic &= ~(CS4231_PLAYBACK_ENABLE | CS4231_PLAYBACK_PIO);
	snd_dma_program(codec->dma1, buffer, size, DMA_MODE_WRITE | DMA_MODE_AUTOINIT);
	codec->set_playback_format(pcm1, codec, new_pdfr);
	count = snd_cs4231_get_count(codec->image.pdfr, count) - 1;
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_PLY_LWR_CNT, (unsigned char) count);
	snd_cs4231_out(codec, CS4231_PLY_UPR_CNT, (unsigned char) (count >> 8));
	snd_spin_unlock(codec, reg, &flags);
#if 0
	snd_cs4231_debug(codec);
#endif
}

static void snd_cs4231_record_prepare(snd_pcm1_t * pcm1,
				      unsigned char *buffer,
				      unsigned int size,
				      unsigned int offset,
				      unsigned int count)
{
	cs4231_t *codec;
	unsigned long flags;
	unsigned char new_cdfr;

	codec = (cs4231_t *) pcm1->private_data;
#if 0
	snd_printk("cs4231_start_record: start - buffer = 0x%lx, offset = 0x%x, size = 0x%x, count = 0x%x\n", (long) buffer, offset, size, count);
#endif
	new_cdfr = snd_cs4231_get_format(codec, pcm1->record.mode, pcm1->record.voices) |
	    snd_cs4231_get_freq(pcm1->record.real_rate);
	codec->image.ic &= ~(CS4231_RECORD_ENABLE | CS4231_RECORD_PIO);
	snd_dma_program(codec->dma2, buffer, size, DMA_MODE_READ | DMA_MODE_AUTOINIT);
#if 0
	printk("cdfr = 0x%x, rport = 0x%x, pport = 0x%x, status = 0x%x\n", codec->image.cdfr, snd_cs4231_in(codec, CS4231_REC_FORMAT), snd_cs4231_in(codec, CS4231_PLAYBK_FORMAT), inb(CS4231P(codec, REGSEL)));
#endif
	codec->set_record_format(pcm1, codec, new_cdfr);
	count = snd_cs4231_get_count(codec->image.cdfr, count) - 1;
	snd_spin_lock(codec, reg, &flags);
	if (codec->single_dma && codec->hardware != CS4231_HW_INTERWAVE) {
		snd_cs4231_out(codec, CS4231_PLY_LWR_CNT, (unsigned char) count);
		snd_cs4231_out(codec, CS4231_PLY_UPR_CNT, (unsigned char) (count >> 8));
	} else {
		snd_cs4231_out(codec, CS4231_REC_LWR_CNT, (unsigned char) count);
		snd_cs4231_out(codec, CS4231_REC_UPR_CNT, (unsigned char) (count >> 8));
	}
	snd_spin_unlock(codec, reg, &flags);
}

static void snd_cs4231_overrange(snd_pcm1_t * pcm1)
{
	cs4231_t *codec;
	unsigned long flags;
	unsigned char res;

	codec = (cs4231_t *) pcm1->private_data;
	snd_spin_lock(codec, reg, &flags);
	res = snd_cs4231_in(codec, CS4231_TEST_INIT);
	snd_spin_unlock(codec, reg, &flags);
	if (res & (0x08 | 0x02))	/* detect overrange only above 0dB; maybe user selectable? */
		pcm1->record.overrange++;
}

void snd_cs4231_interrupt(snd_pcm_t * pcm, unsigned char status)
{
	unsigned long flags;
	snd_pcm1_t *pcm1;
	cs4231_t *codec;
	snd_timer_t *timer;

#if 0
	snd_printk("snd_cs4231_interrupt: status=0x%x\n", status);
#endif
	pcm1 = (snd_pcm1_t *) pcm->private_data;
	codec = (cs4231_t *) pcm1->private_data;
	if (!codec)
		return;
	status = snd_cs4231_in(codec, CS4231_IRQ_STATUS);

	if (status & CS4231_TIMER_IRQ) {
		timer = codec->timer;
		if (!timer->cticks) {
			if (timer->flags & SND_TIMER_FLG_AUTO) {
				timer->cticks = timer->ticks;
				snd_cs4231_timer_start(timer);
			} else {
				snd_cs4231_timer_stop(timer);
			}
			if (timer->callback)
				timer->callback(timer, timer->callback_data);
		} else {
			if (timer->cticks > 65535) {
				timer->cticks -= 65535;
			} else {
				snd_cs4231_timer_stop(timer);
				snd_cs4231_timer_start(timer);
			}
		}
	}
	if (codec->single_dma && codec->hardware != CS4231_HW_INTERWAVE) {
		if (status & CS4231_PLAYBACK_IRQ) {
			if ((codec->mode & CS4231_MODE_PLAY) && pcm1->playback.ack)
				pcm1->playback.ack(pcm1);
			if ((codec->mode & CS4231_MODE_RECORD) && pcm1->record.ack) {
				snd_cs4231_overrange(pcm1);
				pcm1->record.ack(pcm1);
			}
		}
	} else {
		if ((status & CS4231_PLAYBACK_IRQ) && pcm1->playback.ack)
			pcm1->playback.ack(pcm1);
		if ((status & CS4231_RECORD_IRQ) && pcm1->record.ack) {
			snd_cs4231_overrange(pcm1);
			pcm1->record.ack(pcm1);
		}
	}

	snd_spin_lock(codec, reg, &flags);
	outb(0, CS4231P(codec, STATUS));	/* clear global interrupt bit */
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, CS4231_ALL_IRQS & ~status);
	snd_spin_unlock(codec, reg, &flags);
}

/*

 */

static int snd_cs4231_playback_open(snd_pcm1_t * pcm1)
{
	snd_card_t *card;
	cs4231_t *codec;
	int err;

	card = pcm1->card;
	codec = (cs4231_t *) pcm1->private_data;
	if ((err = snd_pcm1_dma_alloc(pcm1, SND_PCM1_PLAYBACK, codec->dmaptr1, "CS4231 (playback)")) < 0)
		return err;
	snd_cs4231_open(codec, CS4231_MODE_PLAY);
	return 0;
}

static int snd_cs4231_record_open(snd_pcm1_t * pcm1)
{
	snd_card_t *card;
	cs4231_t *codec;
	int err;

	card = pcm1->card;
	codec = (cs4231_t *) pcm1->private_data;
	if ((err = snd_pcm1_dma_alloc(pcm1, SND_PCM1_RECORD, codec->dmaptr2, "CS4231 (record)")) < 0)
		return err;
	snd_cs4231_open(codec, CS4231_MODE_RECORD);
	return 0;
}

static void snd_cs4231_playback_close(snd_pcm1_t * pcm1)
{
	snd_card_t *card;
	cs4231_t *codec;

	card = pcm1->card;
	codec = (cs4231_t *) pcm1->private_data;
	snd_cs4231_close(codec, CS4231_MODE_PLAY);
	snd_pcm1_dma_free(pcm1, SND_PCM1_PLAYBACK, codec->dmaptr1);
}

static void snd_cs4231_record_close(snd_pcm1_t * pcm1)
{
	snd_card_t *card;
	cs4231_t *codec;

	card = pcm1->card;
	codec = (cs4231_t *) pcm1->private_data;
	snd_cs4231_close(codec, CS4231_MODE_RECORD);
	snd_pcm1_dma_free(pcm1, SND_PCM1_RECORD, codec->dmaptr2);
}

static void snd_cs4231_playback_dma(snd_pcm1_t * pcm1,
			      unsigned char *buffer, unsigned int offset,
				 unsigned char *user, unsigned int count)
{
	cs4231_t *codec;

	codec = (cs4231_t *) pcm1->private_data;
	if ((pcm1->playback.mode & SND_PCM1_MODE_ULAW) &&
	    codec->hardware == CS4231_HW_INTERWAVE && codec->dma1 > 3) {
		snd_translate_from_user(snd_ulaw_dsp_loud, &buffer[offset], user, count);
	} else {
		copy_from_user(&buffer[offset], user, count);
	}
}

static unsigned int snd_cs4231_playback_pointer(snd_pcm1_t * pcm1, unsigned int used_size)
{
	cs4231_t *codec;

	codec = (cs4231_t *) pcm1->private_data;
	if (!(codec->image.ic & CS4231_PLAYBACK_ENABLE))
		return 0;
	return used_size - snd_dma_residue(codec->dma1);
}

static unsigned int snd_cs4231_record_pointer(snd_pcm1_t * pcm1, unsigned int used_size)
{
	cs4231_t *codec;

	codec = (cs4231_t *) pcm1->private_data;
	if (!(codec->image.ic & CS4231_RECORD_ENABLE))
		return 0;
	return used_size - snd_dma_residue(codec->dma2);
}

/*

 */

#if 0
void snd_cs4231_info(snd_card_t * card, snd_info_buffer_t * buffer)
{
	if (!card->use_codec)
		return;
	snd_iprintf(buffer, "CS4231:\n");
	snd_iprintf(buffer, "  mode               : %s\n", card->codec.hardware != CS4231_HW_INTERWAVE ? "2" : "3");
	snd_iprintf(buffer, "  record overflow    : %i\n", card->codec.record_overflow);
	if (card->pnp_flag) {
		snd_iprintf(buffer, "  playback fifo      : %i\n", card->codec.playback_fifo_size);
		snd_iprintf(buffer, "  record fifo        : %i\n", card->codec.record_fifo_size);
	}
}
#endif

/*

 */

static int snd_cs4231_probe(snd_pcm1_t * pcm1)
{
	unsigned long flags;
	snd_pcm_t *pcm;
	cs4231_t *codec;
	int i, id, rev;
	unsigned char *ptr;
	unsigned int hw;

	codec = (cs4231_t *) pcm1->private_data;
	if (!codec)
		return -EINVAL;
	pcm = pcm1->pcm;
	if (!pcm)
		return -EINVAL;

#if 0
	snd_cs4231_debug(codec);
#endif
	id = 0;
	for (i = 0; i < 1000; i++) {
		mb();
		if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
			snd_delay(50 * 8);
		else {
			snd_spin_lock(codec, reg, &flags);
			snd_cs4231_out(codec, CS4231_MISC_INFO, CS4231_MODE2);
			id = snd_cs4231_in(codec, CS4231_MISC_INFO) & 0x0f;
			snd_spin_unlock(codec, reg, &flags);
			if (id == 0x0a)
				break;	/* this is valid value */
		}
	}
	if (id != 0x0a)
		return -ENODEV;	/* no valid device found */

	if (((hw = codec->hardware) & CS4231_HW_TYPE_MASK) == CS4231_HW_DETECT) {
		rev = snd_cs4231_in(codec, CS4231_VERSION) & 0xe7;
		snd_printdd("CS4231: VERSION (I25) = 0x%x\n", rev);
		if (rev == 0x80) {
			codec->hardware = CS4231_HW_CS4231;
		} else if (rev == 0xa0) {
			codec->hardware = CS4231_HW_CS4231A;
		} else if (rev == 0xa2) {
			codec->hardware = CS4231_HW_CS4232;
		} else if (rev == 0xb2) {
			codec->hardware = CS4231_HW_CS4232A;
		} else if (rev == 0x03) {
			codec->hardware = CS4231_HW_CS4236;
		} else {
			snd_printk("cs4231: unknown CS chip with version 0x%x\n", rev);
			return -ENODEV;		/* unknown CS4231 chip? */
		}
	}
	snd_spin_lock(codec, reg, &flags);
	inb(CS4231P(codec, STATUS));	/* clear any pendings IRQ */
	outb(0, CS4231P(codec, STATUS));
	mb();
	snd_spin_unlock(codec, reg, &flags);

	codec->image.mi = CS4231_MODE2;
	switch (codec->hardware) {
	case CS4231_HW_INTERWAVE:
		codec->image.mi = CS4231_IW_MODE3;
		break;
	case CS4231_HW_CS4235:
	case CS4231_HW_CS4236:
	case CS4231_HW_CS4236B:
	case CS4231_HW_CS4237B:
	case CS4231_HW_CS4238B:
	case CS4231_HW_CS4239:
		if (hw == CS4231_HW_DETECT3)
			codec->image.mi = CS4231_4236_MODE3;
		else
			codec->hardware = CS4231_HW_CS4232A;
		break;
	}

	codec->image.ic =
	    (codec->image.ic & ~CS4231_SINGLE_DMA) |
	    (codec->single_dma ? CS4231_SINGLE_DMA : 0);
	codec->image.afei = 0x80;
	codec->image.afeii = codec->hardware == CS4231_HW_INTERWAVE ? 0xc2 : 0x01;
	ptr = (unsigned char *) &codec->image;
	snd_cs4231_mce_down(codec);
	snd_spin_lock(codec, reg, &flags);
	for (i = 0; i < 32; i++)	/* ok.. fill all CS4231 registers */
		snd_cs4231_out(codec, i, *ptr++);
	snd_spin_unlock(codec, reg, &flags);
	snd_cs4231_mce_up(codec);
	snd_cs4231_mce_down(codec);

	/* ok.. try check hardware version for CS4236+ chips */
	if ((hw & CS4231_HW_TYPE_MASK) == CS4231_HW_DETECT) {
		if (codec->hardware == CS4231_HW_CS4236) {
			rev = snd_cs4236_ext_in(codec, CS4236_VERSION);
			snd_cs4236_ext_out(codec, CS4236_VERSION, 0xff);
			id = snd_cs4236_ext_in(codec, CS4236_VERSION);
			snd_cs4236_ext_out(codec, CS4236_VERSION, rev);
			snd_printdd("CS4231: ext version; rev = 0x%x, id = 0x%x\n", rev, id);
			if ((id & 0x1f) == 0x1d) {	/* CS4235 */
				codec->hardware = CS4231_HW_CS4235;
				switch (id >> 5) {
				case 4:
				case 5:
				case 6:
					break;
				default:
					snd_printk("cs4231: unknown CS4235 chip (enhanced version = 0x%x)\n", id);
				}
			} else if ((id & 0x1f) == 0x0b) {	/* CS4236/B */
				switch (id >> 5) {
				case 4:
				case 5:
				case 6:
				case 7:
					codec->hardware = CS4231_HW_CS4236B;
					break;
				default:
					snd_printk("cs4231: unknown CS4236 chip (enhanced version = 0x%x)\n", id);
				}
			} else if ((id & 0x1f) == 0x08) {	/* CS4237B */
				codec->hardware = CS4231_HW_CS4237B;
				switch (id >> 5) {
				case 4:
				case 5:
				case 6:
				case 7:
					break;
				default:
					snd_printk("cs4231: unknown CS4237B chip (enhanced version = 0x%x)\n", id);
				}
			} else if ((id & 0x1f) == 0x09) {	/* CS4238B */
				codec->hardware = CS4231_HW_CS4238B;
				switch (id >> 5) {
				case 5:
				case 6:
				case 7:
					break;
				default:
					snd_printk("cs4231: unknown CS4238B chip (enhanced version = 0x%x)\n", id);
				}
			} else if ((id & 0x1f) == 0x1e) {	/* CS4239 */
				codec->hardware = CS4231_HW_CS4239;
				switch (id >> 5) {
				case 4:
				case 5:
				case 6:
					break;
				default:
					snd_printk("cs4231: unknown CS4239 chip (enhanced version = 0x%x)\n", id);
				}
			} else {
				snd_printk("cs4231: unknown CS4236/CS423xB chip (enhanced version = 0x%x)\n", id);
			}
		}
	}
	return 0;		/* all things are ok.. */
}

/*

 */

static struct snd_stru_pcm1_hardware snd_cs4231_playback =
{
	NULL,			/* private data */
	NULL,			/* private_free */
	SND_PCM1_HW_AUTODMA,	/* flags */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_A_LAW | SND_PCM_FMT_IMA_ADPCM |
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE | SND_PCM_FMT_S16_BE,	/* formats */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_A_LAW | SND_PCM_FMT_IMA_ADPCM |
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE | SND_PCM_FMT_S16_BE,	/* hardware formats */
	0,			/* align value */
	6,			/* minimal fragment */
	5510,			/* min. rate */
	48000,			/* max. rate */
	2,			/* max. voices */
	snd_cs4231_playback_open,
	snd_cs4231_playback_close,
	snd_cs4231_playback_ioctl,
	snd_cs4231_playback_prepare,
	snd_cs4231_playback_trigger,
	snd_cs4231_playback_pointer,
	snd_cs4231_playback_dma,
	snd_pcm1_dma_move,
	snd_pcm1_playback_dma_neutral
};

static struct snd_stru_pcm1_hardware snd_cs4231_record =
{
	NULL,			/* private data */
	NULL,			/* private free */
	SND_PCM1_HW_AUTODMA,	/* flags */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_A_LAW | SND_PCM_FMT_IMA_ADPCM |
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE | SND_PCM_FMT_S16_BE,	/* formats */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_A_LAW | SND_PCM_FMT_IMA_ADPCM |
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE | SND_PCM_FMT_S16_BE,	/* hardware formats */
	0,			/* align value */
	6,			/* minimal fragment */
	5510,			/* min. rate */
	48000,			/* max. rate */
	2,			/* max. voices */
	snd_cs4231_record_open,
	snd_cs4231_record_close,
	snd_cs4231_record_ioctl,
	snd_cs4231_record_prepare,
	snd_cs4231_record_trigger,
	snd_cs4231_record_pointer,
	snd_pcm1_record_dma,
	snd_pcm1_dma_move,
	NULL
};

static void snd_cs4231_free(void *private_data)
{
	cs4231_t *codec;

	codec = (cs4231_t *) private_data;
	if (codec->timer)
		snd_timer_unregister(codec->timer);
	snd_free(codec, sizeof(cs4231_t));
}

snd_pcm_t *snd_cs4231_new_device(snd_card_t * card,
				 unsigned short port,
				 snd_irq_t * irqptr,
				 snd_dma_t * dmaptr1,
				 snd_dma_t * dmaptr2,
				 unsigned short hardware)
{
	snd_pcm_t *pcm;
	snd_pcm1_t *pcm1;
	cs4231_t *codec;
	snd_timer_t *timer;
	char *str;

	pcm = snd_pcm1_new_device(card, "CS4231");
	if (!pcm)
		return NULL;
	pcm1 = (snd_pcm1_t *) pcm->private_data;
	codec = (cs4231_t *) snd_calloc(sizeof(cs4231_t));
	if (!codec)
		return NULL;
	snd_spin_prepare(codec, reg);
	snd_mutex_prepare(codec, mce);
	snd_mutex_prepare(codec, open);
	snd_sleep_prepare(codec, mce);
	snd_sleep_prepare(codec, iec958);
	codec->pcm = pcm;
	codec->card = pcm->card;
	codec->port = port;
	codec->irq = irqptr->irq;
	codec->irqptr = irqptr;
	codec->dmaptr1 = dmaptr1;
	codec->dma1 = dmaptr1->dma;
	codec->dmaptr2 = dmaptr2;
	codec->dma2 = dmaptr2->dma;
	if (codec->dma1 == codec->dma2)
		codec->single_dma = 1;
	codec->hardware = hardware;
	codec->set_playback_rate =
	    codec->set_record_rate = snd_cs4231_xrate;
	codec->set_playback_format = snd_cs4231_playback_format;
	codec->set_record_format = snd_cs4231_record_format;
	memcpy(&codec->image, &snd_cs4231_original_image, sizeof(snd_cs4231_original_image));
	memcpy(&pcm1->playback.hw, &snd_cs4231_playback, sizeof(snd_cs4231_playback));
	memcpy(&pcm1->record.hw, &snd_cs4231_record, sizeof(snd_cs4231_record));
	if (codec->hardware == CS4231_HW_INTERWAVE && codec->dma1 > 3)
		pcm1->playback.hw.hw_formats &= ~SND_PCM_FMT_MU_LAW;	/* bug reached */
	pcm1->private_data = codec;
	pcm1->private_free = snd_cs4231_free;
	pcm->info_flags = SND_PCM_INFO_CODEC | SND_PCM_INFO_MMAP |
	    SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_RECORD;
	if (!codec->single_dma)
		pcm->info_flags |= SND_PCM_INFO_DUPLEX;
	if (codec->hardware != CS4231_HW_INTERWAVE)
		pcm->info_flags |= SND_PCM_INFO_DUPLEX_LIMIT;
	if (snd_cs4231_probe(pcm1) < 0) {
		snd_pcm_free(pcm);
		return NULL;
	}
	switch (codec->hardware) {
	case CS4231_HW_CS4231:
		str = "CS4231";
		break;
	case CS4231_HW_CS4231A:
		str = "CS4231A";
		break;
	case CS4231_HW_CS4232:
		str = "CS4232";
		break;
	case CS4231_HW_CS4232A:
		str = "CS4232A";
		break;
	case CS4231_HW_CS4235:
		str = "CS4235";
		break;
	case CS4231_HW_CS4236B:
		str = "CS4236B";
		break;
	case CS4231_HW_CS4237B:
		str = "CS4237B";
		break;
	case CS4231_HW_CS4238B:
		str = "CS4238B";
		break;
	case CS4231_HW_CS4239:
		str = "CS4239";
		break;
	case CS4231_HW_INTERWAVE:
		str = "AMD InterWave";
		break;
	case CS4231_HW_OPL3SA:
		str = card->shortname;
		break;
	default:
		str = "???";
	}
	strcpy(pcm->name, str);
	snd_cs4231_init(codec);
	timer = snd_timer_new_device(card, "CS4231");
	if (timer) {
		strcpy(timer->name, pcm->name);
		timer->private_data = codec;
		memcpy(&timer->hw, &snd_cs4231_timer, sizeof(snd_cs4231_timer));
		if (snd_timer_register(timer) < 0) {
			snd_timer_free(timer);
			timer = NULL;
		}
	}
	codec->timer = timer;
	return pcm;
}

/*
 *  MIXER part
 */

/*
 *    private_value
 *              0x000000ff - mute mask
 *              0xff000000 - left register (or mono register)
 *              0x00ff0000 - right register
 *              0x00000f00 - shift (from right)
 *              0x00001000 - invert mute
 *              0x00002000 - invert value
 */

static void snd_cs4231_record_source(snd_kmixer_t * mixer,
				     snd_kmixer_channel_t * channel,
				     unsigned int rec)
{
	unsigned long flags;
	unsigned char mixs = CS4231_MIXS_LINE, mask = 0;
	cs4231_t *codec;

	switch (channel->hw.priority) {
	case SND_MIXER_PRI_AUXA:
		mask = 1;
		break;
	case SND_MIXER_PRI_MIC:
	case SND_MIXER_PRI_MONO:
		mask = 2;
		break;
	case SND_MIXER_PRI_LINE:
		mask = 4;
		break;
	default:
		mask = 8;
		break;		/* master */
	}
	if (rec & SND_MIX_REC)
		mixer->private_value |= mask;
	else
		mixer->private_value &= ~mask;

	if (mixer->private_value == 0)
		mixs = CS4231_MIXS_LINE;
	else if (mixer->private_value & 8)
		mixs = CS4231_MIXS_ALL;
	else if (mixer->private_value & 2)
		mixs = CS4231_MIXS_MIC;
	else if (mixer->private_value & 4)
		mixs = CS4231_MIXS_LINE;
	else
		mixs = CS4231_MIXS_AUX1;

	codec = (cs4231_t *) mixer->private_data;
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_outm(codec, CS4231_LEFT_INPUT, 0x3f, mixs);
	snd_cs4231_outm(codec, CS4231_RIGHT_INPUT, 0x3f, mixs);
	snd_spin_unlock(codec, reg, &flags);
}

static void snd_cs4231_mute(snd_kmixer_t * mixer,
			    snd_kmixer_channel_t * channel,
			    unsigned int mute)
{
	unsigned long flags;
	unsigned char regl, regr, mask;
	cs4231_t *codec;

	regl = (unsigned char) (channel->hw.private_value >> 24);
	regr = (unsigned char) (channel->hw.private_value >> 16);
	mask = (unsigned char) (channel->hw.private_value);
	if (channel->hw.private_value & 0x1000)
		mute ^= SND_MIX_MUTE;
#if 0
	snd_printk("mute: regl = 0x%x(0x%x), regr = 0x%x(0x%x), mask = 0x%x, mute = 0x%x\n", regl, snd_cs4231_in(mixer->private_data, regl), regr, snd_cs4231_in(mixer->private_data, regr), mask, mute);
#endif
	codec = (cs4231_t *) mixer->private_data;
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_outm(codec, regl, ~mask, (mute & SND_MIX_MUTE_LEFT) ? mask : 0);
	if (channel->hw.caps & SND_MIXER_CINFO_CAP_STEREO)
		snd_cs4231_outm(codec, regr, ~mask, (mute & SND_MIX_MUTE_RIGHT) ? mask : 0);
	snd_spin_unlock(codec, reg, &flags);
}

static void snd_cs4231_volume_level(snd_kmixer_t * mixer,
				    snd_kmixer_channel_t * channel,
				    int left, int right)
{
	unsigned long flags;
	unsigned char regl, regr, shift, mask;
	cs4231_t *codec;

	regl = (unsigned char) (channel->hw.private_value >> 24);
	regr = (unsigned char) (channel->hw.private_value >> 16);
	shift = (channel->hw.private_value >> 8) & 0x0f;
	mask = ~(unsigned char) (channel->hw.max << shift);
#if 0
	snd_printk("volume: regl = 0x%x(0x%x), regr = 0x%x(0x%x), mask = 0x%x, shift = %i, left = %i, right = %i\n", regl, snd_cs4231_in(mixer->private_data, regl), regr, snd_cs4231_in(mixer->private_data, regr), mask, shift, left, right);
#endif
	if (!(channel->hw.private_value & 0x2000)) {
		left = channel->hw.max - left;
		right = channel->hw.max - right;
	}
	codec = (cs4231_t *) mixer->private_data;
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_outm(codec, regl, mask, left << shift);
	if (channel->hw.caps & SND_MIXER_CINFO_CAP_STEREO)
		snd_cs4231_outm(codec, regr, mask, right << shift);
	snd_spin_unlock(codec, reg, &flags);
}

#define CS4231_MIXS \
	  (sizeof(snd_cs4231_mixs)/sizeof(struct snd_stru_mixer_channel_hw))
#define CS4231_PRIVATE( left, right, shift, mute ) \
	  ((left << 24)|(right << 16)|(shift<<8)|mute)

static struct snd_stru_mixer_channel_hw snd_cs4231_mixs[] =
{
	{
		SND_MIXER_PRI_GAIN,	/* priority */
		SND_MIXER_PRI_PARENT,	/* parent priority */
		SND_MIXER_ID_GAIN,	/* device name */
		SND_MIXER_OSS_IMIX,	/* OSS device # */
		SND_MIXER_CINFO_CAP_STEREO,
		0, 15,			/* min, max value */
		0, 2250, 150,		/* min, max, step - dB */
		CS4231_PRIVATE(CS4231_LEFT_INPUT, CS4231_RIGHT_INPUT, 0, 0x00) | 0x2000,
		NULL,			/* compute dB -> linear */
		NULL,			/* compute linear -> dB */
		NULL,			/* record source */
		NULL,			/* set mute */
		snd_cs4231_volume_level,/* set volume level */
		NULL,			/* set route */
	},
	{
		SND_MIXER_PRI_AUXA,	/* priority */
		SND_MIXER_PRI_PARENT,	/* parent priority */
		SND_MIXER_ID_AUXA,	/* device name */
		SND_MIXER_OSS_LINE1,	/* OSS device # */
		SND_MIXER_CINFO_CAP_STEREO | SND_MIXER_CINFO_CAP_JOINRECORD | SND_MIXER_CINFO_CAP_INPUT,
		0, 31,			/* min, max value */
		-3450, 1200, 150,	/* min, max, step - dB */
		CS4231_PRIVATE(CS4231_AUX1_LEFT_INPUT, CS4231_AUX1_RIGHT_INPUT, 0, 0x80),
		NULL,			/* compute dB -> linear */
		NULL,			/* compute linear -> dB */
		snd_cs4231_record_source, /* record source */
		snd_cs4231_mute,	/* set mute */
		snd_cs4231_volume_level,/* set volume level */
		NULL,			/* set route */
	},
	{
		SND_MIXER_PRI_AUXB,	/* priority */
		SND_MIXER_PRI_PARENT,	/* parent priority */
		SND_MIXER_ID_AUXB,	/* device name */
		SND_MIXER_OSS_LINE2,	/* OSS device # */
		SND_MIXER_CINFO_CAP_STEREO | SND_MIXER_CINFO_CAP_JOINRECORD | SND_MIXER_CINFO_CAP_INPUT,
		0, 31,			/* min, max value */
		-3450, 1200, 150,	/* min, max, step - dB */
		CS4231_PRIVATE(CS4231_AUX2_LEFT_INPUT, CS4231_AUX2_RIGHT_INPUT, 0, 0x80),
		NULL,			/* compute dB -> linear */
		NULL,			/* compute linear -> dB */
		snd_cs4231_record_source, /* record source */
		snd_cs4231_mute,	/* set mute */
		snd_cs4231_volume_level,/* set volume level */
		NULL,			/* set route */
	},
	{
		SND_MIXER_PRI_PCM,	/* priority */
		SND_MIXER_PRI_PARENT,	/* parent priority */
		SND_MIXER_ID_PCM,	/* device name */
		SND_MIXER_OSS_PCM,	/* OSS device # */
		SND_MIXER_CINFO_CAP_STEREO,
		0, 63,			/* min, max value */
		-9450, 0, 150,		/* min, max, step - dB */
		CS4231_PRIVATE(CS4231_LEFT_OUTPUT, CS4231_RIGHT_OUTPUT, 0, 0x80),
		NULL,			/* compute dB -> linear */
		NULL,			/* compute linear -> dB */
		NULL,			/* record source */
		snd_cs4231_mute,	/* set mute */
		snd_cs4231_volume_level,/* set volume level */
		NULL,			/* set route */
	},
	{
		SND_MIXER_PRI_LOOPBACK,	/* priority */
		SND_MIXER_PRI_PARENT,	/* parent priority */
		SND_MIXER_ID_LOOPBACK,	/* device name */
		SND_MIXER_OSS_MONITOR,	/* OSS device # */
		SND_MIXER_CINFO_CAP_STEREO | SND_MIXER_CINFO_CAP_DIGITAL,
		0, 63,			/* min, max value */
		-9450, 0, 150,		/* min, max, step - dB */
		CS4231_PRIVATE(CS4231_LOOPBACK, 0, 0x12, 0x01),
		NULL,			/* compute dB -> linear */
		NULL,			/* compute linear -> dB */
		NULL,			/* record source */
		snd_cs4231_mute,	/* set mute */
		snd_cs4231_volume_level,/* set volume level */
		NULL,			/* set route */
	},
	{
		SND_MIXER_PRI_LINE,	/* priority */
		SND_MIXER_PRI_PARENT,	/* parent priority */
		SND_MIXER_ID_LINE,	/* device name */
		SND_MIXER_OSS_LINE,	/* OSS device # */
		SND_MIXER_CINFO_CAP_STEREO | SND_MIXER_CINFO_CAP_JOINRECORD | SND_MIXER_CINFO_CAP_INPUT,
		0, 31,			/* min, max value */
		-3450, 1200, 150,	/* min, max, step - dB */
		CS4231_PRIVATE(CS4231_LEFT_LINE_IN, CS4231_RIGHT_LINE_IN, 0, 0x80),
		NULL,			/* compute dB -> linear */
		NULL,			/* compute linear -> dB */
		snd_cs4231_record_source, /* record source */
		snd_cs4231_mute,	/* set mute */
		snd_cs4231_volume_level,/* set volume level */
		NULL,			/* set route */
	},
	{
		SND_MIXER_PRI_MONO,	/* priority */
		SND_MIXER_PRI_PARENT,	/* parent priority */
		SND_MIXER_ID_MONO,	/* device name */
		SND_MIXER_OSS_LINE3,	/* OSS device # */
		SND_MIXER_CINFO_CAP_INPUT,
		0, 15,			/* min, max value */
		-4500, 0, 300,		/* min, max, step - dB */
		CS4231_PRIVATE(CS4231_MONO_CTRL, 0, 0, 0x80),
		NULL,			/* compute dB -> linear */
		NULL,			/* compute linear -> dB */
		snd_cs4231_record_source, /* record source */
		snd_cs4231_mute,	/* set mute */
		snd_cs4231_volume_level,/* set volume level */
		NULL,			/* set route */
	},
};

static int snd_cs4231_mic_gain_get(snd_kmixer_t * mixer,
				   snd_kmixer_switch_t * kswitch,
				   snd_mixer_switch_t * uswitch)
{
	unsigned long flags;
	cs4231_t *codec = (cs4231_t *) mixer->private_data;

	uswitch->type = SND_MIXER_SW_TYPE_BOOLEAN;
	snd_spin_lock(codec, reg, &flags);
	uswitch->value.enable = snd_cs4231_in(codec, CS4231_LEFT_INPUT) & 0x20 ? 1 : 0;
	snd_spin_unlock(codec, reg, &flags);
	return 0;
}

static int snd_cs4231_mic_gain_set(snd_kmixer_t * mixer,
				   snd_kmixer_switch_t * kswitch,
				   snd_mixer_switch_t * uswitch)
{
	unsigned long flags;
	cs4231_t *codec = (cs4231_t *) mixer->private_data;

	if (uswitch->type != SND_MIXER_SW_TYPE_BOOLEAN)
		return -EINVAL;
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_outm(codec, CS4231_LEFT_INPUT, ~0x20, uswitch->value.enable ? 0x20 : 0);
	snd_cs4231_outm(codec, CS4231_RIGHT_INPUT, ~0x20, uswitch->value.enable ? 0x20 : 0);
	snd_spin_unlock(codec, reg, &flags);
	return 0;
}

static snd_kmixer_switch_t snd_cs4231_mic_gain =
{
	SND_MIXER_SW_MIC_GAIN,
	snd_cs4231_mic_gain_get,
	snd_cs4231_mic_gain_set,
	0,
	NULL
};

static int snd_cs4231_mono_output_get(snd_kmixer_t * mixer,
				      snd_kmixer_switch_t * kswitch,
				      snd_mixer_switch_t * uswitch)
{
	unsigned long flags;
	cs4231_t *codec = (cs4231_t *) mixer->private_data;

	uswitch->type = SND_MIXER_SW_TYPE_BOOLEAN;
	snd_spin_lock(codec, reg, &flags);
	uswitch->value.enable = snd_cs4231_in(codec, CS4231_MONO_CTRL) & 0x40 ? 1 : 0;
	snd_spin_unlock(codec, reg, &flags);
	return 0;
}

static int snd_cs4231_mono_output_set(snd_kmixer_t * mixer,
				      snd_kmixer_switch_t * kswitch,
				      snd_mixer_switch_t * uswitch)
{
	unsigned long flags;
	cs4231_t *codec = (cs4231_t *) mixer->private_data;

	if (uswitch->type != SND_MIXER_SW_TYPE_BOOLEAN)
		return -EINVAL;
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_outm(codec, CS4231_MONO_CTRL, ~0x40, uswitch->value.enable ? 0x40 : 0);
	snd_spin_unlock(codec, reg, &flags);
	return 0;
}

static snd_kmixer_switch_t snd_cs4231_mono_output =
{
	"Mono Output",
	snd_cs4231_mono_output_get,
	snd_cs4231_mono_output_set,
	0,
	NULL
};

snd_kmixer_t *snd_cs4231_new_mixer(snd_pcm_t * pcm)
{
	int idx;
	snd_pcm1_t *pcm1;
	cs4231_t *codec;
	snd_kmixer_t *mixer;
	snd_kmixer_channel_t *channel;

	if (!pcm || !pcm->card)
		return NULL;
	pcm1 = (snd_pcm1_t *) pcm->private_data;
	codec = (cs4231_t *) pcm1->private_data;
	if (!codec)
		return NULL;
	mixer = snd_mixer_new(pcm->card, pcm->id);
	if (!mixer)
		return NULL;
	strcpy(mixer->name, pcm->name);
	for (idx = 0; idx < CS4231_MIXS; idx++) {
		channel = snd_mixer_new_channel(mixer, &snd_cs4231_mixs[idx]);
		if (!channel) {
			snd_mixer_free(mixer);
			return NULL;
		}
	}
	mixer->hw.caps = SND_MIXER_INFO_CAP_EXCL_RECORD;
	mixer->private_data = codec;
	codec->mixer = mixer;
	if (!(codec->hardware & CS4231_HW_CS4236_MASK) &&
	    codec->hardware != CS4231_HW_INTERWAVE) {
		snd_mixer_new_switch(mixer, &snd_cs4231_mic_gain);
		snd_mixer_new_switch(mixer, &snd_cs4231_mono_output);
	}
	return mixer;
}

/*
 *  INIT part
 */

#ifndef LINUX_2_1
extern struct symbol_table snd_symbol_table_cs4231_export;
#endif

int init_module(void)
{
#ifndef LINUX_2_1
	if (register_symtab(&snd_symbol_table_cs4231_export) < 0)
		return -ENOMEM;
#endif
	return 0;
}

void cleanup_module(void)
{
}
