/*
 *  Routines for control of TEA6330T circuit over i2c bus
 *  Sound fader control circuit for car radios by Philips Semiconductors
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "tea6330t.h"

#define TEA6330T_ADDR			0x80	/* fixed address */

#define TEA6330T_SADDR_VOLUME_LEFT	0x00	/* volume left */
#define TEA6330T_SADDR_VOLUME_RIGHT	0x01	/* volume right */
#define TEA6330T_SADDR_BASS		0x02	/* bass control */
#define TEA6330T_SADDR_TREBLE		0x03	/* treble control */
#define TEA6330T_SADDR_FADER		0x04	/* fader control */
#define   TEA6330T_MFN			0x20	/* mute control for selected channels */
#define   TEA6330T_FCH			0x10	/* select fader channels - front or rear */
#define TEA6330T_SADDR_AUDIO_SWITCH	0x05	/* audio switch */
#define   TEA6330T_GMU			0x80	/* mute control, general mute */
#define   TEA6330T_EQN			0x40	/* equalizer switchover (0=equalizer-on) */

int snd_tea6330t_detect(struct snd_i2c_bus *bus, int equalizer)
{
	equalizer = equalizer ? 0 : TEA6330T_EQN;
	/* mute & turn on/off equalizer */
	if (snd_i2c_write(bus, TEA6330T_ADDR, TEA6330T_SADDR_AUDIO_SWITCH, TEA6330T_GMU | equalizer, 1) < 0)
		return -ENODEV;
	/* fader off */
	if (snd_i2c_write(bus, TEA6330T_ADDR, TEA6330T_SADDR_FADER, 0x3f, 1) < 0)
		return -ENODEV;
	return 0;
}

static void snd_tea6330t_set_master(struct snd_i2c_bus *bus,
				    unsigned short value)
{
	unsigned short l, r;

	l = (unsigned char) value;
	if (l & 0x80)
		l = 0;
	r = (unsigned char) (value >> 8);
	if (r & 0x80)
		r = 0;
	snd_i2c_write(bus, TEA6330T_ADDR, TEA6330T_SADDR_VOLUME_LEFT, l, 1);
	snd_i2c_write(bus, TEA6330T_ADDR, TEA6330T_SADDR_VOLUME_RIGHT, r, 1);
}

static void snd_tea6330t_set(struct snd_i2c_bus *bus,
			     unsigned char addr, unsigned char value)
{
#if 0
	printk("set - 0x%x/0x%x\n", addr, value);
#endif
	snd_i2c_write(bus, TEA6330T_ADDR, addr, value, 1);
}

static void snd_tea6330t_set_mute(snd_kmixer_t * mixer,
				  snd_kmixer_channel_t * channel,
				  unsigned int mute)
{
	struct snd_i2c_bus *bus;
	unsigned char regl, regr;
	unsigned int value;
	unsigned long flags;

	bus = (struct snd_i2c_bus *) channel->private_data;
	snd_spin_lock(bus, lock, &flags);
	regl = (unsigned char) channel->hw.private_value;
	regr = (unsigned char) (channel->hw.private_value >> 8);
	value = channel->private_value;
#if 0
	printk("mute - name = '%s' (0x%x), regl = 0x%x, regr = 0x%x, value = 0x%x, mute = 0x%x\n", channel->hw.name, channel->hw.private_value, regl, regr, value, mute);
#endif
	switch (regl) {
	case TEA6330T_SADDR_VOLUME_LEFT:
		if (mute & SND_MIX_MUTE_LEFT)
			value |= 0x80;
		else
			value &= ~0x80;
		if (mute & SND_MIX_MUTE_RIGHT)
			value |= 0x8000;
		else
			value &= ~0x8000;
		snd_tea6330t_set_master(bus, value);
		break;
	case TEA6330T_SADDR_FADER:
		if (mute & SND_MIX_MUTE)
			value |= TEA6330T_MFN;
		else
			value &= ~TEA6330T_MFN;
		snd_tea6330t_set(bus, TEA6330T_SADDR_FADER, value);
		break;
	}
	channel->private_value = value;
	snd_spin_unlock(bus, lock, &flags);
}

static void snd_tea6330t_volume_level(snd_kmixer_t * mixer,
				      snd_kmixer_channel_t * channel,
				      int left, int right)
{
	struct snd_i2c_bus *bus;
	unsigned char regl, regr;
	unsigned int value;
	unsigned long flags;

	bus = (struct snd_i2c_bus *) channel->private_data;
	snd_spin_lock(bus, lock, &flags);
	regl = (unsigned char) channel->hw.private_value;
	regr = (unsigned char) (channel->hw.private_value >> 8);
	value = channel->private_value;
#if 0
	printk("volume level - name = '%s' (0x%x), regl = 0x%x, regr = 0x%x, value = 0x%x, left = %i, right = %i\n", channel->hw.name, channel->hw.private_value, regl, regr, value, left, right);
#endif
	switch (regl) {
	case TEA6330T_SADDR_VOLUME_LEFT:
		value &= 0x8080;
		value |= left + 0x14;
		value |= (right + 0x14) << 8;
		snd_tea6330t_set_master(bus, value);
		break;
	case TEA6330T_SADDR_BASS:
		value &= ~0x0f;
		value |= left + (bus->private_value & 1) ? 7 : 3;
		snd_tea6330t_set(bus, TEA6330T_SADDR_BASS, value);
		break;
	case TEA6330T_SADDR_TREBLE:
		value &= ~0x0f;
		value |= left + 3;
		snd_tea6330t_set(bus, TEA6330T_SADDR_TREBLE, value);
		break;
	case TEA6330T_SADDR_FADER:
		value &= ~0x0f;
		value |= left;
		snd_tea6330t_set(bus, TEA6330T_SADDR_FADER, value);
		break;
	}
	channel->private_value = value;
	snd_spin_unlock(bus, lock, &flags);
}

static void snd_tea6330t_copy_mapping(snd_kmixer_t * mixer,
				      snd_kmixer_channel_t * channel,
				      struct snd_tea6330t_mapping *mapping,
				      int idx)
{
	if (!mapping)
		return;
	channel->hw.priority = mapping[idx].priority;
	channel->hw.parent_priority = mapping[idx].parent_priority;
	strcpy(channel->hw.name, mapping[idx].name);
	channel->hw.ossdev = mapping[idx].ossdev;
	snd_mixer_reorder_channel(mixer, channel);
}

int snd_tea6330t_update_mixer(snd_kmixer_t * mixer,
			      struct snd_i2c_bus *bus,
			      struct snd_tea6330t_mapping *mapping,
			      int equalizer, int fader)
{
	static struct snd_stru_mixer_channel_hw master_mix =
	{
		SND_MIXER_PRI_MASTER,	/* priority */
		SND_MIXER_PRI_PARENT,	/* parent priority */
		SND_MIXER_ID_MASTER,	/* device name */
		SND_MIXER_OSS_VOLUME,	/* OSS device # */
		SND_MIXER_CINFO_CAP_STEREO,
		0, 43,			/* min, max value */
		-6600, 2000, 200,	/* min, max, step - dB */
		TEA6330T_SADDR_VOLUME_LEFT | (TEA6330T_SADDR_VOLUME_RIGHT << 8),	/* private value */
		NULL,			/* compute dB -> linear */
		NULL,			/* compute linear -> dB */
		NULL,			/* record source */
		snd_tea6330t_set_mute,	/* set mute */
		snd_tea6330t_volume_level, /* set volume level */
		NULL,			/* set route */
	};
	static struct snd_stru_mixer_channel_hw bass_mix =
	{
		SND_MIXER_PRI_BASS,	/* priority */
		SND_MIXER_PRI_MASTER,	/* parent priority */
		SND_MIXER_ID_BASS,	/* device name */
		SND_MIXER_OSS_BASS,	/* OSS device # */
		0,			/* capabilities */
		0, 9,			/* min, max value */
		-1200, 1500, 300,	/* min, max, step - dB */
		TEA6330T_SADDR_BASS,	/* private value */
		NULL,			/* compute dB -> linear */
		NULL,			/* compute linear -> dB */
		NULL,			/* record source */
		NULL,			/* set mute */
		snd_tea6330t_volume_level, /* set volume level */
		NULL,			/* set route */
	};
	static struct snd_stru_mixer_channel_hw treble_mix =
	{
		SND_MIXER_PRI_TREBLE,	/* priority */
		SND_MIXER_PRI_MASTER,	/* parent priority */
		SND_MIXER_ID_TREBLE,	/* device name */
		SND_MIXER_OSS_TREBLE,	/* OSS device # */
		0,			/* capabilities */
		0, 8,			/* min, max value */
		-1200, 1200, 300,	/* min, max, step - dB */
		TEA6330T_SADDR_TREBLE,	/* private value */
		NULL,			/* compute dB -> linear */
		NULL,			/* compute linear -> dB */
		NULL,			/* record source */
		NULL,			/* set mute */
		snd_tea6330t_volume_level, /* set volume level */
		NULL,			/* set route */
	};
	static struct snd_stru_mixer_channel_hw fader_mix =
	{
		SND_MIXER_PRI_FADER,	/* priority */
		SND_MIXER_PRI_FADER,	/* parent priority */
		SND_MIXER_ID_FADER,	/* device name */
		SND_MIXER_OSS_UNKNOWN,	/* OSS device # */
		0,			/* capabilities */
		0, 15,			/* min, max value */
		-3000, 0, 200,		/* min, max, step - dB */
		TEA6330T_SADDR_FADER,	/* private value */
		NULL,			/* compute dB -> linear */
		NULL,			/* compute linear -> dB */
		NULL,			/* record source */
		NULL,			/* set mute */
		snd_tea6330t_volume_level, /* set volume level */
		NULL,			/* set route */
	};
	snd_kmixer_channel_t *channel;

	bus->private_value = 0;
	if (equalizer)
		bus->private_value |= 1;
	if (fader)
		bus->private_value |= 2;

	channel = snd_mixer_new_channel(mixer, &master_mix);
	if (!channel)
		return -ENOMEM;
	snd_tea6330t_copy_mapping(mixer, channel, mapping, 0);
	channel->private_data = bus;

	channel = snd_mixer_new_channel(mixer, &bass_mix);
	if (!channel)
		return -ENOMEM;
	snd_tea6330t_copy_mapping(mixer, channel, mapping, 1);
	channel->private_data = bus;
	if (equalizer) {
		channel->hw.min = 0;
		channel->hw.max = 5;
		channel->hw.min_dB = 0;
	}
	if (!equalizer) {
		channel = snd_mixer_new_channel(mixer, &treble_mix);
		if (!channel)
			return -ENOMEM;
		snd_tea6330t_copy_mapping(mixer, channel, mapping, 2);
		channel->private_data = bus;
	}
	if (fader) {
		channel = snd_mixer_new_channel(mixer, &fader_mix);
		if (!channel)
			return -ENOMEM;
		snd_tea6330t_copy_mapping(mixer, channel, mapping, 3);
		channel->private_data = bus;
	}
	equalizer = equalizer ? 0 : TEA6330T_EQN;
	/* mute off & turn on/off equalizer */
	if (snd_i2c_write(bus, TEA6330T_ADDR, TEA6330T_SADDR_AUDIO_SWITCH, equalizer, 1) < 0)
		return -ENODEV;

	strcat(mixer->name, " + TEA6330T");

	return 0;
}

/*
 *  INIT part
 */

#ifndef LINUX_2_1
extern struct symbol_table snd_symbol_table_tea6330t_export;
#endif

int init_module(void)
{
#ifndef LINUX_2_1
	if (register_symtab(&snd_symbol_table_tea6330t_export) < 0)
		return -ENOMEM;
#endif
	return 0;
}

void cleanup_module(void)
{
}
