/*
 *   ALSA sequencer Timer
 *   Copyright (c) 1998 by Frank van de Pol <frank@vande-pol.demon.nl>
 *                         Jaroslav Kysela <perex@jcu.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "driver.h"
#include "timer.h"
#include "seq_timer.h"
#include "seq_queue.h"
#include "seq_info.h"

/* static data */
snd_mutex_define_static(global_timer);
snd_spin_define_static(global_timer);
static int global_timer_use = 0;
static snd_timer_t *global_timer = NULL;
/* default timer resolution - 250 Hz - should be configurable !! */
static unsigned int global_timer_resolution = 250;
static unsigned int global_timer_ticks;
static volatile int global_timer_running = 0;
static int global_timer_change = 0;

/* create new timer (constructor) */
timer_t *snd_seq_timer_new(void)
{
	timer_t *tmr;
	
	tmr = snd_calloc(sizeof(timer_t));
	if (tmr == NULL) {
		snd_printd("malloc failed for snd_seq_timer_new() \n");
		return NULL;
	}

	/* setup defaults */
	tmr->ppq = 96;		/* 96 PPQ */
	tmr->tempo = 500000;	/* 120 BPM */	
	tmr->running = 0;

	/* reset time */
	snd_seq_timer_reset(tmr);
	
	return tmr;
}

/* delete timer (destructor) */
void snd_seq_timer_delete(timer_t **tmr)
{
	timer_t *t = *tmr;
	*tmr = NULL;

	if (t == NULL) {
		snd_printd("oops: snd_seq_timer_delete() called with NULL timer\n");
		return;
	}
	t->running = 0;

	/* reset time */
	snd_seq_timer_stop(t);
	snd_seq_timer_reset(t);

	snd_free(t,sizeof(timer_t));
}


void snd_seq_timer_reset(timer_t * tmr)
{
	/* reset time & songposition */
	tmr->cur_tick = 0;
	tmr->cur_time.tv_sec = 0;
	tmr->cur_time.tv_nsec = 0;

	tmr->tempo_tick = 0;
	tmr->tempo_time.tv_sec = 0;
	tmr->tempo_time.tv_nsec = 0;

	tmr->sync_tmp = 0;
}


/* called by timer interrupt routine. the period time since previous invocation is passed */
static void snd_seq_timer_interrupt(snd_timer_t *timer, void *data)
{
	volatile int timers_running = 0;	/* flag if we did increment times */
	unsigned int resolution;
	snd_seq_real_time_t period;
	int c;
	queue_t *q;
	timer_t *tmr;

	resolution = snd_timer_resolution(timer) * global_timer_ticks;
	period.tv_sec = 0;
	period.tv_nsec = resolution % 1000000000;

	/* process all timers... */
	for (c = 0; c < SND_SEQ_MAX_QUEUES; c++) {
		q = queueptr(c);
		if (q == NULL)
			continue;

		tmr = q->timer;

		if (tmr != NULL) {
			if (tmr->running) {
				double delta_time;

				/* update timer */
				snd_seq_inc_real_time(&tmr->cur_time, &period);
				timers_running = 1;

				/* calculate current tick */
				/* FIXME: use of floating point stuff, illegal in kernel (??) */
				if (tmr->tempo > 0) {
					delta_time = tmr->cur_time.tv_sec - tmr->tempo_time.tv_sec + (1.0E-9 * (double) (tmr->cur_time.tv_nsec - tmr->tempo_time.tv_nsec));
					tmr->cur_tick = tmr->tempo_tick + (tmr->ppq * delta_time * 1.0E6 / tmr->tempo);
				}
			}
		}
	}

	if (timers_running) {

#if 0
		/* sync timer has expired *//* FIXME: hack to send midi sync info */
		if ((tmr->cur_tick > tmr->sync_tmp) && (snd_seq_unused_cells() > 10)) {
			snd_seq_event_t ev;

			ev.type = SND_SEQ_EVENT_CLOCK;
			ev.flags = SND_SEQ_TIME_STAMP_TICK | SND_SEQ_TIME_MODE_ABS;
			ev.time.tick = 0;
			ev.source.queue = 0;
			ev.source.client = 0;
			ev.source.port = 0;
			ev.dest.queue = 1;
			ev.dest.client = 255;
			ev.dest.port = 255;
			if (snd_seq_kernel_client_dispatch(0, &ev)) {
				tmr->sync_tmp += tmr->ppq / 24;
			}
		}
#endif
		/* check queues and dispatch events */
		snd_seq_check_queues(1);
	}
}


void snd_seq_timer_set_tempo(timer_t * tmr, int tempo)
{
	if (tmr) {
		/* store location of tempo change */
		tmr->tempo_tick = tmr->cur_tick;
		tmr->tempo_time.tv_sec = tmr->cur_time.tv_sec;
		tmr->tempo_time.tv_nsec = tmr->cur_time.tv_nsec;

		tmr->tempo = tempo;
	}
}


void snd_seq_timer_set_ppq(timer_t * tmr, int ppq)
{
	if (tmr) {
		if (tmr->running & (ppq != tmr->ppq)) {
			/* refuse to change ppq on running timers, because it wil upset the song position (ticks) */
			snd_printd("seq: cannot change ppq of a running timer\n");
		} else {
			tmr->ppq = ppq;
		}
	}
}


static int snd_seq_timer_open1(void)
{
	unsigned long flags;
	snd_timer_t *t;
	unsigned int tmp, tmp1;

	t = snd_timer_open_always("sequencer", global_timer_resolution);
	if (!t) {
		snd_printk("sequencer fatal error: cannot create timer\n");
		return -ENODEV;
	}
	t->callback = snd_seq_timer_interrupt;
	t->callback_data = NULL;	/* not used */
	tmp = 1000000000UL / global_timer_resolution;
	tmp1 = snd_timer_resolution(t);
	if (tmp1 >= tmp) {
		global_timer_ticks = 1;
	} else {
		global_timer_ticks = tmp / tmp1;
	}
	t->flags |= SND_TIMER_FLG_AUTO;
	snd_spin_lock_static(global_timer, &flags);
	global_timer = t;
	snd_spin_unlock_static(global_timer, &flags);
	snd_timer_change(&global_timer_change);
	return 0;
}

static void snd_seq_timer_close1(void)
{
	unsigned long flags;
	snd_timer_t *t = global_timer;
	
	snd_spin_lock_static(global_timer, &flags);
	global_timer = NULL;
	snd_spin_unlock_static(global_timer, &flags);
	snd_timer_stop(t);
	snd_timer_close_always(t);
}

int snd_seq_timer_open(void)
{
	snd_mutex_down_static(global_timer);
	if (global_timer_use++ == 0) {
		if (snd_seq_timer_open1() < 0)
			global_timer_use--;
	}
	snd_mutex_up_static(global_timer);
	return 0;
}

int snd_seq_timer_close(void)
{
	snd_mutex_down_static(global_timer);
	if (--global_timer_use == 0)
		snd_seq_timer_close1();
	snd_mutex_up_static(global_timer);
	return 0;
}

int snd_seq_timer_reopen(void)
{
	unsigned long flags;

	if (snd_timer_change(&global_timer_change)) {
		snd_seq_timer_close1();
		snd_seq_timer_open1();
		snd_spin_lock_static(global_timer, &flags);
		if (global_timer_running && global_timer)
			snd_timer_start(global_timer, global_timer_ticks);
		snd_spin_unlock_static(global_timer, &flags);
	}
	return 0;
}


void snd_seq_timer_stop(timer_t * tmr)
{
	unsigned long flags;

	if (!tmr->running)
		return;
	snd_spin_lock_static(global_timer, &flags);
	tmr->running = 0;
	if (--global_timer_running == 0 && global_timer)
		snd_timer_stop(global_timer);
	snd_spin_unlock_static(global_timer, &flags);
}

void snd_seq_timer_start(timer_t * tmr)
{
	unsigned long flags;

	/* reset time & songposition */
	if (tmr->running)
		snd_seq_timer_stop(tmr);
	snd_seq_timer_reset(tmr);
	snd_spin_lock_static(global_timer, &flags);
	if (++global_timer_running == 1 && global_timer)
		snd_timer_start(global_timer, global_timer_ticks);
	tmr->running = 1;
	snd_spin_unlock_static(global_timer, &flags);
}

void snd_seq_timer_continue(timer_t * tmr)
{
	unsigned long flags;

	if (tmr->running)
		return;
	snd_spin_lock_static(global_timer, &flags);
	if (++global_timer_running == 1 && global_timer)
		snd_timer_start(global_timer, global_timer_ticks);
	tmr->running = 1;	
	snd_spin_unlock_static(global_timer, &flags);
}

/* exported to seq_info.c */
void snd_seq_info_timer_read(snd_info_buffer_t * buffer, void *private_data)
{
	unsigned int resolution;

	snd_mutex_down_static(global_timer);
	snd_iprintf(buffer, "Timer       : %s\n", global_timer ? global_timer->name : "None");
	if (global_timer) {
		resolution = snd_timer_resolution(global_timer) * global_timer_ticks;
		snd_iprintf(buffer, "Period time : %d.%09d\n", resolution / 1000000000, resolution % 1000000000);
 	}
	snd_mutex_up_static(global_timer);
}
