/*
 *  ALSA sequencer Client Manager
 *  Copyright (c) 1998 by Frank van de Pol <frank@vande-pol.demon.nl>
 *                        Jaroslav Kysela <perex@jcu.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "driver.h"
#include "minors.h"

#include "seq_kernel.h"
#include "seq_clientmgr.h"
#include "seq_memory.h"
#include "seq_queue.h"
#include "seq_timer.h"
#include "seq_info.h"
#include "seq_system.h"

#undef TRACK_SUBSCRIBE

/* Client Manager

 * this module handles the connections of userland and kernel clients
 * 
 */

#define SND_SEQ_LFLG_INPUT	0x0001
#define SND_SEQ_LFLG_OUTPUT	0x0002
#define SND_SEQ_LFLG_OPEN	(SND_SEQ_LFLG_INPUT|SND_SEQ_LFLG_OUTPUT)

snd_spin_define_static(clients);
snd_mutex_define_static(register);

static char clienttablock[SND_SEQ_MAX_CLIENTS];
static client_t *clienttab[SND_SEQ_MAX_CLIENTS];
static usage_t client_usage = {0, 0};
static snd_info_entry_t *seq_dev = NULL;

static inline unsigned short snd_seq_file_flags(struct file *file)
{
#ifdef LINUX_2_1
        switch (file->f_mode & (FMODE_READ | FMODE_WRITE)) {
        case FMODE_WRITE:
                return SND_SEQ_LFLG_OUTPUT;
        case FMODE_READ:
                return SND_SEQ_LFLG_INPUT;
        default:
                return SND_SEQ_LFLG_OPEN;
        }
#else
        switch (file->f_flags & O_ACCMODE) {
        case O_WRONLY:
                return SND_SEQ_LFLG_OUTPUT;
        case O_RDONLY:
                return SND_SEQ_LFLG_INPUT;
        default:
                return SND_SEQ_LFLG_OPEN;
        }
#endif
}

/* return pointer to client structure for specified id */
static client_t *clientptr(int clientid)
{
	if (clientid < 0 || clientid >= SND_SEQ_MAX_CLIENTS) {
		snd_printd("Seq: oops. Trying to get pointer to client %d\n", clientid);
		return NULL;
	}
	return clienttab[clientid];
}

client_t *snd_seq_client_use_ptr(int clientid)
{
	unsigned long flags;
	client_t *client;

	if (clientid < 0 || clientid >= SND_SEQ_MAX_CLIENTS) {
		snd_printd("Seq: oops. Trying to get pointer to client %d\n", clientid);
		return NULL;
	}
	snd_spin_lock_static(clients, &flags);
	client = clienttab[clientid];
	if (client) {
		snd_spin_lock(client, use, &flags);
		client->use++;
		snd_spin_unlock(client, use, &flags);
	}
	snd_spin_unlock_static(clients, &flags);
	return client;
}

void snd_seq_client_unlock(client_t *client)
{
	unsigned long flags;

	if (!client)
		return;
	snd_spin_lock(client, use, &flags);
	if (!(--client->use) && (snd_getlock(client, use) & SND_WK_SLEEP)) {
	    	snd_getlock(client, use) &= ~SND_WK_SLEEP;
	    	snd_wakeup(client, use);
	}
	snd_spin_unlock(client, use, &flags);
}

static void usage_alloc(usage_t * res, int num)
{
	res->cur += num;
	if (res->cur > res->peak)
		res->peak = res->cur;
}

static void usage_free(usage_t * res, int num)
{
	res->cur -= num;
}

static int snd_seq_output_watermark_queue(client_t *client, queue_t *q)
{
	int unused;

	if (!test_bit(client->number, &q->clients_bitmap))
		return 0;	/* fail */
	unused = snd_seq_unused_cells(q->pool);
	if (unused < client->low[q->queue])
		return 0;
	if (snd_seq_total_cells(q->pool) - unused > client->high[q->queue])
		return 0;
	return 1;		/* continue */
}

static int snd_seq_output_watermark(client_t *client)
{
	int c, unused;
	queue_t *q;

	for (c = 0; c < SND_SEQ_MAX_QUEUES; c++) {
		q = queueptr(c);
		if (q == NULL)
			continue;
		if (!test_bit(client->number, &q->clients_bitmap))
			continue;
		unused = snd_seq_unused_cells(q->pool);
		if (unused < client->low[c])
			return 0;
		if (snd_seq_total_cells(q->pool) - unused > client->high[c])
			return 0;
	}
	return 1;		/* all used queues are ok */
}

/* initialise data structures */
void client_init_data(void)
{
	/* zap out the client table */
	memset(&clienttablock, 0, sizeof(clienttablock));
	memset(&clienttab, 0, sizeof(clienttab));
}


static client_t *seq_create_client1(int client_index)
{
	unsigned long flags;
	int c;
	client_t *client;

	/* init client data */
	client = snd_calloc(sizeof(client_t));
	if (!client)
		return NULL;
	client->type = NO_CLIENT;
	client->outqueue = snd_seq_fifo_new();
	if (!client->outqueue) {
		snd_free(client, sizeof(client_t));
		return NULL;
	}
	snd_spin_prepare(client, use);
	snd_sleep_prepare(client, use);
	snd_spin_prepare(client, ports);
	snd_sleep_prepare(client, ports);
	for (c = 0; c < SND_SEQ_MAX_QUEUES; c++) {
		client->low[c] = 100;
		client->high[c] = 0x7fffffff;
	}
	/* find free slot in the client table */
	snd_spin_lock_static(clients, &flags);
	if (client_index < 0) {
		for (c = 128; c < SND_SEQ_MAX_CLIENTS; c++) {
			if (clienttab[c] || clienttablock[c])
				continue;
			clienttab[client->number = c] = client;
			snd_spin_unlock_static(clients, &flags);
			return client;
		}
	} else {
		if (!clienttab[client_index] && !clienttablock[client_index]) {
			clienttab[client->number = client_index] = client;
			snd_spin_unlock_static(clients, &flags);
			return client;
		}
	}
	snd_spin_unlock_static(clients, &flags);
	snd_seq_fifo_delete(&client->outqueue);
	snd_free(client, sizeof(client_t));
	return NULL;	/* no free slot found or busy, return failure code */
}


static int seq_free_client1(client_t *client)
{
	unsigned long flags;

	if (!client)
		return -EINVAL;
	snd_spin_lock_static(clients, &flags);
	clienttablock[client->number] = 1;
	clienttab[client->number] = NULL;
	snd_spin_unlock_static(clients, &flags);
	snd_spin_lock(client, use, &flags);
	while (client->use) {
		snd_getlock(client, use) |= SND_WK_SLEEP;
		snd_spin_unlock(client, use, &flags);
		snd_sleep(client, use, HZ);
		snd_spin_lock(client, use, &flags);
		snd_getlock(client, use) &= ~SND_WK_SLEEP;
	}
	snd_spin_unlock(client, use, &flags);
	snd_seq_queue_client_termination(client->number);
	snd_seq_fifo_delete(&client->outqueue);
	snd_seq_delete_ports(client);
	snd_seq_queue_client_leave(client->number);
	clienttablock[client->number] = 0;
	snd_free(client, sizeof(client_t));
	return 0;
}


static void seq_free_client(client_t * client)
{
	if (!client)
		return;

	snd_mutex_down_static(register);
	switch (client->type) {
		case NO_CLIENT:
			snd_printk("Seq: Trying to free unused client %d\n", client->number);
			break;

		case USER_CLIENT:
		case KERNEL_CLIENT:
			seq_free_client1(client);
			usage_free(&client_usage, 1);
			break;

		default:
			snd_printk("Seq: Trying to free client %d with undefined type = %d\n", client->number, client->type);
	}
	snd_mutex_up_static(register);

	snd_seq_system_client_ev_client_exit(client->number);
}



/* -------------------------------------------------------- */

static int snd_seq_open(unsigned short minor, int cardnum, int device, struct file *file)
{
	int c, mode;			/* client id */
	client_t *client;

	snd_mutex_down_static(register);
	client = seq_create_client1(-1);
	if (!client) {
		snd_mutex_up_static(register);
		return -ENOMEM;	/* failure code */
	}
	snd_spin_prepare(&client->data.user, input_sleep);
	usage_alloc(&client_usage, 1);
	client->type = USER_CLIENT;
	snd_mutex_up_static(register);

	mode = snd_seq_file_flags(file);
	if (mode & SND_SEQ_LFLG_INPUT)
		client->accept_input = 1;
	if (mode & SND_SEQ_LFLG_OUTPUT)
		client->accept_output = 1;

	c = client->number;
	(user_client_t *) file->private_data = client;

	/* fill client data */
	client->data.user.file = file;
	sprintf(client->name, "Client-%d", c);

	/* make others aware this new client */
	snd_seq_system_client_ev_client_start(c);

	snd_seq_timer_open();

	MOD_INC_USE_COUNT;

	return 0;
}

static int snd_seq_release(unsigned short minor, int cardnum, int device, struct file *file)
{
	client_t *client = (client_t *) file->private_data;

	snd_seq_timer_close();

	if (client)
		seq_free_client(client);
	MOD_DEC_USE_COUNT;
	return 0;
}


/* handle client read() */
static long snd_seq_read(struct file *file, char *buf, long count)
{
	client_t *client = (client_t *) file->private_data;
	long result = 0;
	unsigned long flags;

	if (!(snd_seq_file_flags(file) & SND_SEQ_LFLG_INPUT))
		return -ENXIO;

	if (verify_area(VERIFY_WRITE, buf, count))
		return -EFAULT;

	/* check client structures are in place */
	if (client == NULL)
		return -EIO;

	if (!client->accept_input)
		return -EIO;

	/* while we have room for at least one event */
	while (count >= sizeof(snd_seq_event_t)) {
		while (snd_seq_fifo_avail(client->outqueue) == 0) {
			/* no data available in outqueue, block */
			if ((file->f_flags & O_NONBLOCK) || result > 0)
				return result;
			snd_spin_lock(&client->data.user, input_sleep, &flags);
			client->outqueue->flags |= SND_WK_SLEEP;
			snd_sleep(client->outqueue, input, 10 * HZ);
			client->outqueue->flags &= ~SND_WK_SLEEP;
			snd_spin_unlock(&client->data.user, input_sleep, &flags);
			if (snd_sleep_abort(client->outqueue, input))
				return result;
		}
		while (snd_seq_fifo_avail(client->outqueue)) {
			/* data available in queue */

			snd_seq_event_cell_t *cell;
			int len;

			/* check if buffer is big enough for the new event */
			cell = snd_seq_fifo_cell_peek(client->outqueue);
			if (cell == NULL) {
				snd_printd("Whoops, snd_seq_read got a NULL from the fifo\n");
				return result;
			}
			switch (cell->event.flags & SND_SEQ_EVENT_LENGTH_MASK) {
				case SND_SEQ_EVENT_LENGTH_VARIABLE:
					/* handle variable length data */
					len = sizeof(snd_seq_event_t) + cell->event.data.ext.len;
					break;
				case SND_SEQ_EVENT_LENGTH_FIXED:
				default:
					/* normal event, no special action needed */
					len = sizeof(snd_seq_event_t);
			}
			if (count < len)
				return result;	/* not enough room for this new event */

			/* get event */
			cell = snd_seq_fifo_cell_out(client->outqueue);
			if (cell == NULL) {
				snd_printd("Whoops, snd_seq_read got a NULL from the fifo\n");
				return result;
			}
			switch (cell->event.flags & SND_SEQ_EVENT_LENGTH_MASK) {
				case SND_SEQ_EVENT_LENGTH_VARIABLE:
					/* handle variable length data */
					len = sizeof(snd_seq_event_t) + cell->event.data.ext.len;
					copy_to_user(buf, &cell->event, sizeof(snd_seq_event_t));
					if (cell->event.data.ext.ptr != NULL) {
						copy_to_user(buf + sizeof(snd_seq_event_t), cell->event.data.ext.ptr, cell->event.data.ext.len);
					}
					break;

				case SND_SEQ_EVENT_LENGTH_FIXED:
				default:
					/* normal event, no special action needed */
					len = sizeof(snd_seq_event_t);
					copy_to_user(buf, &cell->event, sizeof(snd_seq_event_t));
			}
			snd_seq_cell_free(cell);

			result += len;
			count -= len;
			buf += len;
		}
	}

	return result;
}


/* handle write() */
static long snd_seq_write(struct file *file, const char *buf, long count)
{
	client_t *client = (client_t *) file->private_data;
	int rd = 0, len;
	snd_seq_event_cell_t *cell = NULL;
	snd_seq_event_t event;
	queue_t *q;

	if (!(snd_seq_file_flags(file) & SND_SEQ_LFLG_OUTPUT))
		return -ENXIO;

	/* check if the data is accessible */
	if (verify_area(VERIFY_READ, buf, count))
		return -EFAULT;

	/* check client structures are in place */
	if (client == NULL)
		return -EIO;
		
	if (!client->accept_output)
		return -EIO;

	/* only process whole event */
	while (count >= sizeof(snd_seq_event_t)) {

		/* we got one event, copy it into our own data structure and */
		/* feed to the prioQ */
		copy_from_user(&event, buf, len = sizeof(event));
		if (event.dest.queue < SND_SEQ_MAX_QUEUES) {
			q = queueptr(event.dest.queue);
			if (q == NULL)
				goto __skip;
			/* client really uses this queue? */
			if (!test_bit(client->number, &q->clients_bitmap))
				goto __skip;
			/* only start processing if we have enough free cells */
			if (!snd_seq_output_watermark_queue(client, q))
				break;
			cell = snd_seq_cell_alloc(q->pool);
		} else {
			if (event.dest.queue != SND_SEQ_ADDRESS_SUBSCRIBERS)
				goto __skip;
			q = NULL;	/* for sure */
			/* allocate temporary cell */
			cell = snd_seq_cell_alloc(client->outqueue->pool);
			if (cell == NULL)
				break;
		}
		if (cell == NULL)
			goto __skip;

		memcpy(&cell->event, &event, sizeof(event));
		cell->event.source.queue = SND_SEQ_ADDRESS_UNKNOWN;
		cell->event.source.client = client->number;	/* fill in client number */

		switch (event.flags & SND_SEQ_EVENT_LENGTH_MASK) {
		case SND_SEQ_EVENT_LENGTH_VARIABLE:
			if (event.data.ext.len < 0 || len > count) {/* wrong event */
				snd_printd("wrong event, len = %i\n",
					event.data.ext.len);
				cell->event.flags = 0;
				snd_seq_cell_free(cell);
				return rd;
			}
			len += event.data.ext.len;
			cell->event.data.ext.ptr = snd_seq_ext_malloc(event.data.ext.len, 0);
			if (cell->event.data.ext.ptr != NULL) {
				copy_from_user(cell->event.data.ext.ptr, buf + sizeof(event), event.data.ext.len);
			} else {
				snd_printd("failed to allocate %d bytes for variable length event\n", event.data.ext.len);
				cell->event.flags = 0;
				snd_seq_cell_free(cell);
				goto __skip;
			}

			break;

		case SND_SEQ_EVENT_LENGTH_FIXED:
		default:
				/* normal event, no special action needed */
		}

		snd_seq_enqueue_event(cell, 0);

		/* up to next event */
		count -= len;
		buf += len;
		rd += len;
		continue;
		
	      __skip:

		len = sizeof(event);
		switch (event.flags & SND_SEQ_EVENT_LENGTH_MASK) {
		case SND_SEQ_EVENT_LENGTH_VARIABLE:
			if (event.data.ext.len < 0) {
				return rd;
			}
		      	len += cell->event.data.ext.len;
		      	break;

		case SND_SEQ_EVENT_LENGTH_FIXED:
		default:
				/* normal event, no special action needed */
		}

		if (len > count) {
			snd_printd("wrong event (skip), len = %i\n", len);
			break;
		}

		/* up to next event */
		count -= len;
		buf += len;
		rd += len;		
	}
	return rd;
}


#ifdef SND_POLL
static unsigned int snd_seq_poll(struct file *file, poll_table * wait)
{
	client_t *client = (client_t *) file->private_data;
	unsigned int mask = 0;
	unsigned long flags;

	/* check client structures are in place */
	if (client == NULL)
		return -EIO;

	if (snd_seq_file_flags(file) & SND_SEQ_LFLG_INPUT) {
		snd_spin_lock(&client->data.user, input_sleep, &flags);
		client->outqueue->flags |= SND_WK_SLEEP;
		snd_poll_wait(file, client->outqueue, input, wait);
		snd_spin_unlock(&client->data.user, input_sleep, &flags);

		/* check if data is available in the outqueue */
		if (snd_seq_fifo_avail(client->outqueue) >= 1)
			mask |= POLLIN | POLLRDNORM;
	}

	if (snd_seq_output_watermark(client))
		mask |= POLLOUT | POLLWRNORM;

	return mask;
}
#else
static int snd_seq_select(struct file *file, int sel_type, select_table * wait)
{
	client_t *client = (client_t *) file->private_data;
	unsigned long flags;

	switch (sel_type) {
		case SEL_IN:	/* check if data is available in the outqueue */

			if (!(snd_seq_file_flags(file) & SND_SEQ_LFLG_INPUT))
				return 0;
			snd_spin_lock(&client->data.user, input_sleep, &flags);
			if (!(snd_seq_fifo_avail(client->outqueue) >= 1)) {
				client->outqueue->flags |= SND_WK_SLEEP;
				snd_select_wait(client->outqueue, input, wait);
				snd_spin_unlock(&client->data.user, input_sleep, &flags);
				return 0;
			}
			client->outqueue->flags &= ~SND_WK_SLEEP;
			snd_spin_unlock(&client->data.user, input_sleep, &flags);
			return 1;

		case SEL_OUT:

			if (!(snd_seq_file_flags(file) & SND_SEQ_LFLG_OUTPUT))
				return 0;
			if (snd_seq_output_watermark(client))
					return 1;
			return 0;

		case SEL_EX:
			break;
	}

	return 0;
}
#endif


/*-----------------------------------------------------*/


/* dispatch a single event to the client */
static void snd_seq_dispatch_single_event(snd_seq_event_cell_t * cell)
{
	client_t *dest_client;
	int source, dest;
	
	if (!cell)
		return;
	source = cell->event.source.client;	/* source client */
	dest = cell->event.dest.client;		/* dest client */

	dest_client = snd_seq_client_use_ptr(dest);
	if (dest_client == NULL)
		return;

	/* use filters here */
	if (!dest_client->accept_input)
		return;
	if (source == SND_SEQ_ADDRESS_BROADCAST) {
		if (!(dest_client->filter & SND_SEQ_FILTER_BROADCAST))
			return;
	} else if (source > 191) {	/* multicast */
		if (!(dest_client->filter & SND_SEQ_FILTER_MULTICAST))
			return;
		if (test_bit(source - 192, &dest_client->multicast_filter))
			return;
	}
	if (dest_client->filter & SND_SEQ_FILTER_USE_EVENT) {
		if (!test_bit(cell->event.type, &dest_client->event_filter))
			return;
	}

	/* deliver event */
	switch (dest_client->type) {
		case USER_CLIENT:

			/* write in client's outgoing fifo */
			snd_seq_fifo_cell_in(dest_client->outqueue, cell);
			break;

		case KERNEL_CLIENT:

			if (dest_client->data.kernel.input != NULL) {
				int result;

				/* drain queue if old events are present, should be improved!! */
				while (snd_seq_fifo_avail(dest_client->outqueue)) {

					snd_seq_event_cell_t *queued_cell;

					queued_cell = snd_seq_fifo_cell_out(dest_client->outqueue);

					result = dest_client->data.kernel.input(&queued_cell->event, dest_client->data.kernel.private_data);
					if (result >= 0) {
						/* event successfully processed, it can now be removed */
						snd_seq_cell_free(queued_cell);
					} else {
						if (result == -EAGAIN)
							snd_seq_fifo_cell_in(dest_client->outqueue, queued_cell);
						break;
					}
				}

				/* directly pass event to client's routine */
				result = dest_client->data.kernel.input(&cell->event, dest_client->data.kernel.private_data);
				if (result >= 0) {
					/* event was successfully processed, it can mow be removed */
					snd_seq_cell_free(cell);
				} else {
					/* client refused the event (shouldn't happen!), queue it only on client request */
					if (result == -EAGAIN) {
						snd_seq_fifo_cell_in(dest_client->outqueue, cell);
					} else {
						snd_seq_cell_free(cell);
					}
				}
			} else {
				/* no input routine registered, get rid of the event */
				snd_printd("seq: kernel client %d has no input processor, event discarded\n", dest);
				snd_seq_cell_free(cell);
			}
			break;

		default:

			/* client not found, discard this event... */
			snd_seq_cell_free(cell);
			break;
	}
	snd_seq_client_unlock(dest_client);
}


/*
 * dispatch event to the client 
 *
 * at this moment we can only send the event to either all clients 
 * (broadcast) or to a specific client.
 * in a later stadium multicast groups will be added 
 *
 */
void snd_seq_dispatch_event(snd_seq_event_cell_t * cell, int atomic)
{
	client_t *dest_client;
	int dest;

	if (!cell)
		return;				/* something screwed up */

	dest = cell->event.dest.client;		/* dest client */

	/* send to specific client */
	if ((dest >= 0) && (dest <= 191)) {
		snd_seq_dispatch_single_event(cell);
		return;
	}
	/* send to a multicast group */
	if ((dest > 191) && (dest < SND_SEQ_ADDRESS_BROADCAST)) {
		snd_printk("Seq: multicast groups not supported yet (dest = %d)\n", dest);
		/* Idea: multicast groups: 
		   a multicast message is send to a list of destinations:
		   per list item map: client, port, channel
		 */

		/* release the original cell */
		snd_seq_cell_free(cell);
		return;
	}
	/* handle broadcasts */
	if (dest == SND_SEQ_ADDRESS_BROADCAST) {
		/* send the event to all clients */
		for (dest = 0; dest < SND_SEQ_MAX_CLIENTS; dest++) {
			dest_client = snd_seq_client_use_ptr(dest);
			if (dest_client) {
				if (dest_client->type != NO_CLIENT) {
					snd_seq_event_cell_t *new_cell;

					/* send duplicates to all the clients */
					new_cell = snd_seq_cell_dup(dest_client->outqueue->pool, cell, atomic);
					if (new_cell) {
						new_cell->event.dest.client = dest;
						snd_seq_dispatch_single_event(new_cell);
					}
				}
				snd_seq_client_unlock(dest_client);
			}
		}
		/* release the original cell */
		snd_seq_cell_free(cell);
		return;
	}
}


/*-----------------------------------------------------*/


/* SYSTEM_INFO ioctl() */
static int snd_seq_ioctl_system_info(client_t *client, snd_seq_system_info_t * _info)
{
	snd_seq_system_info_t info;

	if (verify_area(VERIFY_WRITE, _info, sizeof(info)))
		return -EFAULT;

	/* fill the info fields */
	info.queues = SND_SEQ_MAX_QUEUES;
	info.clients = SND_SEQ_MAX_CLIENTS;
	info.ports = 256;	/* fixed limit */
	info.channels = 256;	/* fixed limit */

	copy_to_user(_info, &info, sizeof(info));
	return 0;
}


/* CLIENT_INFO ioctl() */
static int snd_seq_ioctl_get_client_info(client_t * client, snd_seq_client_info_t * _client_info)
{
	client_t *cptr;
	snd_seq_client_info_t client_info;

	if (verify_area(VERIFY_READ, _client_info, sizeof(client_info)) ||
	    verify_area(VERIFY_WRITE, _client_info, sizeof(client_info)))
		return -EFAULT;
	copy_from_user(&client_info, _client_info, sizeof(client_info));

	/* requested client number */
	cptr = snd_seq_client_use_ptr(client_info.client);
	if (!cptr)
		return -ENOENT;		/* don't change !!! */
	memset(&client_info, 0, sizeof(client_info));
	client_info.client = cptr->number;

	/* fill the info fields */
	client_info.type = cptr->type;
	strcpy(client_info.name, cptr->name);
	client_info.filter = cptr->filter;
	memcpy(client_info.multicast_filter, cptr->multicast_filter, sizeof(client_info.multicast_filter));
	memcpy(client_info.event_filter, cptr->event_filter, sizeof(client_info.event_filter));

	snd_seq_client_unlock(cptr);

	copy_to_user(_client_info, &client_info, sizeof(client_info));
	return 0;
}


/* CLIENT_INFO ioctl() */
static int snd_seq_ioctl_set_client_info(client_t * client, snd_seq_client_info_t * _client_info)
{
	snd_seq_client_info_t client_info;

	if (verify_area(VERIFY_READ, _client_info, sizeof(client_info)) ||
	    verify_area(VERIFY_WRITE, _client_info, sizeof(client_info)))
		return -EFAULT;

	copy_from_user(&client_info, _client_info, sizeof(client_info));

	/* fill the info fields */
	client_info.type = client->type;
	strncpy(client->name, client_info.name, sizeof(client->name)-1);
	client->name[sizeof(client->name)-1] = '\0';
	client->filter = client_info.filter;
	memcpy(client->multicast_filter, client_info.multicast_filter, sizeof(client->multicast_filter));
	memcpy(client->event_filter, client_info.event_filter, sizeof(client->event_filter));

	copy_to_user(_client_info, &client_info, sizeof(client_info));
	return 0;
}


/* 
 * CREATE PORT ioctl() 
 */
static int snd_seq_ioctl_create_port(client_t * client, snd_seq_port_info_t * _info)
{
	client_port_t *port;
	snd_seq_port_info_t info;
	snd_seq_port_callback_t *callback;

	if (!client || !_info)
		return -EINVAL;
	port = snd_seq_create_port(client);
	if (!port)
		return -ENOMEM;

	/* set passed parameters */
	if (verify_area(VERIFY_READ, _info, sizeof(snd_seq_port_info_t)) ||
	    verify_area(VERIFY_WRITE, _info, sizeof(snd_seq_port_info_t))) {
		snd_seq_delete_port(client, port->port);
		return -EFAULT;
	}

	copy_from_user(&info, _info, sizeof(snd_seq_port_info_t));

	if (client->type == USER_CLIENT && info.kernel) {
		snd_seq_delete_port(client, port->port);
		return -EINVAL;
	}
	if (client->type == KERNEL_CLIENT) {
		if ((callback = info.kernel) != NULL) {
			port->private_data = callback->private_data;
			port->subscribe = callback->subscribe;
			port->unsubscribe = callback->unsubscribe;
			port->use = callback->use;
			port->unuse = callback->unuse;
		} else {
			port->private_data = NULL;
			port->subscribe = NULL;
			port->unsubscribe = NULL;
			port->use = NULL;
			port->unuse = NULL;
		}
	}

	info.client = client->number;
	info.port = port->port;

	copy_to_user(_info, &info, sizeof(snd_seq_port_info_t));

	snd_seq_set_port_info(port, &info);
	snd_seq_system_client_ev_port_start(client->number, port->port);
	
	return 0;
}

/* 
 * DELETE PORT ioctl() 
 */
static int snd_seq_ioctl_delete_port(client_t * client, snd_seq_port_info_t * info)
{
	int ret = -EINVAL;
	snd_seq_port_info_t _info;

	if (!client || !info)
		return -EINVAL;

	/* set passed parameters */
	if (verify_area(VERIFY_READ, info, sizeof(snd_seq_port_info_t)) ||
	    verify_area(VERIFY_WRITE, info, sizeof(snd_seq_port_info_t)))
		return -EFAULT;

	copy_from_user(&_info, info, sizeof(snd_seq_port_info_t));
	ret = snd_seq_delete_port(client, _info.port);
	copy_to_user(info, &_info, sizeof(snd_seq_port_info_t));
	return ret;
}


/* 
 * GET_PORT_INFO ioctl() (on any client) 
 */
static int snd_seq_ioctl_get_port_info(client_t *client, snd_seq_port_info_t * info)
{
	client_t *cptr;
	client_port_t *port;
	snd_seq_port_info_t _info;

	if (!client)
		return -EINVAL;

	if (verify_area(VERIFY_READ, info, sizeof(snd_seq_port_info_t)) ||
            verify_area(VERIFY_WRITE, info, sizeof(snd_seq_port_info_t)))
		return -EFAULT;

	copy_from_user(&_info, info, sizeof(snd_seq_port_info_t));
	cptr = snd_seq_client_use_ptr(_info.client);
	if (!cptr)
		return -ENXIO;

	port = snd_seq_port_use_ptr(cptr, _info.port);
	if (!port) {
		snd_seq_client_unlock(cptr);
		return -ENOENT;			/* don't change */
	}

	/* get port info */
	snd_seq_get_port_info(port, &_info);
	snd_seq_ports_unlock(cptr);
	snd_seq_client_unlock(cptr);

	copy_to_user(info, &_info, sizeof(snd_seq_port_info_t));
	return 0;
}


/* 
 * SET_PORT_INFO ioctl() (only ports on this/own client) 
 */
static int snd_seq_ioctl_set_port_info(client_t * client, snd_seq_port_info_t * info)
{
	client_port_t *port;
	snd_seq_port_info_t _info;

	if (!client || !info)
		return -EINVAL;
	if (verify_area(VERIFY_READ, info, sizeof(snd_seq_port_info_t)) ||
	    verify_area(VERIFY_WRITE, info, sizeof(snd_seq_port_info_t)))

	copy_from_user(&_info, info, sizeof(snd_seq_port_info_t));

	if (_info.client != client->number) /* only set our own ports ! */
		return -EPERM;
	port = snd_seq_port_use_ptr(client, _info.port);
	if (port) {
		snd_seq_set_port_info(port, &_info);
		snd_seq_ports_unlock(client);
	}
	copy_to_user(info, &_info, sizeof(snd_seq_port_info_t));
	return 0;
}


/* 
 * add to port's subscription list IOCTL interface 
 */
static int snd_seq_ioctl_subscribe_port(client_t * client, snd_seq_port_subscribe_t * _subs)
{
	int result = -ENXIO;
	client_t *receiver = NULL, *sender = NULL;
	client_port_t *sport = NULL, *dport = NULL;
	snd_seq_port_subscribe_t subs;

	if (!client)
		return -EINVAL;

	if (verify_area(VERIFY_READ, _subs, sizeof(snd_seq_port_subscribe_t)))
		return -EFAULT;
	copy_from_user(&subs, _subs, sizeof(snd_seq_port_subscribe_t));

	if (subs.sender.queue != subs.dest.queue)
		return -EINVAL;

	receiver = snd_seq_client_use_ptr(subs.dest.client);
	if (!receiver)
		goto __end;
	sender = snd_seq_client_use_ptr(subs.sender.client);
	if (!sender)
		goto __end;

	sport = snd_seq_port_use_ptr(sender, subs.sender.port);
	if (sport == NULL)
		goto __end;
	dport = snd_seq_port_use_ptr(receiver, subs.dest.port);
	if (dport == NULL)
		goto __end;

	if (sport == dport || subs.sender.client == subs.dest.client) {
		result = -EINVAL;
		goto __end;
	}
	
	if (client->number == subs.dest.client) {

		if (!(sport->capability & SND_SEQ_PORT_CAP_SUBSCRIPTION)) {
			result = -EINVAL;
			goto __end;
		}
		
		if ((result = snd_seq_port_subscribe(sport)) < 0)
			goto __end;

#ifdef TRACK_SUBSCRIBE
		snd_printd("Add subscription; %d:%d will send to %d:%d via queue %d\n",
			   subs.sender.client, subs.sender.port,
		  	   subs.dest.client, subs.dest.port, subs.dest.queue);
#endif

		result = snd_seq_port_add_subscriber(&sport->input, &subs.dest, subs.exclusive, subs.realtime);
		if (result < 0) {
			snd_seq_port_unsubscribe(sport);
		} else {
			result = snd_seq_port_add_subscriber(&dport->itrack, &subs.sender, 0, 0);
			if (result < 0) {
				snd_seq_port_remove_subscriber(&sport->output, &subs.dest);
				snd_seq_port_unsubscribe(sport);
			}
		}

	} else if (client->number == subs.sender.client) {

		if (!(dport->capability & SND_SEQ_PORT_CAP_SUBSCRIPTION)) {
			result = -EINVAL;
			goto __end;
		}
		
		if ((result = snd_seq_port_use(dport)) < 0)
			goto __end;
	
#ifdef TRACK_SUBSCRIBE
		snd_printd("Add use; %d:%d will use %d:%d via queue %i\n",
			   subs.sender.client, subs.sender.port,
		  	   subs.dest.client, subs.dest.port, subs.dest.queue);
#endif
	
		result = snd_seq_port_add_subscriber(&dport->output, &subs.sender, subs.exclusive, subs.realtime);
		if (result < 0) {
			snd_seq_port_unuse(dport);
		} else {
			result = snd_seq_port_add_subscriber(&sport->otrack, &subs.dest, 0, 0);
			if (result < 0) {
				snd_seq_port_remove_subscriber(&sport->output, &subs.dest);
				snd_seq_port_unuse(dport);
			}
		}
	
	} else {

		result = -EPERM;

	}

	if (result >= 0 && client->type == USER_CLIENT)
		snd_seq_timer_reopen();

      __end:
      	if (sport)
		snd_seq_ports_unlock(sender);
	if (dport)
		snd_seq_ports_unlock(receiver);
	if (sender)
		snd_seq_client_unlock(sender);
	if (receiver)
		snd_seq_client_unlock(receiver);
	return result;
}


/* 
 * remove from port's subscription list 
 */
static int snd_seq_ioctl_unsubscribe_port(client_t * client, snd_seq_port_subscribe_t * _subs)
{
	int result = -ENXIO;
	client_t *receiver = NULL, *sender = NULL;
	client_port_t *sport = NULL, *dport = NULL;
	snd_seq_port_subscribe_t subs;

	if (!client)
		return -EINVAL;

	if (verify_area(VERIFY_READ, _subs, sizeof(snd_seq_port_subscribe_t)))
		return -EFAULT;
	copy_from_user(&subs, _subs, sizeof(snd_seq_port_subscribe_t));

	receiver = snd_seq_client_use_ptr(subs.dest.client);
	if (!receiver)
		goto __end;
	sender = snd_seq_client_use_ptr(subs.sender.client);
	if (!sender)
		goto __end;

	sport = snd_seq_port_use_ptr(sender, subs.sender.port);
	if (sport == NULL)
		goto __end;
	dport = snd_seq_port_use_ptr(receiver, subs.dest.port);
	if (dport == NULL)
		goto __end;

	if (sport == dport) {
		result = -EINVAL;
		goto __end;
	}
	if (!(dport->capability & SND_SEQ_PORT_CAP_SUBSCRIPTION)) {
		result = -EINVAL;
		goto __end;
	}
	
	if (client->number == subs.dest.client) {

		snd_printd("Remove subscription; %d:%d will send to %d:%d via queue %d\n",
			   subs.sender.client, subs.sender.port,
		  	   subs.dest.client, subs.dest.port, subs.dest.queue);

		snd_seq_port_remove_subscriber(&sport->input, &subs.dest);
		snd_seq_port_remove_subscriber(&dport->itrack, &subs.sender);
		snd_seq_port_unsubscribe(sport);

	} else if (client->number == subs.sender.client) {

		snd_printd("Remove use; %d:%d will use %d:%d via queue %i\n",
			   subs.sender.client, subs.sender.port,
		  	   subs.dest.client, subs.dest.port, subs.dest.queue);
	
		snd_seq_port_remove_subscriber(&dport->output, &subs.sender);
		snd_seq_port_remove_subscriber(&sport->otrack, &subs.dest);
		snd_seq_port_unuse(dport);
		
	} else {

		result = -EPERM;

	}

      __end:
      	if (sport)
		snd_seq_ports_unlock(sender);
	if (dport)
		snd_seq_ports_unlock(receiver);
	if (sender)
		snd_seq_client_unlock(sender);
	if (receiver)
		snd_seq_client_unlock(receiver);
	return result;
}


/* GET_QUEUE_INFO ioctl() */
static int snd_seq_ioctl_get_queue_info(client_t * client, snd_seq_queue_info_t * _info)
{
	snd_seq_queue_info_t info;
	queue_t *queue;
	timer_t *tmr;

	if (!client)
		return -EINVAL;

	if (verify_area(VERIFY_READ, _info, sizeof(snd_seq_queue_info_t)) ||
	    verify_area(VERIFY_WRITE, _info, sizeof(snd_seq_queue_info_t)))
		return -EFAULT;

	copy_from_user(&info, _info, sizeof(snd_seq_queue_info_t));

	queue = queueptr(info.queue);
	if (!queue)
		return -EINVAL;
	
	tmr = NULL;

	tmr = queue->timer;
	info.events = snd_seq_total_cells(queue->pool);

	info.tick = tmr->cur_tick;
	info.time.tv_sec = tmr->cur_time.tv_sec;
	info.time.tv_nsec = tmr->cur_time.tv_nsec;

	info.running = tmr->running;
	info.tempo = tmr->tempo;
	info.ppq = tmr->ppq;
	info.flags = 0;	/* not yet used... */
	
	info.owner = queue->owner;
	info.locked = queue->locked;

	copy_to_user(_info, &info, sizeof(snd_seq_queue_info_t));
	return 0;
}


/* SET_QUEUE_INFO ioctl() */
static int snd_seq_ioctl_set_queue_info(client_t * client, snd_seq_queue_info_t * _info)
{
	snd_seq_queue_info_t info;

	if (!client)
		return -EINVAL;

	if (verify_area(VERIFY_READ, _info, sizeof(snd_seq_queue_info_t)))
		return -EFAULT;
	copy_from_user(&info, _info, sizeof(snd_seq_queue_info_t));

	if (snd_seq_queue_check_access(info.queue, client->number)) {
		if (info.tempo >= 0)
			if (snd_seq_queue_timer_set_tempo(info.queue, client->number, info.tempo) < 0)
				return -EPERM;

		if (info.ppq >= 0)
			if (snd_seq_queue_timer_set_ppq(info.queue, client->number, info.ppq) < 0)
				return -EPERM;

		if (info.owner >= 0) {
			if (info.locked)
				snd_seq_queue_use(info.queue, client->number, 1);
			if (snd_seq_queue_set_locked(info.queue, client->number, info.locked) < 0)
				return -EPERM;
			if (snd_seq_queue_set_owner(info.queue, info.owner) < 0)
				return -EPERM;
		}

	} else {
		return -EPERM;
	}
	

	return snd_seq_ioctl_get_queue_info(client, _info);
}


/* GET_QUEUE_CLIENT ioctl() */
static int snd_seq_ioctl_get_queue_client(client_t * client, snd_seq_queue_client_t * _info)
{
	snd_seq_queue_client_t info;
	queue_t *queue;

	if (!client)
		return -EINVAL;

	if (verify_area(VERIFY_READ, _info, sizeof(snd_seq_queue_client_t)) ||
	    verify_area(VERIFY_WRITE, _info, sizeof(snd_seq_queue_client_t)))
		return -EFAULT;

	copy_from_user(&info, _info, sizeof(snd_seq_queue_client_t));

	queue = queueptr(info.queue);
	if (!queue)
		return -EINVAL;
	info.client = client->number;
	info.low = client->low[info.queue];
	info.high = client->high[info.queue];
	info.used = test_bit(client->number, &queue->clients_bitmap) ? 1 : 0;

	copy_to_user(_info, &info, sizeof(snd_seq_queue_client_t));
	return 0;
}


/* SET_QUEUE_CLIENT ioctl() */
static int snd_seq_ioctl_set_queue_client(client_t * client, snd_seq_queue_client_t * _info)
{
	queue_t *queue;
	snd_seq_queue_client_t info;

	if (!client)
		return -EINVAL;

	if (verify_area(VERIFY_READ, _info, sizeof(snd_seq_queue_client_t)))
		return -EFAULT;
	copy_from_user(&info, _info, sizeof(snd_seq_queue_client_t));

	queue = queueptr(info.queue);
	if (!queue)
		return -EINVAL;

	if (info.low >= 0) {
		if (!info.low)
			info.low = 1;
		client->low[info.queue] = info.low;
	}
	if (info.high >= 0) {
		if (!info.high)
			info.high = 1;		/* it's correct? */
		client->high[info.queue] = info.high;
	}
	if (info.used >= 0)
		snd_seq_queue_use(info.queue, client->number, info.used);

	return snd_seq_ioctl_get_queue_client(client, _info);
}



static int snd_seq_do_ioctl(client_t *client, unsigned int cmd, unsigned long arg)
{
	switch (cmd) {

		case SND_SEQ_IOCTL_PVERSION:
			/* return sequencer version number */
			return snd_ioctl_out((long *) arg, SND_SEQ_VERSION);

		case SND_SEQ_IOCTL_CLIENT_ID:
			/* return the id of this client */
			return snd_ioctl_out((long *) arg, client->number);

		case SND_SEQ_IOCTL_SYSTEM_INFO:
			/* return system information */
			return snd_seq_ioctl_system_info(client, (snd_seq_system_info_t *) arg);

		case SND_SEQ_IOCTL_GET_CLIENT_INFO:
			/* return info on specified client */
			return snd_seq_ioctl_get_client_info(client, (snd_seq_client_info_t *) arg);

		case SND_SEQ_IOCTL_SET_CLIENT_INFO:
			/* set info on specified client */
			return snd_seq_ioctl_set_client_info(client, (snd_seq_client_info_t *) arg);

		case SND_SEQ_IOCTL_CREATE_PORT:
			/* create a port for this client */
			return snd_seq_ioctl_create_port(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_DELETE_PORT:
			/* remove a port from this client */
			return snd_seq_ioctl_delete_port(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_GET_PORT_INFO:
			/* get info for specified port */
			return snd_seq_ioctl_get_port_info(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_SET_PORT_INFO:
			/* set info for specified port in this client */
			return snd_seq_ioctl_set_port_info(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_SUBSCRIBE_PORT:
			/* add to port's subscription list */
			return snd_seq_ioctl_subscribe_port(client, (snd_seq_port_subscribe_t *) arg);

		case SND_SEQ_IOCTL_UNSUBSCRIBE_PORT:
			/* remove from port's subscription list */
			return snd_seq_ioctl_unsubscribe_port(client, (snd_seq_port_subscribe_t *) arg);

		case SND_SEQ_IOCTL_GET_QUEUE_INFO:
			/* get info for specified queue */
			return snd_seq_ioctl_get_queue_info(client, (snd_seq_queue_info_t *) arg);

		case SND_SEQ_IOCTL_SET_QUEUE_INFO:
			/* set info for specified queue */
			return snd_seq_ioctl_set_queue_info(client, (snd_seq_queue_info_t *) arg);

		case SND_SEQ_IOCTL_GET_QUEUE_CLIENT:
			/* get client specific info for specified queue */
			return snd_seq_ioctl_get_queue_client(client, (snd_seq_queue_client_t *) arg);

		case SND_SEQ_IOCTL_SET_QUEUE_CLIENT:
			/* set client specfic info for specified queue */
			return snd_seq_ioctl_set_queue_client(client, (snd_seq_queue_client_t *) arg);

		default:
			snd_printd("seq unknown ioctl() 0x%x (type='%c', number=0x%2x)\n",
				   cmd, _IOC_TYPE(cmd), _IOC_NR(cmd));
	}
	return -ENXIO;
}


static int snd_seq_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	client_t *client = (client_t *) file->private_data;

	if (!client)
		return -ENXIO;
		
	return snd_seq_do_ioctl(client, cmd, arg);
}


/* -------------------------------------------------------- */


/* exported to kernel modules */
int snd_seq_create_kernel_client(snd_card_t *card, int client_index, snd_seq_client_callback_t * callback)
{
	client_t *client;

	if (!callback)
		return -EINVAL;
	if (card && client_index > 7)
		return -EINVAL;
	if (!card && client_index > 63)
		return -EINVAL;
	if (card)
		client_index += 64 + (card->number << 3);

	snd_mutex_down_static(register);
	client = seq_create_client1(client_index);
	if (!client) {
		snd_mutex_up_static(register);
		return -EBUSY;	/* failure code */
	}
	usage_alloc(&client_usage, 1);

	client->accept_input = callback->allow_output;
	client->accept_output = callback->allow_input;
		
	/* fill client data */
	client->data.kernel.card = card;
	client->data.kernel.private_data = callback->private_data;
	client->data.kernel.input = callback->input;
	sprintf(client->name, "Client-%d", client->number);

	client->type = KERNEL_CLIENT;
	snd_mutex_up_static(register);

	/* make others aware this new client */
	snd_seq_system_client_ev_client_start(client->number);
	
	/* return client number to caller */
	return client->number;
}

/* exported to kernel modules */
int snd_seq_delete_kernel_client(int client)
{
	client_t *ptr;

	ptr = clientptr(client);
	if (!ptr)
		return -EINVAL;

	seq_free_client(ptr);
	return 0;
}


/* exported, called by kernel clients to enqueue events */
int snd_seq_kernel_client_enqueue(int client, snd_seq_event_t * ev, int atomic)
{
	queue_t *q;
	client_t *cptr;
	snd_seq_event_cell_t *cell;

	/* we got one event, copy it into our own data structure and */
	/* feed it to the prioQ */
	if (ev->dest.queue < SND_SEQ_MAX_QUEUES) {
		q = queueptr(ev->dest.queue);
		if (!q)
			return -EINVAL;
		cell = snd_seq_event_dup(q->pool, ev, atomic);
	} else {
		if (ev->dest.queue != SND_SEQ_ADDRESS_SUBSCRIBERS)
			return -EINVAL;
		cptr = snd_seq_client_use_ptr(client);
		if (!cptr)
			return -EINVAL;
		cell = snd_seq_event_dup(cptr->outqueue->pool, ev, atomic);
		snd_seq_client_unlock(cptr);
	}
	if (cell) {
		cell->event.source.queue = SND_SEQ_ADDRESS_UNKNOWN;
		cell->event.source.client = client;	/* fill in client index */
		snd_seq_enqueue_event(cell, atomic);
	} else {
		return -ENOMEM;
	}
	return 0;	/* success */
}


/* 
 * exported, called by kernel clients to dispatch events directly to other
 * clients, bypassing the queues. No processing of time stamps is done.
 */
int snd_seq_kernel_client_dispatch(int client, snd_seq_event_t * ev, int atomic)
{
	client_t *cptr;
	snd_seq_event_cell_t *cell;

	/* we got one event, copy it into our own data structure and */
	/* send it directly to the client */
	if (ev->dest.client < 192) {
		cptr = snd_seq_client_use_ptr(ev->dest.client);
		if (!cptr)
			return -EINVAL;
	} else {	/* multicast or broadcast - use own pool (temporary) */
		cptr = snd_seq_client_use_ptr(ev->source.client);
		if (!cptr)
			return -EINVAL;
	}
	cell = snd_seq_event_dup(cptr->outqueue->pool, ev, atomic);
	snd_seq_client_unlock(cptr);
	if (cell) {
		snd_seq_dispatch_event(cell, atomic);
		return 1;	/* success */
	} else {
		return -1;
	}
}


/*
 * exported, called by kernel clients to perform same functions as with
 * userland ioctl() 
 */
int snd_seq_kernel_client_ctl(int clientid, unsigned int cmd, void *arg)
{
	client_t *client;
	mm_segment_t fs;
	int result;

	client = clientptr(clientid);
	if (!client)
		return -ENXIO;
	fs = snd_enter_user();
	result = snd_seq_do_ioctl(client, cmd, (unsigned long)arg);
	snd_leave_user(fs);
	return result;
}


/*---------------------------------------------------------------------------*/

/*
 *  /proc interface
 */
static void snd_seq_info_dump_subscribers(snd_info_buffer_t * buffer, subscribers_group_t * group)
{
	subscribers_t *s;

	snd_seq_subscribers_lock(group);
	for (s = group->list; s; s = s->next) {
		snd_iprintf(buffer, "%d:%d:%d",
				s->addr.queue,
				s->addr.client,
				s->addr.port);
		if (s->next)
			snd_iprintf(buffer, ", ");
	}
	snd_seq_subscribers_unlock(group);
}

static void snd_seq_info_dump_ports(snd_info_buffer_t * buffer, client_port_t * ports)
{
	client_port_t *p = ports;

	while (p) {
#if SHOW_CAPABILITY
		snd_iprintf(buffer, "  Port %3d : \"%s\" (%s,%s,%s,%s)\n",
			    p->port, p->name,
		(p->capability & SND_SEQ_PORT_CAP_MIDI_IN ? "MIDI_in" : ""),
			    (p->capability & SND_SEQ_PORT_CAP_MIDI_OUT ? "MIDI_out" : ""),
		(p->capability & SND_SEQ_PORT_CAP_SYNC_IN ? "Sync_in" : ""),
			    (p->capability & SND_SEQ_PORT_CAP_SYNC_OUT ? "Sync_out" : ""),
			    (p->capability & SND_SEQ_PORT_CAP_SUBSCRIPTION ? "Subscription" : ""));
#else
		snd_iprintf(buffer, "  Port %3d : \"%s\"\n",
			    p->port, p->name);
#endif
		snd_iprintf(buffer, "    Subscribers: ");
		snd_seq_info_dump_subscribers(buffer, &p->input);
		snd_iprintf(buffer, "\n    Subscribers tracking: ");
		snd_seq_info_dump_subscribers(buffer, &p->itrack);
		snd_iprintf(buffer, "\n    Users: ");
		snd_seq_info_dump_subscribers(buffer, &p->output);
		snd_iprintf(buffer, "\n    Users tracking: ");
		snd_seq_info_dump_subscribers(buffer, &p->otrack);
		snd_iprintf(buffer, "\n");

		p = p->next;
	}
}


/* exported to seq_info.c */
void snd_seq_info_clients_read(snd_info_buffer_t * buffer, void *private_data)
{
	extern void snd_seq_info_pool(snd_info_buffer_t * buffer, pool_t * pool, char *space);
	int c;
	client_t *client;

	snd_iprintf(buffer, "Client info\n");
	snd_iprintf(buffer, "  cur  clients : %d\n", client_usage.cur);
	snd_iprintf(buffer, "  peak clients : %d\n", client_usage.peak);
	snd_iprintf(buffer, "  max  clients : %d\n", SND_SEQ_MAX_CLIENTS);
	snd_iprintf(buffer, "\n");

	/* list the client table */
	for (c = 0; c < SND_SEQ_MAX_CLIENTS; c++) {
		client = snd_seq_client_use_ptr(c);
		if (!client)
			continue;
		snd_seq_ports_lock(client);
		switch (client->type) {
			case NO_CLIENT:
				/*snd_iprintf(buffer, "Client %3d : Free\n", c); */
				break;

			case USER_CLIENT:
			case KERNEL_CLIENT:
				snd_iprintf(buffer, "Client %3d : \"%s\" [%s]\n",
					c, client->name,
					client->type == USER_CLIENT ? "User" : "Kernel");
				snd_seq_info_dump_ports(buffer, client->ports);
				snd_iprintf(buffer, "  Event pool :\n");
				snd_seq_info_pool(buffer, client->outqueue->pool, "    ");
				break;
		}
		snd_seq_ports_unlock(client);
		snd_seq_client_unlock(client);
	}
}


/*---------------------------------------------------------------------------*/


/*
 *  REGISTRATION PART
 */
static snd_minor_t snd_seq_reg =
{
	"sequencer",
	NULL,			/* unregister */
	NULL,			/* lseek */
	snd_seq_read,		/* read */
	snd_seq_write,		/* write */
	snd_seq_open,		/* open */
	snd_seq_release,	/* release */
#ifdef SND_POLL
	snd_seq_poll,		/* poll */
#else
	snd_seq_select,		/* select */
#endif
	snd_seq_ioctl,		/* ioctl */
	NULL			/* mmap */
};


/* 
 * register sequencer device 
 */
int snd_sequencer_device_init(void)
{
	int err;

	snd_mutex_down_static(register);

	if ((err = snd_register_minor(SND_MINOR_SEQUENCER, &snd_seq_reg)) < 0) {
		snd_mutex_up_static(register);
		return err;
	}
	if ((seq_dev = snd_info_create_device("seq", SND_MINOR_SEQUENCER, 0)) == NULL) {
		snd_unregister_minor(SND_MINOR_SEQUENCER);
		snd_mutex_up_static(register);
		return err;
	}
	
	snd_mutex_up_static(register);

	return 0;
}



/* 
 * unregister sequencer device 
 */
void snd_sequencer_device_done(void)
{
	if (seq_dev)
		snd_info_free_device(seq_dev);
	snd_unregister_minor(SND_MINOR_SEQUENCER);
}
