
/*
    opti9xx.c - ALSA driver for OPTi 82C9xx based soundcards.
    Copyright (C) 1998-1999 by Massimo Piccioni <piccio@caronte.csr.unibo.it>

    This code has been developed at the Italian Ministry of Air Defence,
    Sixth Division (oh, che gioia ...), Rome.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
    This code provides support for most OPTi 82C9xx chips based soundcards.
    All non-PnP chips (i.e. 82C928, 82C929 and 82C930) are fully supported.
    All PnP-capable chips (82C924, 82C925 and 82C931) should be supported
    in non-PnP mode. The PnP support is still uncertain (see below).
    I have no documentation about the OPTi 82C933 chip (82C931 compatible?).

    This source is not complete! I still have to do a lot of work (see the
    todo section below).

    This driver supports only one soundcard per system, due to an harware
    limitation (i/o at fixed locations).

    This driver allow the user to enable and fully configure the on-board
    cdrom drive interface (i.e. IDE/ATAPI, Sony, Mitsumi and Panasonic)
    of several OPTi based soundcards.
    Please, search for the NON-SCSI/ATAPI cdrom support documentation in
    the Linux kernel sources.

    Sources:
        OPTi Inc. 82C9xx databooks (available from ftp.opti.com).
        Analog Devices Inc. AD1845/8 databooks.
        Advanced Linux Sound Architecture documentation and sources.

    Version history:
        0.1:  82C9xx detection and configuration only
        1.0:  basic sound support added (ad1848 and opl3)
        1.1:  cdrom and game port configuration added
        1.1c: cdrom and game port configuration improved

    Todo:
        improve 82C9xx detection and configuration (in non-PnP mode)
        add PnP support (ALSA native)
        add cs42xx support (full duplex)
        add MPU-401 support for 82C929+ chips
        obtain 82C933 documentation
*/

/*
    Chips description coming soon ...
*/

/*
    PnP considerations:
        At the moment, this driver is not reliable with PnP-capable chips,
        because the MC registers configuration doesn't work very well when
        PnP is enabled.
        Several soundcards are PnP-only, others have a jumper that enables
        or disables PnP. If you can, disable PnP, if not, don't worry, I'm
        working on PnP support ;)

        With a non-PnP BIOS any OPTi PnP soundcard will not work, until a
        PnP activation procedure is applied (all logical devices will remain
        inactive after power-on).
        At the moment, you will need to use isapnptools to configure and
        activate your PnP card.

        If you don't have isapnptools already installed on your system, try
        at the following web site: http://www.roestock.demon.co.uk/isapnptools
*/

/*
    Sound cards (being) tested:
        ExpertColor MED-3931 (82C931 based)
        ExpertMedia Sound 16 MED-1600 (82C928 based)
        Mozart (OTI601 based)
        Sound Player S-928 (82C928 based)

    Cdrom drives tested:
        CreativeLabs CD200F (Funai E2550UA), Panasonic i/f
        Matsushita-Kotobuki CR-562-B, Panasonic i/f

    Things of interest:
        Some soundcards (such as ExpertMedia Sound 16) use jumpers to configure
        the on-board cdrom drive interface, so software configuration doesn't
        work (of course).

        I have no problems with my CreativeLabs CD200F cdrom drive (a Funai
        E2550UA drive with Panasonic proprietary interface) connected to the
        on-board interface of a S-928 soundcard (see the cdrom configuration
        section below). I simply load the sbpcd kernel module after this.
        See the sbpcd documentation for more informations (available in the
        /usr/src/linux/Documentation/cdrom directory).

        Another interesting thing: the contemporary presence of an 82C928
        based (not PnP) sound card and an IBM Auto 16/4 Token Ring ISA PnP
        adapter, may cause several problems (i.e. Linux might not detect the
        token ring adapter), due to an i/o conflict between the soundcard
        game port (0xa20 fixed?) and the tr adapter base port (0xa20 or 0xa24).
        It happens because PnP BIOS is not able to detect what i/o ports the
        soundcard is using, so the first 'available' i/o port (i.e. 0xa20) is
        assigned to the tr adapter (i/o conflict).
        A fine solution is to use the isapnptools to reconfigure your PnP
        adapters (then you can load the ibmtr kernel module without any
        problem).
        Another solution is to disable the sound card game port in this module
        (loaded before the ibmtr module, see the game port configuration
        section below).

        Finally, Mozart soundcards are also supported, due to some kind of
        compatibility between OPTi 82C928 and OAK OTI601 chips (see mozart.c
        in the ALSA sources).
        I was not able to get the on-board cdrom interface working with a
        Mozart soundcard (broken board?). Unfortunately I have no documentation
        about the OTI601 chip, so any suggestion is welcome.
*/

/*
    I strongly need feedback from users (there is a lot of OPTi 82C9xx based
    soundcards, that I don't know) and your experiences may be very useful,
    so feel free to contact me by e-mail.

    Acknowledgments:
        Elliot Lee <sopwith@redhat.com> reported an 82C930 based soundcard
        working well.
*/

/*
    Module options:
        snd_port        codec base i/o port: 0x530,0xe80,0xf40,0x604
        snd_irq         WSS irq num: 5*,7,9,10,11 (*: 82C930+ only) 
        snd_dma         WSS dma num: 0,1,3
        snd_dma_size    dma size in Kb
        cd_if           cdrom interface: disabled,ide1,ide2,mitsumi,panasonic,
                        sony
        cd_port         cdrom i/o port: 0x320,0x330,0x340,0x360
        cd_irq          cdrom irq num: -1,5,7,9,10,11          (-1 disables)
        cd_drq          cdrom drq num: -1,0,1,3                (-1 disables)
        game            game port: enabled,disabled

    Default values:
        snd_xxx         auto (probed and configured)
        cd_if           panasonic
        cd_port         0x340
        cd_irq          disabled
        cd_drq          disabled
        game            disabled

    Examples:
        modprobe snd-opti9xx snd_irq=9
        modprobe snd-opti9xx cd_port=0x360 game=enabled snd_port=0xe80
        modprobe snd-opti9xx snd_port=0xf40 snd_irq=9 snd_dma=1
*/

#define OPTi9XX_VERSION	"opti9xx.c: v1.1c, 1/23/99 <piccio@caronte.csr.unibo.it>"

#define OPTi9XX_DEBUG
#undef OPTi9XX_DEBUG_MC

#define __SND_OSS_COMPAT	/* why not? */
#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "ad1848.h"
#include "initval.h"
#include "isapnp.h"
#include "opl3.h"

#ifndef ESUCCESS
#define ESUCCESS	0
#endif	/* ESUCCESS */

#define opti9xx_printk( args... )	snd_printk( "opti9xx: " ##args )

/* on-board interfaces configuration section */
/* cdrom type selection */
static struct {
	char *name;
	unsigned char bits;
} cd_if_table[]= {
	{ "disabled",	0x00 },		/* all */
	{ "ide1",	0x05 },		/* c930 */
	{ "ide2",	0x04 },		/* c925, c930, c931 */
	{ "mitsumi",	0x02 },		/* c924, c928, c929 */
	{ "panasonic",	0x03 },		/* c924, c928, c929 */
	{ "sony",	0x01 }		/* c924, c928, c929 */
};

/* cdrom ioports (Sony, Mitsumi or Panasonic interfaces only) */
static struct {
	int port;
	unsigned char bits;
} cd_port_table[]= {
	{ 0x320,	0x03 },		
	{ 0x330,	0x01 },
	{ 0x340,	0x00 },		
	{ 0x360,	0x02 }
};

/* cdrom irq (Sony, Mitsumi, Panasonic) */
static struct {
	int irq;
	unsigned char bits;
} cd_irq_table[]= {
	{ -1,	0x00 },		/* disabled */
	{ 5,	0x01 },
	{ 7,	0x02 },
	{ 9,	0x04 },
	{ 10,	0x05 },
	{ 11,	0x06 }
};

/* cdrom drq (Sony, Mitsumi, Panasonic) */
static struct {
	int drq;
	unsigned char bits;
} cd_drq_table[]= {
	{ -1,	0x03 },		/* disabled */
	{ 0,	0x01 },
	{ 1,	0x02 },
	{ 3,	0x00 }
};

/* game port */
static struct {
	char *name;
	unsigned char bits;
} game_en_table[]= {
	{ "enabled",	0x00 },	
	{ "disabled",	0x01 }
};

#define OPTi9XX_OPL3_LPORT	0x388
#define OPTi9XX_OPL3_RPORT	OPTi9XX_OPL3_LPORT+2

/* OPTi chips supported */
enum chipsets {
	c928=0,	c929=1,	c924=2,	c925=2,
	c930=3,	c931=4,	c933=5,	unsupported=c933
};

/* chip-specific parameters */
static struct {
	char *name;
	unsigned char pwd;
	int mc_base;
	int pwd_reg;
	char mc_regs;
} chip_info[]= {
	{ "82C928",		0xe2,	0xf8c,	3,	7 },
	{ "82C929",		0xe3,	0xf8c,	3,	7 },
	{ "82C924/5",		0xe5,	0xf8c,	3,	10 },
	{ "82C930",		0xe4,	0xf8f,	0,	1 },
	{ "82C931",		0xe4,	0xf8d,	0,	1 },
	{ "unsupported",	0x00,	0x000,	0,	0 }
};

#define OPTi9XX_CHIP		chip_info[opti9xx_chip].name
#define OPTi9XX_MC_BASE		chip_info[opti9xx_chip].mc_base
#define OPTi9XX_PASSWD		chip_info[opti9xx_chip].pwd

#define OPTi9XX_MC(x)		x
#define OPTi92X_MC_REG(x)	OPTi9XX_MC_BASE+OPTi9XX_MC(x)
#define OPTi9XX_PWD_REG		OPTi9XX_MC_BASE+chip_info[opti9xx_chip].pwd_reg

#define OPTi93X_MC_INDEX	opti93x_mc_index
#define OPTi93X_MC_DATA		OPTi93X_MC_INDEX+1

#define OPTi9XX_WSS_ENABLE	0x80
#define OPTi93X_OPL4_ENABLE     0x20
#define OPTi93X_PWD_ENABLE	0x00
#define OPTi93X_IDX_BITS	((OPTi93X_MC_INDEX &(1<<8))>>4)|\
				((OPTi93X_MC_INDEX &(0xf0))>>4)

#define WSS_ADDRESS_PORT	WSS_BASE_PORT+0
#define WSS_CONFIG_PORT		snd_port
#define WSS_BASE_PORT		WSS_CONFIG_PORT+4
#define WSS_STATUS_PORT		WSS_BASE_PORT+2
#define WSS_SIZE		8

static int wss_valid_ports[]= {
	0x530,	0xe80,	0xf40,	0x604,	SND_AUTO_PORT
};

static int wss_valid_dmas[]= {
	0,	1,	3,	-1
};

static int wss_valid_irqs[]= {
	5,	/* 82C93x based soundcards only */
	9,	10,	11,	7,	-1
};

/* WSS configuration (module options) */
int snd_dma = SND_DEFAULT_DMA1;
int snd_dma_size = SND_DEFAULT_DMA_SIZE1;
char *snd_id = SND_DEFAULT_STR1;
int snd_index = SND_DEFAULT_IDX1;
int snd_irq = SND_DEFAULT_IRQ1;
int snd_port=SND_DEFAULT_PORT1;

/* cdrom drive and game port configuration section (module options) */
int cd_drq = -1;
/* default: Panasonic i/f at ox340, no irq and drq needed (see sbpcd.c) */
char *cd_if = "panasonic";
int cd_irq = -1;
int cd_port = 0x340;
/* default: game port disabled (i/o conflict with an ibmtr adapter) */
char *game = "disabled";

#ifdef MODULE_PARM
MODULE_AUTHOR("Massimo Piccioni <piccio@caronte.csr.unibo.it>");
MODULE_DESCRIPTION("OPTi 82C9xx based soundcards support.");
/* cdrom interface configuration */
MODULE_PARM(cd_drq, "i");
MODULE_PARM_DESC(cd_drq, "cdrom drive dma number");
MODULE_PARM(cd_irq, "i");
MODULE_PARM_DESC(cd_irq, "cdrom drive irq number");
MODULE_PARM(cd_port, "i");
MODULE_PARM_DESC(cd_port, "cdrom drive interface port");
MODULE_PARM(cd_if, "s");
MODULE_PARM_DESC(cd_if, "cdrom drive interface type");
/* game port configuration */
MODULE_PARM(game, "s");
MODULE_PARM_DESC(game, "game port enable");
/* WSS configuration */
MODULE_PARM(snd_dma, "i");
MODULE_PARM_DESC(snd_dma, "WSS dma number");
MODULE_PARM(snd_dma_size, "i");
MODULE_PARM_DESC(snd_dma_size, "WSS dma size in kB");
MODULE_PARM(snd_irq, "i");
MODULE_PARM_DESC(snd_irq, "WSS irq number");
MODULE_PARM(snd_port, "i");
MODULE_PARM_DESC(snd_port, "sound base port");
#endif	/* MODULE_PARM */

/* internal data */
snd_card_t *opti9xx_card = NULL;
snd_dma_t *snd_dma_ptr = NULL;
snd_irq_t *snd_irq_ptr = NULL;
snd_kmixer_t *snd_mixer = NULL;
snd_synth_t *snd_opl3 = NULL;
snd_pcm_t *snd_pcm = NULL;

unsigned char cd_drq_sel = 0xff;
unsigned char cd_if_sel = 0xff;
unsigned char cd_irq_sel = 0xff;
unsigned int cd_port_sel = 0xff;
unsigned char game_sel = 0xff;

int opti9xx_chip = unsupported;
int opti93x_mc_index = 0xe0e;

/* ad1848 irq (snd_irq) handling function, inspired from mozart.c */
void snd_opti9xx_ad1848_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	register unsigned char status = inb(WSS_STATUS_PORT);
	if (status & 0x01)
		snd_ad1848_interrupt(snd_pcm, status);
}

/* OPTi chips specific i/o function, MC registers read */ 
unsigned char snd_opti9xx_read(int reg_idx)
{
	unsigned long flags;
	unsigned char retval = 0xff;
	snd_cli(&flags);
	outb(OPTi9XX_PASSWD, OPTi9XX_PWD_REG);
	switch (opti9xx_chip) {
	case c924:
	/* case c925: */
	case c928:
	case c929:
		retval = inb(OPTi92X_MC_REG(reg_idx));
		break;
	case c930:
	case c931:
		outb(reg_idx, OPTi93X_MC_INDEX);
		outb(OPTi9XX_PASSWD, OPTi9XX_PWD_REG);
		retval = inb(OPTi93X_MC_DATA);
		break;
	default:	/* uff */
	}
	snd_sti(&flags);
#ifdef OPTi9XX_DEBUG_MC
	opti9xx_printk("0x%x read from MC%d reg\n", retval, reg_idx);
#endif	/* OPTi9XX_DEBUG_MC */
	return retval;
}

/* OPTi chips specific i/o function, MC registers write */ 
void snd_opti9xx_write(int reg_idx, unsigned char value)
{
	unsigned long flags;
	snd_cli(&flags);
	outb(OPTi9XX_PASSWD, OPTi9XX_PWD_REG);
	switch (opti9xx_chip) {
	case c924:
	/* case c925: */
	case c928:
	case c929:
		outb(value, OPTi92X_MC_REG(reg_idx));
		break;
	case c930:
	case c931:
		outb(reg_idx, OPTi93X_MC_INDEX);
		outb(OPTi9XX_PASSWD, OPTi9XX_PWD_REG);
		outb(value, OPTi93X_MC_DATA);
		break;
	default:	/* uff */
	}
	snd_sti(&flags);
#ifdef OPTi9XX_DEBUG_MC
	opti9xx_printk("written 0x%x to MC%d reg\n", value, reg_idx);
#endif	/* OPTi9XX_DEBUG_MC */
}

void snd_opti9xx_use_inc(snd_card_t *card)
{
	MOD_INC_USE_COUNT;	/* wow, someone is using this driver! */
}

void snd_opti9xx_use_dec(snd_card_t *card)
{
	MOD_DEC_USE_COUNT;	/* see you soon ;) */
}

/* MC indirect registers index i/o port */
/* 0x[ef][0..f]e are valid values */
int snd_opti93x_register_index_reg(void)
{
	int hb, lb;
	for (hb = 0x00; hb < 0x02; hb++)
		for (lb = 0x00; lb < 0x10; lb++) {
			opti93x_mc_index = 0xe00 + (hb << 8) + (lb << 4) + 0x0e;
			if (snd_register_ioport(opti9xx_card,
				opti93x_mc_index, 2,
				"opti93x - mc regs (indirect)", NULL) == 0)
				return ESUCCESS;
		}
	return -EBUSY;
}

int snd_opti9xx_register_wss_dma(void)
{
	return snd_register_dma_channel(opti9xx_card,
		"opti9xx - wss",
		snd_dma,SND_DMA_TYPE_ISA,snd_dma_size,
		wss_valid_dmas,&snd_dma_ptr);
}

int snd_opti9xx_register_wss_irq(void)
{
	return snd_register_interrupt(opti9xx_card,
		"opti9xx - wss",
		snd_irq, SND_IRQ_TYPE_ISA, snd_opti9xx_ad1848_interrupt, NULL,
		wss_valid_irqs + ((opti9xx_chip >= c930) ? 0 : 1),
		&snd_irq_ptr);
}

int snd_opti9xx_register_wss_port(void)
{
	int bits;
	for (bits = 0; wss_valid_ports[bits] != SND_AUTO_PORT; bits++)
		if (snd_port == SND_AUTO_PORT)
			if (snd_register_ioport(opti9xx_card,
				wss_valid_ports[bits], WSS_SIZE,
				"opti9xx - wss", NULL) >= 0) {
				snd_port = wss_valid_ports[bits];
				break;
			} else
				continue;
		else if (snd_port == wss_valid_ports[bits]) {
				if (snd_register_ioport(opti9xx_card,
					snd_port, WSS_SIZE,
					"opti9xx - wss", NULL) < 0)
					snd_port=SND_AUTO_PORT;
				break;
			}
	if (snd_port == SND_AUTO_PORT)
		return -EBUSY;
	return bits;
}

int snd_opti92x_detect(void)
{
	for (opti9xx_chip = c928; opti9xx_chip < c930; opti9xx_chip++) {
		unsigned char value;
		if (snd_register_ioport(opti9xx_card,
			OPTi9XX_MC_BASE, chip_info[opti9xx_chip].mc_regs,
			"opti92x - mc regs", NULL) < 0)
			continue;
		value = snd_opti9xx_read(OPTi9XX_MC(1));
		if (value != 0xff) 
			if (value != inb(OPTi9XX_MC_BASE + OPTi9XX_MC(1)))
				if (value == snd_opti9xx_read(OPTi9XX_MC(1)))
					break;
		snd_unregister_ioports(opti9xx_card);
	}
	return (opti9xx_chip < c930) ? ESUCCESS : -ENODEV;
}

/* temporary version */
int snd_opti93x_detect(void)
{
	for (opti9xx_chip = c930; opti9xx_chip < unsupported; opti9xx_chip++) {
		unsigned char value;
		if (snd_register_ioport(opti9xx_card,
			OPTi9XX_MC_BASE, chip_info[opti9xx_chip].mc_regs,
			"opti93x - mc regs", NULL)<0)
			continue;
		if (snd_opti93x_register_index_reg() == ESUCCESS) {
			outb(OPTi9XX_PASSWD, OPTi9XX_PWD_REG);
			outb(OPTi93X_PWD_ENABLE | OPTi93X_IDX_BITS,
				OPTi9XX_MC_BASE);
			value = snd_opti9xx_read(OPTi9XX_MC(7));
			snd_opti9xx_write(OPTi9XX_MC(7), value ^ 0xff);
			if (snd_opti9xx_read(OPTi9XX_MC(7)) == (value ^ 0xff))
				break;
		}
		snd_unregister_ioports(opti9xx_card);
	}
	return (opti9xx_chip == unsupported) ? -ENODEV : ESUCCESS;
}

int snd_opti9xx_detect(void)
{
	return (snd_opti92x_detect()) ? snd_opti93x_detect() : ESUCCESS;
}

int snd_opti9xx_setup(void)
{
	unsigned char mc1, mc2, mc3, mc4, mc5, mc6;
	int wss_bits;
	if ((wss_bits = snd_opti9xx_register_wss_port()) < 0)
		return -EBUSY;
	/* read current configuration */
	mc1=snd_opti9xx_read(OPTi9XX_MC(1));
	mc2=snd_opti9xx_read(OPTi9XX_MC(2));
	mc3=snd_opti9xx_read(OPTi9XX_MC(3));
	mc4=snd_opti9xx_read(OPTi9XX_MC(4));
	mc5=snd_opti9xx_read(OPTi9XX_MC(5));
	mc6=snd_opti9xx_read(OPTi9XX_MC(6));
	switch (opti9xx_chip) {
	case c924:
	/* case c925: */
	case c928:
	case c929:
		mc1 = OPTi9XX_WSS_ENABLE | (wss_bits << 4) | (cd_if_sel << 1) |
			game_sel;
		mc2 = (cd_port_sel << 6) | (cd_irq_sel << 2) | cd_drq_sel;
		mc3 = 0xf2;
		mc4 = 0xf2;
		break;
	case c930:
	case c931:
		mc1 &= (~0x3f);
		mc1 |= (wss_bits << 4) | (cd_if_sel << 1) | game_sel;
		mc3 = mc5 = 0x00;
		mc4 &= (~0x30);
		mc4 |= OPTi93X_OPL4_ENABLE;
		mc6 &= (~0x03);
		mc6 |= (OPTi9XX_WSS_ENABLE >> 6);
	}
	/* write new configuration */
	snd_opti9xx_write(OPTi9XX_MC(1), mc1);
	snd_opti9xx_write(OPTi9XX_MC(2), mc2);
	snd_opti9xx_write(OPTi9XX_MC(3), mc3);
	snd_opti9xx_write(OPTi9XX_MC(4), mc4);
	snd_opti9xx_write(OPTi9XX_MC(5), mc5);
	snd_opti9xx_write(OPTi9XX_MC(6), mc6);
	return ESUCCESS;
}

/* inspired from mozart.c */
int snd_opti9xx_attach(void)
{
	static int dma_bits[] = { 1, 2, 0, 3 };
	static int irq_bits[] = { 0, 0, 0, 0, 0, 5, 0, 1, 0, 2, 3, 4 };
	if (snd_opti9xx_register_wss_irq())
		return -EBUSY;
	snd_irq = snd_irq_ptr->irq;
	if (snd_opti9xx_register_wss_dma())
		return -EBUSY;
	snd_dma = snd_dma_ptr->dma;
	outb((irq_bits[snd_irq] << 3) | (dma_bits[snd_dma]), WSS_CONFIG_PORT);
	snd_pcm = snd_ad1848_new_device(opti9xx_card, WSS_ADDRESS_PORT,
		snd_irq_ptr, snd_dma_ptr, AD1848_HW_DETECT);
	if (!snd_pcm)
		return -ENXIO;
	if (snd_pcm_register(snd_pcm, 0) < 0)
		return -ENOMEM;
	snd_mixer = snd_ad1848_new_mixer(snd_pcm);
	if (!snd_mixer)
		return -ENXIO;
	if (snd_mixer_register(snd_mixer, 0) < 0)
		return -ENOMEM;
	snd_opl3=snd_opl3_new_device(opti9xx_card,
		OPTi9XX_OPL3_LPORT, OPTi9XX_OPL3_RPORT,
		OPL3_HW_OPL3, 0);
	if (snd_opl3)
		snd_synth_register(snd_opl3);
	strcpy(opti9xx_card->abbreviation, OPTi9XX_CHIP);
	sprintf(opti9xx_card->shortname, "OPTi %s based soundcard",
		OPTi9XX_CHIP);
	sprintf(opti9xx_card->longname,
		"OPTi %s based soundcard at 0x%x, irq %i, dma %i",
		OPTi9XX_CHIP, snd_port, snd_irq, snd_dma);
	if (snd_card_register(opti9xx_card))
		return -ENOMEM;
#ifdef OPTi9XX_DEBUG
	opti9xx_printk("found an %s\n", opti9xx_card->longname);
#endif	/* OPTi9XX_DEBUG */
	return ESUCCESS;
}

int snd_opti9xx_probe(void)
{
	int error;
	opti9xx_card = snd_card_new(snd_index, snd_id,
		snd_opti9xx_use_inc, snd_opti9xx_use_dec);
	if (!opti9xx_card)
		return -ENOMEM;
	opti9xx_card->type=SND_CARD_TYPE_OPTI9XX;
	if ((error = snd_opti9xx_detect()))
		return error;
	if ((error = snd_opti9xx_setup()))
		return error;
	return snd_opti9xx_attach();
}

int snd_opti9xx_parse_options(void) {
	int i;
#ifdef OPTi9XX_DEBUG
	opti9xx_printk("cdrom: %s i/f, port 0x%x, irq %d, drq %d\n",
		cd_if, cd_port, cd_irq, cd_drq);
	opti9xx_printk("game port %s\n", game);
#endif	/* OPTi9XX_DEBUG */
	for (i = 0; i < 6; i++)
		if (!strcmp(cd_if_table[i].name, cd_if))
			break;
	if (i == 6)
		return -EINVAL;
	cd_if_sel = cd_if_table[i].bits;
	for (i = 0; i < 4; i++)
		if (cd_port_table[i].port == cd_port)
			break;
	if (i == 4)
		return -EINVAL;
	cd_port_sel = cd_port_table[i].bits;
	for (i = 0; i < 6; i++)
		if (cd_irq_table[i].irq == cd_irq)
			break;
	if (i == 6)
		return -EINVAL;
	cd_irq_sel = cd_irq_table[i].bits;
	for (i = 0; i < 4; i++)
		if (cd_drq_table[i].drq == cd_drq)
			break;
	if (i == 4)
		return -EINVAL;
	cd_drq_sel = cd_drq_table[i].bits;
	for (i = 0; i < 2; i++)
		if (!strcmp(game_en_table[i].name, game))
			break;
	if (i == 2)
		return -EINVAL;
	game_sel = game_en_table[i].bits;
	if ((cd_irq != -1) && (cd_irq == snd_irq))
		return -EINVAL;
	if ((cd_drq != -1) && (cd_drq == snd_dma))
		return -EINVAL;
	if (snd_port != SND_AUTO_PORT)
		for (i = 0; wss_valid_ports[i] != SND_AUTO_PORT; i++)
			if (wss_valid_ports[i] == snd_port)
				break;
	if (wss_valid_ports[i] == SND_AUTO_PORT)
		return -EINVAL;
	return ESUCCESS;
}

void cleanup_module(void) {
	if (snd_opl3) {
		snd_synth_unregister(snd_opl3);
		snd_synth_free(snd_opl3);
	}
	if (snd_mixer) {
		snd_mixer_unregister(snd_mixer);
		snd_mixer_free(snd_mixer);
	}
	if (snd_pcm) {
		snd_pcm_unregister(snd_pcm);
		snd_pcm_free(snd_pcm);
	}
	if (opti9xx_card) {
		snd_card_unregister(opti9xx_card);
		snd_card_free(opti9xx_card);
	}
}

int init_module(void) {
	int error;
	if ((error = snd_opti9xx_parse_options()))
		opti9xx_printk("please check module options\n");
	else if ((error = snd_opti9xx_probe()))
		 cleanup_module();
	return error;
}

