/*
 *  Driver for ESS Solo-1 (ES1938, ES1946) soundcard
 *  Copyright (c) by Jaromir Koutek <miri@punknet.cz>,
 *                   Jaroslav Kysela <perex@jcu.cz>
 * 
 * Rewritted from sonicvibes.c source.
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "esssolo1.h"
#include "opl3.h"
#include "initval.h"

#ifndef PCI_VENDOR_ID_ESS
#define PCI_VENDOR_ID_ESS             0x125d
#endif
#ifndef PCI_DEVICE_ID_ESS_ES1938
#define PCI_DEVICE_ID_ESS_ES1938  0x1969
#endif

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;	/* Index 1-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;	/* ID for this card */
#ifdef MODULE_PARM
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for ESS Solo-1 soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for ESS Solo-1 soundcard.");
#endif

struct snd_solo {
	struct snd_pci_dev pci;
	snd_irq_t *irqptr;
	snd_dma_t *dma1ptr;
	snd_dma_t *dma2ptr;
	snd_card_t *card;
	esssolo_t *solo;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *midi_uart;
	snd_synth_t *synth;
};

static struct snd_solo *snd_solo_cards[SND_CARDS] = SND_DEFAULT_PTR;

static void snd_solo_use_inc(snd_card_t * card)
{
	MOD_INC_USE_COUNT;
}

static void snd_solo_use_dec(snd_card_t * card)
{
	MOD_DEC_USE_COUNT;
}

static int snd_solo_detect(snd_card_t * card, struct snd_solo *scard, int pci_index)
{
	int idx;

	if (snd_pci_find_device(PCI_VENDOR_ID_ESS, PCI_DEVICE_ID_ESS_ES1938, pci_index, &scard->pci) < 0)
		return -ENODEV;
	for (idx = 0; idx < 5; idx++) {
		if (scard->pci.base_address[idx] == 0 ||
		    (scard->pci.base_address[idx] & PCI_BASE_ADDRESS_SPACE) != PCI_BASE_ADDRESS_SPACE_IO)
			return -ENODEV;
	}
/*  if ( snd_register_ioport( card, scard -> pci.base_address[ 1 ] & ~PCI_BASE_ADDRESS_SPACE, 0x10, "ESS Solo-1 I/O" ) < 0 ) goto __nodev; */
	if (snd_register_ioport(card, scard->pci.base_address[1] & ~PCI_BASE_ADDRESS_SPACE, 0x10, "ESS Solo-1 SB", NULL) < 0)
		goto __nodev;
	if (snd_register_ioport(card, scard->pci.base_address[2] & ~PCI_BASE_ADDRESS_SPACE, 0x10, "ESS Solo-1 VC (DMA)", NULL) < 0)
		goto __nodev;
	if (snd_register_ioport(card, scard->pci.base_address[3] & ~PCI_BASE_ADDRESS_SPACE, 4, "ESS Solo-1 MIDI", NULL) < 0)
		goto __nodev;
	if (snd_register_ioport(card, scard->pci.base_address[4] & ~PCI_BASE_ADDRESS_SPACE, 4, "ESS Solo-1 Game port", NULL) < 0)
		goto __nodev;
	return 0;
      __nodev:
	snd_unregister_ioports(card);
	return -ENODEV;
}

static void snd_solo_interrupt2(int irq, void *dev_id, struct pt_regs *regs)
{
	struct snd_solo *scard = (struct snd_solo *) dev_id;

	if (!scard || !scard->solo)
		return;
	snd_solo_interrupt(scard->solo);
}

#define SOLO_DMA_SIZE 32

static int snd_solo_resources(snd_card_t * card, struct snd_solo *scard, int dev)
{
	int err;

	if ((err = snd_register_interrupt(card, "ESS Solo-1", scard->pci.irq,
				SND_IRQ_TYPE_PCI, snd_solo_interrupt2, scard,
				NULL, &scard->irqptr)) < 0)
		return err;
	if ((err = snd_register_dma_channel(card, "ESS Solo-1 DMA 1", 0,
					    SND_DMA_TYPE_PCI_16MB,
					    SOLO_DMA_SIZE, NULL, &scard->dma1ptr)) < 0)
		return err;
#if 0
	if ((err = snd_register_dma_channel(card, "ESS Solo-1 DMA 2", 1,
					    SND_DMA_TYPE_PCI_16MB,
					    SOLO_DMA_SIZE, NULL, &scard->dma2ptr)) < 0)
		return err;
#endif
	return 0;
}

static int snd_solo_probe(int dev, struct snd_solo *scard)
{
	snd_card_t *card;
	snd_pcm_t *pcm = NULL;
	snd_kmixer_t *mixer = NULL;
	snd_rawmidi_t *midi_uart = NULL;
	snd_synth_t *synth = NULL;
	int pci_index;

	card = snd_card_new(snd_index[dev], snd_id[dev],
			    snd_solo_use_inc, snd_solo_use_dec);
	if (!card)
		return -ENOMEM;
	card->type = SND_CARD_TYPE_ESS_ES1938;
	card->static_data = scard;
	for (pci_index = 0; pci_index < SND_CARDS; pci_index++) {
		if (!snd_solo_detect(card, scard, pci_index))
			break;
	}
	if (pci_index >= SND_CARDS) {
		snd_card_free(card);
		return -ENODEV;
	}
	if (snd_solo_resources(card, scard, dev) < 0) {
		snd_card_free(card);
		return -ENODEV;
	}
	scard->solo = snd_solo_create(card, &scard->pci, scard->dma1ptr,
				      scard->dma2ptr, scard->irqptr, 0, 0);
	if (!scard->solo)
		goto __nodev;
	mixer = snd_solo_mixer(scard->solo);
	if (!mixer)
		goto __nodev;
	pcm = snd_solo_pcm(scard->solo);
	if (!pcm)
		goto __nodev;
#if 0
	midi_uart = snd_mpu401_uart_new_device(card, MPU401_HW_SONICVIBES,
					       scard->sonic->midi_port,
					       scard->irqptr->irq);
	if (!midi_uart)
		goto __nodev;
	snd_sonicvibes_midi(scard->sonic, (mpu401_t *) midi_uart->private_data);
	synth = snd_opl3_new_device(card, scard->solo->synth_port, scard->solo->synth_port + 2, OPL3_HW_OPL3_SV, 0);
	if (!synth)
		goto __nodev;
#endif
	if (snd_mixer_register(mixer, 0) < 0)
		goto __nodev;
	if (snd_pcm_register(pcm, 0) < 0) {
		snd_mixer_unregister(mixer);
		mixer = NULL;
		goto __nodev;
	}
#if 0
	if (snd_rawmidi_register(midi_uart, 0) < 0) {
		snd_pcm_unregister(pcm);
		pcm = NULL;
		snd_mixer_unregister(mixer);
		mixer = NULL;
		goto __nodev;
	}
	if (snd_synth_register(synth) < 0) {
		snd_rawmidi_unregister(midi_uart);
		midi_uart = NULL;
		snd_pcm_unregister(pcm);
		pcm = NULL;
		snd_mixer_unregister(mixer);
		mixer = NULL;
		goto __nodev;
	}
#endif
	strcpy(card->abbreviation, "Solo-1");
	strcpy(card->shortname, "ESS Solo-1");
	sprintf(card->longname, "%s rev %i, irq %i",
		card->shortname,
		scard->solo->revision,
		scard->irqptr->irq);

	if (!snd_card_register(card)) {
		scard->card = card;
		scard->mixer = mixer;
		scard->pcm = pcm;
		scard->midi_uart = scard->solo->rmidi = midi_uart;
		scard->synth = synth;
		return 0;
	}
	goto __nodev;

      __nodev:
	if (synth)
		snd_synth_free(synth);
	if (midi_uart)
		snd_rawmidi_free(midi_uart);
	if (mixer)
		snd_mixer_free(mixer);
	if (pcm)
		snd_pcm_free(pcm);
	if (scard->solo) {
		snd_solo_free(scard->solo);
		scard->solo = NULL;
	}
	snd_card_free(card);
	return -ENXIO;
}

static int snd_solo_free2(int dev)
{
	struct snd_solo *scard;
	snd_pcm_t *pcm;

	scard = snd_solo_cards[dev];
	snd_solo_cards[dev] = NULL;
	if (scard) {
		snd_card_unregister(scard->card);
		if (scard->synth)
			snd_synth_unregister(scard->synth);
		if (scard->midi_uart) {
			scard->solo->rmidi = NULL;
			snd_rawmidi_unregister(scard->midi_uart);
		}
		if (scard->mixer)
			snd_mixer_unregister(scard->mixer);
		if (scard->pcm) {
			pcm = scard->pcm;
			scard->pcm = NULL;	/* turn off interrupts */
			snd_pcm_unregister(pcm);
		}
		if (scard->solo) {
			snd_solo_free(scard->solo);
			scard->solo = NULL;
		}
		snd_card_free(scard->card);
		snd_free(scard, sizeof(struct snd_solo));
	}
	return 0;
}

int init_module(void)
{
	int dev, cards;
	struct snd_solo *scard;

#ifndef LINUX_2_1
	register_symtab(NULL);
#endif
	for (dev = cards = 0; dev < SND_CARDS; dev++) {
		scard = (struct snd_solo *) snd_malloc(sizeof(struct snd_solo));
		if (!scard)
			continue;
		memset(scard, 0, sizeof(struct snd_solo));
		if (snd_solo_probe(dev, scard) < 0) {
			snd_free(scard, sizeof(struct snd_solo));
			break;
		}
		snd_solo_cards[dev] = scard;
		cards++;
	}
	if (!cards) {
		snd_printk("ESS Solo-1 soundcard #%i not found or device busy\n", dev + 1);
		return -ENODEV;
	}
	return 0;
}

void cleanup_module(void)
{
	int dev;

	for (dev = 0; dev < SND_CARDS; dev++)
		snd_solo_free2(dev);
}
