/*
 * RCF named extensions:
 * (C) Research Computing Facility,
 *     Department of Computer Science
 *     Umass/Amherst
 *
 * Matt Kimmel    (kimmel@oit.umass.edu)
 * Craig I. Hagan (hagan@cih.com)
 *
 * You are free to redistribute this code as long as the above
 * notice remains. Please send up any changes that you make.
 */

/* This module allows the nameserver to give out different aliases for
 * given hostnames, depending on weights assigned in a configuration file.
 * For example, the alias 'www.cs.umass.edu' could actually point to
 * 'www-1.cs.umass.edu', with a weight of 1, and to 'www-2.cs.umass.edu'
 * with a weight of 10.  Then, 'www.cs.umass.edu' would resolve to 
 * 'www-1.cs.umass.edu' on 10 out of 11 requests, and to 'www-2.cs.umass.edu'
 * on 1 out of 11 requests.
 *
 * The configuration file, whose filename is defined in ns_service.h,
 * should have the following format:
 *
 * Blank lines and lines starting with '#' are ignored.
 *
 * service aliasname   // Define alias 'aliasname' at ip port ipport
 * name ipaddr weight
 * name ipaddr weight
 *  .
 *  .
 * end
 *
 * guaranteed aliasname ipport // Define guaranteed alias 'aliasname' at
 * name ipaddr weight          // ip port ipport.  The port is used to
 * name ipaddr weight          // determine if a given host is available
 *  .                          // before returning it.
 *  .
 * end
 *
 * A word on shared memory--
 * This program forks off a separate process to continually check
 * "guaranteed" machines for reachability.  The main named process and
 * this process share a dynamically allocated array of ints, one for
 * each guaranteed machine (plus one extra).  The FIRST value in this
 * array is 0 until the machines have been scanned through once.  The
 * rest of the the values are GURANTEED_HOSTUP if the corresponding 
 * machine is up, and GUARANTEED_HOSTDOWN if it's down.  Each 
 * "guaranteed" machine has an index into the
 * array in its namenode structure.  These are assigned as the config file
 * is parsed.
 */
#include <sys/param.h>
#include <sys/uio.h>
#include <sys/time.h>
#include <sys/file.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#ifndef SHM_W /* sunos doesn't define these unless you are in kernel mode */
#define  SHM_W   0200
#endif SHM_W
#ifndef SHM_R
#define  SHM_R   0400
#endif SHM_R

#include <sys/socket.h>
#include <netinet/in.h>
#include <syslog.h>
#include <arpa/nameser.h>
#include <resolv.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <netdb.h>
#include "named.h"
#include "ns_service.h"

char *SERVICE_FILE="services";

int lookup_service_was_used = 0;
int lookup_service_ttl = -1;

int shmid;  /* Shared memory identifier */
key_t shmkey; /* Shared memory key */
int *uptable; /* Shared up table */
int uptablesize; /* Size of uptable */
struct aliasnode *aliaslist = NULL;

/* local prototypes */
void strtolower(char *);
void lookup_service_addalias(struct aliasnode **,struct aliasnode *);
void lookup_service_addname(struct namenode **,struct namenode *);
void lookup_service_findservice(struct namenode *,char *);
void lookup_service_findguaranteed(struct namenode *,int,char *);
void lookup_service_syscheck();
int lookup_service_hostup(char *,int);

char *lookup_service(char *name)
{
  struct aliasnode *ap;
  char newname[40], *n;

  lookup_service_was_used = 0;

  /*
   * Try to find this name in the alias list.  If we have it, invoke a
   * function to get the proper name and return it.
   */
  ap = aliaslist;
  while(ap != NULL) {
    if(!strcmp(name,ap->alias)) {
      /* The find functions don't touch newname if nothing is found. */
      strcpy(newname,name);
#ifdef USE_GUARANTEED
      if(ap->guaranteed)
	lookup_service_findguaranteed(ap->namelist,ap->ipport,newname);
      else
#endif USE_GUARANTEED
	lookup_service_findservice(ap->namelist,newname);
      n = (char *)malloc(sizeof(char) * (strlen(newname)+1));
      if(n != NULL) {
	strcpy(n,newname);
/*	free(newname);*/
	lookup_service_was_used = 1;
	return(n);
      }
      else
	return(name);
    }
    ap = ap->next;
  }

  /* If we get here, no match was found.  Just return the original name */
  return(name);
}

int lookup_service_init(char *service_file)
{
  FILE *fd;
  char line[255], line2[255];
  char w1[255], w2[255], w3[255], w4[255];
  char *s, *s2;
  struct aliasnode *tmpalias;
  struct namenode *tmpname;
  int index = 1, i;

#ifdef DEBUG
  if(debug)
    fprintf(ddt,"lookup_service_init\n");
#endif

  lookup_service_ttl = zones[0].z_minimum*TTLMUL; 

  /* Parse up the configuration file */
  if((fd = fopen(service_file,"r")) == NULL) {
    fprintf(stderr,"Couldn't open %s!  Aborting!\n",service_file);
    exit(1);
  }

  /* Go through the file one line at a time, parsing. */
  while(fgets(line,255,fd) != NULL) {

    /* First, eliminate the \n */
    s = strchr(line,'\n');
    if(s != NULL)
      *s = '\0';

    /* Skip past any leading whitespace */
    s = line;
    while((*s == ' ') || (*s == '\t'))
      s++;

    /* If it's a blank or a comment, ignore it */
    if((s[0] == '#') || (s[0] == '\0'))
      continue;

    /* Ok, try to parse this line.  It should be either a 'service'
     * or 'guaranteed' statement (which are handled similarly).
     */
    strtolower(s);
    sscanf(s,"%s %s %s",w1,w2,w3);

    if((!strcmp(w1,"service")) || (!strcmp(w1,"guaranteed"))) {
      /* First, malloc a new aliasnode.
       */
      if((tmpalias = (struct aliasnode *)malloc(sizeof(struct aliasnode))) == NULL) {
	perror("malloc");
	exit(1);
      }

      /* Fill it in */
      strcpy(tmpalias->alias,w2);
      if(!strcmp(w1,"guaranteed")) {
	tmpalias->guaranteed = 1;
#ifdef GETSERVBYNAME_NOT_BROKEN
	tmpalias->ipport = getservbyname(w3,"tcp")->s_port;
	printf("got %d\n",tmpalias->ipport);
	if(!(tmpalias->ipport))
#endif  GETSERVBYNAME_NOT_BROKEN
	  tmpalias->ipport = atoi(w3);
      }
      else {
	tmpalias->guaranteed = 0;
	tmpalias->ipport = 0;
      }

      tmpalias->namelist = NULL;
      tmpalias->next = NULL;

      /* Now grab all the names and add them. */
      while(fgets(line2,255,fd) != NULL) {
	/* First, eliminate the \n */
	s2 = strchr(line2,'\n');
	if(s2 != NULL)
	  *s2 = '\0';
	
	/* Skip past any leading whitespace */
	s2 = line2;
	while((*s2 == ' ') || (*s2 == '\t'))
	  s2++;
	
	/* If it's a blank or a comment, ignore it */
	if((s2[0] == '#') || (s2[0] == '\n'))
	  continue;

	strtolower(s2);
	sscanf(s2,"%s %s %s",w1,w2,w3);

	/* Is this the end of the names? */
	if(!strcmp(w1,"end"))
	  break;

	/* Nope, make a new name node and add it */
        if((tmpname = (struct namenode *)malloc(sizeof(struct namenode))) == NULL) {
	  perror("malloc");
	  exit(1);
	}

	strcpy(tmpname->name,w1);
	strcpy(tmpname->ipaddr,w2);
	tmpname->weight = atoi(w3);
	tmpname->accesses = (unsigned long)0;
	tmpname->next = NULL;

	if(tmpalias->guaranteed) {
	  tmpname->upindex = index;
	  index++;
	}
	else
	  tmpname->upindex = 0;

	lookup_service_addname(&(tmpalias->namelist),tmpname);
      }

      /* Ok, we have all the names, now add the alias */
      lookup_service_addalias(&aliaslist,tmpalias);

      /* Go on parsing */
      continue;
    }
    else { /* A bad line */
      fprintf(stderr,"Unknown keyword %s in %s\n",w1,service_file);
      exit(1);
    }
  }

  /* The configuration file has been read. */
  fclose(fd);

#ifdef DEBUG
  if (debug) {
    fprintf(ddt,"Configuration file read, aliases are:\n");
    tmpalias = aliaslist;
    while(tmpalias != NULL) {
      fprintf(ddt,"\n%s guaranteed: %d port %d\n",tmpalias->alias,
	      tmpalias->guaranteed,tmpalias->ipport);
      tmpname = tmpalias->namelist;
      while(tmpname != NULL) {
	fprintf(ddt,"  %s %s %d\n",tmpname->name,tmpname->ipaddr,tmpname->weight);
	tmpname = tmpname->next;
      }
      tmpalias = tmpalias->next;
    }
  }
#endif

#ifdef GUARANTEED_AUTOCHECK
  /* Now create the shared memory if necessary, and start the syscheck process */
  if(index > 1) {
#ifdef DEBUG
    if(debug)
      fprintf(ddt,"Creating and initializing shared memory\n");
#endif

    /* Create shared memory */
    if((shmid = shmget(0,(index*sizeof(int)),(SHM_R | SHM_W | IPC_PRIVATE))) == -1) {
      perror("shmget");
      exit(1);
    }

    /* Attach it to this process */
    if((uptable = (int *)shmat(shmid,NULL,0)) == (int *)-1) {
      perror("shmat");
      shmctl(shmid,IPC_RMID,NULL);
      exit(1);
    }
    
    /* Now initialize it to all 0s */
    for(i=0;i<index;i++)
      uptable[i] = GUARANTEED_HOSTDOWN;

    uptable[0] = GUARANTEED_UNCHECKED; 
    /* just in case someone alters GUARANTEED_HOSTDOWN */

    uptablesize = index - 1;

    /* Finally, start the new process */
#ifdef DEBUG
    if(debug)
      fprintf(ddt,"Starting syscheck process\n");
#endif

    if(fork()==0) { /* Child */
      close(0);
      close(1);
      for(i=3;i<256;i++)
	close(i);

      lookup_service_syscheck();
      exit(0);
    }

    /* Wait until the first syscheck pass is complete */
    while(uptable[0] == GUARANTEED_UNCHECKED)
      sleep(1);

#ifdef DEBUG
    if (debug) {
      fprintf(ddt,"Syscheck done, hosts:\n");
      tmpalias = aliaslist;
      while(tmpalias != NULL) {
	tmpname = tmpalias->namelist;
	while(tmpname != NULL) {
	  if(tmpalias->guaranteed)
	    fprintf(ddt,"%s %d\n",tmpname->name,uptable[tmpname->upindex]);
	  tmpname = tmpname->next;
	}
	tmpalias = tmpalias->next;
      }
    }
#endif
  }
#endif GUARANTEED_AUTOCHECK
}

/* Convert a string to lowercase. */
void strtolower(char *s)
{
  while((s != NULL) && (*s != NULL)) {
    *s = tolower(*s);
    s++;
  }
}

/* Add an alias to a list of aliases */
void lookup_service_addalias(struct aliasnode **head,struct aliasnode *node)
{
  struct aliasnode *ap;

  /* We do this the slow way to insure that order is maintained */
  if(*head == NULL) {
    *head = node;
    node->next = NULL;
    return;
  }

  ap = *head;
  while(ap->next != NULL)
    ap = ap->next;
  ap->next = node;
  node->next = NULL;
}

/* Add a name to a list of names */
void lookup_service_addname(struct namenode **head,struct namenode *node)
{
  struct namenode *np;

  /* We do this the slow way to insure that order is maintained */
  if(*head == NULL) {
    *head = node;
    node->next = NULL;
    return;
  }

  np = *head;
  while(np->next != NULL)
    np = np->next;
  np->next = node;
  node->next = NULL;
}

/* Find the proper name to return for a "service" alias */
void lookup_service_findservice(struct namenode *namelist,char *newname)
{
  struct namenode *np, *leastused;

  /* Find the least-used name in the list.  If there is a tie, the last
   * name found that ties will be used.
   */
  leastused = NULL;
  np = namelist;
  while(np != NULL) {
    if(leastused == NULL) {
      leastused = np;
      np = np->next;
      continue;
    }
    if(np->accesses < leastused->accesses)
      leastused = np;
    np = np->next;
  }

  if(leastused == NULL) /* We didn't find anything at all */
    return;

  /* Make sure that adding this node's weight to its accesses won't
   * roll over accesses.  If it won't, add it.  If it will, zero ALL
   * accesses.
   */
  if((leastused->accesses + (unsigned long)leastused->weight) > leastused->accesses)
    leastused->accesses += (unsigned long)leastused->weight;
  else {
    np = namelist;
    while(np != NULL) {
      np->accesses = (unsigned long)0;
      np = np->next;
    }
  }

  /* Now copy the name and return */
  strcpy(newname,leastused->name);
  return;
}

void lookup_service_findguaranteed(struct namenode *namelist,int ipport,char *newname)
{
  struct namenode *np, *leastused;

  /* Find the least-used name in the list.  If there is a tie, the last
   * name found that ties will be used.  The host must be up to be considered!
   */
  leastused = NULL;
  np = namelist;

  while(np != NULL) {

#ifdef GUARANTEED_AUTOCHECK
    if(uptable[np->upindex] == GUARANTEED_HOSTDOWN) {
      np = np->next;
      continue;
    }
#endif GUARANTEED_AUTOCHECK

#ifdef GUARANTEED_STRICT
    if((leastused == NULL) 
       && (lookup_service_hostup(np->ipaddr,ipport) == GUARANTEED_HOSTUP)) {
      leastused = np;
    } else
#else
    if(leastused == NULL) {
      leastused = np;
    } else
#endif GUARANTEED_STRICT
#ifdef GUARANTEED_STRICT
      if((np->accesses < leastused->accesses) 
	 && (lookup_service_hostup(np->ipaddr,ipport) == GUARANTEED_HOSTUP)) {
	leastused = np;
	}
#else
      if(np->accesses < leastused->accesses) {
         leastused = np;
      }
#endif GUARANTEED_STRICT
      np = np->next;
  } /* end while */
  
  if(leastused == NULL) { /* We didn't find anything at all */
    /* Return the first address in the list, if possible */
    if(namelist != NULL)
      strcpy(newname,namelist->name);
    return;
  }

  /* Make sure that adding this node's weight to its accesses won't
   * roll over accesses.  If it won't, add it.  If it will, zero ALL
   * accesses.
   */
  if((leastused->accesses + (unsigned long)leastused->weight) > leastused->accesses)
    leastused->accesses += (unsigned long)leastused->weight;
  else {
    np = namelist;
    while(np != NULL) {
      np->accesses = (unsigned long)0;
      np = np->next;
    }
  }

  /* Now copy the name and return */
  strcpy(newname,leastused->name);
  return;
}

/* This function runs asynchronously and keeps the uptable up to date */
void lookup_service_syscheck()
{
  struct aliasnode *ap;
  struct namenode *np;
  int rc;

  while(1) {
    ap = aliaslist;
    while(ap != NULL) {
      if(ap->guaranteed) {
	np = ap->namelist;
	while(np != NULL) {
#ifdef GUARANTEED_PESSIMISTIC
	  uptable[np->upindex] = GUARANTEED_HOSTDOWN;
#endif
  if(lookup_service_hostup(np->ipaddr,ap->ipport) == GUARANTEED_HOSTUP) 
	    uptable[np->upindex] = GUARANTEED_HOSTUP;
	  else
	    uptable[np->upindex] = GUARANTEED_HOSTDOWN;
	  np = np->next;
	}
      }
      ap = ap->next;
    }
    uptable[0] = GUARANTEED_CHECKED;
    sleep(GUARANTEED_AUTOCHECK_PERIOD);
  }
}

/* rc = hostup(hostname)
 * int rc;
 * char *hostname;
 *
 * Check to see if the ip address specified in hostip is up.  Return nonzero if
 * it is, zero if it isn't.  Attempts to connect to specified port.  Only
 * works for TCP ports!!
 *
 * Remember, it's your environment--recycle your code!
 */
int lookup_service_hostup(char *hostip,int ipport)
{
  int sd;
  pid_t child,grandchild;
  int *status;
  struct sockaddr_in addr;

  /* Open socket */
  if((sd = socket(AF_INET,SOCK_STREAM,0)) < 0) {
#ifdef notdef
    perror("socket");
#endif
    return(GUARANTEED_HOSTDOWN);
  }

  /* Set up addr structure */
  bzero((char *)&addr,sizeof(addr));
  addr.sin_family = AF_INET;
  addr.sin_addr.s_addr = inet_addr(hostip);
  addr.sin_port = htons(ipport);

  /* Try to connect; return 0 if unsuccessful or 1 if successful */
  if(connect(sd,(struct sockaddr *)&addr,sizeof(addr)) < 0)
    {
      return(GUARANTEED_HOSTDOWN);
    }
  close(sd);
  return(GUARANTEED_HOSTUP);
}
