/*
 * Initialize an MSDOS diskette.  Read the boot sector, and switch to the
 * proper floppy disk device to match the format on the disk.  Sets a bunch
 * of global variables.  Returns 0 on success, or 1 on failure.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

/*#define volatile unsigned*/
#include <sys/types.h>			/* fix silly DEC include files, without
					 * breaking them for sun*/
#undef volatile

#include <sys/stat.h>
#include <errno.h>
#include <string.h>
#ifdef BSD
#include <sys/time.h>
#else /* BSD */
#include <time.h>
#endif /* BSD */
#include <sys/param.h>
#include <fcntl.h>
#include "msdos.h"

#ifdef USE_2M
#include <linux/fd.h>
#endif

extern int errno;

#ifdef NO_STRERROR
/* fixup braindead include philes */
extern char *sys_errlist[];
#define strerror(x) ( sys_errlist[x] )
#endif

#define FULL_CYL
#define WORD(x) ((boot->x)[0] + ((boot->x)[1] << 8))
#define DWORD(x) ((boot->x)[0] + ((boot->x)[1] << 8) + ((boot->x)[2] << 16) + ((boot->x)[3] << 24))

unsigned int num_clus;			/* total number of cluster */
int num_fat;				/* the number of FAT tables */
long disk_offset;			/* skip this many bytes */
int fat_bits;				/* the FAT encoding scheme */
int is_2m;

#ifdef USE_SERIAL
int serialized;
unsigned long serial_number;
#endif
 
static struct bootsector *read_boot();
static int lock_dev();

int
init(drive, mode)
char drive;
int mode;
{
#ifdef USE_2M
	int BootP, Infp0, InfpX, InfTm, use_2m;
#endif

#ifdef MULTISIZE
	int ssize;
#endif
	unsigned int tot_sectors;
	unsigned int fat_start, tracks, heads, sectors;
	int old_dos;
	char *name;

	struct bootsector *boot;
	struct device *dev;
	char errmsg[80];

	fat_start = tracks = heads = sectors = old_dos = 0;
	boot = 0;

	sprintf(errmsg, "Drive '%c:' not supported", drive);	
					/* open the device */
	for (dev=devices; dev->name; dev++) {
		if (fd != -1) {
			close(fd);
			if(fat_buf) free((char *) fat_buf);
			if(disk_buf) free((char *) disk_buf);
			if(dir_buf) free((char *) dir_buf);
			fat_buf = disk_buf = dir_buf = 0;
		}
		if (dev->drive != drive)
			continue;
		
		name = expand(dev->name);

#ifdef O_SYNC
                if(getenv("MTOOLS_USE_SYNC"))
			fd = open(name, mode | dev->mode | O_SYNC | O_EXCL);
		else
#endif
			fd = open(name, mode | dev->mode | O_EXCL);

		if (fd < 0) {
			sprintf(errmsg,"init: open: \"%s\"", strerror(errno));
			continue;
		}
                                        /* lock the device on writes */
                if (mode == O_RDWR && lock_dev(fd)) {
                        fprintf(stderr, "Device \"%s\" is busy\n", dev->name);
                        exit(1);
                }

#ifdef MULTISIZE
		ssize=0x80;
#endif

#ifdef USE_2M
		use_2m=0;
#endif

					/* set default parameters, if needed */
		if ((dev->gioctl) &&
		    (*(dev->gioctl)) (fd, dev->tracks,
				      dev->heads,dev->sectors PAR_SIZE PAR_2M)){
			sprintf(errmsg,"init: set default params");
			continue;
		}

					/* read the boot sector */
		disk_offset = dev->offset;
		if ((boot = read_boot()) == NULL){
			sprintf(errmsg,"init: read boot");
			continue;
		}
		
		msector_size = WORD(secsiz);
		heads = WORD(nheads);
		sectors = WORD(nsect);
		if (heads && sectors)
			tracks = WORD(psect) / (heads * sectors);

					/* sanity checking */
		old_dos = 0;
		if (!heads || heads > 100 || !sectors || sectors > 500 ||
		    tracks > 5000 || !boot->clsiz) {
			/*
			 * The above technique will fail on diskettes that
			 * have been formatted with very old MSDOS, so we
			 * resort to the old table-driven method using the
			 * media signature (first byte in FAT).
			 */
			unsigned char temp[MAX_SECTOR];
			if (read(fd, (char *) temp, msector_size) != 
			    msector_size)
				temp[0] = '0';

			switch (temp[0]) {
				case 0xfe:	/* 160k */
					tracks = 40;
					sectors = 8;
					heads = 1;
					dir_start = 3;
					dir_len = 4;
					clus_size = 1;
					fat_len = 1;
					num_clus = 313;
					break;
				case 0xfc:	/* 180k */
					tracks = 40;
					sectors = 9;
					heads = 1;
					dir_start = 5;
					dir_len = 4;
					clus_size = 1;
					fat_len = 2;
					num_clus = 351;
					break;
				case 0xff:	/* 320k */
					tracks = 40;
					sectors = 8;
					heads = 2;
					dir_start = 3;
					dir_len = 7;
					clus_size = 2;
					fat_len = 1;
					num_clus = 315;
					break;
				case 0xfd:	/* 360k */
					tracks = 40;
					sectors = 9;
					heads = 2;
					dir_start = 5;
					dir_len = 7;
					clus_size = 2;
					fat_len = 2;
					num_clus = 354;
					break;
				default:
					fprintf(stderr, "Probable non-MSDOS disk\n");
					close(fd);
					fd = -1;
					return(1);
			}
			fat_start = 1;
			num_fat = 2;
			old_dos = 1;
		}
					/* check the parameters */
		if (dev->tracks && !dev->gioctl) {
			if (dev->tracks == tracks && dev->heads == heads && dev->sectors == sectors)
				break;
			else
				sprintf(errmsg, "No support for %d tracks, %d heads, %d sector diskettes\n", tracks, heads, sectors);
				continue;
		}


#ifdef USE_SERIAL
 		serialized = boot->dos4 == 0x29;
 		serial_number = *((long *)&(boot->serial));	
#endif
 
#ifdef USE_2M
		ssize = 0x80;
		use_2m = 1;
		BootP = WORD(BootP);
		Infp0 = WORD(Infp0);
		InfpX = WORD(InfpX);
		InfTm = WORD(InfTm);

		if (strncmp( boot->banner,"2M", 2 ) == 0 &&
		    BootP < 512 && Infp0 < 512 && InfpX < 512 && InfTm < 512 &&
		    BootP >= InfTm + 2 && InfTm >= InfpX && InfpX >= Infp0 && 
		    Infp0 >= 76 ){
			ssize = boot->junk[InfTm - 76];
			if ( ssize < 0 || ssize > 7 )
				ssize = 0x80;
			else
				use_2m= FD_2M;
		}
#endif

					/* set new parameters, if needed */
		if ((dev->gioctl) &&
		    (*(dev->gioctl)) (fd, tracks, heads, 
				      sectors * msector_size / 512
				      PAR_SIZE PAR_2M)){
			sprintf(errmsg, "Can't set disk parameters: %s", 
				strerror(errno));
			close(fd);
			fd = -1;
			continue;
		}
		break;

	}

					/* print error msg if needed */	
	if ( dev->drive == 0 ){
		fprintf(stderr,"%s\n",errmsg);
		return(1);
	}

#ifdef USE_2M
	is_2m = use_2m;
#endif

	/*
	 * all numbers are in sectors, except num_clus (which is in clusters)
	 */
	if (!old_dos) {
		clus_size = boot->clsiz;
		fat_start = WORD(nrsvsect);
		fat_len = WORD(fatlen);
		dir_start = fat_start + (boot->nfat * fat_len);
		dir_len = WORD(dirents) * MDIR_SIZE / msector_size;
		/*
		 * For DOS partitions > 32M
		 */
		if (WORD(psect) == 0)
			tot_sectors = DWORD(bigsect);
		else
			tot_sectors = WORD(psect);
		if ( !tot_sectors ){
			/* this should only happen on hard disks, so let's
			 * assume a 16 bit fat for now */
			tot_sectors = dir_start + dir_len + 
				fat_len * msector_size / 2 * clus_size;
			tot_sectors -= (tot_sectors + DWORD(nhs)) % 
				(sectors * heads);
			fprintf(stderr,
				"Warning: disk size unknown, assuming %d mbytes\n", 
				tot_sectors * msector_size / 1024 / 1024);
		}
		num_clus = (tot_sectors - dir_start - dir_len) / clus_size;
		if(!getenv("MTOOLS_SKIP_CHECK") && tot_sectors % sectors){
			fprintf(stderr,"Total number of sectors not a multiple of sectors per track!\n");
			exit(1);
		}

		num_fat = boot->nfat;
	}
					/* more sanity checking */
	if (clus_size * msector_size > MAX_CLUSTER) {
		fprintf(stderr, "Cluster size of %d is larger than max of %d\n",
			clus_size * msector_size, MAX_CLUSTER);
		close(fd);
		fd = -1;
		return(1);
	}
	if (!old_dos && WORD(secsiz) != msector_size) {
		fprintf(stderr, "Sector size of %d is not supported\n", WORD(secsiz));
		close(fd);
		fd = -1;
		return(1);
	}
					/* full cylinder buffering */
#ifdef FULL_CYL
	disk_size = (dev->tracks) ? (sectors * heads) : 1;
#else /* FULL_CYL */
	disk_size = (dev->tracks) ? sectors : 1;
#endif /* FULL_CYL */
  
#ifdef DELL201
	/*
	 * The driver in Dell's SVR4 v2.01 is unreliable with large writes.
	 */
        disk_size = 1;
#endif /* DELL201 */

#ifdef linux
	if ( disk_size == 1 )
		disk_size = sectors * heads;
#endif
	if ( disk_size % 2 && disk_size != 1)
		disk_size *= 2;
	disk_buf = (unsigned char *) malloc(disk_size * msector_size);
	if (disk_buf == NULL) {
		perror("init: malloc");
		exit(1);
	}
					/* read the FAT sectors */
	disk_current = -1000L;
	disk_dirty = 0;
	fat_error = 0;
	fat_bits = dev->fat_bits;
	if (fat_bits == -12)
		fat_bits = 12;
	else if (fat_bits == -16)
		fat_bits = 16;
	else if (strncmp (boot->fat_type, "FAT12   ", 8)==0)
		fat_bits = 12;
	else if (strncmp (boot->fat_type, "FAT16   ", 8)==0)
		fat_bits = 16;
	else if (fat_bits == 0 ){
		if (num_clus > FAT12)
			fat_bits = 16;
		else
			fat_bits = 12;		
	} else if ((fat_bits == 12 && num_clus > FAT12) || 
		   (fat_bits == 16 && num_clus <= FAT12 )
		   || (fat_bits != 12 && fat_bits != 16)) {
		fprintf(stderr,"%d bit FAT on %c: sure ? (Use -%d in \
the device config file to bypass.)\n",fat_bits,drive,fat_bits);
		exit(1);
	}

	fat_read(fat_start);
					/* set dir_chain[] to root directory */
	dir_dirty = 0;
	reset_chain(NEW);
	return(0);
}

/*
 * Fix the info in the MCWD file to be a proper directory name.  Always
 * has a leading separator.  Never has a trailing separator (unless it is
 * the path itself).
 */

char *
fix_mcwd()
{
	FILE *fp;
	struct stat sbuf;
	char *s, *mcwd_path;
	char buf[BUFSIZ], *file;
	static char ans[MAX_PATH];
	long now;
	char mcwd_file[MAX_PATH + sizeof("/.mcwd")];
	char *homedir;

	mcwd_path = getenv("MCWD");
	if (mcwd_path == NULL || *mcwd_path == '\0'){
		homedir= get_homedir();
		if ( homedir ){
			strncpy(mcwd_file, homedir, MAXPATHLEN);
			mcwd_file[MAXPATHLEN]='\0';
			strcat( mcwd_file, "/.mcwd");
			file = mcwd_file;
		} else
			file = "/tmp/.mcwd";
	} else
		file = expand(mcwd_path);
	if (stat(file, &sbuf) < 0)
		return("A:/");
	/*
	 * Ignore the info, if the file is more than 6 hours old
	 */
	time(&now);
	if (now - sbuf.st_mtime > 6 * 60 * 60) {
		fprintf(stderr, "Warning: \"%s\" is out of date, contents ignored\n", file);
		return("A:/");
	}
	
	if (!(fp = fopen(file, "r")))
		return("A:/");

	if (!fgets(buf, BUFSIZ, fp))
		return("A:/");

	buf[strlen(buf) -1] = '\0';
	fclose(fp);
					/* drive letter present? */
	s = buf;
	if (buf[0] && buf[1] == ':') {
		strncpy(ans, buf, 2);
		ans[2] = '\0';
		s = &buf[2];
	}
	else 
		strcpy(ans, "A:");
					/* add a leading separator */
	if (*s != '/' && *s != '\\') {
		strcat(ans, "/");
		strcat(ans, s);
	}
	else
		strcat(ans, s);
					/* translate to upper case */
	for (s = ans; *s; ++s) {
		if (islower(*s))
			*s = toupper(*s);
		if (*s == '\\')
			*s = '/';
	}
					/* if only drive, colon, & separator */
	if (strlen(ans) == 3)
		return(ans);
					/* zap the trailing separator */
	if (*--s == '/')
		*s = '\0';
	return(ans);
}

/*
 * Read the boot sector.  We glean the disk parameters from this sector.
 */

static struct bootsector *
read_boot()
{
	/*static struct bootsector boot;*/
	static char boot[512];

	if (lseek(fd, disk_offset, 0) < 0)
		return(NULL);
					/* read the first sector, or part of
					 * it */
	if (read(fd, boot, 512) != 512)
		return(NULL);

	return (struct bootsector *) boot;
}

/*
 * Create an advisory lock on the device to prevent concurrent writes.
 * Uses either lockf, flock, or fcntl locking methods.  See the Makefile
 * and the Configure files for how to specify the proper method.
 */

static int
lock_dev(fd)
int fd;
{
#ifdef LOCKF
#include <unistd.h>

	if (lockf(fd, F_TLOCK, 0) < 0)
		return(1);
#endif /* LOCKF */

#ifdef FLOCK
#include <sys/file.h>

	if (flock(fd, LOCK_EX|LOCK_NB) < 0)
		return(1);
#endif /* FLOCK */

#ifdef FCNTL
#include <fcntl.h>
	struct flock flk;

	flk.l_type = F_WRLCK;
	flk.l_whence = 0;
	flk.l_start = 0L;
	flk.l_len = 0L;

	if (fcntl(fd, F_SETLK, &flk) < 0)
		return(1);
#endif /* FCNTL */
	return(0);
}
