/*----------------------------------------------------------------------
 *
 * Program:     dcf77clock
 *
 * File:  	parse.c   Extract the date and time information of
 *                        the data stream, received from the DCF 77
 *			  standard time transmitter Mainflingen near
 *			  Frankfurt/Main Germany.
 * 		
 *
 * Edition	History:  
 *                                                                
 * #    Date        			Comments                   by  
 * -- --------  ----------------------------------------------    ---- 
 * 01 02/05/94	Created                                           wpsk
 *
 * Last Edition:   9. May 1994
 *
 *----------------------------------------------------------------------*/

/*
 * 	The DCF77 time code
 *
 *
 *	SECOND		MEANING
 *
 *
 * 	 0 - 14		unused (always zero)
 *	15		alternate transmitter
 *	16		expect zone change (1 hour before)
 *	17              MEZ  (MET) 	    winter time
 *	18              MESZ (MED, MET DST) summer time
 *	19		leap insertion/deletion (1 hour before)
 *	20		start of time code (always 1)
 *	21 - 24		 1 Minutes
 *	25 - 27		10 Minutes
 *	28		Minute Parity
 *	29 - 32		 1 Hours
 *	33 - 34        	10 Hours
 *	35		Hour Parity
 *	36 - 39		 1 Days
 *	40 - 41		10 Days
 *	42 - 44		Day of week  Sunday .... Saturday
 *	45 - 49 	 1 Month
 *	50              10 Months
 *	51 - 53	50-53	 1 Years
 *	54 - 57		10 Years
 *	58		Date Parity
 *	59		Minute ready, except for leap insertion
 *
 *	all data are binary coded decimal
 */


#include <stdio.h>
#include <strings.h>
#include "clock.h"

#define	Summer 		17	/* MET DST   	*/
#define Winter 		18	/* MET       	*/

#define DataStart	20

#define Min1Start	21
#define Min1End		24
#define Min10Start	25
#define Min10End	27
#define MinParity	28

#define Hour1Start	29 
#define Hour1End	32
#define Hour10Start	33
#define Hour10End	34
#define HourParity	35

#define Day1Start	36
#define Day1End		39
#define Day10Start	40
#define Day10End	41

#define WeekDayStart	42
#define WeekDayEnd	44


#define Month1Start	45
#define Month1End	48
#define Month10		49

#define Year1Start	50
#define Year1End	53
#define Year10Start	54
#define Year10End	57
#define DateParity	58

struct  dcfsdata {
	char smin[3];
	char shour[3];
	char sday[3];
	char sweekday[2];
	char smonth[3];
	char syear[5];
} dcfs;

struct  dcfidata {
	int imin;
	int ihour;
	int iday;
	int iweekday;
	int imonth;
	int iyear;
} dcfi;

char *Month[] = {
	" ",
	"January",
	"February",
	"March",
	"April",	
	"May",
	"June",
	"July",
	"August",
	"September",
	"October",
	"November",
	"December",
	""
};

char *DayOfWeek[] = {
	" ",
	"Mon",
	"Tues",
	"Wednes",
	"Thurs",
	"Friy",
	"Sater",
	"Sun",
	""
};

char datastr[STRLEN];

extern int xmax, ymax;
extern int digitsize;	/* size of big digit from file bignum.c */

ParseDCFdata(buf)
char buf[];
{
REG int i, j; 
char *ptr;


	/*
	 *  Clear all bytes of the data structure
	 */

	ptr = (char *) &dcfs;
	for(i = 0; i < sizeof(dcfs); i++)
		*ptr++ = 0;

	ptr = (char *) &dcfi;
	for(i = 0; i < sizeof(dcfi); i++)
		*ptr++ = 0;		

	parity_flg = TRUE;

	/*
	 *  Parity check and other error check possibility's
	 */

	if(!CheckParity(buf, Min1Start, MinParity))	/* Minutes Parity */
		parity_flg = FALSE;

	if(!CheckParity(buf, Hour1Start, HourParity))	/* Hour Parity */
		parity_flg = FALSE;

	if(!CheckParity(buf, Day1Start, DateParity))	/* Date Parity */
		parity_flg = FALSE;

	if((buf[Summer] == '1') && (buf[Winter] == '1')) /* only 1 state allowed */
		parity_flg = FALSE;

	if(buf[DataStart] != '1' )			 /* must allway '1'	 */
		parity_flg = FALSE;
		

	if(parity_flg) {
		success++;
		goxy(xmax - 6,2);
		bgc(BLUE);
		printf("OK   ");
		fflush(stdout);
		goxy(xmax - 6,4);
		printf("%d",success);
		fflush(stdout);
		bgc(WHITE);

		/*---- Build Minute ----*/

		for(i = Min1Start,  j = 1; i <= Min1End ; i++ ) {
			if(buf[i] ==  '1') {
				dcfs.smin[1] +=  j; 
			}
			j <<= 1 ;
		}
		dcfs.smin[1] += 0x30;
	
		for(i = Min10Start, j = 1; i <= Min10End ; i++ ) {
			if(buf[i] ==  '1') {
				dcfs.smin[0] +=  j; 
			}
			j <<= 1 ;
		}
		dcfs.smin[0] += 0x30;
		dcfi.imin = atoi(dcfs.smin);

		/*---- Build Hour ----*/
	
		for(i = Hour1Start, j = 1; i <= Hour1End ; i++ ) {
			if(buf[i] ==  '1') {
				dcfs.shour[1] +=  j; 
			}
			j <<= 1 ;
		}
		dcfs.shour[1] += 0x30;

		for(i = Hour10Start, j = 1; i <= Hour10End ; i++ ) {
			if(buf[i] ==  '1') {
				dcfs.shour[0] +=  j; 
			}
			j <<= 1 ;
		}
		dcfs.shour[0] += 0x30;
		dcfi.ihour = atoi(dcfs.shour);

		/*---- Build Day ----*/
	
		for(i = Day1Start, j = 1; i <= Day1End ; i++ ) {
			if(buf[i] ==  '1') {
				dcfs.sday[1] +=  j; 
			}
			j <<= 1 ;
		}
		dcfs.sday[1] += 0x30;
	
		for(i = Day10Start, j = 1; i <= Day10End ; i++ ) {
			if(buf[i] ==  '1') {
				dcfs.sday[0] +=  j; 
			}
			j <<= 1 ;
		}
		dcfs.sday[0] += 0x30;
		dcfi.iday = atoi(dcfs.sday);
	
		/*---- Build Day of Week ----*/
	
		for(i = WeekDayStart, j = 1; i <= WeekDayEnd ; i++ ) {
			if(buf[i] ==  '1') {
				dcfs.sweekday[0] +=  j; 
			}
			j <<= 1 ;
		}
		dcfs.sweekday[0] += 0x30;
		dcfi.iweekday = atoi(dcfs.sweekday);

		/*---- Build Month ----*/
		
		for(i = Month1Start, j = 1; i <= Month1End ; i++ ) {
			if(buf[i] ==  '1') {
				dcfs.smonth[1] +=  j; 
			}
			j <<= 1 ;
		}
		dcfs.smonth[1] += 0x30;	
		dcfs.smonth[0] += buf[Month10];
		dcfi.imonth = atoi(dcfs.smonth);


		/*---- Build Year ----*/
	
		for(i = Year1Start, j = 1; i <= Year1End ; i++ ) {
			if(buf[i] ==  '1') {
				dcfs.syear[1] +=  j; 
			}
			j <<= 1 ;
		}
		dcfs.syear[1] += 0x30;
		
		for(i = Year10Start, j = 1; i <= Year10End ; i++ ) {
			if(buf[i] ==  '1') {
		 		dcfs.syear[0] +=  j; 
			}
			j <<= 1 ;
		}
		dcfs.syear[0] += 0x30;
		dcfi.iyear = atoi(dcfs.syear);

		/*
		 *  Build the argument for the 'date' command
		 */	

		sprintf(date,"%s%s%s%s%s.00", dcfs.smonth, dcfs.sday, dcfs.shour, dcfs.smin, dcfs.syear);

		sprintf(datastr," %sday  %s   %d.%d.19%s   %s ",
			  DayOfWeek[dcfi.iweekday],  Month[dcfi.imonth],  dcfi.iday,
			  dcfi.imonth,  dcfs.syear ,(buf[Summer] == '1') ? "MET DST": "MET");
		datamsg(datastr, 4, WHITE, RED);
		sprintf(msgstr,"%s:%s", dcfs.shour, dcfs.smin);
		putbnum(msgstr,half_x - 24 , (win2end + digitsize + (ymax - win2end)) /2);
	}
	else {
		success = 0;
		goxy(xmax - 6,2);
		bgc(BLUE);
		printf("ERROR");
		fflush(stdout);
		goxy(xmax - 6,4);
		printf("0");
		fflush(stdout);
		clearline(datastr,4);
		datamsg("  CLOCK  HAS  READ  BAD  DATA   ", 4, WHITE, RED);
		putbnum("00:00",half_x - 24 , (win2end + digitsize + (ymax - win2end)) /2);
		fflush(stdout);
	}
}

/*--------------------------------------------------
 * Check the given range of 'buf' for EVEN parity.
 * Return TRUE if parity is ok.
 *--------------------------------------------------*/
 
CheckParity(buf, FirstByte, ParityByte)
char *buf;
int  FirstByte, ParityByte;
{
REG char *parity, *tstr;
REG int i, sum; 

	sum    = 0;
	parity = buf + ParityByte;
	tstr   = buf + FirstByte;

	for( i = 0 ; i <= (ParityByte - FirstByte) ; i++ ) {
		if(*tstr == '1')
			sum ^= *tstr;
		tstr++; 
    	}
	if(sum)
		return(FALSE);
	else 	return(TRUE);
}
