/*

  rt.c

  Runs process on realtime priority.

  $Id$

  */

# include <stdio.h>
# include <sched.h>
# include <unistd.h>
# include <stdlib.h>
# include <errno.h>

void Usage( void )
{
	fprintf( stderr,
			 "Experimental Linux real time scheduling client, v1.0\n"
			 "Copyright (c) 1996, Boris Tobotras <boris@xtalk.msk.su>\n"
			 "\n"
			 "This program is free software; you can redistribute it and/or modify\n"
			 "it under the terms of the GNU General Public License as published by\n"
			 "the Free Software Foundation; either version 2 of the License, or\n"
			 "(at your option) any later version.\n"
			 "\n"
			 "This program is distributed in the hope that it will be useful,\n"
			 "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
			 "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
			 "GNU General Public License for more details.\n"
			 "\n"
			 "You should have received a copy of the GNU General Public License\n"
			 "along with this program; if not, write to the Free Software\n"
			 "Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n"
			 "\n\n"
			 "Usage: rt [options] command\n"
			 "\n"
			 "Recognized options are:\n"
			 "\t-r   Use SCHED_RR scheduling policy (default);\n"
			 "\t-f   Use SCHED_FIFO scheduling policy;\n"
			 "\t-p N Set priority level N. If N<0, set priority level\n"
			 "\t      to sched_get_priority_min(POLICY)-N.\n"
			 "\t      Default is (sched_get_priority_max()+sched_get_priority_min())/2;\n"
			 "\t--   End of options;\n"
			 "\t-v   Be verbose;\n"
			 "\t-h   This help screen.\n"
		);
}

main( int argc, char *argv[] )
{
	struct sched_param param;
	int opt, policy = SCHED_RR, priority = 0, verbose = 0;

	if ( argc == 1 ) {
		Usage();
		return 1;
	}

	while ( ( opt = getopt( argc, argv, "frp:hv" ) ) != EOF ) {
		switch ( opt ) {
		case 'f':
			policy = SCHED_FIFO;
			break;
		case 'r':
			policy = SCHED_RR;
			break;
		case 'p':
			priority = atoi( optarg );
			break;
		default:
		case 'h':
			Usage();
			return 1;
		case 'v':
			verbose = 1;
			break;
		}
	}

	if ( optind >= argc ) {
		Usage();
		return 1;
	}

	
	{
		int prio_min = sched_get_priority_min( policy );
		int prio_max = sched_get_priority_max( policy );
		if ( priority == 0 )
			priority = ( prio_min + prio_max ) / 2;
		else {
			if ( priority < 0 )
				priority = prio_min - priority;
			if ( priority < prio_min || priority > prio_max ) {
				fprintf( stderr, "Bad priority value, should be %d..%d\n",
						 prio_min, prio_max );
				return 1;
			}
		}

	}

	param.sched_priority = priority;
	
	if ( verbose )
		fprintf( stderr, "Starting %s with policy %s, priority %d.\n",
				argv[ optind ], policy == SCHED_RR ? "SCHED_RR" :
				"SCHED_FIFO", priority );

	if ( sched_setscheduler( getpid(), policy, &param ) ) {
		perror( "sched_setscheduler" );
		return 1;
	}
	
	execvp( argv[ optind ], argv + optind );
	fprintf( stderr, "Can't exec %s: %s\n", argv[ optind ],
			 sys_errlist[ errno ] );
	return 1;
}
				

