/* Copyright (c) 1997 Thorsten Kukuk

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this program; see the file COPYING.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <pwd.h>
#include <netdb.h>
#include <ctype.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <rpc/des_crypt.h>
#include <rpc/key_prot.h>
#include <rpcsvc/nis.h>
#include <rpcsvc/nislib.h>
#include <rpcsvc/nispasswd.h>

extern struct passwd *_nis_getpwuid (uid_t uid);
extern int key_get_conv (char *pkey, des_block *deskey);

static const char *
npderr2str (nispasswd_code error)
{
  switch (error)
    {
    case NPD_NOTMASTER:
      return "Server is not master of this domain";
    case NPD_NOSUCHENTRY:
      return "No passwd entry exists for this user";
    case NPD_IDENTINVALID:
      return "Identifier invalid";
    case NPD_NOPASSWD:
      return "No password stored";
    case NPD_NOSHDWINFO:
      return "No shadow information stored";
    case NPD_SHDWCORRUPT:
      return "Shadow information corrupted";
    case NPD_NOTAGED:
      return "Passwd has not aged sufficiently";
    case NPD_CKGENFAILED:
      return "Common key could not be generated";
    case NPD_VERFINVALID:
      return "verifier mismatch";
    case NPD_PASSINVALID:
      return "All auth attempts incorrect";
    case NPD_ENCRYPTFAIL:
      return "Encryption failed";
    case NPD_DECRYPTFAIL:
      return "Decryption failed";
    case NPD_KEYSUPDATED:
      return "New key-pair generated for user";
    case NPD_KEYNOTREENC:
      return "Could not reencrypt secret key";
    case NPD_PERMDENIED:
      return "Permission denied";
    case NPD_SRVNOTRESP:
      return "Server not responding";
    case NPD_NISERROR:
      return "NIS+ server error";
    case NPD_SYSTEMERR:
      return "System error";
    case NPD_BUFTOOSMALL:
      return "Buffer too small";
    case NPD_INVALIDARGS:
      return "Invalid args to function";
    default:
      return "Unknown error!";
    }
}

static int
verifypassword (const char *oldpwd, const char *pwdstr, const char *user,
		uid_t gotuid)
{
  /* this function will verify the user's password
     for some silly things.
     return values: 0 = not ok, 1 = ok */

  const char *p, *q;
  int ucase, lcase, other, r;

  if ((strlen (pwdstr) < 6) && gotuid)
    {
      fputs ("The password must have at least 6 characters.\n", stdout);
      return 0;
    }

  other = ucase = lcase = 0;
  for (p = pwdstr; *p; p++)
    {
      ucase = ucase || isupper (*p);
      lcase = lcase || islower (*p);
      other = other || !isalpha (*p);
    }

  if ((!ucase || !lcase) && !other && gotuid)
    {
      fputs ("The password must have both upper and lowercase ", stdout);
      fputs ("letters, or non-letters.\n", stdout);
      return 0;
    }

  if (oldpwd && !strncmp (oldpwd, pwdstr, 8) && gotuid)
    {
      printf ("You cannot reuse the old password.\n");
      return 0;
    }
  r = 0;
  for (p = pwdstr, q = user; *q && *p; q++, p++)
    {
      if (tolower (*p) != tolower (*q))
        {
          r = 1;
          break;
        }
    }

  for (p = pwdstr + strlen (pwdstr) - 1, q = user;
       *q && p >= pwdstr; q++, p--)
    {
      if (tolower (*p) != tolower (*q))
        {
          r += 2;
          break;
        }
    }

  if (gotuid && r != 3)
    {
      printf ("Please don't use something like your username as password.\n");
      return 0;
    }
  return 1;                     /* OK */
}

static char *
getnewpwd (const char *oldpwd, const char *user)
{
  char *pwdstr = NULL;
  char pwdstr1[10];
  uid_t gotuid;

  gotuid = getuid ();

redo_it:
  pwdstr = getpass ("Enter new password: ");
  if (pwdstr[0] == '\0' && gotuid != 0)
    {
      fputs ("Password not changed.\n", stdout);
      exit (3);
    }
  if (verifypassword (oldpwd, pwdstr, user, gotuid) != 1)
    {
      goto redo_it;
    }
  strncpy (pwdstr1, pwdstr, 9);
  pwdstr1[9] = '\0';
  pwdstr = getpass ("Re-type new password: ");

  if (strncmp (pwdstr, pwdstr1, 8))
    {
      fputs ("You misspelled it. Password not changed.\n", stdout);
      exit (3);
    }

  return pwdstr;
}

static void
Usage (void)
{
  fputs ("nispasswd [-m masterhost] [-D domainname]\n", stdout);
  fputs ("nispasswd --version\n", stdout);
}

int
main (int argc, char *argv[])
{
  CLIENT *clnt;
  struct timeval timeout;
  char oldpwd[17];
  char domain[NIS_MAXNAMELEN + 1];
  npd_request request;
  npd_update update;
  nispasswd_authresult result;
  nispasswd_updresult updresult;
  struct passwd *pwd;
  char pkey_host[HEXKEYBYTES + 1];
  char pkey_user[HEXKEYBYTES + 1];
  char skey_data[HEXKEYBYTES + 1];
  char usernetname[MAXNETNAMELEN + 1], servernetname[MAXNETNAMELEN + 1];
  des_block CK;
  const char *masterhost = "localhost";
  des_block cryptbuf;
  char ivec[8];
  long *ixdr;
  int err;

  strncpy (domain, nis_local_directory (), NIS_MAXNAMELEN);
  domain[NIS_MAXNAMELEN] = '\0';

  /* Parse program arguments */

  while (1)
    {
      int c;
      int option_index = 0;
      static struct option long_options[] =
      {
	{"version", no_argument, NULL, '\255'},
	{"usage", no_argument, NULL, 'u'},
	{"help", no_argument, NULL, 'h'},
	{NULL, 0, NULL, '\0'}
      };

      c = getopt_long (argc, argv, "uhD:m:", long_options, &option_index);
      if (c == EOF)
	break;
      switch (c)
	{
	case 'D':
	  strncpy (domain, optarg, NIS_MAXNAMELEN);
	  break;
	case 'm':
	  masterhost = optarg;
	  break;
	case '\255':
	  fputs ("nispasswd version 1.0\n", stdout);
	  exit (0);
	case 'u':
	case 'h':
	  Usage ();
	  exit (0);
	default:
	  Usage ();
	  exit (1);
	}
    }

  argc -= optind;
  argv += optind;

  /* Get old passwd information for caller*/
  pwd = _nis_getpwuid (getuid ());
  if (pwd == NULL)
    {
      fprintf (stderr, "Who are you ?\n");
      return 4;
    }

  /* build netname for user or if caller == root, host */
  if (getuid () == 0)
    {
      char hostname[MAXHOSTNAMELEN + 1];

      if (gethostname (hostname, MAXHOSTNAMELEN) != 0)
	{
	  fprintf (stderr, "Could not determine hostname !\n");
	  return 3;
	}
      host2netname (usernetname, hostname, NULL);
    }
  else
    user2netname (usernetname, getuid (), NULL);

  /* get old password for decrypting secret key and further use */
  memset (oldpwd, '\0', sizeof (oldpwd));
  strncpy (oldpwd, getpass ("Enter old NIS+ password: "), 11);
  if (!getsecretkey (usernetname, skey_data, oldpwd))
    {
      if (!getsecretkey (usernetname, skey_data,
			 getpass ("Enter RPC secure password: ")))
	{
	  fprintf (stderr, "Can't find %s's secret key\n", usernetname);
	  return 3;
	}
    }

  /* fill in request struct */
  memset (&request, '\0', sizeof (request));
  request.ident = 0;
  request.key_type = (char *)"DES";
  request.domain = domain;
  request.username = pwd->pw_name;

  /* get publickey of the user */
  memset (pkey_user, '\0', sizeof (pkey_user));
  if (getpublickey (usernetname, pkey_user) == 0 || pkey_user[0] == '\0')
    {
      fprintf (stderr, "Could not get public key for %s!\n", usernetname);
      return 3;
    }
  request.user_pub_key.user_pub_key_len = HEXKEYBYTES;
  request.user_pub_key.user_pub_key_val = pkey_user;
  /* get publickey of the server running rpc.nispasswdd */
  host2netname (servernetname, masterhost, NULL);
  memset (pkey_host, '\0', sizeof (pkey_host));
  if (getpublickey (servernetname, pkey_host) == 0 || pkey_host[0] == '\0')
    {
      fprintf (stderr, "Could not get public key for %s!\n", servernetname);
      return 3;
    }

  /* Generate common DES key from public server key and secret user key */
  if (key_get_conv (pkey_host, &CK) != 0)
    {
      fputs ("Could not create conversion key !\n", stderr);
      return 3;
    }

  /* encrypt old clear password. Don't know why Sun needs 16 byte,
     normaly a password could only by 8 byte. And later in the protocol
     SUN allows only 12 byte. */
  memset (ivec, 0, 8);
  err = cbc_crypt ((char *) &CK, oldpwd,
		   16, DES_ENCRYPT | DES_HW, ivec);
  if (DES_FAILED (err))
    {
      fputs ("DES encryption failure", stderr);
      return 3;
    }
  request.npd_authpass.npd_authpass_len = 16;
  request.npd_authpass.npd_authpass_val = oldpwd;

  /* Try to authenticate us and the server.
     XXX This should be done in a loop,
     since the server could be bussy or the password wrong */
  clnt = clnt_create (masterhost, NISPASSWD_PROG, NISPASSWD_VERS, "tcp");
  if (clnt == NULL)
    {
      fprintf (stderr, "rpc.nispasswd not running on %s ?\n", masterhost);
      return 3;
    }
  memset ((char *) &result, 0, sizeof (result));
  timeout.tv_sec = 25;
  timeout.tv_usec = 0;
  err = clnt_call (clnt, NISPASSWD_AUTHENTICATE,
		   (xdrproc_t) xdr_npd_request, (char *) &request,
		   (xdrproc_t) xdr_nispasswd_authresult, (char *) &result,
		   timeout);
  if (err)
    {
      clnt_perrno (err);
      fputs ("\n", stderr);
      return 3;
    }
  if (result.status != NPD_SUCCESS)
    {
      if (result.status == NPD_TRYAGAIN)
	fputs ("ERROR: password incorrect, try again\n", stderr);
      else
	{
	  fprintf (stderr, "ERROR: %s\n",
		   npderr2str (result.nispasswd_authresult_u.npd_err));
	  fputs ("       password not changed\n", stderr);
	}
      return 3;
    }

  /* Decrypt the ID and the random value. Not easy, since Sparc's are
     big endian, i?86 little endian, we have to revert to big endian
     for decrypt */
  memset (&cryptbuf, '\0', sizeof (cryptbuf));
  ixdr = (long *) &cryptbuf;
  IXDR_PUT_LONG (ixdr, result.nispasswd_authresult_u.npd_verf.npd_xid);
  IXDR_PUT_LONG (ixdr, result.nispasswd_authresult_u.npd_verf.npd_xrandval);
  err = ecb_crypt ((char *) &CK,
		   (char *) &cryptbuf,
		   8, DES_DECRYPT | DES_HW);
  if (DES_FAILED (err))
    {
      fputs ("DES decryption failure!\n", stderr);
      return 3;
    }

  /* fill out update request */
  memset (&update, 0, sizeof (update));
  update.ident = ntohl (cryptbuf.key.high);
  update.xnewpass.npd_xrandval = cryptbuf.key.low;
  memset (update.xnewpass.pass, '\0', __NPD_MAXPASSBYTES);
  strncpy (update.xnewpass.pass, getnewpwd (oldpwd, pwd->pw_name),
	   __NPD_MAXPASSBYTES);
  update.pass_info.pw_gecos = pwd->pw_gecos;
  update.pass_info.pw_shell = pwd->pw_shell;

  memset (ivec, 0, 8);
  err = cbc_crypt ((char *) &CK, (char *)&update.xnewpass,
                   16, DES_ENCRYPT | DES_HW, ivec);
  if (DES_FAILED (err))
    {
      fputs ("DES decryption failure!\n", stderr);
      return 3;
    }

  /* update.xnewpass.npd_xrandval will be chagned in XDR again */
  update.xnewpass.npd_xrandval = ntohl (update.xnewpass.npd_xrandval);

  memset ((char *) &updresult, 0, sizeof (updresult));
  timeout.tv_sec = 25;
  timeout.tv_usec = 0;
  err = clnt_call (clnt, NISPASSWD_UPDATE,
		   (xdrproc_t) xdr_npd_update, (char *) &update,
		   (xdrproc_t) xdr_nispasswd_updresult, (char *) &updresult,
		   timeout);
  if (err)
    {
      clnt_perrno (err);
      fprintf (stderr, "\n");
      return 3;
    }
  clnt_destroy (clnt);
  if (updresult.status != NPD_SUCCESS)
    {
      if (updresult.status == NPD_FAILED)
	{
	  fprintf (stderr, "ERROR: %s\n",
		   npderr2str (updresult.nispasswd_updresult_u.npd_err));
	  fputs ("       password not changed\n", stderr);
	}
      else
	if (updresult.status == NPD_PARTIALSUCCESS)
	  {
	    nispasswd_error *err;
	    fputs ("ERROR: Only partial success\n", stderr);
	    err = &updresult.nispasswd_updresult_u.reason;
	    while (err != NULL)
	      {
		switch (err->npd_field)
		  {
		  case NPD_PASSWD:
		    fprintf (stderr, "\tpassword field: %s\n",
			     npderr2str (err->npd_code));
		    break;
		  case NPD_GECOS:
		    fprintf (stderr, "\tgecos field: %s\n",
			     npderr2str (err->npd_code));
		  case NPD_SHELL:
		    fprintf (stderr, "\tshell field: %s\n",
			     npderr2str (err->npd_code));
		    break;
		  case NPD_SECRETKEY:
		    fprintf (stderr, "\tsecret key: %s\n",
			     npderr2str (err->npd_code));
		    break;
		  }
		err = err->next;
	      }
	  }
	else
	  {
	    fputs ("ERROR: Unknown error, don't know what happend\n", stderr);
	  }
      return 3;
    }

  fputs ("SUCCESS!\n", stdout);

  return 0;
}
