/* ttcd - Terminal Type Control Daemon.
 * Version 1.40 26.5.1997
 * 
 * Copyright (C) 1997 Pekka Riikonen, priikone@fenix.pspt.fi.
 * Distributable under the terms of GNU GPL, version 2 or later.
 *
 * This code comes 'as is' and gives no warranty to work. You use this
 * software at your own risk. You can freely modify this code, but remember
 * to give me the credit for original code.
 *
 * Receives messages from socket. Messages are sent by ttc. After 
 * receiving, daemon forks a new child of timing(), which sets timing for 
 * user. After that, daemon continues receiving messages. Child dies if user 
 * uses his/her all time limit or logs off. Check README for more info.
 *
 */

/*
 * Revision 1.40  26.5.1997  pekka
 *	Fixed recvfrom()'s for other systems in receive_messages()
 *	Fixed some typos
 *	Added max PID (SID) number for flood test in timing()
 *	Added logins count per day for users -feature
 *
 * Revision 1.39  15.2.1997  pekka
 *	'First' release version.
 *
 */

#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include <time.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/un.h>
#include "config.h"		/* configuration file */

int logspd;			/* logins allowed per day for users */

main(argc, argv)
int argc;
char *argv[];
{
	char opt;			/* option */

	if (argc < 2) {			/* if no options */
		if (getuid() != 0) {
		    printf("ttcd: Sorry, only root may run ttcd.\n");
		    exit(1); }
		if(fork())
		   exit(0);
		log("ttc daemon started", "root");
		setsid();
		receive_messages();	/* child (main daemon)*/
	} else {
	    while ((opt = getopt(argc, argv, "chV")) != EOF) /* get options */
		switch(opt) {
		case 'c':
		 clean_files();
		 exit(0);
		case 'V':
		 printf("Terminal Type Control Daemon, version 1.40 (C) 26.5.1997 Pekka Riikonen\n");
		 exit(0);
		case 'h':
		default:
		 printf("usage: ttcd [-c] [-hV]\n");
		 printf("    -c    clean time limit files\n");      
		 printf("    -h    display this message\n");      
		 printf("    -V    display version\n");      
		 exit(0);
	    }
	}
}	

/* Kill vanity child processes. */

void sigchld(int signum)	/* called when SIGCHLD is received */
{				/* this way we prevent child zombies */
	int stat;
	wait(&stat);
	receive_messages();	/* I have to call this here, since if not
				   called, the whole daemon gets screwed up,  
				   dunno why, thou ??? Overload maybe?? 
				   or this doesn't return properly? */
}

/* This function receives messages from ttc-socket. It stops the process 
until some data is received from socket. After that it forks a new child 
which then handels the timing. Then, it remains waiting new messages. */

receive_messages()
{
	char cmesg[2][12];		/* [0] = username
					 * [1] = current tty */
	int sock;			/* socket */
	int pid;			/* child's pid */
	int imesg[4];			/* [0] = time limit
					 * [1] = check status
					 * [2] = PPID
					 * [3] = logs per day */

	char fname[14];			/* filename for socket */
	struct sockaddr_un addr;	/* socket address */
	int *len;			/* receive length buffer */
	int i;				/* counter */

	sprintf(fname, "/var/lock/ttcd");/* socket path */
	unlink(fname);			/* remove old socket */

	/* create socket */
	if((sock = socket(AF_UNIX, SOCK_DGRAM, 0)) == -1) {
		log("Couldn't create socket.", "socket()");
		exit(1); }

	addr.sun_family = AF_UNIX;	/* protocol family */
	strcpy(addr.sun_path, fname); 	/* path to address */

	/* bind socket - create socket file */
	if((bind(sock, (struct sockaddr *)&addr, sizeof(addr))) == -1) {
		log("Couldn't bind the socket.", "bind()");
		exit(1); } 

	/* correct permissions */
	chown(fname, getuid(), getgid());
	chmod(fname, 0766);

	signal(SIGCHLD, sigchld);

	/* let's loop forever and be ready to receive messages */
	for (;;) {
	    for (i = 0; i < 2; i++) {
 		recvfrom(sock, &cmesg[i], 12, 0, (struct sockaddr *)&addr, len);
	    }

	    for (i = 0; i < 4; i++) {
 		recvfrom(sock, &imesg[i], 6, 0, (struct sockaddr *)&addr, len);
	    }	

	    if ((pid = fork()) == 0)
		break;
	    if (pid == -1)
	    	log("fork(): Couldn't fork timing()", (char *)cmesg[0]);
	    else
	  	log("Timing started", (char *)cmesg[0]);
	}					/* parent continues here */

	signal(SIGCHLD, SIG_DFL);		/* default action to SIGCHLD */
	setsid();				/* create new session */
	timing((char *)cmesg, (int *)imesg);	/* call child */
}

/* Child and timing routine for ttc. This is the routine which is forked 
as new process by ttcd. This handles the timing for ttc. */

timing(char *cmesg[][12], int *imesg)
{
	time_t curtime; 		/* time_t datatype */
	struct tm *time_point;		/* pointer to tm structure */
	char filename[18] = "/tmp/ttc/";/* filename for time limit file */
	char *term = (char *)cmesg[1]; 	/* received tty */

	FILE *cn;			/* file handler */
	char *logname = (char *)cmesg[0]; /* received logname */
	int dtmp;
	
	int hour, 	/* hours */
	min, 		/* minutes */
	cmins, 		/* current time formed to minutes */
	dmins, 		/* destination time formed to minutes */
	sid, 		/* session ID == parent process PID (shell) */
	signal,		/* signal from kill() */
	check = imesg[1], /* check status for notimebank() */
	loginspd = imesg[3]; /* logins allowed per day for users */

	int a_time = A_TIME,
	a_minutes = A_MINUTES,
	k_seconds = K_SECONDS,
	s_seconds = A_MINUTES * 60; 	/* form A_MINUTES to seconds */

	logspd = loginspd;
	if (logspd > 0)
		logspd -= 1;		/* minus 1 login */

	sid = imesg[2];			/* ppid */
	curtime = time(0);		/* get time */
 	time_point = localtime(&curtime); /* localtime address to pointer */
	hour = time_point->tm_hour;	/* point to get hour */
	min = time_point->tm_min;	/* point to get minutes */

	strcat(filename, logname);	/* filename */

	cmins = (hour * 60) + min;	/* current minutes */
	if (check == 0)
	    a_time = imesg[0];
	dmins = cmins + a_time; 	/* destination minutes */

	/* this way we prevent some flood messages perhaps */
	if (a_time > A_TIME || a_time < 0) {
	    log("Bad destination time error", logname);
	    exit(1); }

	if (check < 0 || check > 1) {
	    log("Bad check status error", logname);
	    exit(1); }

	if (sid < 0 || sid > 32767) {		/* max 32767 ? */
	    log("Bad pid number error", logname);
	    exit(1); }

	if (logspd < 0 || logspd > 1000) {
	    log("Bad logins per day number error", logname);
	    exit(1); }

	while(cmins < dmins) { /* true as long as cmins is smaller than dmins */
	    cmins = cmins + a_minutes;	/* current time + a_minutes */
	    if (check == 0) {
		dtmp = dmins - cmins;	/* time remaining */
		create_tlfile(dtmp, logname); }
	    signal = kill(sid, 15);	/* is the process alive? */
	    if (signal == -1) {		/* nope, if -1, so user is logged off */
		if (check == 0) {
		    dmins = dmins - cmins; /* time remaining */
		    create_tlfile(dmins, logname); }
		    log("Timing stopped", logname);
		    exit(0);
	    }
	    if (cmins > 1439) {		/* if current time is 23:59 */
		printo(N_MESSAGE, term);/* a new time limit is starting... */
		sleep(60);		/* sleep for 60 seconds */
		cmins = 0;		/* current mins to zero */
		dmins = A_TIME; 	/* a new time limit */
		log("Day changed. New time limit starting", logname);
	        if (check == 0) {
		    create_tlfile(dmins, logname); }
	    } else {
	    	sleep(s_seconds); }	/* sleep before next round */
	}
	signal = kill(sid, 15);		/* is the process alive? */
	if (signal != -1) {
	  printo(MESSAGE, term);	/* message we print to user */
	  sleep(k_seconds); }		/* wait k_seconds before killing */
	  log("Time limit reached", logname); 
	kill(sid, 9);	 		/* let's kill SID == shell's PID */
	exit(0);
}

/* Clear's a day old time limit files */

clean_files()
{
	if(getuid() != 0)
		printf("ttcd: Sorry, only root may use this option.\n");

	system("rm -rf /tmp/ttc");
	mkdir("/tmp/ttc", 0755);
	log("Time limit files removed", "root");
}

/* Creates time limit file for user. */

create_tlfile(int dmins, char *logname)
{
	FILE *cn;
	char filename[18] = "/tmp/ttc/"; /* filename for time limit file */
	int b;
	int k;
	
	strcat(filename, logname);	/* filename */

	if ((cn = fopen(filename, "w")) == 0)
	    log("Couldn't create time limit file", logname);
	for (k = 0; k < logspd; k++)
	    fputc(26, cn);
	for (b = 0; b < dmins; b++)
	    fputc(27, cn);
	fclose(cn);
	return;
}

/* Prints message to specific tty */

printo(char *message[], char *tty)
{
	FILE *fd;

	if ((fd = fopen(tty, "w")) == NULL)	/* open dev */
		log("Couldn't open tty for writing", "fopen()");
	fprintf(fd, "%s\007\n", (char *)message);/* write it */
	fclose(fd);				/* close dev */
	return;
}

/* Makes log messages into ttc logfile if DEBUG is defined. */

log(char *mesg[], char *lname)
{
#ifdef DEBUG			/* logs only if DEBUG defined */
	time_t cutime;
	char *ctime();
	FILE *fp;
	char *logfile = "/var/log/ttcdlog";	/* logfile */

	cutime = time(0);			/* current time */

	if ((fp = fopen(logfile, "a+")) == NULL)
		printf("ttcd: %s.\n", strerror(errno));

	fprintf(fp, "ttcd: %s by %s %s", (char *)mesg, lname, ctime(&cutime));
	fclose(fp);
#endif
	return;
}
