/* ttc - Terminal Type Control.
 * Version 2.6 28.5.1997
 * 
 * Copyright (C) 1997 Pekka Riikonen, priikone@fenix.pspt.fi.
 * Distributable under the terms of GNU GPL, version 2 or later.
 *
 * This code comes 'as is' and gives no warranty to work. You use this
 * software at your own risk. You can freely modify this code, but remember
 * to give me the credit for original code.
 *
 * Takes tty's by argument, and check's if they are going to be denied or 
 * timed. If terminal type is denied only those users who has  
 * authorization, can use that terminal type. If terminal type is timed, 
 * ttc will send messages to ttc daemon socket and daemon sets timing. 
 * After time is up, session will be killed. Check README for more info.
 *
 */

/*
 * Revision 2.6  28.5.1997  pekka
 *	Fixed sendto() for other systems in send_message()
 *	Added -l option for login count checks per day
 *	 (idea by: Eric Beyer <beyer@cis.nmclites.edu>)
 *	Added ^C signal to be set default before exiting
 *
 * Revision 2.5  24.2.1997  pekka
 *	Removed suid-root
 *	Made ttcd.c, ttc daemon to handle timing
 *	ttc now uses sockets to communicate between ttc and ttcd
 *	 so we don't need to run it as suid-root
 *	Made options.c for better option handling
 * 
 * Revision 2.2  7.2.1997  pekka
 *	added -static flag in compiling for suid-root
 *
 * Revision 2.1  5.2.1997  pekka
 *	Added timing routine into ttc.c
 *	ttc runs now as suid-root
 *
 * Revision 1.7  26.1.1997  pekka
 *	Made all new timing system	
 *
 * Revision 1.0  23.1.1997  pekka
 *	First release version
 */

#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include "config.h"	/* configuration file */

static char *tauth_file = AUTH_FILE, *dauth_file = AUTH_FILE;
int logspd = 0;				/* logins allowed per day */
int fl = 0;				/* flag for logins count 
					   0 = no limit
					   1 = limit 
					*/
main(argc, argv)
int argc;
char *argv[];
{
	int c = 1;			/* counter for options */
	char opt;			/* option */
	char *filename;
	char *ldb_filename;
	extern char *tauth_file, *dauth_file;

	char *options[7];		/* options */
	char *params[100][1024];	/* parameters for options */
	int optnum;			/* number of options */
	int parnum[100];		/* number of parameters */

	char *loptions[500];		/* login database options */
	char *lparams[500][1024];	/* ldb parameters for options */
	int loptnum;			/* ldb number of options */
	int lparnum[500];		/* ldb number of parameters */

	filename = DEFOPTION;
	ldb_filename = LDB_FILE;

	signal(SIGINT, SIG_IGN);	/* ignore ^C */

	if (argc < 2) {			/* if no options */
		optnum = read_options(filename, options, params, parnum, 0);
		do_options(options, params, parnum, optnum);	
	} else {
	    while ((opt = getopt(argc, argv, "Vsldtfh")) != EOF) /* get options */
		switch(opt) {
		case 'V':
		 printf("Terminal Type Control, version 2.6 (C) 28.5.1997 Pekka Riikonen\n");
		 exit(0);
		case 's':
		 dauth_file = DAUTH_FILE;
		 tauth_file = TAUTH_FILE;
		 optnum = read_options(filename, options, params, parnum, 0);
		 do_options(options, params, parnum, optnum);
		 break;
		case 'l':
		 /* get number of logins allowed per day for user(s) */
		 loptnum = read_options(ldb_filename, loptions, lparams, lparnum, 1);
		 do_loptions(loptions, lparams, lparnum, loptnum);	
		 break;
		repeat:
		case 'd':
		 for (c = c + 1; c < argc; c++) {
			if (!strcmp(argv[c], "-t")) break;
			deny_tty(argv[c], dauth_file); } /* check if there is a match */ 
		 break;
		case 't':
		 for (c = c + 1; c < argc; c++) {
			if (!strcmp(argv[c], "-d")) goto repeat; /* hate to use this */
			time_tty(argv[c], tauth_file); } /* check if there is a match */
		 break;
		case 'f':
		 filename = argv[c + 1];
		 optnum = read_options(filename, options, params, parnum, 0);
		 do_options(options, params, parnum, optnum);
		 break;
		case 'h':
		default:
		 usage();
	    }
	}
	signal(SIGINT, SIG_DFL);	/* default action for ^C */
}	

/* Parses options from options file for ttc */

do_options(char *options[], char *argv[][1024], int *parnum[], int *optnum) 
{
	int i, c = 0, d = 0, y = 0, a = 1;

	char *loptions[500];		/* login database options */
	char *lparams[500][1024];	/* ldb parameters for options */
	int loptnum;			/* ldb number of options */
	int lparnum[500];		/* ldb number of parameters */

	char *ldb_filename = LDB_FILE;


	for (i = 0; i < (int)optnum; i++) {
	    switch((int)*options[i]) {
	    case 's':
		dauth_file = DAUTH_FILE;
		tauth_file = TAUTH_FILE;
		d++;
	        break;
	    case 'l':
		/* get number of logins allowed per day for user(s) */
		loptnum = read_options(ldb_filename, loptions, lparams, lparnum, 1);
		do_loptions(loptions, lparams, lparnum, loptnum);
		d++;
		break;
	    case 'd':
		c = c + a;
	    	for (y = 0; y < (int)parnum[d]; y++) {
	    	    deny_tty(argv[c], dauth_file); 
		    c++; }
	    	if (d == 0) d = 1;
		if (a == 1) a = 0;
		y = 0; 
	        break;
	    case 't':
		c = c + a;
	    	for (y = 0; y < (int)parnum[d]; y++) {
	    	    time_tty(argv[c], tauth_file); 
		    c++; }
	    	if (d == 0) d = 1;
		if (a == 1) a = 0;
		y = 0; 
		break;
	    case 'V':
	    	printf("You can't use -V option in options file");
		exit(1);
	    case 'f':
	    	printf("You can't use -f option in options file");
		exit(1);
	    default:
		usage();
	    }
	}
}

/* Parses login count options from database file */

do_loptions(char *options[], char *argv[][1024], int *parnum[], int *optnum) 
{
	int i, c = 0, d = -1, y = 0;

	for (i = 0; i < (int)optnum; i++) {
	    switch((int)*options[i]) {
	    case 'a':
		fl = 1;
		logspd = atoi((char *)argv[c + 1]);
		if (logspd > 1000)
		    logspd = 1000;
		if (logspd < 1)
		    logspd = 1;
		c++;
		break;
	    case 'u':
		d++;
		if (fl)
		    c++;
	    	for (y = 0; y < (int)parnum[d]; y++) {
		    if(!strcmp((char *)argv[c], getlogin())) {
			logspd = atoi((char *)argv[c + 1]);
			if (logspd > 1000)
		    	    logspd = 1000;
			if (logspd < 1)
			    logspd = 1;
			fl = 1;
			break;
		    }
		    c++;
		}
		break;
	    default:
		break;
	    }
	}
}

usage()
{
	printf("usage: ttc [-s -l -dt [/dev/tty ...] -f [options file] -hV]\n");
	printf("    -s    seperated authorization files for deny and timing\n");      
	printf("          (defaults: deny: %s timing: %s)\n", DAUTH_FILE, TAUTH_FILE);      
	printf("          NOTE: without -s option deny and timing will use same\n");
	printf("          authorization file (default: %s)\n", AUTH_FILE);
	printf("    -l    logins allowed per day for users (default: %s)\n", LDB_FILE);
	printf("    -d    deny tty's (e.g. -d /dev/tty1 /dev/tty2 ...)\n");
	printf("    -t    timing for tty's (e.g. -t /dev/ttyS1 ...)\n");
	printf("    -f    read options from file (default: %s)\n", DEFOPTION);      
	printf("    -h    display this message\n");
	printf("    -V    display version\n");
	exit(1);
}

/* Do the deny function. This checks if user has right to access denied 
tty. If not he/she will be killed. */

deny_tty(char *ttyd, char *dauth_file)
{
	char *ttype;
	FILE *nm;
	char login[10]; 		/* 8 + 2 chars for logname */
	int pid, sid;

	ttype = ttyname(0);		/* get current tty */
	
	if(!strcmp(ttype, ttyd)) { /* if current tty and argument matches? */
	     if((nm=fopen(dauth_file,"r")) == NULL) {	/* open file */
		printf("ttc: Deny options file doesn't exist.\n");
		_exit(1); }
	     while((fscanf(nm, "%s", login)) != EOF) { /* names from file */
		if(!strcmp(login, getlogin())) { /* if my name was there */
			exit(0); } 	/* your name was there, let's quit */
	     }
	     fclose(nm);
	     printf(D_MESSAGE"\007\n");	/* "This terminal is denied" */
	     pid = getpid();
	     sid = getsid(pid);
	     kill(sid, 9);		/* let's kill this session */
	} else {
		return;		/* wasn't a match, let's return to main() */
	}
}

/* Do time time check function. Checks if user has access to timed tty 
without timing. If not message will be sent to ttcd to start timing for 
user. */

time_tty(char *ttyt, char *tauth_file)
{
	FILE *nm;
	int a_time = A_TIME,
	check = 0;
	char *ttype;
	char login[10]; 		/* 8 + 2 chars for logname */

	ttype = ttyname(0);		/* get current tty */

	if(!strcmp(ttype, ttyt)) { /* if current tty and argument matches? */

	     if((nm=fopen(tauth_file,"r")) == NULL) { 	/* open file */
		printf("ttc: Timing options file doesn't exist.\n");
		_exit(1); 
	     }
	     while((fscanf(nm, "%s", login)) != EOF) { /* names from file */
		if(!strcmp(login, getlogin())) { /* if my name was there */
			printf(A_MESSAGE"\n"); 	/* print message to user */
			exit(0); 
		}
	     }
	     fclose(nm);

	     check = check_notimebank(check);	/* checf if no time bank */
	     if (check == 0) 
	     	a_time = check_timing(a_time);	/* check for time limit */
	     send_message(a_time, check); /* message to daemon socket */
	} else {
		return;		/* wasn't a match, let's return to main() */
	}
}

/* Sends message to ttc-socket what ttcd listens for incoming messages. 
This sends to ttcd logname, tty, time limit, check status and session 
process id. */

send_message(int a_time, int check)
{
	int pid = getsid(getpid());
	int loginspd = logspd;

	/* messages we send to ttc daemon socket */
	char *cmsg[2] = { (char *)getlogin(),		/* username */ 
			 (char *)ttyname(1) };		/* current tty */
	int imsg[4] = { a_time,		/* time limit */
			check,		/* check status */
			pid,			/* SID == PPID */
			loginspd };		/* logs per day */

	struct sockaddr_un addr;		/* socket address */
	int sock;				/* socket */
	char fname[14];				/* filename for socket */
	int chs, i;

	sprintf(fname, "/var/lock/ttcd");	/* socket path */
	
	sock = socket(AF_UNIX, SOCK_DGRAM, 0);	/* create socket */

	addr.sun_family = AF_UNIX;		/* protocol family */
	strcpy(addr.sun_path, fname);		/* path to address */

	/* send messages to daemon socket */
	for (i = 0; i < 2; i++) {
 	if((chs = sendto(sock, cmsg[i], 12, 0,
	 (struct sockaddr *)&addr, sizeof(addr))) == -1) {
		printf("sendto: %s\n", strerror(errno));
		exit(1); } 	
	}
	for (i = 0; i < 4; i++) {
 	if((chs = sendto(sock, &imsg[i], 6, 0,
	 (struct sockaddr *)&addr, sizeof(addr))) == -1) {
		printf("sendto: %s\n", strerror(errno));
		exit(1); } 	
	}
	return;
}

/* Checks out users time limit, if any. This also checks the logins count 
remaining for today, if any. */

check_timing(int a_time)
{
	int pid, sid, check;
	char *logname, filename[18] = "/tmp/ttc/";
	FILE *tl;

	pid = getpid();
	sid = getsid(pid);

	logname = getlogin();			/* get logname */
	strcat(filename, logname);		/* filename */

	if ((tl = fopen(filename, "r")) == NULL) { /* open file */
	    return(a_time);			/* return old value */
	} else {
	    a_time = 0;
	    logspd = 0;
	    while((check = fgetc(tl)) != 27) {
		if (check == 26) {
		   logspd++;
		} else {
		    if (check == EOF) {		/* already at end? */
			printf(TLMESSAGE"\007\n");/* time limit message */
			kill(sid, 9);		/* kill session */
			exit(1);
		    }
		    printf(FMESSAGE"\007\n");	/* illegal file type */
		    kill(sid, 9); 		/* kill session */
		    exit(1); }
	      }
	    if (check == 27)
		a_time = 1;
	    while((check = fgetc(tl)) != EOF) {
		if (check == 27) {
		    a_time++;
		} else {
		    printf(FMESSAGE"\007\n");
		    kill(sid, 9); 		/* kill session */
		    exit(1); }
	    }
	    fclose(tl);
	    if (fl) {
	    	if (logspd < 1) {
		    printf(LLMESSAGE"\007\n");	/* log limit message */
		    kill(sid, 9);		/* kill session */
		    exit(1); }
	    }
	    printf(TIMELEFT"\n", a_time);	/* time left message */
	    if (fl)
	      printf(LOGLEFT"\n", logspd);	/* logins left today */
	    return(a_time);			/* return new value */
	}
}

/* Checks if user belongs to notimebank system and so can access terminal 
without time bank system. Allthough this doesn't mean that user doesn't 
have time limit. */

check_notimebank(int check)
{
	char login[10]; 	/* login name */
	FILE *tl;

	check = 0;

	if ((tl = fopen(NOTIMEBANK, "r")) == NULL) { /* open file */
		return(check);			/* return old value */
	} else {
		while((fscanf(tl, "%s", login)) != EOF) { 
		if(!strcmp(login, getlogin())) { /* if name is there */
			check = 1;
			fclose(tl);
			return(check); }	/* return new value */
		}
		fclose(tl);
		return(check);			/* return old value */
	}
}
