/*	clnwtmp.c										Oct. 95
	Mark Santelli

	Reads in records from the wtmp file and writes them out if the 
	date from wtmp is greater than the date received in the command 
	line. The time of '00:00:00' is used in the date spec. The 
	wtmp or utmp filename is passed as the first parameter to the
	program and the cutoff date is passed as the second. The new 
	system file is created in the current directory.

	LOG:

	August 96.				Mark Santelli

	added option of passing the number of days to keep in the
	output file.
*/

#include <ctype.h>
#include <sys/types.h> 
#include <sys/stat.h>
#include <time.h>
#include <utmp.h>
#include <fcntl.h>

#define BLOCK	140

struct utmp filebuf[BLOCK];
struct utmp outbuf [BLOCK];
int num_days;
time_t time_threshold;
int infile_handle, outfile_handle;

int  OpenFiles(char *);
int  ValidInput(int, char * []);
void ConvertDate(int, int, int);
void WriteFile(void);
void ShowUsage(char *);

void main(int argcnt, char *args[])
{								/* validate input */
	if (!ValidInput(argcnt, args))
		exit(1);
							
	if (!OpenFiles(args[1]))
		exit(1);
								
	WriteFile();					/* do it */

	exit(0); 			
} 

void WriteFile(void)
{
	int bytes, sub, outsub = 0;

	while ((bytes = read(infile_handle, &filebuf, sizeof(filebuf))) > 0)
	{	
		bytes /= sizeof(struct utmp);
		for (sub = 0; sub < bytes; sub++) 
		{
			if (filebuf[sub].ut_time > time_threshold)
				outbuf[outsub++] = filebuf[sub];
		}

		if (write(outfile_handle, &outbuf, sizeof(struct utmp) * outsub) 
		    != sizeof(struct utmp) * outsub)
		{
			perror("write error.");
			exit(1);
		}

		outsub = 0;
	}
}

int ValidInput(int parmcnt, char * parms[])
{
	int day, mon, yr, check, factor;
	time_t num_seconds;
	struct stat st;
	char * num_save, * n_s, errbuf[100];
							
	if (parmcnt < 3 || parmcnt > 4)	/* correct # of parms */
	{
		ShowUsage(parms[0]);
		return 0;
	}
								/* does file exists */
	if (stat(parms[1], &st) == -1)
	{
		sprintf(errbuf, "\nstat error: %s", parms[1]);
		perror(errbuf);
		return 0;
	}

	if (parms[2][0] == '-')
	{								/* # of days */
		if      (parms[2][1] == 'd')
			factor = 1;
		else if (parms[2][1] == 'w')
			factor = 7;
		else
		{
			ShowUsage(parms[0]);
			return 0;
		}
		
		if (parmcnt == 3)
			num_save = &parms[2][2];
		else		
		{		/* parmcnt is 4, check for eg: '-dcrap 55' */
			if (parms[2][2])
			{
				ShowUsage(parms[0]);
				return 0;
			}
			else
				num_save = parms[3];
		}

		n_s = num_save;				/* check for junk */
		while (*num_save != ' ' && *num_save != '\t' && *num_save)
		{						
			if (*num_save < '0' || *num_save > '9')
			{
				printf("\nnumber contains non-numeric characters\n");
				return 0; 
			}
			num_save++;
		}
		num_days = atoi(n_s);
		num_days *= factor;
		num_seconds = num_days * 86400;
		time_threshold = time('\0') - num_seconds;
	}
	else
	{								/* date validation */
		if (strlen(parms[2]) == 6) 
		{							/* is date numeric */
			for (check = 0; check < 6; check++)
			{
				if (parms[2][check] < '0' || parms[2][check] > '9')
				{
					printf("\ndate contains non-numeric characters\n");
					return 0; 
				}		
			}	
									/* test ranges */
			day = atoi(&parms[2][4]);
			parms[2][4] = '\0';		
			if (day < 1 || day > 31)
			{
				printf("\nday must be an integer from (01 <--> 31)\n");
				return 0;
			}
			mon = atoi(&parms[2][2]) - 1;
			parms[2][2] = '\0';		
			if (mon < 0 || mon > 11)
			{
				printf("\nmonth must be an integer from (01 <--> 12)\n");
				return 0;
			}
			yr = atoi(parms[2]);
			if (yr < 1 || yr > 99)
			{
				printf("\nyear must be an integer from (01 <--> 99)\n");
				return 0;
			}
									/* convert date received into	  */
			ConvertDate(day, mon, yr);	/* a time_t format like in wtmp */
		}		                         
		else
		{
			printf("\ndate must be in YYMMDD format\n");
			return 0;
		}
	}

	return 1;
}


void ConvertDate(int d, int m, int y)
{
	struct tm cutdate;
	  
	cutdate.tm_sec   = 0;
	cutdate.tm_min   = 0;
	cutdate.tm_hour  = 0;
	cutdate.tm_mday  = d;
	cutdate.tm_mon	  = m;
	cutdate.tm_year  = y;
	cutdate.tm_isdst = -1; 

	time_threshold = mktime(&cutdate);
}
					
int OpenFiles(char * path)
{
	if ((infile_handle = open(path, O_RDONLY )) == -1)
	{
		perror("\nerror on open 'wtmp'\n");
		return 0;
	}
	
	if ((outfile_handle = open("new.wtmp", O_WRONLY | O_CREAT | O_TRUNC, 
								    S_IWRITE | S_IREAD | S_IRGRP)) == -1)
	{
		perror("\nerror on open 'new.wtmp'\n");
		return 0;
	}

	return 1;
}

void ShowUsage(char * prg_name)
{
	printf("\nusage:  %s /path/wtmp [ YYMMDD | -dw N ]\n\nwhere:", prg_name);
	printf("\n- '/path/wtmp' is the pathname of the system login file");
	printf("\n- YYMMDD is the cutoff date\n");
	printf("\tYY\t01-99\n");           
	printf("\tMM\t01-12\n");
	printf("\tDD\t01-31\n");
	printf("\nIf the 'd' or 'w' switches are used. The N most recent records");
	printf("\nare save in the output file.");
	printf("\n\td  save N days of records");
	printf("\n\tw  save N weeks of records\n");
	printf("\nA new wtmp file is created containing only the records\n");
	printf("as specified on the command line.\n");
	printf("'utmp' can also be rewritten.\n");
}

