#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <limits.h>
#include <netdb.h>
#include <net/if.h>
#include "netconf.h"
#include "internal.h"
#include <misc.h>
#include "netconf.m"


/* #Specification: devices / management
	netconf knows how to load a devices module (probing) and how
	to configure it (ifconfig). It also knows how to update the
	different routes.
*/
#

/*
	Programme a network interface (ifconfig) only if needed,

	Compare the current configuration with the expected configuration.
	If there is a mismatch (or the interface is not configure), then
	it is reconfigure with the expected values.

	Return -1 if any error.
*/
static int devices_ifconfig(
	const char *device,
	const char *expected_hst,
	const char *expected_msk,
	const char *expected_bcast)
{
	IFCONFIG_INFO info;
	int ret = ifconfig_getinfo (device,info);
	if (ret == -1){
		xconf_error (MSG_U(E_CANTPROBE
			,"Can't probe current setup of network device %s\n"
			 "No way to activate the network\n")
			,device);
	}else{
		DAEMON *dae = daemon_find ("ifconfig");
		if (dae != NULL){
			if (strcmp(info.ip_addr,expected_hst)==0
				&& strcmp(info.netmask,expected_msk)==0
				&& strcmp(info.bcast,expected_bcast)==0
				&& info.flags & IFF_UP){
				ret = 0;			
			}else{
				char cmd[300];
				/* #Specification: netconf / ifconfig / down
					We always force an interface DOWN
					before reconfiguring it. It has
					the side effect of cleaning the route
					table for that device.
				*/
				sprintf (cmd,"%s down",device);
				netconf_system_if ("ifconfig",cmd);
				sprintf (cmd,"%s %s netmask %s broadcast %s"
					,device
					,expected_hst
					,expected_msk
					,expected_bcast);
				ret = netconf_system_if ("ifconfig",cmd);
			}
		}
	}
	return ret;
}
/*
	Configure the loopback device.
	Return -1 if any error.
*/
int netconf_setloopback()
{
	/* #Specification: loopback / strategy
		As far as I know, there is only a way to configure the
		loopback device.

		/sbin/ifconfig lo 127.0.0.1
		/sbin/route add -net loopback
	*/
	int ret = devices_ifconfig ("lo","127.0.0.1","255.0.0.0","127.255.255.255");
	if (ret == 0){
		char gateway[PATH_MAX];
		if (!route_isactive("127.0.0.0",gateway)){
			ret = netconf_system_if ("route","add -net 127.0.0.0 lo");
		}
	}
	return ret == 0 ? 0 : -1;
}
#if 0
/*
	Perform a gethostbyname with error reporting.
	Return -1 if not found.
*/
static int devices_gethostbyname(
	const char *host,
	const char *device,
	char *ip_num,		// Will contain the formatted IP address
	int  silent)		// Ok if not set, no error message
{
	struct hostent *ent = gethostbyname(host);
	int ret = -1;
	if (ent == NULL){
		if (!silent){
			xconf_error ("%s is not defined\n"
				"Can't configured device %s\n",host,device);
		}
	}else{
		ret = 0;
		ipnum_ip2a (ent,ip_num);
	}
	return ret;
}
/*
	Perform a getnetbyname with error reporting.
	Return -1 if not found.
*/
static int devices_getnetbyname(
	const char *net,
	const char *device,
	char *ip_num,		// Will contain the formatted IP address
	int  silent)		// Ok if not set, no error message
{
	struct netent *ent = getnetbyname(net);
	int ret = -1;
	if (ent == NULL){
		if (!silent){
			xconf_error ("%s is not defined\n"
				"Can't configured device %s\n",net,device);
		}
	}else{
		ret = 0;
		ipnum_ip2a (ent,ip_num);
	}
	return ret;
}
#endif
struct IFACE_NUMS{
	char hst[20];
	char net[20];
	char msk[20];
	char bcast[20];
};

void device_setstdnetmask(
	const char *hostip,	// IP number of a HOST
	char *def_msk)		// Default netmask
{
	int num4[4];
	ipnum_aip24 (hostip,num4);
	if (num4[0] == 0){
		strcpy (def_msk,"0.0.0.0");
	}else if (num4[0] <= 127){
		strcpy (def_msk,"255.0.0.0");
	}else if (num4[0] <= 191){
		strcpy (def_msk,"255.255.0.0");
	}else{
		strcpy (def_msk,"255.255.255.0");
	}
}

static int device_copystr (char *dst, SSTRING &src)
{
	src.copy (dst);
	return src.is_empty() ? -1 : 0;
}
/*
	Get the IP number assigned to an interface (host,network and mask)
	Fill default values if some information is not fully specified.
	Return -1 if to much is missing.
*/
static int device_ifaceinfo (
	const char *device,		// Name of the device (for error report)
	INTER_INFO &itf,
	IFACE_NUMS &nums)
{
	int ret = -1;
	int h_ok = device_copystr (nums.hst,itf.ipaddr);
	int n_ok = device_copystr (nums.net,itf.network);
	int m_ok = device_copystr (nums.msk,itf.netmask);
	int b_ok = device_copystr (nums.bcast,itf.bcast);
	if (h_ok != -1){
		/* #Specification: ifconfig / eth devices / 127.0.0.1
			Linuxconf won't configure an ethernet
			interface to 127.0.0.1. It will simply ignore it.

			This case happen when a properly configured
			linux system without an ethernet adaptor get
			one.
		*/
		if (strcmp(nums.hst,"127.0.0.1")!=0){
			/* #Specification: ifconfig / netmask and network / optionnal
				The netmask and network and broadcast address of a network
				specification are optionnal. The kernel (or ifconfig
				or route, I don't know) should compute suitable defaults.

				Given if it not the case, linuxconf is computing the
				defaults. If you only give a host IP number, the netmask
				and the net number and broadcast will be computed. If you
				supply the IP and the netmask, the net number and the
				broadcast will be computed.

				The logic of linuxconf goes like this. If you supply
				only the left part, linuxconf compute the right part
				#
				IP number -> netmask, net num, broacast
				IP number + netmask -> net num broadcast
				IP number + netmask + net num -> broadcast
				#
			*/
			ret = 0;
			char def_msk[20];
			device_setstdnetmask(nums.hst,def_msk);
			if (m_ok == -1){
				strcpy (nums.msk,def_msk);
				if (n_ok != -1){
					xconf_error (MSG_U(E_NETMASKNEEDED
						,"A netmask is needed for interface %s\n"
						 "because you have supplied a network number")
						,device);
					ret = -1;
				}
			}
			unsigned long nummsk = ipnum_aip2l(nums.msk);
			if (n_ok == -1){
				unsigned long numip = ipnum_aip2l(nums.hst);
				unsigned long numnet = numip & nummsk;
				ipnum_ip2a (numnet,nums.net);
			}
			if (b_ok == -1){
				unsigned long n_nummsk = ~nummsk;
				unsigned long numnet = ipnum_aip2l(nums.net);
				unsigned long numbcast = numnet | (n_nummsk & 0xffffffffl);
				ipnum_ip2a (numbcast,nums.bcast);
			}
			if (!ipnum_validip(nums.hst,true)
				|| !ipnum_validip(nums.net,false)
				|| !ipnum_validip(nums.msk,false)
				|| !ipnum_validip(nums.bcast,false)){
				ret = -1;
				xconf_error (MSG_U(E_IVLIPINTER,
					"Invalid IP number for interface %s")
					,device);
			}
		}
	}
	return ret;
}

/*
	Load a network device module.
	Return -1 if any error
*/
int device_insmod (const char *devname)
{
	return netconf_system_if ("modprobe",devname);
}	


	

/*
	Initialise a network device and set the basic route to it.
*/
static int device_set (
	int dev_exist,		// Is the device configured in the kernel ?
	const char *device,	// Name of the device
	INTER_INFO &itf)
{
	int ret = -1;
	IFACE_NUMS nums;
	int ip_needed = device_ifaceinfo (device,itf,nums) != -1;
	if (ip_needed){
		/* #Specification: netconf / devices / modules
			If a device (eth?) is not available in the kernel
			linuxconf does a "modprobe device" to load the module.
		*/
		if (!dev_exist)	dev_exist = device_insmod (device) != -1;
		if (dev_exist){
			if (ip_needed){
				ret = devices_ifconfig(device,nums.hst,nums.msk,nums.bcast);
				if (ret == 0){
					char cmd[300];
					char gateway[PATH_MAX];
					int status = route_isactive(nums.net,gateway);
					if (status && strcmp(gateway,device)!=0){
						sprintf (cmd,"del -net %s",nums.net);
						ret = netconf_system_if ("route",cmd);
						status = 0;
					}
					if (!status){
						sprintf (cmd,"add -net %s %s",nums.net,device);
						ret = netconf_system_if ("route",cmd);
					}
				}
			}
		}else{
			xconf_error (MSG_U(E_NODEVICE
				,"Networking was configured\n"
				 "for the interface %s\n"
				 "but this device is not configured\n"
				 "in the kernel\n")
				,device);
		}
	}else if (!dev_exist){
		// No configuration (IP or IPX) for this device
		// but it is not there anyway
		ret = 0;
	}
	return ret;
}
const char *tb_netdevices[]={
	"eth0","eth1","eth2","eth3"
};
/*
	Configure the main communication devices (ethernet) and set the
	basic route.

	Return -1 if any error.
*/
int netconf_setdevices()
{
	/* #Specification: network devices / lookup
		netconf finds out which permanent network devices do exist
		by reading /proc/net/dev. Slip devices are ignored because
		they are not managed properply by netconf right now.

		In fact, netconf deal only with ETHx devices.
	*/
	int	ret = 0;
	HOSTINFO info;
	if (netconf_loadinfos(info) != -1){
		for (int i=0; i<NB_ETH; i++){
			const char *devname = tb_netdevices[i];
			ret |= device_set (devlist_devexist(devname)
				,devname
				,info.a[i]);
		}
	}
	return ret;
}




