#pragma implementation
/* #Specification: modules / elf only
	Module support in linuxconf only work for ELF systems.
*/
#include <stdio.h>
#ifndef LINUXCONF_AOUT
	#include <dlfcn.h>
#endif
#include <limits.h>
#include <ctype.h>
#include <stdlib.h>
#include <sys/stat.h>
#include "misc.h"
#include "module.h"
#include "misc.m"
#include <dialog.h>

static HELP_FILE help_modules ("misc","modules");

class LINUXCONF_MODULES: public ARRAY{
	/*~PROTOBEG~ LINUXCONF_MODULES */
public:
	LINUXCONF_MODULE *getitem (int no);
	/*~PROTOEND~ LINUXCONF_MODULES */
};

PUBLIC LINUXCONF_MODULE *LINUXCONF_MODULES::getitem(int no)
{
	return (LINUXCONF_MODULE*)ARRAY::getitem(no);
}

static LINUXCONF_MODULES modules;

PUBLIC LINUXCONF_MODULE::LINUXCONF_MODULE()
{
	modules.add (this);
}

PUBLIC LINUXCONF_MODULE::~LINUXCONF_MODULE()
{
	modules.remove (this);
}

/*
	Let the module add its own option to one menu.
	The "context" let the module identify which dialog it is
	The module is not forced to add options to the menu.
*/
PUBLIC VIRTUAL void LINUXCONF_MODULE::setmenu (
	DIALOG &,
	MENU_CONTEXT)
{
}

/*
	Let the module take control for some html query.
*/
PUBLIC VIRTUAL int LINUXCONF_MODULE::dohtml (const char *)
{
	return -1000;
}

/*
	Check if the user has selected one menu option related to this module
	Do nothing most of the time.
*/
PUBLIC VIRTUAL int LINUXCONF_MODULE::domenu (
	MENU_CONTEXT,		// context
	const char *)		// key
{
	return 0;
}

/*
	Check some file permissions related to the module.
	Do nothing most of the time.
*/
PUBLIC VIRTUAL int LINUXCONF_MODULE::fixperm (bool )
{
	return 0;
}

/*
	Give control to the module based on argv[0].
	A module foo may setup a symlink to linuxconf like this

	ln -s /bin/linuxconf /bin/foo

	and the execution of foo ..., will give control directly to the
	module.

	A module not supporting this, or which does not accept argv[0] as an
	alias to itself will return -1000. Any other value is the return
	code and the program (linuxconf) will terminate.
*/
PUBLIC VIRTUAL int LINUXCONF_MODULE::execmain (
	int,		// argc
	char *[])	// argv
{
	return -1;
}

/*
	Check if any module has something to add to this menu
*/
void module_setmenu (DIALOG &dia, MENU_CONTEXT context)
{
	int n = modules.getnb();
	for (int i=0; i<n; i++) modules.getitem(i)->setmenu (dia,context);
}
/*
	Probe the module for some update after configuration changes.
*/
int module_probe (
	int state,		// networking level 0, 1 or 2
					// at which state are we checking
	int target)		// idem, but the target of the general probe
{
	int ret = 0;
	int n = modules.getnb();
	for (int i=0; i<n; i++){
		if (modules.getitem(i)->probe (state,target)==-1){
			ret = -1;
			break;
		}
	}
	return 0;
}

/*
	Check if any module has something to do with this menu selection.
	Return != 0 if this menu event was managed by a module.
*/
int module_domenu (MENU_CONTEXT context, const char *key)
{
	int ret = 0;
	int n = modules.getnb();
	for (int i=0; i<n; i++){
		if (modules.getitem(i)->domenu (context,key)){
			ret = 1;
			break;
		}
	}
	return ret;
}

/*
	Tell the different module to check their file permissions.
	Most of the time, those modules will call the function
	Return != 0 if there was any errors.
*/
int module_fixperm (bool boottime)
{
	int ret = 0;
	int n = modules.getnb();
	for (int i=0; i<n; i++){
		ret |= modules.getitem(i)->fixperm (boottime);
	}
	return ret;
}

/*
	Try to pass control to a module based on argv[0]
	Return -1000 if no module accept control.
*/
int module_execmain (int argc, char *argv[])
{
	int ret = -1000;
	int n = modules.getnb();
	for (int i=0; i<n; i++){
		int code = modules.getitem(i)->execmain (argc, argv);
		if (code != -1000){
			ret = code;
			break;
		}
	}
	return ret;
}


/*
	Dispatch an HTML request to a module
	Return -1000 if no module accepted control.
*/
int module_dohtml(const char *key)
{
	int ret = -1000;
	int n = modules.getnb();
	for (int i=0; i<n; i++){
		int code = modules.getitem(i)->dohtml (key);
		if (code != -1000){
			ret = code;
			break;
		}
	}
	if (ret == -1000) html_ivldurl();
	return ret;
}


class MODULE_INFO: public ARRAY_OBJ{
public:
	char path[PATH_MAX+1];
	char active;
	/*~PROTOBEG~ MODULE_INFO */
public:
	MODULE_INFO (const char *buf);
	MODULE_INFO (void);
	int locate (char realpath[PATH_MAX]);
	int write (void);
	/*~PROTOEND~ MODULE_INFO */
};

static const char K_MODULE[]="module";
static const char K_LIST[]="list";

PUBLIC MODULE_INFO::MODULE_INFO(const char *buf)
{
	int a;
	sscanf (buf,"%d %s",&a,path);
	active = (char)(a != 0);
}

PUBLIC MODULE_INFO::MODULE_INFO()
{
	path[0] = '\0';
	active = 1;
}

PUBLIC int MODULE_INFO::write ()
{
	if (path[0] != '\0'){
		char buf[PATH_MAX+4];
		sprintf (buf,"%d %s",active,path);
		linuxconf_add (K_MODULE,K_LIST,buf);
	}
	return 0;
}


/*
	Locate a module based on the version of linuxconf
	Return -1 if the module can't be found.
	realpath will contain the path linuxconf was looking for.

	Return the subrevision number.
*/
PUBLIC int MODULE_INFO::locate (char realpath[PATH_MAX])
{
	/* #Specification: modules / revision management
		Linuxconf's module follow a strict numbering scheme to avoid
		strange incompatibilities. For each linuxconf version, linuxconf
		expect a module using the same version number. Module may
		use a seperate number to differentiate internal releases. Linuxconf
		will always used the highest internal release.

		For example, given a module "/var/project/dummy",
		linuxconf 1.6 will look for /var/project/dummy.so.1.6.0. If it
		exist, linuxconf will look for /var/project/dummy.so.1.6.1 and
		so on. It will use the highest found.

		To give some flexibility to the user, linuxconf allows one
		to specify the full path of the module. So linuxconf first
		try to open this file. If it exist, it is believe to be the module.
		If not, the extension .so.LINUXCONF_REV.SUBREV is tried.

		The idea is to let someone say "Use /var/project/dummy.so.1.7.4"
		just to see if it is better than 1.7.5 for example.
	*/
	int ret = -1;
	struct stat st;
	if (stat(path,&st)!=-1 && (st.st_mode & S_IXUSR)){
		ret = 0;
		strcpy (realpath,path);
	}else{
		char trypath[PATH_MAX];
		extern char *revision;
		int lentry = sprintf (trypath,"%s.so.%s.",path,revision);
		SSTRINGS lst;
		int nb = dir_getlist_p (trypath,lst);
		for (int i=0; i<nb; i++){
			const char *p = lst.getitem(i)->get();
			const char *ext = p + lentry;
			if (isdigit(ext[0])){
				int subrev = atoi(ext);
				if (subrev > ret) ret = subrev;
			}
		}
		if (ret != -1){
			sprintf (realpath,"%s%d",trypath,ret);
		}
	}
	return ret;
}

class MODULE_INFOS: public ARRAY{
	/*~PROTOBEG~ MODULE_INFOS */
public:
	MODULE_INFOS (void);
	MODULE_INFO *getitem (int no);
	int write (void);
	/*~PROTOEND~ MODULE_INFOS */
};

PUBLIC MODULE_INFO *MODULE_INFOS::getitem(int no)
{
	return (MODULE_INFO *)ARRAY::getitem(no);
}

PUBLIC MODULE_INFOS::MODULE_INFOS()
{
	SSTRINGS tb;
	int n = linuxconf_getall (K_MODULE,K_LIST,tb,0);
	for (int i=0; i<n; i++) add (new MODULE_INFO(tb.getitem(i)->get()));
}

PUBLIC int MODULE_INFOS::write ()
{
	linuxconf_removeall (K_MODULE,K_LIST);
	for (int i=0; i<getnb(); i++) getitem(i)->write();
	return linuxconf_save();
}

static void module_addfields (MODULE_INFOS &infos, DIALOG &dia, int f)
{
	MODULE_INFO *in = infos.getitem(f);
	dia.newf_str(MSG_U(F_MODPATH,"Module path"),in->path,PATH_MAX);
	dia.newf_chk("",in->active,MSG_U(F_MODACTIVE,"This module is active"));
}

void module_config()
{
	MODULE_INFOS infos;
	int nof = 0;
	infos.add (new MODULE_INFO);
	DIALOG dia;
	int i;
	int n = infos.getnb();
	for (i=0; i<n; i++){
		module_addfields(infos,dia,i);
	}
	dia.addwhat (MSG_U(I_EMPTYSLOT,"one empty slot in the dialog"));
	while (1){
		MENU_STATUS code = dia.edit (MSG_U(T_MODLIST,"List of modules")
			,MSG_U(I_MODLIST
				,"You can add Linuxconf's module here.\n"
				 "The modules are loaded each time linuxconf is used\n"
				 "and enhance its functionnality.")
			,help_modules
			,nof,MENUBUT_CANCEL|MENUBUT_ACCEPT|MENUBUT_ADD);
		if (code == MENU_CANCEL || code == MENU_ESCAPE){
			break;
		}else if (code == MENU_ADD){
			infos.add (new MODULE_INFO);
			module_addfields(infos,dia,n);
			n++;
		}else if (code == MENU_ACCEPT){
			for (i=0; i<n; i++){
				MODULE_INFO *in = infos.getitem(i);
				char path[PATH_MAX];
				if (in->active
					&& in->path[0] != '\0'
					&& in->locate(path)==-1){
					xconf_error (MSG_U(E_MODNOPATH
						,"Module %s does not exist"),in->path);
					nof = i*2;
					break;
				}
			}
			if (i == n){
				infos.write();
				/* #Specification: modules / activation
					Whenever we accept the change in the module list,
					linuxconf will try to load and activate a module.

					Doing a dlopen on a module twice is not a problem.
					Unloading a module is not really possible during
					a session, so removal or desactivation of a module
					will take effect at the next session.
				*/
				module_load();
				break;
			}
		}
	}
}

void module_load ()
{
	#ifndef LINUXCONF_AOUT
		MODULE_INFOS infos;
		for (int i=0; i<infos.getnb(); i++){
			MODULE_INFO *in = infos.getitem(i);
			if (in->active){
				char path[PATH_MAX];
				if (in->locate(path)==-1){
					fprintf (stderr,MSG_R(E_MODNOPATH),in->path);
				}else{
					dlopen (path,RTLD_LAZY);
				}
			}
		}
	#endif
}

