#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "diadef.h"
#include "dialog.h"
#include "dialog.m"

PUBLIC FIELD::FIELD(const char *_prompt)
{
	readonly = 0;
	prompt = strdup(_prompt);
	box.width = 30;		// Default visible width
	maybeempty = 1;
	may_select = 0;
}


PUBLIC VIRTUAL FIELD::~FIELD()
{
	free (prompt);
}

/*
	Indicate that this field may not be empty.
	This does not apply to all field. For checkbox, it is meaningless.
*/
PUBLIC VIRTUAL void FIELD::set_noempty()
{
	maybeempty = 0;
}

/*
	Indicate that this field may be selected or not.
*/
PUBLIC VIRTUAL void FIELD::set_selectable(int _may_select)
{
	may_select = (char)(_may_select != 0);
}
/*
	Return if this field is selectable
*/
PUBLIC VIRTUAL int FIELD::is_selectable()
{
	return may_select;
}



/*
	This function is called when the user quits the dialog (with accept)
	each field may return -1 if something is not appropriate.

	The field is responsible to signal the error and it will become
	the current field immediatly.
*/
PUBLIC VIRTUAL int FIELD::post_validate()
{
	return 0;
}
/*
	Is the field editable or is it write protect.
	Title field do use this feature.
*/
PUBLIC int FIELD::is_readonly()
{
	return readonly;
}
/*
	Set the readonly status of a field.
*/
PUBLIC void FIELD::set_readonly()
{
	readonly = 1;
}

/*
	Process conditionnally a message and conditionnally redraw
	yourself if needed. See radio.c for the intended application.
*/
PROTECTED VIRTUAL void FIELD::processmsg(WINDOW *, FIELD_MSG &, int)
{
}
/*
	Get the width of the first column of a field.
	This is only meaningful for a Menu item.
*/	  
PROTECTED VIRTUAL int FIELD::getwidths (int [])
{
	return 0;
}
/*
	Set the width of the first column of a field.
	This is only meaningful for a Menu item.
*/	  
PROTECTED VIRTUAL void FIELD::setwidths (int, int [])
{
}
/*
	Return the second string of a menu item
*/
PUBLIC VIRTUAL const char *FIELD::getmenustr()
{
	return NULL;
}

/*
	Build a key that uniquely identify this field in the dialog
*/
PUBLIC VIRTUAL void FIELD::format_htmlkey(char *key, int nof)
{
	html_formatkey (key,"%s-%d",prompt,nof);
}


PRIVATE void FIELD_STRING_BASE::init (int maxsiz)
{
	x.input = x.scroll = 0;
	password_mode = 0;
	size = maxsiz;
	buf = (char*)malloc_err (size+1);
	maybeempty = 1;
}

PROTECTED FIELD_STRING_BASE::FIELD_STRING_BASE(
	const char *_prompt,
	const char *_str,
	int maxsiz)
	: FIELD (_prompt)
{
	init(maxsiz);
	strncpy (buf,_str,maxsiz);
	buf[maxsiz] = '\0';
}

PROTECTED FIELD_STRING_BASE::FIELD_STRING_BASE(
	const char *_prompt,
	int maxsiz)
	: FIELD (_prompt)
{
	init(maxsiz);
	buf[0] = '\0';
}
PUBLIC FIELD_STRING_BASE::~FIELD_STRING_BASE()
{
	free (buf);
}

/*
	Return != 0 if the field is empty
*/
PUBLIC int FIELD_STRING_BASE::is_empty()
{
	int ret = 1;
	char *pt = buf;
	while (*pt != '\0'){
		if (*pt != ' '){
			ret = 0;
			break;
		}
		pt++;
	}
	return ret;
}

PUBLIC int FIELD_STRING_BASE::post_validate()
{
	int ret = 0;
	if (!maybeempty && is_empty()){
		xconf_error (MSG_U(E_NOEMPTY,"This field may not be empty"));
		ret = -1;
	}
	return ret;
}
PUBLIC FIELD_STRING::FIELD_STRING(
	const char *_prompt,
	char *_str,
	int maxsiz)
	: FIELD_STRING_BASE (_prompt, _str, maxsiz)
{
	str = _str;
	backup = strdup(_str);
}

PUBLIC FIELD_STRING::~FIELD_STRING()
{
	free (backup);
}

PUBLIC FIELD_PASSWORD::FIELD_PASSWORD(
	const char *_prompt,
	SSTRING &_str)
	: FIELD_SSTRING (_prompt,_str)
{
	buf[0] = '\0';
	password_mode = 1;
}

PUBLIC void FIELD_STRING::save()
{
	strcpy (str,buf);
	strip_end (str);
}

PUBLIC void FIELD_STRING::restore()
{
	strcpy (str,backup);
}

/*
	Draw only the input part of a field
*/
PUBLIC void FIELD_STRING_BASE::drawtxt (WINDOW *dialog)
{
	int blank_start;
	wattrset(dialog, inputbox_attr);
	wmove(dialog, box.y,box.x);
	if (password_mode){
		blank_start = 0;
	}else{
		// Make sure the string is not too long
		char *instr = buf + x.scroll;
		char *last_visible = instr + box.width;
		char tmp = *last_visible;

		*last_visible = '\0';
		waddstr (dialog,instr);
		blank_start = strlen(instr);
		*last_visible = tmp;
	}
	const int box_width = box.width;
	for (int i=blank_start; i<box_width; i++) waddch (dialog,' ');
}

/*
	Draw the field with the prompt
*/
PUBLIC void FIELD_STRING_BASE::html_draw(int nof)
{
	html_printf ("<tr><td>%s<td>",prompt);
	if (readonly){
		html_printf ("%s\n",buf);
	}else{
		char key[100];
		format_htmlkey (key,nof);
		html_defvar (password_mode ? "password" : "text",key,buf
			,"size=30 maxlength=256");
		html_defvarcur (key,buf);
	}
}

/*
	Grab the input of the user received from the www browser.
	Validate that the Draw the field with the prompt
	and compare the old value with the content of the buffer. If there
	is a mismatch, it means this www form was too old and the input
	is refused.
*/
PUBLIC int FIELD_STRING_BASE::html_validate(int nof)
{
	int ret = -1;
	char key[100];
	format_htmlkey (key,nof);
	const char *old_val = html_getoldval (key);
	const char *new_val = html_getval (key);
	fprintf (stderr,"validate %s val :%s: old :%s: buf :%s:\n",key,new_val,old_val,buf);
	if (strcmp(buf,old_val)==0){
		strcpy (buf,new_val);
		ret = 0;
	}
	return ret;
}



/*
	Draw the field with the prompt and the surrounding box
*/
PUBLIC void FIELD::draw (WINDOW *dialog)
{
	int posy = box.y;
	wmove(dialog, posy,1);
	wattrset(dialog, dialog_attr);
	waddstr (dialog,prompt);
	int len = strlen(prompt);
	int lastx = box.x-2;
	for ( ; len < lastx; len++) waddch (dialog,' ');
	drawtxt(dialog);
}

/*
	Position the cursor in the field
*/
PUBLIC VIRTUAL void FIELD::setcursor(WINDOW *dialog)
{
	wmove (dialog,box.y,box.x);
}
/*
	Called when the field loose selection. This was
	done mostly for menu entries which is a very special case.
	The default behavior does nothing.
*/
PUBLIC VIRTUAL void FIELD::unselect(WINDOW *)
{
}
PUBLIC void FIELD_STRING_BASE::setcursor(WINDOW *dialog)
{
	wmove (dialog,box.y,box.x+x.input-x.scroll);
}

PUBLIC void FIELD_STRING_BASE::dokey (
	WINDOW *dialog,
	int key,
	FIELD_MSG &)
{
	if (readonly) return;
	int input_x = x.input;
	int scroll = x.scroll;
	const int box_width = box.width;
	const int box_y = box.y;
	const int box_x = box.x;
	char *instr = buf;
	int curlen = strlen(instr);
	int last_scroll = scroll;
	int mustdraw = 0;
	switch (key){
	case 1:		/* ^A like emacs */
	case KEY_HOME:
		input_x = 0;
		scroll = 0;
		break;
	case 5:		/* ^E like Emacs */
	case KEY_END:
		input_x = curlen;
		if (input_x > box_width){
			scroll = input_x - box_width + 1;
		}else{
			scroll = 0;
		}
		break;
	case KEY_LEFT:
		if (input_x > 0) input_x--;
		break;
	case KEY_RIGHT:
		if (input_x < curlen) input_x++;
		break;
	case KEY_BACKSPACE:
	case DEL:
		if (input_x > 0) {
			int i;
			input_x--;
			for (i=input_x; i<curlen; i++) instr[i] = instr[i+1];
			mustdraw = 1;
		}
		break;
	case 4:		/* ^d like Emacs */
	case KEY_DC:
		if (input_x < curlen){
			int i;
			for (i=input_x; i<curlen; i++) instr[i] = instr[i+1];
			mustdraw = 1;
		}
		break;
	default:
		if (isprint(key)) {
			if (input_x < MAX_LEN) {
				wattrset(dialog, inputbox_attr);
				if (input_x < curlen){
					int i;
					for (i=curlen+1; i>input_x; i--) instr[i] = instr[i-1];
					mustdraw = 1;
				}
				instr[input_x] = key;
				instr[curlen+1] = '\0';
				input_x++;
				if (input_x - scroll == box_width) {
					scroll++;
				}else{
					wmove(dialog, box_y, input_x-1 + box_x);
					if (!password_mode) waddch(dialog, key);
				}
			}
		}
	}
	if (input_x < scroll){
		scroll--;
	}else if (input_x - scroll == box_width){
		scroll++;
	}
	x.input = input_x;
	x.scroll = scroll;
	if (mustdraw || last_scroll != scroll) drawtxt (dialog);
}


PUBLIC FIELD_SSTRING::FIELD_SSTRING(
	const char *_prompt,
	SSTRING &_str)
	: FIELD_STRING_BASE (_prompt, _str.get(), _str.getmaxsiz()), str(_str)
{
	backup.setfrom (_str);
}

PUBLIC void FIELD_SSTRING::save()
{
	str.setfrom(buf);
}

PUBLIC void FIELD_SSTRING::restore ()
{
	str.setfrom(backup);
}
/*
	Draw the field with the prompt
*/
PUBLIC void FIELD_SSTRING::html_draw(int nof)
{
	html_printf ("<tr><td>%s<td>",prompt);
	if (readonly){
		html_printf ("%s\n",buf);
	}else{
		char key[100];
		format_htmlkey (key,nof);
		html_defvar (password_mode ? "password" : "text",key,buf
			,"size=30 maxlength=256");
		html_defvarcur (key,backup.get());
	}
}

/*
	Add a string field to the dialog.
*/
PUBLIC FIELD_STRING *DIALOG::newf_str(
	const char *prompt,
	char *str,
	int maxsiz)
{
	FIELD_STRING *s = new FIELD_STRING(prompt,str,maxsiz);
	add (s);
	return s;
}

/*
	Add a SSTRING field to the dialog.
*/
PUBLIC FIELD_SSTRING *DIALOG::newf_str(
	const char *prompt,
	SSTRING &str)
{
	FIELD_SSTRING *s = new FIELD_SSTRING(prompt,str);
	add (s);
	return s;
}
/*
	Add a password field to the dialog.
*/
PUBLIC FIELD_PASSWORD *DIALOG::newf_pass(
	const char *prompt,
	SSTRING &str)
{
	FIELD_PASSWORD *s = new FIELD_PASSWORD(prompt,str);
	add (s);
	return s;
}
