//==========================================================
// klogwidget.cpp
// Part of Kwatch
// (c) 1997 by Ralph C. Weichert

#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>

#include <kurl.h>
#include <kapp.h>

#include "utils.h"
#include "klogwidget.moc"

#define maxlines 500
#define bufsize 200
#define numlines 10

KLogFile::KLogFile( const char * name, QObject * parent ) :
    QObject( parent )
{
    initMetaObject();

    // debug( "Adding %s", name );
    file = new QFile( name );
    activate();
}

KLogFile::~KLogFile() {
    delete file;
}

void KLogFile::activate() {
    if ( file->isOpen() ) return;
    file->open( IO_ReadOnly | IO_Raw );
    if ( file->isOpen() )
        // Go to EOF
        file->at( file->size() );
    else {
        // Try again in a minute
        QTimer::singleShot( 60000, this, SLOT(activate()) );
    }
}

bool KLogFile::readMore() {
    char buf[bufsize];
    QString text;
    int len;

    if ( !file->isOpen() ) return FALSE;
    
    do {
        len = file->readBlock( buf, bufsize-1 );
        if (len < 0) {
            text += "\nRead Error!\n";
            file->close();
            QTimer::singleShot( 60000, this, SLOT(activate()) );
            break;
        }
        buf[len] = 0;
        text += buf;
    } while (len == bufsize-1);
    if ( !text.isEmpty() ) {
        emit moreLines( this, text );
        return TRUE;
    }
    return FALSE;
}
    

KLogWidget::KLogWidget( KConfigBase * cfg, QWidget * parent, const char * name ) :
    QMultiLineEdit( parent, name )
{
    initMetaObject();

    setFrameStyle( WinPanel | Sunken );
    setReadOnly( TRUE );

    installEventFilter( this );

    rmbmenu = NULL;
    r = 0; c = 0;
    
    loglist = new QList<KLogFile>();
    loglist->setAutoDelete( TRUE );
    current = NULL;

    setupWindow( cfg );
    
    timer = new QTimer( this );
    timer->start(0);
    connect( timer, SIGNAL(timeout()), SLOT(timeout()) );
}

KLogWidget::~KLogWidget() {
    delete loglist;
    delete logfiles;
}

void KLogWidget::timeout() {
    bool text_added = FALSE;
    QListIterator<KLogFile> it( *loglist );
    for ( ; it.current(); ++it )
        if ( it.current()->readMore() )
            text_added = TRUE;
    if ( text_added )
        timer->changeInterval(0);
    else
        timer->changeInterval(1000);
}

//leon
void KLogWidget::SetShowDate(bool d) {
        ShowDate = d;
}
//leon

void KLogWidget::moreText( KLogFile * f, QString s ) {
    if (s.isEmpty()) return;

    QString more = s;

    if (f != current ) {
        current = f;
        more.prepend( "***\n " );
//Leon Van Heerden {Dec 7 1997}
if(ShowDate == TRUE) {
        QDateTime tmpDate = tmpDate.currentDateTime();          //Leon Van Heerden {Dec 7 1997}
        more.prepend( " ] " );
                more.prepend( tmpDate.toString() );
                more.prepend( " [ " );
                }
//Leon Van Heerden {Dec 7 1997}
        more.prepend( f->file->name() );
        more.prepend( "\n***" );
    }
    insertAt( more, r, c );
    for (int l = numLines() - maxlines; l > 0; l--)
        removeLine(0);
    r = numLines() - 1;
    while (0 == (c = lineLength( r )) && r > 0) r--;
    if (more.right(1) == "\n") r++;
    setCursorPosition ( numLines()-1, 0 );
}

void KLogWidget::setRMBMenu( QPopupMenu *m ) {
    rmbmenu = m;
    m->installEventFilter( this );
}

// dropEvent: file is added to file list
void KLogWidget::dropEvent( KDNDDropZone * dz ) {
    QStrList & list = dz -> getURLList();
    for (char * s = list.first(); s; s = list.next()) {
        KURL url( s );
        if (url.isMalformed()) continue;
        if (strcmp( url.protocol(), "file")) continue;
        addFile( url.path() );

//         KConfig * cfg = kapp->getConfig();
//         cfg->setGroup( "Log" );
//         QString files = cfg->readEntry( "Files", "" );

        logfiles->append( url.path() );

//             cfg->writeEntry( "Files", url.path() );
//         else
//             cfg->writeEntry( "Files", files + ":" + url.path() );
    }
}

void KLogWidget::setupWindow( KConfigBase * cfg ) {
    logfiles = getFileList( cfg );

    QListIterator<KLogFile> it( *loglist );
    for ( ; it.current(); ++it )
        if ( logfiles -> find( it.current()->file->name() ) < 0 )
            loglist->removeRef( it.current() );
    for ( char * c = logfiles -> first(); c; c = logfiles -> next() )
        addFile( c );
    // delete files;

    // KConfig * cfg = kapp -> getConfig();
    cfg -> setGroup( "Window" );
    QFont f = cfg -> readFontEntry( "Font", &font() );
    setFont( f );
    QColor fg = cfg -> readColorEntry( "Foreground", &foregroundColor() );
    QColor bg = cfg -> readColorEntry( "Background", &backgroundColor() );

    // This is stolen from kedit:
    QPalette mypalette = palette().copy();

    QColorGroup cgrp = mypalette.normal();
    QColorGroup ncgrp(fg,cgrp.background(),
                      cgrp.light(),cgrp.dark(),cgrp.mid(),fg,bg);

    mypalette.setNormal(ncgrp);
    mypalette.setDisabled(ncgrp);
    mypalette.setActive(ncgrp);

    setPalette(mypalette);
    setBackgroundColor(bg);
}

void KLogWidget::reOpen() {
    QListIterator<KLogFile> it( *loglist );
    KLogFile * f;

    for ( ; (f = it.current()); ++it ) {
        if ( f->file->isOpen() ) f->file->close();
        f->activate();
    }
}

void KLogWidget::saveOptionsTo( KConfigBase * cfg ) {
    putFileList( cfg, *logfiles );

    cfg->setGroup( "Window" );
    cfg->writeEntry( "Font", font() );
    cfg->writeEntry( "Foreground", foregroundColor() );
    cfg->writeEntry( "Background", backgroundColor() );
}

void KLogWidget::reConfig( KConfigBase * cfg ) {
    setupWindow( cfg );
}

void KLogWidget::addFile( const char * n ) {
    KLogFile *f = findName( n );
    if (!f) {
        f = new KLogFile( n );
        if ( ! f->file->isOpen() )
            moreText( f, "Error: cannot open file\n" );
        connect( f, SIGNAL(moreLines(KLogFile*,QString)),
                 SLOT(moreText(KLogFile*,QString)) );
        loglist->append( f );
    }
}

void KLogWidget::delFile( const char * n ) {
    KLogFile *f = findName( n );
    if (f) loglist->removeRef( f );
}

KLogFile * KLogWidget::findName( const char * n ) {
    QString path = absFilePath( n );
    QListIterator<KLogFile> it( *loglist );

    for ( ; it.current(); ++it ) {
        KLogFile * f = it.current();
        if ( path == f->file->name() ) return f;
    }
    return NULL;
}

bool KLogWidget::eventFilter( QObject *obj, QEvent *ev ){
    static QPoint rmbpoint( -10, -10 );

    if ( ev->type() == Event_MouseButtonRelease ) { 
        QPoint tmp = rmbpoint;
        rmbpoint = QPoint( -10, -10 );
        if ( obj == rmbmenu && QCursor::pos() == tmp )
            return TRUE;
    }
        
    if ( ev->type() != Event_MouseButtonPress || obj != this )
        return FALSE;
    
    QMouseEvent *e = (QMouseEvent *) ev;
    if ( e->button() != RightButton )
        return FALSE;

    rmbpoint = QCursor::pos();

    if ( rmbmenu )
        rmbmenu -> popup ( rmbpoint );

    return TRUE;
} 

// EOF
