// $Id: greetings.cpp,v 1.4 1998/01/11 15:53:13 knoll Exp $
#include <qdir.h>
#include "greetings.moc"

#include "data.h"
#include "messages.h"
#include "kvfiledlg.h"
#include "messagelist.h"

#include <unistd.h>

#include <kmsgbox.h>
#include <kiconloader.h>
#include <kmenubar.h>
#include <kstdaccel.h>
#include <qlayout.h>
#include <qstring.h>
#include <qregexp.h>
#include <qpopmenu.h>

#include "global.h"

Greetings::Greetings(const char *name,
		     Messages  *m)
  : KTopLevelWidget(name)
{
  data = Data::getData();
  messages = m;
  greetingDir = 0;

  setupMenuBar();
  setupToolBar();
  init();

  initNewDialog();
  readConfig();

}

Greetings::~Greetings()
{
  writeConfig();

  delete greetingDir;
  delete recorddialog;
}

void
Greetings::readConfig()
{
  QString str;
  int w,h;

  setMinimumSize(200, 100);
  cfg->setGroup("geometry");
  str = cfg->readEntry("greetings", "250 300");
  sscanf(str.data(), "%d %d", &w, &h);
  resize(w, h);

  str = cfg->readEntry("greetings-new", "300 350");
  sscanf(str.data(), "%d %d", &w, &h);
  recorddialog->resize(w, h);
}

void
Greetings::writeConfig(bool sync)
{
  QString str;

  cfg->setGroup("geometry");
  str.sprintf("%d %d", width(), height());
  cfg->writeEntry("greetings", str);

  str.sprintf("%d %d", recorddialog->width(), recorddialog->height());
  cfg->writeEntry("greetings-new", str);

  if(sync) cfg->sync();
}

void
Greetings::setupMenuBar()
{
  KMenuBar *menubar = new KMenuBar(this);

  QPopupMenu *file = new QPopupMenu();
  file->insertItem( nls("New"), this, SLOT(slotNew()) );
  file->insertItem( nls("Play") , this, SLOT(slotPlay()) );
  file->insertItem( nls("Delete") , this, SLOT(slotDelete()) );
  file->insertSeparator();
  file->insertItem( nls("&Close"), this, SLOT(slotFinished()), keys->close() );
  menubar->insertItem(nls("&File"),file);

  menubar->insertSeparator();
  QPopupMenu *help = new QPopupMenu();
  help->insertItem( nls("Help"), this, SLOT(slotHelp()) );
  menubar->insertItem(nls("&Help"),help);

  setMenu(menubar);
}

void
Greetings::setupToolBar()
{
  KIconLoader* loader = kapp->getIconLoader();

  KToolBar *toolbar = new KToolBar(this);

  toolbar->insertButton(loader->loadIcon("filenew2.xpm"),0,
			SIGNAL(clicked()),this,
			SLOT(slotNew()),TRUE,nls("New Message"));
  toolbar->insertButton(loader->loadIcon("kvplay.xpm"),0,
			SIGNAL(clicked()),this,
			SLOT(slotPlay()),TRUE,nls("Play Message"));
  toolbar->insertButton(loader->loadIcon("filedel2.xpm"),0,
			SIGNAL(clicked()),this,
			SLOT(slotDelete()),TRUE,nls("Delete Message"));
  toolbar->insertSeparator();
  toolbar->insertButton(loader->loadIcon("exit.xpm"),0,
			SIGNAL(clicked()),this,
			SLOT(slotFinished()),TRUE,nls("Close"));
  addToolBar(toolbar);
}

void
Greetings::init()
{
  view = new QWidget(this);
  setView(view);

  QBoxLayout *layout = new QHBoxLayout(view,4);

  greetings = new MessageList(0, view, "greetings");
  greetings->setColumn(1, nls("Name"), greetings->columnWidth(1) );
  greetings->setColumn(3, nls("Created"), greetings->columnWidth(3) );
  layout->addWidget(greetings,1);

  getGreetings();

  dropZone = new KDNDDropZone(greetings, DndURL);
  connect(dropZone, SIGNAL(dropAction(KDNDDropZone *)), 
	  SLOT(slotDrop()) );
}

void
Greetings::initNewDialog()
{
  recorddialog = new QWidget();
  recorddialog->setCaption(nls("New greeting message"));
  recorddialog->setMinimumSize(250,250);

  QBoxLayout *reclayout = new QVBoxLayout(recorddialog,5);
  QLabel *text1 = new QLabel(nls("Import existing sound file:")
			     ,recorddialog);
  text1->setMargin(0);
  text1->setMinimumSize(text1->sizeHint());
  reclayout->addWidget(text1);
  
  filedlg = new KVFileDlg(recorddialog);
  filedlg->setMinimumSize(100,100);
  reclayout->addWidget(filedlg,1);
  connect(filedlg, SIGNAL(fileSelected(const char *)), 
	  this, SLOT(slotImportFile(const char *)) );
  connect(filedlg, SIGNAL(canceled()),
	  this, SLOT(slotNewFinished()) );
  
  QFrame *frame1 = new QFrame(recorddialog);

  frame1->setMinimumSize(0,2);
  frame1->setMaximumSize(10000,2);
  frame1->setFrameStyle(QFrame::HLine | QFrame::Plain);
  frame1->setLineWidth(2);
  reclayout->addWidget(frame1);
  
  QLabel *text2 = new QLabel(nls("Record a message with local headset:"), 
			     recorddialog);
  text2->setMargin(0);
  text2->setMinimumSize(text2->sizeHint());
  reclayout->addWidget(text2);

  QBoxLayout *layout2 = new QHBoxLayout();
  reclayout->addLayout(layout2);

  entergreeting = new QLineEdit(recorddialog);
  entergreeting->setMinimumSize(80,20);
  entergreeting->setMaximumSize(500,20);
  connect(entergreeting, SIGNAL(returnPressed()), 
	  this, SLOT(slotRecFromHeadset()) );
  layout2->addWidget(entergreeting);

  layout2->addStrut(20);
  
  QPushButton *recordbutton = new QPushButton(nls("Record"),
					      recorddialog,"record");
  recordbutton->setMinimumSize(recordbutton->sizeHint());
  recordbutton->setMaximumSize(recordbutton->sizeHint());  
  layout2->addWidget(recordbutton);
  connect(recordbutton,SIGNAL(clicked()), this, SLOT(slotRecFromHeadset()) );
}

void Greetings::getGreetings()
{
  QString aStr;

  if(greetingDir != 0)
    delete greetingDir;
  gcfg->setGroup("location");
  greetingDir = new QDir(gcfg->readEntry("voiceMessages"));
  greetingDir->setFilter(QDir::Files);

  greetings->clear();

  indexfilename=greetingDir->path();
  indexfilename+="/Index";
  makeIndex();

  QFile *index = new QFile(indexfilename);
  if(!index->open(IO_ReadOnly))
    {
      KMsgBox::message(0, nls("Warning"),
		       nls("Could not get Index file"));
      return;
    }

  char line[128];
  while(1)
    {
      if(index->readLine(line,128) == 0) break;
      aStr.sprintf(" \n%s",line);
      greetings->insertItem(aStr);
    }
  index->close();
  delete index;
}

void Greetings::slotFinished()
{
  emit finished();
}
  
void Greetings::slotPlay()
{
  if(greetings->currentItem() == -1) return;

  QString aFileName;
  aFileName=greetingDir->path();
  aFileName+="/";
  aFileName+=(*greetingDir)[greetings->currentItem()];

  messages->playVoice((const char *)aFileName);
}

void Greetings::slotDelete()
{
  if(greetings->currentItem() == -1) return;

  int yes = KMsgBox::yesNo(0,nls("Delete Voice Mail"),
			   nls("Are you sure, you\n"
			       "want to delete this message?") );
  if(yes == 1)
    {
      //remove file
      QString aFileName;
      aFileName = greetingDir->path();
      aFileName += "/";
      aFileName += greetings->text(greetings->currentItem(), 1);
      debug("deleting %s\n",(const char *)aFileName);
      if(!greetingDir->remove( (const char *)aFileName ))
      	debug("couldn't delete\n");

      getGreetings();
    }
}

void Greetings::slotNew()
{
  hide();
  recorddialog->show();
}

void Greetings::slotNewFinished()
{
  show();
  recorddialog->hide();
}

void Greetings::slotImportFile(const char *name)
{
  gcfg->setGroup("modem");
  if(gcfg->readEntry("type") == "UMC")
  {
        KMsgBox::message(0,nls("Sorry"),
		     nls("vgetty does not support conversion\n"
		       "to the UMC voice format.") );
	return;
  }

  //get extension and base
  QString filename;
  filename = strdup(name);
  int index  = filename.findRev('.');
  QString extension = filename.right(filename.size() - index - 1);

  gcfg->setGroup("location");
  QString greetingfile = gcfg->readEntry("voiceMessages");
  int index2 = filename.findRev('/');
  greetingfile = greetingfile + filename.mid(index2, index-index2);

  debug("greetingfile:%s",(const char *) greetingfile);
  QString tmpfile = "/tmp/__kvoice__wavtopvf.pvf";

  if(extension == ".wav") {
    KProcess proc;
    proc.setExecutable("wavtopvf");
    proc << (const char *) filename << tmpfile;
    proc.start(KProcess::Block);
  } else {
    KMsgBox::message(0,nls("Error"),
		     nls("Unknown file format") );
    return;
  }
  
  KProcess proc;
  proc.setExecutable("pvftormd");
  proc << data->getModemType() << data->getCompression();
  proc << tmpfile << greetingfile;
  proc.start(KProcess::Block);

  getGreetings();

  slotNewFinished();
}

// records a greeting message
void Greetings::slotRecFromHeadset()
{
  recorddialog->hide();

  QString file=entergreeting->text();
  if(file.isEmpty()) return;
  file =  "/" + file;
  file = greetingDir->path() + file;
  debug("%s\n",(const char *)file);


  int yes= KMsgBox::yesNo(0,nls("Record the Greeting message"), 
			  nls("Lift up your telphone handset and\n"
			      "start talking after the signal.\n\n"
			      "Hang up the phone after you"
			      " finished your message."), 
				0, nls("Start"), nls("Cancel") );
  if ( yes ==1 ) {
    slotNewFinished();
	  
    proc = new KProcess();
    proc->setExecutable("vm");
    *proc << "record" << "-H" << "-w" << (const char *)file;
    connect(proc, SIGNAL(processExited(KProcess *)), 
	    this, SLOT(slotRecordFinished(KProcess *)) );
    connect(proc, SIGNAL(receivedStdout(KProcess *, char *, int)),
	    this, SLOT(slotRecordStdout(KProcess *, char *, int)) );
    connect(proc, SIGNAL(receivedStderr(KProcess *, char *, int)),
	    this, SLOT(slotRecordStdout(KProcess *, char *, int)) );
    // fixme: setup communication
    proc->start(KProcess::NotifyOnExit, KProcess::AllOutput);
  }
}

void Greetings::slotRecordStdout(KProcess *, char * buffer, int buflen)
{
  QString buf = buffer;
  buf.truncate(buflen);
  debug("in recordStdout\n");
  fprintf(stdout,"%s",(const char *)buf);
}

void Greetings::slotRecordFinished(KProcess *proc)
{
  debug("finished recording the message.\n");
  delete proc;
  getGreetings();
}

void
Greetings::slotDrop()
{
  QString  file;
  QStrList *urlList;

  urlList = &dropZone->getURLList();

  for (file=urlList->first(); !file.isEmpty() ; file=urlList->next())
  {
    file.replace("file:","");
    slotImportFile(file);
  }
}

void
Greetings::slotHelp()
{
  kapp->invokeHTMLHelp("kvoice/greetings.html",0);
}

bool 
Greetings::makeIndex()
{
  unsigned int i;

  QFile *index = new QFile(indexfilename);

  if(!index->exists())
    {
      QString string;
      string.sprintf(nls("vgetty needs a file named\n %s \n"), 
		     (const char *)indexfilename);
      string +=nls("Do you want me to create this file now?");
      int yes = KMsgBox::yesNo(0,nls("Greeting Messages index file"),
				   (const char *) string);
      if(yes != 1) return false;
    }

  index->open(IO_WriteOnly);
  debug("number of files in greeting dir %d",greetingDir->count());
  for(i=0;i<greetingDir->count();i++)
    {
      QString file = (*greetingDir)[i];
      if(strcmp(file,"Index"))
	{
	  debug("%s",(const char *)file);
	  file += "\n";
	  index->writeBlock((const char *)file, file.length());
	}
    }
  index->close();
  delete index;

  return true;
}

