/* File     : xnew_func.c
 * Author   : Karyl F. Stein <xenon@xenos.net>
 * Purpose  : Collection of functions for use with xnew.  The individual
 *            functions are detailed before the function below.
 *
 * Xnew is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include "xnew.h"       /* Global variables    */
#include "xnew_func.h"  /* Function prototypes */
#include "list_func.h"  /* Function prototypes */
#include <stdio.h>

/* Set defaults if not defined previously or not valid.  Do not change. */
#ifndef INBUF
# define INBUF 512
#elsif (INBUF < 2)
# define INBUF 512
#endif


/* Function: bind_result
 * Input   : The command run and its result
 * Output  : The passed data is added to the result list
 */
void bind_result (char *command, char *result) {
  data_node ptr = result_list;

  /* Ignore system commands */
  if (strcmp(command, "system") == 0)
    return;

  /* Ignore NULL results, (but not strings that are just \0) */
  if (result == NULL)
    return;

  /* Find an appropiate place to store the new result */
  if (ptr == NULL) {
    if ((result_list = (data_node) malloc(sizeof(struct data_node_t)))
	== NULL) {
      fprintf(stderr, "Out Of Memory\n");
      exit(1);
    }
    ptr = result_list;
  } else do {
    if (strcmp(ptr->command, command) == 0) {
      free(ptr->result);
      ptr->result = result;
      return;
    }
    if (ptr->next == NULL) {
      if ((ptr->next = (data_node) malloc(sizeof(struct data_node_t)))
	  == NULL) {
	fprintf(stderr, "Out Of Memory\n");
	exit(1);
      }
      ptr = ptr->next;
      break;
    }
    ptr = ptr->next;
  } while (1);

  ptr->command = command;
  ptr->result = result;
  ptr->next = NULL;
}


/* Function: clear_arguments
 * Input   : An array of character strings to clear.
 * Output  : The passed character strings are freed.
 * Return  : A pointer to the head of the passed array.
 */
char **clear_arguments (char **args) {
  char *ptr;
  int count = 0;

  while ((ptr = args[count++]) != NULL)
    free(ptr);
  return(args);
}


/* Function: get_command
 * Input   : A file pointer from which to read data.
 * Return  : The next command in the file or NULL if error, EOF, or white
 *           space.
 */
char *get_command (FILE *ifp) {
  char c;
  int size = 0;
  char *inbuf = NULL, *retval;
  int count = 0;

  /* Read and store input until whitespace or EOF is found */
  while ((c = getc(ifp)) != EOF) {
    if ((c == ' ') || (c == '\n') || (c == '\t'))
      break;
    if ((count >= size) &&
	((inbuf = (char *) realloc(inbuf, size += INBUF)) == NULL)) {
      fprintf(stderr, "Out Of Memory\n");
      exit(1);
    }
    inbuf[count++] = c;
  }

  if (count == 0)
    return(NULL);

  if (ungetc(c, ifp) == EOF) {
    fprintf(stderr, "Failed ungetc() call\n");
    exit(1);
  }
  inbuf[count] = '\0';

  /* Return the data read */
  if ((retval = (char *) malloc(strlen(inbuf) + 1)) == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }
  strcpy(retval, inbuf);
  free(inbuf);
  return(retval);
}


/* Function: signal_handler
 * Input   : The signal number caught.
 * Output  : Any temporary files are removed.
 */
void signal_handler (int signal_type) {
  if (tmp_file != NULL)
    remove(tmp_file);
  exit(0);
}


/* Function: skip
 * Input   : An open file to read from and a flag telling what to skip over.
 *           The flag may be an or of any of the following values:
 *
 *           COMMENTS - Skip lines starting with a # character
 *           SPACE    - Skip over spaces and tabs, but not new lines
 *           TO_EOL   - Skip over everything to and including the first new
 *                      line
 *           WHITE    - Skip over white space including new lines
 *
 * Output  : File pointer is moved over things to "skip."
 */
void skip (FILE *ifp, int flag) {
  char c;
  extern int line;

  /* Make sure a valid flag and non-null file pointer was given */
  if (!((flag & COMMENTS) || (flag & SPACE) || (flag & TO_EOL) ||
	(flag & WHITE)) || (ifp == NULL))
    return;

  /* Loop until EOF is found */
  while ((c = getc(ifp)) != EOF) {

    /* Skip over comments */
    if ((flag & COMMENTS) && (c == '#')) {
      skip_line(ifp, 1);
      continue;
    }

    /* Skip over white space */
    if (((flag & WHITE) || (flag & SPACE)) &&
	((c == ' ') || (c == '\t') || (c == '\r')))
      continue;

    /* Test for EOL */
    if (c == '\n') {
      ++line;
      if (flag & WHITE)
	continue;
      else if (flag & TO_EOL)
	return;
    }

    /* Ignore everything if TO_EOL (new line is detected above) */
    if (flag & TO_EOL)
      continue;

    /* Place the last character read back into stream */
    if (ungetc(c, ifp) == EOF) {
      fprintf(stderr, "Fatal Error: Unable to unget (line %d, char %d)\n",
	      line, c);
      exit(1);
    }
    return;
  }
}


/* Function: skip_line
 * Input   : A file pointer to read the data from and the number of lines
 *           to skip.
 * Ouptut  : File pointer is moved appropiately
 */
void skip_line (FILE *ifp, int count) {
  char tmpin;
  int line = 0;

  while (line++ < count)
    skip(ifp, TO_EOL);
}


/* Function: test_bool
 * Input   : An argument array containing the first argument to test, the
 *           operator to test with, and the second argument to test.
 * Output  : -1 if the passed arguments are not passed correctly, 0 if the
 *           test is false, otherwise 1.
 */
int test_bool (char *args[]) {

  /* Test that the appropriate number of arguments exist */
  if ((args == NULL) || (args[1] == NULL) || (args[2] == NULL))
    return(-1);

  /* Test the expression */
  if ((strcmp(args[1], "=") == 0) && (strcmp(args[0], args[2]) == 0))
    return(1);
  else if ((strcmp(args[1], "!=") == 0) && (strcmp(args[0], args[2]) != 0))
    return(1);
  else if ((strcmp(args[1], "<") == 0) && (strcmp(args[0], args[2]) < 0))
    return(1);
  else if ((strcmp(args[1], ">") == 0) && (strcmp(args[0], args[2]) > 0))
    return(1);
  else if ((strcmp(args[1], ">=") == 0) && (strcmp(args[0], args[2]) >= 0))
    return(1);
  else if ((strcmp(args[1], "<=") == 0) && (strcmp(args[0], args[2]) == 0))
    return(1);

  return(0);
}
