/* File     : xnew.c
 * Author   : Karyl F. Stein <steinkf@cs.purdue.edu>
 * Purpose  : Program designed to allow users to setup and create an account
 *            on a UNIX system.
 *
 * Xnew is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include <stdio.h>
#include "xnew.h"           /* Global variables    */
#include "xnew_func.h"      /* Function prototypes */
#include "stack_func.h"     /* Function prototypes */
#include "get_arguments.h"  /* Function prototypes */
#include "do_command.h"     /* Function prototypes */
#include <signal.h>         /* For signal handler  */
#include <unistd.h>         /* getopt() and more   */
#include <pwd.h>            /* getpwnam()          */
#include <sys/types.h>      /* getpwnam()          */
#include <termios.h>        /* For setting env     */

/* Flags values for local use */
#define NORMAL          0x01
#define IN_ELSE         0x02

/* Set defaults if not defined previously or not valid.  Change in config.h */
/* and not here.                                                            */
#ifndef INBUF
# define INBUF 512
#elsif (INBUF < 2)
# define INBUF 512
#endif  /* INBUF */


/* Function: Usage
 * Input   : The name that the main program was called as.
 * Output  : Print a usage message and exit.
 */
void usage(char *name) {
  fprintf(stderr, "Usage: %s [-h] [-f infile]\n", name);
  fprintf(stderr, " -h  Print this message and exit\n");
  fprintf(stderr, " -f  Use the file <infile> as the configuration script\n");
  exit(0);
}


void main (int argc, char *argv[]) {
  char **argument, *command, *filename = NULL, *tmparg;
  extern char *optarg;
  extern data_node result_list;
  extern int line;
  fpos_t position;
  FILE *ifp;
  int do_else_depth, flag, if_depth, tmpint, while_depth, while_do_depth;
  int opt;
  stack_node stack = NULL;
  struct passwd *pw_ent;
  struct termios save_scr;

  /* Initialize global variables */
  tmp_file = NULL;
  result_list = NULL;
  line = 1;

  /* Define a signal handler here to catch escapes and cleanup */
  if (signal(SIGINT, signal_handler) == SIG_ERR) {
    fprintf(stderr, "Unable to set signal handler\n");
    exit(1);
  }
  if (signal(SIGHUP, signal_handler) == SIG_ERR) {
    fprintf(stderr, "Unable to set signal handler\n");
    exit(1);
  }

  /* Parse the arguments */
  while ((opt = getopt(argc, argv, "hf:")) != EOF) {
    switch (opt) {
    case 'f':
      filename = optarg;
      continue;
    default:
      usage(argv[0]);
    }
  }
      
  /* Open the configuration script */
  if (filename == NULL) {
#ifndef INPUT_FILE
    fprintf(stderr, "You must specify a configuration file to use\n");
    exit(1);
#else
    filename = INPUT_FILE;
#endif  /* INPUT_FILE */
  }
  if ((ifp = fopen(filename, "r")) == NULL) {
    fprintf(stderr, "Unable to open configuration file %s\n", filename);
    exit(1);
  }

  /* Initialize local variables */
  do_else_depth = if_depth = 0;
  while_depth = while_do_depth = 0;
  flag = NORMAL;

  /* Read and act upon the input */
  while (1) {

    /* Skip over white space and comments */
    skip(ifp, WHITE | COMMENTS);

    /* Save the current file position in case looping is needed */
    if (fgetpos(ifp, &position) == -1) {
      fprintf(stderr, "Unable to get position in file %s\n", filename);
      exit(1);
    }

    /* Get the command */
    if ((command = get_command(ifp)) == NULL)
      break;

    /* Test for an else and act upon it if found */
    if (strcmp(command, "else") == 0) {
      free(command);

      /* Test if there was a preceeding if or else (won't catch everything) */
      if (if_depth == 0) {
	fprintf(stderr, "Syntax Error: else with no if (%s: line %d)\n",
		filename, line);
	exit(1);
      }

      flag |= IN_ELSE;

      /* Test if we should ignore this else */
      if (if_depth != do_else_depth) {
	flag &= ~NORMAL;
	continue;
      }
      do_else_depth = 0;
      flag |= NORMAL;
      continue;
    }

    /* Test for an fi and act upon it if found */
    if (strcmp(command, "fi") == 0) {
      free(command);
      if (--if_depth < 0) {
	fprintf(stderr, "Syntax Error: fi with no if (%s - line %d)\n",
		filename, line);
	exit(1);
      }
      flag &= !IN_ELSE;
      if (do_else_depth > if_depth)
	do_else_depth = 0;
      if ((while_do_depth == 0) && (do_else_depth == 0))
	flag |= NORMAL;
      continue;
    }

    /* Test for a done and act upon it if found */
    if (strcmp(command, "done") == 0) {
      free(command);
      if (--while_depth < 0) {
	fprintf(stderr, "Syntax Error: done with no while (%s line %d)\n",
		filename, line);
	exit(1);
      }
      if ((while_do_depth == 0) && (flag & NORMAL)) {
	if (stack == NULL) {
	  fprintf(stderr, "Fatal Error: Stack Corrupt\n");
	  exit(1);
	}
	position = stack->position;
	line = stack->line;
	if (fsetpos(ifp, &position) == -1) {
	  fprintf(stderr, "Fatal Error: Unable to position file %s", filename);
	  exit(1);
	}
	stack = pop(stack);
      }

      else if (while_depth <= while_do_depth) {
	while_do_depth = 0;
	if (do_else_depth == 0)
	  flag |= NORMAL;
      }
      continue;
    }

    /* Read the command arguments if needed */
    if (flag & NORMAL) {
      skip(ifp, SPACE);
      argument = get_arguments(ifp);
    } else skip(ifp, TO_EOL);

    /* Test if we are doing a variable assignment */
    if (*command == '$') {
      if (flag & NORMAL) {
	if (argument[0] == NULL) {
	  fprintf(stderr, "Fatal Error: Syntax Error in %s at line %d\n",
		  filename, line);
	  exit(1);
	}
	if (argument[1] == NULL) {
	  if ((tmparg = (char *) malloc(strlen("NULL") + 1)) == NULL) {
	    fprintf(stderr, "Out of Memory\n");
	    exit(1);
	  }
	  strcpy(tmparg, "NULL");
	} else {
	  if ((tmparg = (char *) malloc(strlen(argument[1]) + 1)) == NULL) {
	    fprintf(stderr, "Out of Memory\n");
	    exit(1);
	  }
	  strcpy(tmparg, argument[1]);
	}

	if (strcmp(argument[0], "=") == 0)
	  bind_result(command, tmparg);
	else {
	  free(tmparg);
	  free(command);
	}
	free(clear_arguments(argument));
      }
      continue;
    }

    /* Test for a while and act upon it if found */
    if (strcmp(command, "while") == 0) {
      free(command);
      ++while_depth;
      if (flag & NORMAL) {
	if ((tmpint = test_bool(argument)) == -1) {
	  fprintf(stderr, "Syntax Error: Arguments for while (%s - line %d)\n",
		  filename, line);
	  exit(1);
	}
	if (tmpint == 0) {
	  flag &= ~NORMAL;
	  while_do_depth = while_depth;
	} else stack = push(position, line, stack);
	free(clear_arguments(argument));
      }
      continue;
    }

    /* Test for an if and act upon it if found */
    if (strcmp(command, "if") == 0) {
      free(command);
      ++if_depth;
      if (flag & NORMAL) {
	if ((tmpint = test_bool(argument)) == -1) {
	  fprintf(stderr, "Syntax Error: Arguments for if (%s - line %d)\n",
		  filename, line);
	  exit(1);
	}
	if (tmpint == 0) {
	  do_else_depth = if_depth;
	  flag &= ~NORMAL;
	}
	free(clear_arguments(argument));
      }
      continue;
    }

    /* Test for a user and act upon it if found */
    if (strcmp(command, "user") == 0) {
      free(command);
      if (flag & NORMAL) {
	if (argument[0] == NULL) {
	  fprintf(stderr, "Syntax Error: No argument to user (%s - line %d)\n",
		  filename, line);
	  exit(1);
	}
	if ((pw_ent = getpwnam(argument[0])) == NULL) {
	  fprintf(stderr, "Unable to find UID for user %s (%s - line %d)\n",
		  argument[0], filename, line);
	  exit(1);
	}
	if (setreuid(0, 0) == -1) {
	  fprintf(stderr, "Unable to gain root (%s - line %d)\n",
		  filename, line);
	  exit(1);
	}
	if (seteuid(pw_ent->pw_uid) == -1) {
	  fprintf(stderr, "Unable to set EUID (%s - line %d)\n",
		  filename, line);
	  exit(1);
	}
	free(clear_arguments(argument));
      }
      continue;
    }

    /* Run the module or system command and store the result */
    if (flag & NORMAL) {
      bind_result(command, do_command(command, argument));
      free(clear_arguments(argument));
    }
  }
}
