/* File     : mod_func.c
 * Author   : Karyl F. Stein <steinkf@cs.purdue.edu>
 * Purpose  : Collection of functions for use with xnew modules, (or other
 *            programs if desired).  The individual functions are detailed
 *            before the function below.
 *
 * This file is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "mod_func.h"
#include <stdio.h>

/* Set defaults if not defined previously or not valid */
#ifndef INBUF
# define INBUF 512
#elsif (INBUF < 2)
# define INBUF 512
#endif


/* Function: printtxt
 * In      : filename and path to print
 * Out     : Prints file on screen
 * Return  : 0 if successful otherwise 1
 */
int printtxt (char *infile) {
    char c;
    FILE *ifp;

    if ((ifp = fopen(infile, "r")) == NULL)
        return 1;

    while ((c = getc(ifp)) != EOF)
        putc(c, stdout);
    return 0;
}


/* Function: center
 * In: string and length of screen
 * Out: Prints the string centered on the screen
 */
void center (char *str, int cols) {
  if (cols < 1)
    return;

  /* Compute number of leading spaces needed */
  if (strlen(str) < cols)
    cols = (cols - strlen(str)) / 2;
  else
    cols = 0;

  /* Print leading spaces if needed */
  while (cols--)
    putc(' ', stdout);

  printf("%s\n", str);
}


/* Function: get_word
 * Input   : A file pointer to read data from.
 * Return  : The next word in the file or NULL if error, EOF, or white
 *           space.
 */
char *get_word (FILE *ifp) {
  char c;
  int size = 0;
  char *inbuf = NULL, *retval;
  extern int line;
  int count = 0;

  /* Read and store input until whitespace or EOF is found */
  while ((c = getc(ifp)) != EOF) {
    if ((c == ' ') || (c == '\n') || (c == '\t'))
      break;
    if ((++count >= size) &&
        ((inbuf = (char *) realloc(inbuf, size += INBUF)) == NULL)) {
      fprintf(stderr, "Fatal Error: Out of Memory\n");
      exit(1);
    }
    inbuf[count - 1] = c;
  }
  if (count == 0)
    return(NULL);

  inbuf[count] = '\0';
  if (ungetc(c, ifp) == EOF) {
    fprintf(stderr, "Fatal Error: Failed ungetc() call\n");
    exit(1);
  }

  /* Return the data read */
  if ((retval = (char *) malloc(strlen(inbuf) + 1)) == NULL) {
    fprintf(stderr, "Fatal Error: Out of Memory\n");
    exit(1);
  }
  strcpy(retval, inbuf);
  free(inbuf);                     /* just in case... */
  return(retval);
}


/* Function: print_char
 * Input   : A character to print and a flag to tell how to print it.  The
 *           flag is one of the following, (defined in mod_func.h):
 *
 *           TEXT_ONLY: If the key is a control key, the word CONTROL- will
 *                      be printed before it, (i.e. CTRL-C becomes CONTROL-C).
 *           CTRL_ONLY: If the key is a control key, the character ^ will
 *                      be printed before it, (i.e. CTRL-C becomes ^C).
 *           FULL     : Both the TEXT and CTRL representations will be
 *                      printed.  If the character is CTRL-C,
 *                      CONTROL-C (^C) will be output.
 *
 *           If the passed character is not a control key, it is just printed.
 * Output  : The passed key is printed to the screen.  If the key is a
 *           control key, expand it to be printable by prefixing it with
 *           something as defined by the flag.
 *
 * Notes   : This function needs work.
 */
void print_char (char c, int flag) {
  if (c < 0x20) {
    c |= 0x40;
    if (flag & TEXT_ONLY)
      printf("CONTROL-%c", c);
    if (flag & CTRL_ONLY)
      printf("^%c", c);
    if (flag & FULL)
      printf("CONTROL-%c (^%c)", c, c);
  } else if (c == 0x7f) {
    if (flag & TEXT_ONLY)
      printf("Delete");
    if (flag & CTRL_ONLY)
      printf("^?");
    if (flag & FULL)
      printf("Delete (^?)");
  } else printf("%c", c);
}


/* Function: skip
 * Input   : An open file to read from and a flag telling what to skip over.
 *           The flag may be an or of any of the following values:
 *
 *           COMMENTS - Skip lines starting with a # character
 *           SPACE    - Skip over spaces and tabs, but not new lines
 *           TO_EOL   - Skip over everything to and including the first new
 *                      line
 *           WHITE    - Skip over white space including new lines
 *
 * Output  : File pointer is moved over things to "skip."
 */
void skip (FILE *ifp, int flag) {
  char c;

  /* Make sure a valid flag and non-null file pointer was given */
  if (!((flag & COMMENTS) || (flag & SPACE) || (flag & TO_EOL) ||
        (flag & WHITE)) || (ifp == NULL))
    return;

  /* Loop until EOF is found */
  while ((c = getc(ifp)) != EOF) {

    /* Skip over comments */
    if ((flag & COMMENTS) && (c == '#')) {
      skip_line(ifp, 1);
      continue;
    }

    /* Skip over white space */
    if (((flag & WHITE) || (flag & SPACE)) &&
        ((c == ' ') || (c == '\t') || (c == '\r')))
      continue;

    /* Test for EOL */
    if (c == '\n')
      if (flag & WHITE)
        continue;
      else if (flag & TO_EOL)
        return;

    /* Ignore everything if TO_EOL (new line is detected above) */
    if (flag & TO_EOL)
      continue;

    /* Place the last character read back into stream */
    if (ungetc(c, ifp) == EOF) {
      fprintf(stderr, "Fatal Error: Unable to unget()\n");
      exit(1);
    }
    return;
  }
}


/* Function: skip_line
 * Input   : A file pointer to read the data from and the number of lines
 *           to skip.
 * Ouptut  : File pointer is moved appropiately
 */
void skip_line (FILE *ifp, int count) {
  char tmpin;
  int line = 0;

  while (line++ < count)
    skip(ifp, TO_EOL);
}
