/* File   : get_shell.c
 * Author : Karyl F. Stein <xenon@xenos.net>
 * Purpose: Get the user's shell.
 *
 * This file is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include "get_shell.h"
#include "get_input.h"
#include "mod_func.h"

/* Local function prototypes */
int test_shell(char *);
char *convert_shell(char *);


main(int argc, char *argv[]) {
  char *shell;
  FILE *pipe_fp;

  /* Make sure we got at least one argument (the pipe to write to) */
  if (argc <= 1)
    exit(1);

  /* Open the pipe for writing */
  if ((pipe_fp = fopen(argv[1], "w")) == NULL) {
    fprintf(stderr, "Error: Child %s unable to access pipe\n", argv[0]);
    exit(1);
  }

  while (1) {
    printf("Enter the shell you want (? for a list)");

#ifdef DEF_SHELL
    if ((argv[2] != NULL) && (!test_shell(argv[2])))
      printf(" [%s]", argv[2]);
    else if (!test_shell(DEF_SHELL))
      printf(" [%s]", DEF_SHELL);
#else
    if (argv[2] != NULL)
      printf(" [%s]", argv[2]);
#endif  /* DEF_SHELL */

#ifdef SHELL_HELP
    shell = get_input(": ", 4, 80, LETTER | HELP | LOWERCASE);
#else
    shell = get_input(": ", 4, 80, LETTER | LOWERCASE);
#endif  /* SHELL_HELP */
    
    putc('\n', stdout);

#ifdef SHELL_HELP
    if (shell == NULL)
      printtxt(SHELL_HELP);
#endif  /* SHELL_HELP */

    else if (*shell == '\0') {
      free(shell);
      shell = NULL;

#ifdef DEF_SHELL
      if (!test_shell(DEF_SHELL)) {
	if ((shell = (char *) malloc(strlen(DEF_SHELL) + 1)) == NULL) {
	  fprintf(stderr, "Fatal Error: Out of Memory\n");
	  exit(1);
	}
	strcpy(shell, DEF_SHELL);
	break;
      } else printf("\n\nYou must enter a shell\n");
#else
      printf("\n\nYou must enter a shell\n");
#endif  /* DEF_SHELL */

    }

    else if (test_shell(shell)) {

#ifdef INVALID_SHELL
      printtxt(INVALID_SHELL);
#endif  /* INVALID_SHELL */

      free(shell);
    }

    else break;
  }

  /* Write the return data to the pipe. */
  fprintf(pipe_fp, "%s", convert_shell(shell));

  /* Close the pipe */
  fclose(pipe_fp);
}


/* Function: test_shell
 * Input   : A shell to test.
 * Output  : 0 if the shell is valid, otherwise 1.
 *
 * Notes: See the convert_shell function for a description of the SHELL_LIST
 *        file.
 */
int test_shell (char *shell) {
  char *tmpstr = convert_shell(shell);

  if (tmpstr == NULL) {
    free(tmpstr);
    return(1);
  }
  free(tmpstr);
  return(0);
}


/* Function: convert_shell
 * Input   : A shell to convert.
 * Output  : The full name of the shell if valid, otherwise NULL.
 *
 * Notes: A file containing a list of shells is parsed to determine if the
 *        passed shell is valid.  This file, defined as SHELL_LIST in
 *        get_shell.h, is described in shell_list.h.  If SHELL_LIST is not
 *        defined, then any shell is considered to be valid.
 *
 *        Only the existance of the shell name, (the first argument on each
 *        line of the SHELL_LIST file unless that line is a comment), is
 *        checked; the existance of the shell on the system is not checked.
 */
char *convert_shell (char *shell) {
  char *input;
  FILE *ifp;

#ifndef SHELL_LIST
  if ((input = (char *) malloc(strlen(shell) + 1)) == NULL) {
    fprintf(stderr, "Fatal Error: Out Of Memory\n");
    exit(1);
  }
  strcpy(input, shell);
  return(input);
#endif  /* SHELL_LIST */

  if ((ifp = fopen(SHELL_LIST, "r")) == NULL) {
    fprintf(stderr, "Fatal Error: Unable to open %s for reading\n",
	    SHELL_LIST);
    exit(1);
  }

  skip(ifp, WHITE | COMMENTS);
  while ((input = get_word(ifp)) != NULL) {
    if (strcmp(input, shell) == 0) {
      free(input);
      skip(ifp, WHITE);
      return(get_word(ifp));
    }
    free(input);
    skip(ifp, TO_EOL);
    skip(ifp, WHITE | COMMENTS);
  }

  return(NULL);
}
