/* File     : get_password.c
 * Author   : Karyl F. Stein <xenon@xenos.net>
 * Purpose  : Module for use with the xnew program to input a password.
 *
 * Notes    : Be sure to edit get_password.h to change anything needed.
 *
 * This file is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * Xnew is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "get_password.h"
#include "get_input.h"
#include "mod_func.h"
#include <sys/time.h>
#include <unistd.h>
#include <stdio.h>

/* Set defaults if not defined previously or not valid */
#ifndef MAXPASSLEN
# define MAXPASSLEN 8
#endif
#ifndef MINPASSLEN
# define MINPASSLEN 4
#endif

/* Local function prototypes */
int testpass(char *);
void clear_str(char *);


main (int argc, char *argv[]) {
  char c, salt[2];
  char *ptr, *inpass, *new_inpass;
  FILE *pipe_fp;
  int cpass, done = 0, try = 0;
  time_t time;

  /* Make sure we got at least one argument (the pipe to write to) */
  if (argc <= 1)
    exit(1);

  /* Open the pipe for writing */
  if ((pipe_fp = fopen(argv[1], "w")) == NULL) {
    fprintf(stderr, "Error: Child %s unable to access pipe\n", argv[0]);
    exit(1);
  }

  /* Loop until done */
  while (done++ < 2) {

    if (done == 1) {
#ifndef PASSWD_HELP
      inpass = get_input("Enter your password: ", MAXPASSLEN, 80,
			 NOSHOW | TEXT);
#else
      inpass = get_input("Enter your password: ", MAXPASSLEN, 80,
			 NOSHOW | TEXT | HELP);
#endif
      putc('\n', stdout);
    } else {
      new_inpass = get_input("Enter your password again: ", MAXPASSLEN, 80,
			     NOSHOW | TEXT);
      putc('\n', stdout);
      if (strcmp(inpass, new_inpass) != 0) {
	printf("\nYour passwords do not match.  Please re-enter.\n");
	done = 0;
	clear_str(inpass);
	clear_str(new_inpass);
	free(inpass);
	free(new_inpass);
	continue;
      }
      break;
    }

#ifdef PASSWD_HELP
    /* Print a help message */
    if (inpass == NULL) {
      clear_str(inpass);
      free(inpass);
      printtxt(PASSWD_HELP);
      done = 0;
      continue;
    }
#endif

    /* Make sure a password was entered */
    if (*inpass == '\0') {
      clear_str(inpass);
      free(inpass);
      puts("\nYou must enter a password.\n"); 
      done = 0;
    }

    /* Test the validity of the password */
    else if (testpass(inpass)) {
      clear_str(inpass);
      free(inpass);
      done = 0;
    }
  }

  /* Make this better */
  salt[0] = (time & 0x0f) + inpass[0];
  salt[1] = ((time & 0xf0) >> 4) + inpass[1];

  /* Encrypt the password */
  /* TODO: Use pw_encrypt() for Linux (other?) users with -lshadow */
  fprintf(pipe_fp, "%s", crypt(inpass, salt));

  
  /* Clear up the memory */
  clear_str(inpass);
  clear_str(new_inpass);

  fclose(pipe_fp);
}


/* Function: clear_str
 * Input   : A string to clear.
 * Output  : The passed string is filled with * marks
 */
void clear_str (char *str) {
  char *ptr = str;

  if (ptr != NULL)
    for (ptr = str; *ptr != '\0'; ++ptr)
      *ptr = '*';
}


/* Function: testpass
 * Input   : A password to test.
 * Output  : An error message if needed.
 * Return  : 0 if the password is "good," otherwise 1.
 *
 * TODO: Add more tests.
 */
int testpass (char *pass) {
  if (strlen(pass) < MINPASSLEN) {
    printf("\nYour password must contain at least %d characters\n\n",
	   MINPASSLEN);
    return(1);
  }

  return(0);
}
