/* File      : get_login.c
 * Author    : Karyl F. Stein <xenon@xenos.net>
 * Purpose   : Module for use with the xnew program to input a login name.
 *
 * Notes     : Be sure to edit get_login.h and change anything needed.
 *
 * This file is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "get_input.h"
#include "mod_func.h"
#include "get_login.h"
#include <stdio.h>
#include <pwd.h>
#include <sys/types.h>

/* Set defaults if not defined previously or not valid */
#ifndef MAXLOGINLEN
# define MAXLOGINLEN 8
#endif

/* Local function prototype definitions */
int testlogin(char *);


main (int argc, char *argv[]) {
  char c;
  char *login;
  FILE *pipe_fp;
  int done = 0, inloop = 0, try = 0;

  /* Make sure we got at least one argument (the pipe to write to) */
  if (argc <= 1)
    exit(1);

  /* Open the pipe for writing */
  if ((pipe_fp = fopen(argv[1], "w")) == NULL) {
    fprintf(stderr, "Error: Child %s unable to access pipe\n", argv[0]);
    exit(1);
  }

  /* Loop until a valid login name or a quit directive is given */
  while (1) {

#ifndef LOGIN_HELP
    login = get_input("Enter your login name: ", MAXLOGINLEN, 80,
		      LOWERCASE | LETTER | NUMBER | FIELD);
#else
    login = get_input("Enter your login name: ", MAXLOGINLEN, 80,
		      LOWERCASE | LETTER | NUMBER | FIELD | HELP);
#endif

    putc('\n', stdout);

#ifdef LOGIN_HELP
    /* Print the help message */
    if (login == NULL) {
      printtxt(LOGIN_HELP);
      continue;
    }
#endif

    /* Test to see if a login was entered */
    if (*login == '\0') {
      printf("\nYou must enter a login name.\n");
      login = get_input("Enter  q  to Quit or press ENTER to input a login: ",
			1, 80, LOWERCASE | LETTER);
      if (*login != '\0')
	exit(1);
      free(login);
      putc('\n', stdout);
      continue;
    }

    /* See if the given login name is in use */
    if (testlogin(login))
      free(login);
    else break;
  }

  /* Write the return data to the pipe.  */
  fprintf(pipe_fp, "%s", login);

  /* Close the pipe */
  fclose(pipe_fp);
}


/* Function: testlogin
 * Input   : A login name to test.
 * Output  : 1 if the login name exists on the system, (or is a mail alias),
 *           otherwise 0.
 */
int testlogin (char *login) {
  char c;
  char *ptr = login;
  FILE *ifp;

  /* Test if the login exists */
  if (getpwnam(login) != NULL) {
    puts("\nSorry, that name is in use.  Please choose another name.\n");
    return(1);
  }

#ifdef ALIASFILE
  /* Scan the aliases file for login */
  if ((ifp = fopen(ALIASFILE, "r")) != NULL) {
    ptr = login;

    /* Loop until EOF is reached */
    while ((c = getc(ifp)) != EOF) {
      if (*ptr != '\0') {
	if (*ptr != c) {
	  ptr = login;

	  /* Scan to the end of the line or EOF */
	  while ((c = getc(ifp)) != '\n')
	    if (c == EOF)
	      return(0);
	} else ++ptr;

      } else if ((c == ':') || (c == ' ') || (c == '\t')) {
	  puts("\nSorry, that name is in use.  Please choose another name.\n");
	  return(1);

      } else {
	while ((c = getc(ifp)) != '\n')
	  if (c == EOF)
	    return(0);
      }
    }
  }
#endif

  return(0);
}
