/* File     : get_interrupt.c
 * Author   : Karyl F. Stein <xenon@xenos.net>
 * Purpose  : Input and set the interrupt key correctly.
 *
 * This file is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <termios.h>
#include <unistd.h>
#include <stdio.h>
#include "mod_func.h"
#include "get_interrupt.h"


main(int argc, char *argv[]) {
  char interrupt;
  FILE *pipe_fp;
  struct termios save_scr, secure_scr;

  /* Make sure we got at least one argument (the pipe to write to) */
  if (argc <= 1)
    exit(1);

  /* Open the pipe for writing */
  if ((pipe_fp = fopen(argv[1], "w")) == NULL) {
    fprintf(stderr, "Error: Child %s unable to access pipe\n", argv[0]);
    exit(1);
  }

  /* Put the terminal into raw mode */
  if (tcgetattr(0, &save_scr) == -1) {
    fprintf(stderr, "Fatal Error: Unable to get terminal settings\n");
    exit(1);
  }
  secure_scr = save_scr;
  cfmakeraw(&secure_scr);
  secure_scr.c_oflag |= OPOST;
  if (tcsetattr(0, 0, &secure_scr) == -1) {
    fprintf(stderr, "Fatal Error: Unable to set terminal settings\n");
    exit(1);
  }

  while (1) {
    printf("Please press your interrupt key");

#ifdef DEF_INTERRUPT
    printf(" [");
    print_char(DEF_INTERRUPT, TEXT_ONLY);
    printf("]");
#endif

    printf(": ");
    interrupt = getc(stdin);
    putc('\n', stdout);

#ifdef DEF_INTERRUPT
    if ((interrupt == '\n') || (interrupt == '\r')) {
      interrupt = DEF_INTERRUPT;
      break;
    }
#endif

#ifdef INTERRUPT_HELP
    /* Print a help message */
    if (interrupt == '?') {
      printtxt(INTERRUPT_HELP);
      continue;
    }
#endif

    break;
  }

  /* Reset the terminal settings */
  if (tcsetattr(0, 0, &save_scr) == -1) {
    fprintf(stderr, "Fatal Error: Unable to set terminal settings\n");
    exit(1);
  }

  putc('\n', stdout);

  fprintf(pipe_fp, "%c", interrupt);

  /* Close the pipe */
  fclose(pipe_fp);
}
