/* File   : get_editor.c
 * Author : Karyl F. Stein <xenon@xenos.net>
 * Purpose: Get and the user's default editor.
 *
 * Notes  : This module relies on the file EDITOR_LIST defined in get_editor.h
 *          to test whether a given editor is valid.  The format of this file
 *          is explained before the convert_editor() function below.
 *
 * This file is Copyright: (c)1997 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include "get_editor.h"
#include "get_input.h"
#include "mod_func.h"

/* Local function prototypes */
int test_editor(char *);
char *convert_editor(char *);

/* Set defaults if not defined previously or not valid.  Do not change. */
#ifndef MAX_EDITOR_LEN
# define MAX_EDITOR_LEN 50
#elsif (MAX_EDITOR_LEN < 2)
# define MAX_EDITOR_LEN 50
#endif


main(int argc, char *argv[]) {
  char *editor;
  FILE *pipe_fp;

  /* Make sure we got at least one argument (the pipe to write to) */
  if (argc <= 1)
    exit(1);

  /* Open the pipe for writing */
  if ((pipe_fp = fopen(argv[1], "w")) == NULL) {
    fprintf(stderr, "Error: Child %s unable to access pipe\n", argv[0]);
    exit(1);
  }

  /* Loop until we get a valid answer */
  while (1) {
    printf("Enter the editor you want (? for a list)");

#ifdef DEF_EDITOR
    if ((argv[2] != NULL) && (!test_editor(argv[2])))
      printf(" [%s]", argv[2]);
    else if (!test_editor(DEF_EDITOR))
      printf(" [%s]", DEF_EDITOR);
#else
    if (argv[2] != NULL)
      printf(" [%s]", argv[2]);
#endif  /* DEF_EDITOR */

#ifdef EDITOR_HELP
    editor = get_input(": ", MAX_EDITOR_LEN, 80, TEXT | HELP);
#else
    editor = get_input(": ", MAX_EDITOR_LEN, 80, TEXT);
#endif  /* EDITOR_HELP */

    putc('\n', stdout);

#ifdef EDITOR_HELP
    if (editor == NULL) {
      printtxt(EDITOR_HELP);
      continue;
    }
#endif

    if (*editor == '\0') {
      free(editor);
      editor = NULL;

#ifdef DEF_EDITOR
      if (!test_editor(DEF_EDITOR)) {
	if ((editor = (char *) malloc(strlen(DEF_EDITOR) + 1)) == NULL) {
	  fprintf(stderr, "Fatal Error: Out of Memory\n");
	  exit(1);
	}
	strcpy(editor, DEF_EDITOR);
	break;
      } else printf("\n\nYou must enter an editor\n");
#else
      printf("\n\nYou must enter an editor\n");
#endif  /* DEF_EDITOR */

    } else if (test_editor(editor)) {

#ifdef EDITOR_INVALID
      printtxt(EDITOR_INVALID);
#endif

      free(editor);
    } else break;
  }

  /* Set the environment.  This needs to be done elsewhere, really. */
  if ((setenv("EDITOR", editor, 1)) == -1) {
    fprintf(stderr, "Fatal Error: Unable to set environment\n");
    exit(1);
  }

  /* Write the return data to the pipe */
  fprintf(pipe_fp, "%s", convert_editor(editor));

  /* Close the pipe */
  fclose(pipe_fp);
}


/* Function: test_editor
 * Input   : An editor to test.
 * Output  : 0 if the editoy is valid, otherwise 1.
 *
 * Notes: See the convert_editor function for a description of the EDITOR_LIST
 *        file.
 */
int test_editor (char *editor) {
  char *tmpstr = convert_editor(editor);

  if (tmpstr == NULL) {
    free(tmpstr);
    return(1);
  }
  free(tmpstr);
  return(0);
}


/* Function: convert_editor
 * Input   : An editor to convert.
 * Output  : The full name of the editor if valid, otherwise NULL.
 *
 * Notes   : This function reads the file EDITOR_LIST defined in get_editor.h
 *           to get valid editor names.  Each line of this file contains a
 *           valid editor name followed by a space and then the real file
 *           name, (including path), of the editor.  Comments may be inserted
 *           into the file by starting them with a # mark.  Anything after the
 *           editor file name on a line is ignored.  A sample file is as
 *           follows:
 *
 *           vi /bin/vim
 *           emacs /usr/bin/jove
 *
 *           This file will accept "vi" or "emacs" as valid editor names.  If
 *           vi is selected, "/bin/vim" will be returned.  If emacs is
 *           selected, "/usr/bin/jove" will be returned.
 */
char *convert_editor (char *editor) {
  char *input;
  FILE *ifp;

#ifndef EDITOR_LIST
  if ((input = (char *) malloc(strlen(editor) + 1)) == NULL) {
    fprintf(stderr, "Fatal Error: Out Of Memory\n");
    exit(1);
  }
  strcpy(input, editor);
  return(input);
#endif

  if ((ifp = fopen(EDITOR_LIST, "r")) == NULL) {
    fprintf(stderr, "Fatal Error: Unable to open %s for reading\n",
	    EDITOR_LIST);
    exit(1);
  }

  skip(ifp, WHITE | COMMENTS);
  while ((input = get_word(ifp)) != NULL) {
    if (strcmp(input, editor) == 0) {
      free(input);
      skip(ifp, WHITE);
      return(get_word(ifp));
    }
    free(input);
    skip(ifp, TO_EOL);
    skip(ifp, WHITE | COMMENTS);
  }

  return(NULL);
}
