/* File     : get_backspace.c
 * Author   : Karyl F. Stein <xenon@xenos.net>
 * Purpose  : Input and set the backspace key correctly.
 *
 * Notes    : This module is not especially bulletproof especially if the
 *            backspace key sends more than one character.  The only case I
 *            know of this happening is with the terminal program shipped with
 *            Windows 3.1[1] in which "^H ^H" is sent.  This will work fine,
 *            though, but perhaps there are other such things out there which
 *            will cause havoc.
 *
 *            Be sure to edit get_backsapce.h and change anything as needed.
 *
 * This file is Copyright (C)1997 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <termios.h>            /* Terminal manipulation */
#include <unistd.h>             /* Terminal manipulation */
#include <stdio.h>
#include "get_backspace.h"


main(int argc, char *argv[]) {
  char bs, new_bs;
  FILE *pipe_fp;
  int flag = 0;
  struct termios save_scr, secure_scr;

  /* Make sure we got at least one argument (the pipe to write to) */
  if (argc <= 1)
    exit(1);

  /* Open the pipe for writing */
  if ((pipe_fp = fopen(argv[1], "w")) == NULL) {
    fprintf(stderr, "Error: Child %s unable to access pipe\n", argv[0]);
    exit(1);
  }

  /* Put the terminal into raw mode */
  if (tcgetattr(0, &save_scr) == -1) {
    fprintf(stderr, "Fatal Error: Unable to get terminal settings\n");
    exit(1);
  }
  secure_scr = save_scr;
  cfmakeraw(&secure_scr);
  secure_scr.c_oflag |= OPOST;
  if (tcsetattr(0, 0, &secure_scr) == -1) {
    fprintf(stderr, "Fatal Error: Unable to set terminal settings\n");
    exit(1);
  }

  /* Loop until a valid erase key is entered */
  while (1) {
    if (!flag) {
      printf("Please press your backspace key: ");
      bs = getc(stdin);
      putc('\n', stdout);
    } else {
      printf("\nYou have entered a non-standard backspace key.\n");
      printf("Please press your backspace key again: ");
      new_bs = getc(stdin);
      putc('\n', stdout);
      if (new_bs == bs)
        break;
      printf("\n\nYour entries do not match!  Let's try this again.\n\n");
      flag = 0;
      continue;
    }

#ifdef BS_HELP
    /* Check to see if help was requested.  Bad news if the erase key is ?. */
    if (bs == '?') {
      printtxt(BS_HELP);
      continue;
    }
#endif  /* BS_HELP */

    /* Check is a standard erase key was pressed.  This relies on ASCII. */
    if ((bs == 127) || (bs == 8))
      break;
    else flag = 1;
  }

  /* Reset the terminal settings.  Setting the backspace key needs to be */
  /* done in the main xnew program (i.e. with the set command) for it to */
  /* take effect during this session.                                    */
  save_scr.c_cc[VERASE] = bs;
  if (tcsetattr(0, 0, &save_scr) == -1) {
    fprintf(stderr, "Fatal Error: Unable to set terminal settings\n");
    exit(1);
  }

  /* Print the result to the pipe */
  fprintf(pipe_fp, "%c", bs);

  /* Close the pipe */
  fclose(pipe_fp);
}
